#' add_osm_objects
#'
#' Adds layers of spatial objects (polygons, lines, or points generated by
#' extract_osm_objects ()) to a graphics object initialised with
#' plot_osm_basemap().
#'
#' @param map A ggplot2 object to which the objects are to be added
#' @param obj A spatial ('sp') data frame of polygons, lines, or points,
#' typically as returned by extract_osm_objects ()
#' @param col Colour of lines or points; fill colour of polygons
#' @param border Border colour of polygons
#' @param size Size argument passed to ggplot2 (polygon, path, point) functions:
#' determines width of lines for (polygon, line), and sizes of points.
#' Respective defaults are (0, 0.5, 0.5).
#' @param shape Shape of points or lines (the latter passed as 'linetype'): see
#' ?ggplot2::shape
#' @return modified version of map (a ggplot object) to which objects have been
#' added
#' @export
#'
#' @seealso \code{\link{plot_osm_basemap}}, \code{\link{extract_osm_objects}}.
#'
#' @examples
#' bbox <- get_bbox (c (-0.13, 51.5, -0.11, 51.52))
#' map <- plot_osm_basemap (bbox=bbox, bg="gray20")
#'
#' \dontrun{
#' # The 'london' data used below were downloaded as:
#' dat_BNR <- extract_osm_objects (bbox=bbox, key='building',
#'                                 value='!residential')
#' dat_HP <- extract_osm_objects (bbox=bbox, key='highway',
#'                                value='primary')
#' dat_T <- extract_osm_objects (bbox=bbox, key='tree')
#' }
#' map <- add_osm_objects (map, obj=london$dat_BNR, col="gray40", border="yellow") 
#' map <- add_osm_objects (map, obj=london$dat_HP, col="gray80",
#'                         size=1, shape=2)
#' map <- add_osm_objects (map, london$dat_T, col="green", size=2, shape=1)
#' print (map)
#' 
#' # Polygons with different coloured borders
#' map <- plot_osm_basemap (bbox=bbox, bg="gray20")
#' map <- add_osm_objects (map, obj=london$dat_HP, col="gray80")
#' map <- add_osm_objects (map, london$dat_T, col="green")
#' map <- add_osm_objects (map, obj=london$dat_BNR, col="gray40", border="yellow", 
#'                         size=0.5)
#' print (map)

add_osm_objects <- function (map, obj, col='gray40', border=NA, size,
                             shape)
{
    # ---------------  sanity checks and warnings  ---------------
    if (missing (map))
        stop ('map must be supplied to add_osm_objects')
    if (!is (map, 'ggplot'))
        stop ('map must be a ggplot object')
    if (missing (obj))
        stop ('object must be supplied to add_osm_objects')
    if (!inherits (obj, 'Spatial'))
        stop ('obj must be Spatial')
    if (!(is.character (col) | is.numeric (col)))
    {
        warning ("col will be coerced to character")
        col <- as.character (col)
    }
    # ---------------  end sanity checks and warnings  ---------------

    lon <- lat <- id <- NULL # suppress 'no visible binding' error
    if (class (obj) == 'SpatialPolygonsDataFrame')
    {
        if (missing (size))
            size <- 0
        xy <- lapply (slot (obj, "polygons"), function (x)
                      slot (slot (x, "Polygons") [[1]], "coords"))
        xy <- list2df (xy)
        map <- map + ggplot2::geom_polygon (ggplot2::aes (group=id), 
                                                      data=xy, size=size,
                                                      fill=col, colour=border)
    } else if (class (obj) == 'SpatialLinesDataFrame')
    {
        if (missing (size))
            size <- 0.5
        if (missing (shape))
            shape <- 1
        xy <- lapply (slot (obj, 'lines'), function (x)
                      slot (slot (x, 'Lines') [[1]], 'coords'))
        xy <- list2df (xy, islines=TRUE)
        map <- map + ggplot2::geom_path (data=xy,
                                   ggplot2::aes (x=lon, y=lat), 
                                   colour=col, size=size, linetype=shape)
    } else if (class (obj) == 'SpatialPointsDataFrame')
    {
        if (missing (size))
            size <- 0.5
        if (missing (shape))
            shape <- 19
        xy <- data.frame (slot (obj, 'coords'))
        map <- map + ggplot2::geom_point (data=xy,
                                    ggplot2::aes (x=lon, y=lat),
                                    col=col, size=size, shape=shape)
    } else
        stop ("obj is not a spatial class")

    return (map)
}

#' list2df
#'
#' Converts lists of coordinates to single data frames
#'
#' @param xy A list of coordinates extracted from an sp object
#' @param islines Set to TRUE for spatial lines, otherwise FALSE
#' @return data frame
list2df <- function (xy, islines=FALSE)
{
    if (islines) # lines have to be separated by NAs
        xy <- lapply (xy, function (i) rbind (i, rep (NA, 2)))
    else # Add id column to each:
        for (i in seq (xy))
            xy [[i]] <- cbind (i, xy [[i]])
    # And rbind them to a single matrix. 
    xy <-  do.call (rbind, xy)
    # And then to a data.frame, for which duplicated row names flag warnings
    # which are not relevant, so are suppressed by specifying new row names
    xy <-  data.frame (xy, row.names=1:nrow (xy))
    if (islines) # remove terminal row of NAs
        xy <- xy [1:(nrow (xy) - 1),]
    else
        names (xy) <- c ("id", "lon", "lat")
    return (xy)
}
