## NOTE: this is similar to rcmdshlib, and if I ever send that to CRAN
## we might work with it.
compile <- function(filename, verbose = TRUE, preclean = FALSE,
                    compiler_warnings = NULL, base = NULL) {
  compiler_warnings <- compiler_warnings %||%
    getOption("odin.compiler_warnings", FALSE)

  ## The actual compilation step should be fairly quick, so it's going
  ## to be OK to record the entire stream of output.
  Sys.setenv(R_TESTS = "")
  path <- dirname(filename)
  owd <- setwd(path)
  on.exit(setwd(owd))
  source <- basename(filename)

  base <- tools::file_path_sans_ext(source)
  dll <- dllname(base)

  ## But then here we can assume a new filename means a new dll
  if (file.exists(dll)) {
    odin_message("Using previously compiled shared library", verbose)
  } else {
    odin_message("Compiling shared library", verbose)
    R <- file.path(R.home(), "bin", "R")
    args <- c("CMD", "SHLIB", source,
              "-o", dll, if (preclean) c("--preclean", "--clean"))
    output <- suppressWarnings(system2(R, args, stdout = TRUE, stderr = TRUE))
    compiler_output_handle(output, verbose, compiler_warnings)
  }

  list(path = path,
       base = base,
       source = normalizePath(source),
       dll = normalizePath(dll, mustWork = TRUE))
}


## TODO: an extension of this will be to work out where the code that
## *generated* a warning error comes from in the R and report that
## too.
compiler_output_classify <- function(x) {
  if (length(x) > 0) {
    compiler <- sub("^(.+?)\\s.*$", "\\1", x[[1]])
  } else {
    compiler <- "gcc"
  }

  ## We really should not be getting here if an error is thrown, so
  ## don't worry too much about that.
  re_command <- sprintf("^(%s|make:)\\s", compiler)
  re_context <- '^([[:alnum:]._]+): (In|At) ([[:alnum:]]+)\\s.*:$'
  re_info <- '^([[:alnum:]._]+):([0-9]+)(:[0-9]+)?: (warning|error|note):.*$'
  is_continue <- grepl('^\\s+', x)

  ## For our case, I think it's only that the first and last line are
  ## going to be the actual command.
  type <- rep_len(NA_character_, length(x))
  n <- length(x)
  types <- character(n)
  values <- vector("list", n)
  i <- 1L
  while (i <= n) {
    xi <- x[[i]]
    if (grepl(re_command, xi)) {
      types[[i]] <- "command"
    } else if (grepl(re_context, xi)) {
      types[[i]] <- "context"
    } else if (grepl(re_info, xi)) {
      if (i < n) {
        j <- rle(is_continue[-seq_len(i)])
        if (j$values[[1]]) {
          m <- j$lengths[[1]]
          xi <- c(xi, x[seq_len(m) + i])
          i <- i + m
        }
      }
      types[[i]] <- "info"
      attr(xi, "type") <- sub(re_info, "\\4", xi[[1L]])
    } else {
      types[[i]] <- "unknown"
    }
    values[[i]] <- xi
    i <- i + 1L
  }

  i <- lengths(values) > 0

  ret <- list(type = types[i], value = values[i])
  class(ret) <- "compiler_output"
  ret
}


##' @export
format.compiler_output <- function(x, ...) {
  cols <- c(error = "red",
            warning = "yellow",
            note = "blue")
  ## For the info *around* the error (generated by desaturating these
  ## colours by 1/2):
  ##
  ##   m <- col2rgb(cols)
  ##   m <- rgb2hsv(m[1, ], m[2, ], m[3, ])
  ##   cols_info <- setNames(hsv(m[1, ], m[2, ] / 2, m[3, ]), names(cols))
  cols_info <- c(error = "#FF8080",
                 warning = "#FFFF80",
                 note = "#8080FF")

  style <- lapply(cols, crayon::make_style)
  style_info <- lapply(cols_info, crayon::make_style)
  style_context <- crayon::make_style("darkgrey")

  str <- collector()

  for (i in seq_along(x$type)) {
    t <- x$type[[i]]
    v <- x$value[[i]]
    if (t == "command" || t == "unknown") {
      str$add(v, literal = TRUE)
    } else if (t == "context") {
      str$add(style_context(v), literal = TRUE)
    } else if (t == "info") {
      cl <- attr(v, "type")
      if (cl %in% names(style)) {
        str$add(style[[cl]](v[[1]]), literal = TRUE)
        if (length(v) > 1L) {
          str$add(style_info[[cl]](v[-1]), literal = TRUE)
        }
      }
    } else {
      str$add(v, literal = TRUE) # nocov
    }
  }
  paste(sprintf("%s\n", str$get()), collapse = "")
}


compiler_output_handle <- function(output, verbose, compiler_warnings) {
  out <- compiler_output_classify(output)

  ok <- attr(output, "status")
  error <- !is.null(ok) && ok != 0L
  if (error) {
    message(format(out))
    stop("Error compiling source")
  }

  i <- vlapply(seq_along(out$type), function(i)
    out$type[i] == "info" && attr(out$value[[i]], "type") == "warning")
  if (compiler_warnings && any(i)) {
    str <- ngettext(sum(i),
                    "There was 1 compiler warning:\n",
                    sprintf("There were %d compiler warnings:\n", sum(i)))
    warning(str, format(out), call. = FALSE)
  } else if (verbose) {
    cat(format(out))
  }
}
