% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/adp.rdi.R
\name{read.adp.rdi}
\alias{read.adp.rdi}
\title{Read a Teledyne/RDI ADP File}
\usage{
read.adp.rdi(
  file,
  from,
  to,
  by,
  tz = getOption("oceTz"),
  longitude = NA,
  latitude = NA,
  type = c("workhorse"),
  monitor = FALSE,
  despike = FALSE,
  processingLog,
  testing = FALSE,
  debug = getOption("oceDebug"),
  ...
)
}
\arguments{
\item{file}{a connection or a character string giving the name of the file
to load.  (For \code{read.adp.sontek.serial}, this is generally a list of
files, which will be concatenated.)}

\item{from}{indication of the first profile to read.  This can be an
integer, the sequence number of the first profile to read, or a POSIXt time
before which profiles should be skipped, or a character string that converts
to a POSIXt time (assuming UTC timezone).  See \dQuote{Examples}, and make
careful note of the use of the \code{tz} argument. If \code{from} is not
supplied, it defaults to 1.}

\item{to}{an optional indication of the last profile to read, in a
format as described for \code{from}.  As a special case, \code{to=0} means
to read the file to the end.  If \code{to} is not supplied, then it defaults
to 0.}

\item{by}{an optional indication of the stride length to use while walking through
the file.  If this is an integer, then \code{by-1} profiles are skipped
between each pair of profiles that is read, e.g. the default \code{by=1}
means to read all the data.  (For RDI files \emph{only}, there are some
extra features to avoid running out of memory; see \dQuote{Memory considerations}.)}

\item{tz}{character string indicating time zone to be assumed in the data.}

\item{longitude}{optional signed number indicating the longitude in degrees
East.}

\item{latitude}{optional signed number indicating the latitude in degrees
North.}

\item{type}{character string indicating the type of instrument.}

\item{monitor}{boolean value indicating whether to indicate the progress
of reading the file, by using \code{\link[=txtProgressBar]{txtProgressBar()}} or otherwise.  The value
of \code{monitor} is changed to \code{FALSE} automatically, for non-interactive
sessions.}

\item{despike}{if \code{TRUE}, \code{\link[=despike]{despike()}} will be used to clean
anomalous spikes in heading, etc.}

\item{processingLog}{if provided, the action item to be stored in the log.
(Typically only provided for internal calls; the default that it provides is
better for normal calls by a user.)}

\item{testing}{logical value (IGNORED).}

\item{debug}{a flag that turns on debugging.  Set to 1 to get a moderate
amount of debugging information, or to 2 to get more.}

\item{\dots}{optional additional arguments that some (but not all)
\verb{read.adp.*()} functions pass to lower-level functions.}
}
\value{
An \linkS4class{adp} object.
The contents of that object make sense for the particular instrument
type under study, e.g. if the data file contains
NMEA strings, then navigational data will be stored in an item
called \code{nmea} in the \code{data} slot).
}
\description{
Read a Teledyne/RDI ADCP file (called 'adp' in oce).
}
\details{
As of 2016-09-25, this function has provisional functionality to
read data from the new "SentinelV" series ADCP -- essentially a
combination of a 4 beam workhorse with an additional vertical
centre beam.

If a heading bias had been set with the \code{EB} command during the setup
for the deployment, then a heading bias will have been stored in the file's
header.  This value is stored in the object's metadata as
\code{metadata$heading.bias}.  \strong{Importantly}, this value is
subtracted from the headings stored in the file, and the result of this
subtraction is stored in the objects heading value (in \code{data$heading}).
It should be noted that \code{read.adp.rdi()} was tested for firmware
version 16.30.  For other versions, there may be problems.  For example, the
serial number is not recognized properly for version 16.28.

In Teledyne/RDI ADP data files, velocities are coded to signed 2-byte integers, with a
scale factor being used to convert to velocity in metres per second.  These
two facts control the maximum recordable velocity and the velocity
resolution, values that may be retrieved for an ADP object name \code{d}
with \code{d[["velocityMaximum"]]} and \code{d[["velocityResolution"]]}.
}
\section{Handling of old file formats}{

\enumerate{
\item Early PD0 file formats stored the year of sampling with a different
base year than that used in modern files.  To accommodate this,
\code{read.adp.rdi} examines the inferred year, and if it is greater than
2050, then 100 years are subtracted from the time. This offset was
inferred by tests with sample files, but \emph{not} from RDI documentation,
so it is somewhat risky.  If the authors can find RDI documentation that
indicates the condition in which this century offset is required, then
a change will be made to the code.  Even if not, the method should
not cause problems for a long time.
}
}

\section{Names of items in data slot}{


The names of items in the \code{data} slot are below. Not all items are present
for ll file varieties; use e.g. \code{names(d[["data"]])} to determine the
names used in an object named \code{d}. In this list, items are either
a vector (with one sample per time of measurement), a
\link{matrix} with first index for time and second for bin number,
or an \link{array} with first index for time, second for bin number,
and third for beam number. Items are of vector type, unless
otherwise indicated.

\tabular{rr}{
\strong{Item} \tab \strong{Meaning} \cr
\code{a}                                    \tab signal amplitude array (units?)\cr
\code{ambientTemp}                          \tab ambient temperature (degC)\cr
\code{attitude}                             \tab attitude (deg)\cr
\code{attitudeTemp}                         \tab (FIXME add a description here)\cr
\code{avgMagnitudeVelocityEast}             \tab (FIXME add a description here)\cr
\code{avgMagnitudeVelocityNorth}            \tab (FIXME add a description here)\cr
\code{avgSpeed}                             \tab (FIXME add a description here)\cr
\code{avgTrackMagnetic}                     \tab (FIXME add a description here)\cr
\code{avgTrackTrue}                         \tab (FIXME add a description here)\cr
\code{avgTrueVelocityEast}                  \tab (FIXME add a description here)\cr
\code{avgTrueVelocityNorth}                 \tab (FIXME add a description here)\cr
\code{br}                                   \tab bottom range matrix (m)\cr
\code{bv}                                   \tab bottom velocity matrix (m/s)\cr
\code{contaminationSensor}                  \tab (FIXME add a description here)\cr
\code{depth}                                \tab depth (m)\cr
\code{directionMadeGood}                    \tab (FIXME add a description here)\cr
\code{distance}                             \tab (FIXME add a description here)\cr
\code{firstLatitude}                        \tab latitude at start of profile (deg)\cr
\code{firstLongitude}                       \tab longitude at start of profile (deg)\cr
\code{firstTime}                            \tab (FIXME add a description here)\cr
\code{g}                                    \tab data goodness matrix (units?)\cr
\code{heading}                              \tab instrument heading (degrees)\cr
\code{headingStd}                           \tab instrument heading std-dev (deg)\cr
\code{lastLatitude}                         \tab latitude at end of profile (deg)\cr
\code{lastLongitude}                        \tab longitude at end of profile (deg)\cr
\code{lastTime}                             \tab (FIXME add a description here)\cr
\code{numberOfHeadingSamplesAveraged}       \tab (FIXME add a description here)\cr
\code{numberOfMagneticTrackSamplesAveraged} \tab (FIXME add a description here)\cr
\code{numberOfPitchRollSamplesAveraged}     \tab (FIXME add a description here)\cr
\code{numberOfSpeedSamplesAveraged}         \tab (FIXME add a description here)\cr
\code{numberOfTrueTrackSamplesAveraged}     \tab (FIXME add a description here)\cr
\code{pitch}                                \tab instrument pitch (deg)\cr
\code{pitchStd}                             \tab instrument pitch std-dev (deg)\cr
\code{pressure}                             \tab pressure (dbar)\cr
\code{pressureMinus}                        \tab (FIXME add a description here)\cr
\code{pressurePlus}                         \tab (FIXME add a description here)\cr
\code{pressureStd}                          \tab pressure std-dev (dbar)\cr
\code{primaryFlags}                         \tab (FIXME add a description here)\cr
\code{q}                                    \tab data quality array\cr
\code{roll}                                 \tab instrument roll (deg)\cr
\code{rollStd}                              \tab instrument roll std-dev (deg)\cr
\code{salinity}                             \tab salinity\cr
\code{shipHeading}                          \tab ship heading (deg)\cr
\code{shipPitch}                            \tab ship pitch (deg)\cr
\code{shipRoll}                             \tab ship roll (deg)\cr
\code{soundSpeed}                           \tab sound speed (m/s)\cr
\code{speedMadeGood}                        \tab speed over ground (?) (m/s)\cr
\code{speedMadeGoodEast}                    \tab (FIXME add a description here)\cr
\code{speedMadeGoodNorth}                   \tab (FIXME add a description here)\cr
\code{temperature}                          \tab temperature (degC)\cr
\code{time}                                 \tab profile time (POSIXct)\cr
\code{v}                                    \tab velocity array (m/s)\cr
\code{xmitCurrent}                          \tab transmit current (unit?)\cr
\code{xmitVoltage}                          \tab transmit voltage\cr
}
}

\section{Memory considerations}{


For \code{RDI} files only, and only in the case where \code{by} is not specified,
an attempt is made to avoid running out of memory by skipping some profiles
in large input files. This only applies if \code{from} and \code{to} are both
integers; if they are times, none of the rest of this section applies.

A key issue is that RDI files store velocities in 2-byte values, which is
not a format that R supports. These velocities become 8-byte (numeric) values
in R. Thus, the R object created by \code{read.adp.rdi} will require more memory
than that of the data file. A scale factor can be estimated by ignoring
vector quantities (e.g. time, which has just one value per profile) and concentrating on matrix properties
such as velocity, backscatter, and correlation. These three elements have equal dimensions.
Thus, each 4-byte slide in the data file (2 bytes + 1 byte + 1 byte)
corresponds to 10 bytes in the object (8 bytes + 1 byte + 1 byte).
Rounding up the resultant 10/4 to 3 for safety, we conclude that any limit on the
size of the R object corresponds to a 3X smaller limit on file size.

Various things can limit the size of objects in R, but a strong upper limit
is set by the space the operating system provides to R. The least-performant machines
in typical use appear to be Microsoft-Windows systems, which limit R objects to
about 2e6 bytes (see \code{?Memory-limits}).  Since R routinely duplicates objects for certain tasks
(e.g. for call-by-value in function evaluation), \code{read.adp.rdi} uses a safety
factor in its calculation of when to auto-decimate a file. This factor is set to 3,
based partly on the developers' experience with datasets in their possession.
Multiplied by the previously stated safety factor of 3,
this suggests that the 2 GB limit on R objects corresponds to approximately a
222 MB limit on file size. In the present version of \code{read.adp.rdi}, this
value is lowered to 200 MB for simplicity. Larger files are considered to be "big",
and are decimated unless the user supplies a value for the \code{by} argument.

The decimation procedure has two cases.
\enumerate{
\item If \code{from=1} and \code{to=0} (or if neither \code{from} or \code{to} is given), then the
intention is to process the full span of the data.  If the input file is
under 200 MB, then \code{by} defaults to 1, so that all profiles are read.
For larger files, \code{by} is set to the \code{\link[=ceiling]{ceiling()}} of the
ratio of input file size to 200 MB.
\item If \code{from} exceeds 1, and/or \code{to} is nonzero, then
the intention is to process only an interior subset of the file. In this
case, \code{by} is calculated as the \code{\link[=ceiling]{ceiling()}} of
the ratio of \code{bbp*(1+to-from)} to 200 MB, where \code{bbp} is the number
of file bytes per profile. Of course, \code{by} is set to 1, if this
ratio is less than 1.
}

If the result of these calculations is that \code{by} exceeds 1, then
messages are printed to alert the user that the file will be decimated,
and also \code{monitor} is set to \code{TRUE}, so that a textual progress bar
is shown (if the session is interactive).
}

\section{Development Notes}{

An important part of the work of this function is to recognize what
will be called "data chunks" by two-byte ID sequences. This function is
developed in a practical way, with emphasis being focussed on
data files in the possession of the developers. Since Teledyne-RDI tends
to introduce new ID codes with new instruments, that means that
\code{read.adp.rdi} may not work on recently-developed instruments.

The following two-byte ID codes are recognized by \code{read.adp.rdi}
at this time (with bytes listed in natural order, LSB byte before
MSB). Items preceded by an asterisk are recognized, but not handled,
and so produce a warning.
\tabular{rrrr}{
\tab \strong{Byte 1} \tab \strong{Byte 2} \tab \strong{Meaning}\cr
\tab 0x00 \tab 0x01 \tab velocity\cr
\tab 0x00 \tab 0x01 \tab velocity\cr
\tab 0x00 \tab 0x02 \tab correlation\cr
\tab 0x00 \tab 0x03 \tab echo intensity\cr
\tab 0x00 \tab 0x04 \tab percent good\cr
\tab 0x00 \tab 0x06 \tab bottom track\cr
\tab 0x00 \tab 0x0a \tab Sentinel vertical beam velocity\cr
\tab 0x00 \tab 0x0b \tab Sentinel vertical beam correlation\cr
\tab 0x00 \tab 0x0c \tab Sentinel vertical beam amplitude\cr
\tab 0x00 \tab 0x0d \tab Sentinel vertical beam percent good\cr
\tab 0x00 \tab 0x20 \tab VMDASS\cr
\tab 0x00 \tab 0x30 \tab Binary Fixed Attitude header\cr
\tab 0x00 \tab 0x32 \tab Sentinel transformation matrix\cr
\tab 0x00 \tab 0x0a \tab Sentinel data\cr
\tab 0x00 \tab 0x0b \tab Sentinel correlation\cr
\tab 0x00 \tab 0x0c \tab Sentinel amplitude\cr
\tab 0x00 \tab 0x0d \tab Sentinel percent good\cr
\tab 0x01 \tab 0x0f \tab ?? something to do with V series and 4-beam\cr
}

Lacking a comprehensive Teledyne-RDI listing of ID codes,
the authors have cobbled together a listing from documents to which they
have access, as follows.
\itemize{
\item Table 33 of reference 3 lists codes as follows:
\tabular{rrr}{
\strong{Standard ID}  \tab  \strong{Standard plus 1D} \tab \strong{DESCRIPTION}\cr
MSB    LSB  \tab        MSB    LSB\tab\cr
---    ---  \tab        ---    ---\tab\cr
7F     7F  \tab         7F     7F \tab   Header\cr
00     00  \tab         00     01 \tab   Fixed Leader\cr
00     80  \tab         00     81 \tab   Variable Leader\cr
01     00  \tab         01     01 \tab   Velocity Profile Data\cr
02     00  \tab         02     01 \tab   Correlation Profile Data\cr
03     00  \tab         03     01 \tab   Echo Intensity Profile Data\cr
04     00  \tab         04     01 \tab   Percent Good Profile Data\cr
05     00  \tab         05     01 \tab   Status Profile Data\cr
06     00  \tab         06     01 \tab   Bottom Track Data\cr
20     00  \tab         20     00 \tab   Navigation\cr
30     00  \tab         30     00 \tab   Binary Fixed Attitude\cr
30  40-F0  \tab         30  40-F0 \tab   Binary Variable Attitude\cr
}
\item Table 6 on p90 of reference 4 lists "Fixed Leader Navigation" ID
codes (none of which are handled by \code{read.adp.rdi} yet)
as follows (the format is reproduced literally; note that
e.g. 0x2100 is 0x00,0x21 in the oce notation):
\tabular{rr}{
\strong{ID} \tab \strong{Description}\cr
0x2100 \tab  $xxDBT\cr
0x2101 \tab  $xxGGA\cr
0x2102 \tab  $xxVTG\cr
0x2103 \tab  $xxGSA\cr
0x2104 \tab  $xxHDT, $xxHGD or $PRDID\cr
}
and following pages in that manual reveal the following meanings
\tabular{rr}{
\strong{Symbol} \tab \strong{Meaning}\cr
\code{DBT}   \tab depth below transducer\cr
\code{GGA}   \tab global positioning system\cr
\code{VTA}   \tab track made good and ground speed\cr
\code{GSA}   \tab GPS DOP and active satellites\cr
\code{HDT}   \tab heading, true\cr
\code{HDG}   \tab heading, deviation, and variation\cr
\code{PRDID} \tab heading, pitch and roll\cr
}
}
}

\section{Error recovery}{


Files can sometimes be corrupted, and \code{read.adp.rdi} has ways to handle two types
of error that have been noticed in files supplied by users.
\enumerate{
\item There are two bytes within each ensemble that indicate the number of bytes to check within
that ensemble, to get the checksum. Sometimes, those two bytes can be erroneous, so that
the wrong number of bytes are checked, leading to a failed checksum. As a preventative
measure, \code{read.adp.rdi} checks the stated ensemble length, whenever it detects a
failed checksum. If that length agrees with the length of the most recent ensemble that
had a good checksum, then the ensemble is declared as faulty and is ignored. However,
if the length differs from that of the most recent accepted ensemble, then \code{read.adp.rdi}
goes back to just after the start of the ensemble, and searches forward for the next two-byte
pair, namely \verb{0x7f 0x7f}, that designates the start of an ensemble.  Distinct notifications
are given about these two cases, and they give the byte numbers in the original file, as a way
to help analysts who want to look at the data stream with other tools.
\item At the end of an ensemble, the next two characters ought to be \verb{0x7f 0x7f}, and if they
are not, then the next ensemble is faulty. If this error occurs, \code{read.adp.rdi} attempts
to recover by searching forward to the next instance of this two-byte pair, discarding any
information that is present in the mangled ensemble.
}

In each of these cases, warnings are printed about ensembles that seem problematic.
Advanced users who want to diagnose the problem further might find it helpful to
examine the original data file using other tools. To this end, \code{read.adp.rdi}
inserts an element named \code{ensembleInFile} into the \code{metadata} slot.
This gives the starting byte number of each inferred ensemble within the original data
file.  For example, if \code{d} is an object read with \code{read.adp.rdi}, then using

\if{html}{\out{<div class="sourceCode">}}\preformatted{plot(d[["time"]][-1], diff(d[["ensembleInFile"]]))
}\if{html}{\out{</div>}}

can be a good way to narrow in on problems.
}

\examples{
adp <- read.adp.rdi(system.file("extdata", "adp_rdi.000", package="oce"))
summary(adp)

}
\references{
\enumerate{
\item Teledyne-RDI, 2007. \emph{WorkHorse commands and output data
format.} P/N 957-6156-00 (November 2007).  (Section 5.3 h details the binary
format, e.g. the file should start with the byte \code{0x7f} repeated twice,
and each profile starts with the bytes \code{0x80}, followed by \code{0x00},
followed by the sequence number of the profile, represented as a
little-endian two-byte short integer.  \code{read.adp.rdi} uses these
sequences to interpret data files.)
\item Teledyne RD Instruments, 2015. \emph{V Series monitor, sentinel Output Data Format.}
P/N 95D-6022-00 (May 2015). \code{SV_ODF_May15.pdf}
\item Teledyne RD Instruments, 2014. \emph{Ocean Surveyor / Ocean Observer Technical Manual.}
P/N 95A-6012-00 (April 2014). \code{OS_TM_Apr14.pdf}
\item Teledyne RD Instruments, 2001. \emph{WinRiver User's Guide International Version.}
P/N 957-6171-00 (June 2001) \verb{WinRiver User Guide International Version.pdf.pdf}
}
}
\seealso{
Other things related to adp data: 
\code{\link{[[,adp-method}},
\code{\link{[[<-,adp-method}},
\code{\link{ad2cpHeaderValue}()},
\code{\link{adp-class}},
\code{\link{adpConvertRawToNumeric}()},
\code{\link{adpEnsembleAverage}()},
\code{\link{adpFlagPastBoundary}()},
\code{\link{adp_rdi.000}},
\code{\link{adp}},
\code{\link{as.adp}()},
\code{\link{beamName}()},
\code{\link{beamToXyzAdpAD2CP}()},
\code{\link{beamToXyzAdp}()},
\code{\link{beamToXyzAdv}()},
\code{\link{beamToXyz}()},
\code{\link{beamUnspreadAdp}()},
\code{\link{binmapAdp}()},
\code{\link{enuToOtherAdp}()},
\code{\link{enuToOther}()},
\code{\link{handleFlags,adp-method}},
\code{\link{is.ad2cp}()},
\code{\link{plot,adp-method}},
\code{\link{read.adp.ad2cp}()},
\code{\link{read.adp.nortek}()},
\code{\link{read.adp.sontek.serial}()},
\code{\link{read.adp.sontek}()},
\code{\link{read.adp}()},
\code{\link{read.aquadoppHR}()},
\code{\link{read.aquadoppProfiler}()},
\code{\link{read.aquadopp}()},
\code{\link{rotateAboutZ}()},
\code{\link{setFlags,adp-method}},
\code{\link{subset,adp-method}},
\code{\link{subtractBottomVelocity}()},
\code{\link{summary,adp-method}},
\code{\link{toEnuAdp}()},
\code{\link{toEnu}()},
\code{\link{velocityStatistics}()},
\code{\link{xyzToEnuAdpAD2CP}()},
\code{\link{xyzToEnuAdp}()},
\code{\link{xyzToEnu}()}
}
\author{
Dan Kelley and Clark Richards
}
\concept{things related to adp data}
