% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/section.R
\name{sectionSmooth}
\alias{sectionSmooth}
\title{Smooth a Section}
\usage{
sectionSmooth(section, method = "spline", x, xg, yg, xgl, ygl, xr, yr,
  df, gamma = 0.5, iterations = 2, trim = 0, pregrid = FALSE,
  debug = getOption("oceDebug"), ...)
}
\arguments{
\item{section}{A \code{section} object containing the section to be smoothed.
For \code{method="spline"}, the pressure levels must match for each station in
the section.}

\item{method}{A string or a function that specifies the method to use; see \sQuote{Details}.}

\item{x}{Optional numerical vector, of the same length as the number of stations in \code{section},
which will be used in gridding in the lateral direction. If not provided, this
defaults to \code{\link{geodDist}(section)}.}

\item{xg, xgl}{ignored in the \code{method="spline"} case, but passed to
\code{\link{interpBarnes}} if \code{method="barnes"} or
to \code{method} if it is a function.
If \code{xg} is supplied, it defines the x component of the grid, i.e. the resultant station
distances, x, along the track of the section.
Alternatively, if \code{xgl} is supplied, the x grid is established using \code{\link{seq}},
to span the data with \code{xgl} elements. If neither of these is supplied, the output
x grid will equal the input x grid.}

\item{yg, ygl}{similar to \code{xg} and \code{xgl}, but for pressure. If \code{yg}
is not given, it is determined from the deepest station in the section. If
\code{ygl} was not given, then a grid is constructed to span the pressures
of that deepest station with \code{ygl} elements. On the other hand,
if \code{ygl} was not given, then the y grid will constructed from the
pressure levels in the deepest station.}

\item{xr, yr}{influence ranges in x (along-section distance) and y (pressure),
passed to \code{\link{interpBarnes}} if \code{method="barnes"} or to
\code{method}, if the latter is a function. If missing, \code{xr} defaults to
1.5X the median inter-station distance and \code{yr} defaults to 0.2X
the pressure range. Since these defaults have changed over the evolution
of \code{sectionSmooth}, analysts ought to supply \code{xr} and \code{yr}
in the function call, tailoring them to particular applications, and
making the code more resistant to changes in \code{sectionSmooth}.}

\item{df}{Degree-of-freedom parameter, passed to \code{\link{smooth.spline}}
if \code{method="spline"}, and ignored otherwise. If \code{df} is not provided,
it defaults to 1/5-th of the number of stations containing non-\code{NA}
data at the particular pressure level being processed, as \code{sectionSmooth}
works its way through the water column.}

\item{gamma, iterations, trim, pregrid}{Values passed to
\code{\link{interpBarnes}}, if \code{method="barnes"}, and
ignored otherwise. \code{gamma} is the factor by which
\code{xr} and \code{yr} are reduced on each of succeeding iterations.
\code{iterations} is the number of iterations to do.
\code{trim} controls whether the gridded data are set to
\code{NA} in regions with sparse data
coverage. \code{pregrid} controls whether data are to be
pre-gridded with \code{\link{binMean2D}} before being passed to
\code{\link{interpBarnes}}.}

\item{debug}{A flag that turns on debugging.  Set to 1 to get a moderate amount
of debugging information, or to 2 to get more.}

\item{...}{Optional extra arguments, passed to either
\code{\link{smooth.spline}}, if \code{method="spline"}, and ignored otherwise.}
}
\value{
An object of \code{\link{section-class}} that has been smoothed in some way.
Every data field that is in even a single station of the input object
is inserted into every station in the returned value, and therefore
the units represent all the units in any of the stations, as do the
flag names. However, the flags are all set to \code{NA} values.
}
\description{
Smooth a section, in any of several ways, working either in the vertical
direction or in both the vertical and lateral directions.
}
\details{
This function produces smoothed fields that might be useful in
simplifying graphical elements created with \code{\link{plot,section-method}}.
As with any smoothing method, a careful analyst will compare the results
against the raw data, e.g. using \code{\link{plot,section-method}}.
In addition the problem of falsely widening narrow features such as
fronts, there is also the potential to get unphysical results with
spars sampling near topographic features such as bottom slopes and ridges.

The \code{method} argument selects between three possible methods.

\itemize{

\item For \code{method="spline"}, i.e. the default, the section is smoothed
laterally, using \code{\link{smooth.spline}} on individual pressure levels.
(If the pressure levels do not match up, \code{\link{sectionGrid}} should
be used first to create a \code{section} object that can be used here.)
The \code{df} argument sets the degree of freedom of the spline, with
larger values indicating less smoothing.

\item For the (much slower) \code{method="barnes"} method, smoothing is done across
both horizontal and vertical coordinates, using \code{\link{interpBarnes}}.
The output station locations are computed by linear interpolation of
input locations, using \code{\link{approx}} on the original
longitudes and longitudes of stations, with the independent variable
being the distance along the track, computed with \code{\link{geodDist}}.
The values of \code{xg}, \code{yg}, \code{xgl} and \code{ygl} control
the smoothing.

\item If \code{method} is a function, then that function is applied to
the (distance, pressure) data for each variable at a grid defined by
\code{xg}, \code{xgl}, \code{yg} and \code{ygl}. The function must
be of the form \code{function(x,y,F,xg,xr,yg,yr)}, and must
return a matrix of the gridded result, with first index indicating
the "grid" station number and second index indicating "grid" pressure.
The \code{x} value that is supplied to this function is set as
the distance along the section, as computed with \code{\link{geodDist}},
and repeated for each of the points at each station.  The corresponding
pressures are provided in \code{y}, and the value to be gridded is
in \code{v}, which will be \code{temperture} on one call to the function,
\code{salinity} on another call, etc. The other quantities
have the meanings as described below.  See the \dQuote{Examples}
for a description of how to set up and use a function for the gridding
method known as Kriging.

}
}
\examples{
# Unsmoothed (Gulf Stream)
library(oce)
data(section)
gs <- subset(section, 115<=stationId&stationId<=125)
par(mfrow=c(2, 2))

plot(gs, which="temperature")
mtext("unsmoothed")

# Spline
gsg <- sectionGrid(gs, p=seq(0, 5000, 100))
gsSpline <- sectionSmooth(gsg, "spline")
plot(gsSpline, which="temperature")
mtext("spline-smoothed")

# Barnes
gsBarnes <- sectionSmooth(gs, "barnes", xr=50, yr=200)
plot(gsBarnes, which="temperature")
mtext("Barnes-smoothed")

# Kriging
if (requireNamespace("automap",quietly=TRUE)&&requireNamespace("sp",quietly=TRUE)) {
 krig <- function(x, y, F, xg, xr, yg, yr) {
   xy <- data.frame(x=x/xr, y=y/yr)
   K <- automap::autoKrige(F~1, remove_duplicates=TRUE,
                           input_data=sp::SpatialPointsDataFrame(xy, data.frame(F)),
                           new_data=sp::SpatialPoints(expand.grid(xg/xr, yg/yr)))
   matrix(K$krige_output@data$var1.pred, nrow=length(xg), ncol=length(yg))
 }
 gsKrig <- sectionSmooth(gs, krig)
 plot(gsKrig, which="temperature")
 mtext("Kriging-smoothed")
}

}
\seealso{
Other things related to section data: \code{\link{[[,section-method}},
  \code{\link{[[<-,section-method}},
  \code{\link{as.section}},
  \code{\link{handleFlags,section-method}},
  \code{\link{initializeFlagScheme,section-method}},
  \code{\link{plot,section-method}},
  \code{\link{read.section}}, \code{\link{section-class}},
  \code{\link{sectionAddStation}},
  \code{\link{sectionGrid}}, \code{\link{sectionSort}},
  \code{\link{section}},
  \code{\link{subset,section-method}},
  \code{\link{summary,section-method}}
}
\author{
Dan Kelley
}
\concept{things related to section data}
