% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/EnKFilter.R
\name{buildEnsembleKF}
\alias{buildEnsembleKF}
\title{Create an Ensemble Kalman filter algorithm to sample from latent states.}
\usage{
buildEnsembleKF(model, nodes, control = list())
}
\arguments{
\item{model}{A NIMBLE model object, typically representing a state space model or a hidden Markov model}

\item{nodes}{A character vector specifying the stochastic latent model nodes 
the Ensemble Kalman Filter will estimate. All provided nodes must be stochastic.
Can be one of three forms: a variable name, in which case all elements in the variable
are taken to be latent (e.g., 'x'); an indexed variable, in which case all indexed elements are taken
to be latent (e.g., 'x[1:100]' or 'x[1:100, 1:2]'); or a vector of multiple nodes, one per time point,
in increasing time order (e.g., c("x[1:2, 1]", "x[1:2, 2]", "x[1:2, 3]", "x[1:2, 4]")).}

\item{control}{A list specifying different control options for the particle filter.  Options are described in the details section below.}
}
\description{
Create an Ensemble Kalman filter algorithm for a given NIMBLE state space model.
}
\details{
The \code{control()} list option is described in detail below:
\describe{
 \item{saveAll}{Indicates whether to save state samples for all time points (TRUE), or only for the most recent time point (FALSE)}
 \item{timeIndex}{An integer used to manually specify which dimension of the latent state variable indexes time.  
 Only needs to be set if the number of time points is less than or equal to the size of the latent state at each time point.}
 \item{initModel}{A logical value indicating whether to initialize the model before running the filtering algorithm.  Defaults to TRUE.}
}

Runs an Ensemble Kalman filter to estimate a latent state given observations at each time point.  The ensemble Kalman filter
is a Monte Carlo approximation to a Kalman filter that can be used when the model's transition euqations do not follow a normal distribution.  
Latent states (x[t]) and observations (y[t]) can be scalars or vectors at each time point, 
and sizes of observations can vary from time point to time point.
In the BUGS model, the observations (y[t]) must be equal to some (possibly nonlinear) deterministic function
of the latent state (x[t]) plus an additive error term.  Currently only normal and multivariate normal
error terms are supported.
The transition from x[t] to x[t+1] does not have to be normal or linear.  Output from the posterior distribution of the latent
states is stored in \code{mvSamples}.
}
\examples{
## For illustration only.
exampleCode <- nimbleCode({
  x0 ~ dnorm(0, var = 1)
  x[1] ~ dnorm(.8 * x0, var = 1)
  y[1] ~ dnorm(x[1], var = .5)
  for(t in 2:10){
    x[t] ~ dnorm(.8 * x[t-1], var = 1)
    y[t] ~ dnorm(x[t], var = .5)
  }
})

model <- nimbleModel(code = exampleCode, data = list(y = rnorm(10)),
                     inits = list(x0 = 0, x = rnorm(10)))
my_enKF <- buildEnsembleKF(model, 'x',
                control = list(saveAll = TRUE, thresh = 1))
## Now compile and run, e.g.,
## Cmodel <- compileNimble(model)
## Cmy_enKF <- compileNimble(my_enKF, project = model)
## Cmy_enKF$run(m = 1000)
## enKF_X <- as.matrix(Cmy_enKF$mvSamples, 'x')
}
\references{
Houtekamer, P.L., and H.L. Mitchell. (1998). Data assimilation using an ensemble Kalman filter technique. \emph{Monthly Weather Review}, 126(3), 796-811.
}
\seealso{
Other particle filtering methods: 
\code{\link{buildAuxiliaryFilter}},
\code{\link{buildBootstrapFilter}},
\code{\link{buildIteratedFilter2}()},
\code{\link{buildLiuWestFilter}}
}
\author{
Nicholas Michaud
}
\concept{filtering methods}
\concept{particle filtering methods}
