% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/filtering_IF2.R
\name{buildIteratedFilter2}
\alias{buildIteratedFilter2}
\title{Create an IF2 algorithm.}
\usage{
buildIteratedFilter2(model, nodes, params = NULL, baselineNode = NULL,
  control = list())
}
\arguments{
\item{model}{A NIMBLE model object, typically representing a state 
space model or a hidden Markov model.}

\item{nodes}{A character vector specifying the latent model nodes 
over which the particle filter will stochastically integrate to
estimate the log-likelihood function.  All provided nodes must be stochastic.
Can be one of three forms: a variable name, in which case all elements in the variable
are taken to be latent (e.g., 'x'); an indexed variable, in which case all indexed elements are taken
to be latent (e.g., 'x[1:100]' or 'x[1:100, 1:2]'); or a vector of multiple nodes, one per time point,
in increasing time order (e.g., c("x[1:2, 1]", "x[1:2, 2]", "x[1:2, 3]", "x[1:2, 4]")).}

\item{params}{A character vector specifying the top-level parameters to obtain maximum likelihood estimates of. 
 If unspecified, parameter nodes are specified as all stochastic top level nodes which
are not in the set of latent nodes specified in \code{nodes}.}

\item{baselineNode}{A character vector specifying the node that is the latent node at the "0th" time step. The first node in \code{nodes} should depend on this baseline, but \code{baselineNode} should have no data depending on it. If \code{NULL} (the default), any initial state is taken to be fixed at the values present in the model at the time the algorithm is run.}

\item{control}{A list specifying different control options for the IF2 algorithm.  Options are described in the \sQuote{details} section below.}
}
\description{
Create an IF2 algorithm for a given NIMBLE state space model.
}
\section{Reparameterization}{


The IF2 algorithm perturbs the parameters using a normal distribution, which may not be optimal for parameters whose support is not the whole real line, such as variance parameters, which are restricted to be positive. We recommend that users reparameterize the model in advance, e.g., writing variances and standard deviations on the log scale and probabilities on the logit scale. This requires specifying priors directly on the transformed parameters.
}

\section{Parameter prior distributions}{


While NIMBLE's IF2 algorithm requires prior distributions on the parameters, the IF2 algorithm produces maximum likelihood estimates and does not directly use those prior distributions. We require the prior distributions to be stated only so that we can automatically determine which model nodes are the parameters. The IF2 algorithm also makes use of any bounds on the parameters.
}

\section{Diagnostics and information stored in the algorithm object}{


The IF2 algorithm stores the estimated MLEs, one from each iteration, in \code{estimates}. It also stores standard deviation of the particles from each iteration, one per parameter, in \code{estSD}. Finally it stores the estimated log-likelihood at the estimated MLE from each iteration in \code{logLik}.
}
\examples{
\dontrun{
model <- nimbleModel(code = ...)
my_IF2 <- buildIteratedFilter2(model, 'x[1:100]', params = 'sigma_x')
Cmodel <- compileNimble(model)
Cmy_IF2 <- compileNimble(my_IF2, project = model)
# MLE estimate of a top level parameter named sigma_x:
sigma_x_MLE <- Cmy_IF2$run(m = 10000, n = 10)
# Continue running algorithm for more precise estimate:
sigma_x_MLE <- Cmy_IF2$continueRun(n = 10)
# visualize progression of the estimated log-likelihood
ts.plot(CmyIF2$logLik)
}
}
\author{
Nicholas Michaud, Dao Nguyen, and Christopher Paciorek
}
\references{
Ionides, E.L., D. Nguyen, Y. Atchad{\'e}, S. Stoev, and A.A. King (2015). Inference for dynamic and latent variable models via iterated, perturbed Bayes maps. \emph{Proceedings of the National Academy of Sciences}, 112(3), 719-724.
}
\seealso{
Other particle filtering methods: \code{\link{buildAuxiliaryFilter}},
  \code{\link{buildBootstrapFilter}},
  \code{\link{buildEnsembleKF}},
  \code{\link{buildLiuWestFilter}}
}

