#include <R.h>
#include <Rdefines.h>
#include <Rinternals.h>
#include <Rmath.h>
#define NODEBUG

/* ===========================================================================
   AUTHOR Yves Deville <deville.yves@alpestat.com>

   LICENSE This program is part of the 'nieve' R package. See the
   LICENSE of the package.
   
   GOAL Compute the probability functions related to the two-parameter
   Generalized Pareto Distribution (GPD), possibly including the
   gradient and the Hessian for the density, the cumulative
   distribution and the quantile functions. The implementation in C is
   faster than a pure R implementation. Unlike some other
   implementations in existing R packages, NAs are returned when the
   scale is negative, which can be a desirable behaviour when
   unconstrained optimisation is used to maximise the log-likelihood.
 
   We are using here the .Call interface, because we want to return 
   the derivatives as attributes of the result. As a side effect, we 
   can not use the macros like `R_Q_P01_boundaries` 

   =========================================================================== */  


/* ==========================================================================
 * Density function
 * ========================================================================== */

SEXP Call_dGPD2(SEXP x,             /*  double                          */
		SEXP scale,         /*  double                          */
		SEXP shape,         /*  double                          */
		SEXP logFlag,       /*  integer                         */
		SEXP derivFlag,     /*  integer                         */ 
		SEXP hessianFlag) { /*  integer                         */
  
  int n, nx, nscale, nshape, i, ix, iscale, ishape,
    logF = INTEGER(logFlag)[0],
    deriv = INTEGER(derivFlag)[0], hessian = INTEGER(hessianFlag)[0];
  
  double eps = 1e-4, z, V, xi;
  
  SEXP val;
  
  PROTECT(x = coerceVector(x, REALSXP));
  PROTECT(scale = coerceVector(scale, REALSXP));
  PROTECT(shape = coerceVector(shape, REALSXP));

  double *rx = REAL(x), *rscale = REAL(scale), 
    *rshape = REAL(shape);

  nx = LENGTH(x);											
  nscale = LENGTH(scale);		
  nshape = LENGTH(shape);
  
  if ((nx == 0) || (nscale == 0) || (nshape == 0)) {
    UNPROTECT(3);
    return(allocVector(REALSXP, 0));				
  }
  
  n = nx;						       			
  if (n < nscale) n = nscale;
  if (n < nshape) n = nshape;
  
  PROTECT(val = allocVector(REALSXP, n));
  double *rval = REAL(val);

  // Rprintf("logFlag %d\n", logF);
  // Rprintf("derivFlag %d\n", deriv);
  
  if (deriv) {

    SEXP grad, attrNm;

    PROTECT(grad = allocVector(REALSXP, n * 2));
    double *rgrad = REAL(grad);

    /* =====================================================================
       NOTE: although some variables that are used only when the
       hessian is required and hence could be declared only in this
       case, it seems that this does not work. It seems that the
       compilator does not see declarations inside two nested 'if'.
       ===================================================================== */
    
    SEXP hess;
    
    PROTECT(hess = allocVector(REALSXP, n * 2 * 2));
    double *rhess = REAL(hess);
    
    // specific auxiliary variables for hessian
    int j, k;
    double A, B, B1, sigma, sigma2;
    
    PROTECT(attrNm = NEW_CHARACTER(1)); 
    SET_STRING_ELT(attrNm, 0, mkChar("gradient"));

    for (i = ix = iscale = ishape = 0;  i < n; 
	 ix = (++ix == nx) ? 0 : ix, 	       
	   iscale = (++iscale == nscale) ? 0 : iscale,
	   ishape = (++ishape == nshape) ? 0 : ishape,
	   ++i) {
      
      if (ISNA(rx[ix]) || (rscale[iscale] <= 0.0)) {
	
	rval[i] = NA_REAL;
	
	rgrad[i] = NA_REAL;
	rgrad[i + n] = NA_REAL;

	if (hessian) {
	  // row 'sigma'
	  rhess[i] = NA_REAL;
	  rhess[i + n] = NA_REAL;
	  rhess[i + 2 * n] = NA_REAL;
	  // row 'xi'
	  rhess[i + 3 * n] = NA_REAL;
	}
	
      } else if (rx[ix] < 0.0) {

	// remind that for now we are evaluating the log-density
	// the exponential transformation will come later.
	rval[i] = R_NegInf;
	
	rgrad[i] = 0.0;
	rgrad[i + n] = 0.0;

	if (hessian) {
	  rhess[i] = 0.0;
	  rhess[i + n] = 0.0;
	  rhess[i + 2 * n] = 0.0;
	  rhess[i + 3 * n] = 0.0;
	}
	
      } else {
	
	z = rx[ix] / rscale[iscale];
	xi = rshape[ishape];
	sigma = rscale[iscale]; 
	sigma2 = sigma * sigma;

	// Rprintf("%d, %d, %6.3f, %6.3f\n", i, ishape, xi, sigma2);
 
	if (fabs(xi) < eps) {

	  if (z < 0.0) {
	    
	    rgrad[i] = 0.0;
	    rgrad[i + n] = 0.0;
	    
	    if (hessian) {
	      
	      // row 'sigma'
	      rhess[i] = 0.0;
	      rhess[i + n] = 0.0;
	      rhess[i + 2 * n] = 0.0;
	      // row 'xi'
	      rhess[i + 3 * n] = 0.0;
	      
	    }

	  } else {
	    
	    // rval[i] = -log(sigma) - z;

	    // better approx
	    rval[i] = -log(sigma) -
	      z * (xi + 1.0) * (1.0 + (z * xi) * (-1.0 / 2.0  + (z * xi) / 3.0));
	    
	    // rgrad[i] = (z - 1.0) / sigma;
	    // rgrad[i + n] = 0.5 * z * (z - 2.0);
	    // better approxs
	    rgrad[i] = ((z - 1.0) * (1 - z * xi)) / sigma;
	    rgrad[i + n] =  z * ((0.5 * z - 1.0) + (-2.0 * z / 3.0 * z + 1.0) * z * xi);
	    
	    if (hessian) {

	      // row 'sigma'
	      rhess[i] =  (1 - 2 * z) / sigma / sigma;
	      rhess[i + n] =  - z * (z - 1.0) / sigma;
	      rhess[i + 2 * n] = rhess[i + n];
	      // row 'xi'
	      rhess[i + 3 * n] = z * z * (-2.0 * z / 3.0 + 1);
	      
	    }
	  }
	  
	} else {
	  
	  V = 1.0 + xi * z;

	  // Rprintf("%d, %d, %6.3f, %6.3f\n", i, ishape, xi, V);
	  
	  if (V > 0.0) {
	
	    A = log(V);
	    B = z / V; 
	    rval[i] = -log(sigma) - (1.0 / xi + 1.0) * A;
	      
	    B1 = (xi + 1.0) * B;
       
	    rgrad[i] = -(1.0 - B1) / sigma;
	    rgrad[i + n] = (A / xi - B1) / xi;
	    
	    if (hessian) {

	      // row 'sigma'
	      rhess[i] = (1.0 - B1 * (2.0 - B * xi)) / sigma2 ;
	      rhess[i + n] =  B * (1.0 - B1) / sigma;
	      rhess[i + 2 * n] = rhess[i + n];
	      // row 'xi'
	      rhess[i + 3 * n] = (-2.0 * A + xi * B * (2.0 + xi * B1)) / xi / xi / xi;
	    }
	    
	  } else {

	    rval[i] = R_NegInf;
	    rgrad[i] = 0.0;
	    rgrad[i + n] = 0.0;

	    if (hessian) {
	      
	      // row 'sigma'
	      rhess[i] = 0.0;
	      rhess[i + n] = 0.0;
	      rhess[i + 2 * n] = 0.0;
	      // row 'xi'
	      rhess[i + 3 * n] = 0.0;
	     
	    }

	  }

	} /* non-exponential case */
	
      }   /* non-NA case     */
      	
      if (logF == 0) {
	
	rval[i] = exp(rval[i]);
	
	rgrad[i] *= rval[i];
	rgrad[i + n] *= rval[i];
	
	if (hessian) {
	  // multiply the hessian by the density and add the
	  // 'tcrossprod' of the gradient
	  for (j = 0; j < 2; j++) {
	    for (k = 0; k < 2; k++) {
	      rhess[i + (j + k * 2) * n] *= rval[i];
	      rhess[i + (j + k * 2) * n] += rgrad[i + j * n] * rgrad[i + k * n];
	    }
	  }
	}
	
      }
      
    }

    SET_ATTR(val, attrNm, grad);

    if (hessian) {
      SET_STRING_ELT(attrNm, 0, mkChar("hessian"));
      SET_ATTR(val, attrNm, hess);
    } 

    UNPROTECT(7);

    return(val);
    
  } else {

    for (i = ix = iscale = ishape = 0;  i < n; 
	 ix = (++ix == nx) ? 0 : ix, 	        
	   iscale = (++iscale == nscale) ? 0 : iscale,
	   ishape = (++ishape == nshape) ? 0 : ishape,
	   ++i) {
      
      if (ISNA(rx[ix]) || (rscale[iscale] <= 0.0)) {
	
	rval[i] = NA_REAL;

      } else if (rx[ix] < 0.0) {
	
	rval[i] = R_NegInf;

      } else {
	
	z = rx[ix] / rscale[iscale];
	xi = rshape[ishape];
	// Rprintf("%d, %d, %6.3f, sigma = %6.3f\n", i, ishape, xi, rscale[iscale]);
	
	if (fabs(xi) < eps) {
 
	  if (z < 0.0) {
	    rval[i] = 0.0;	  
	  } else {
	    rval[i] = -log(rscale[iscale]) - z;
	    rval[i] = -log(rscale[iscale]) -
	      z * (xi + 1.0) * (1.0 + (z * xi) * (-1.0 / 2.0  + (z * xi) / 3.0));
	  }
	  
	} else {
	  
	  if (z < 0.0) {
	    rval[i] = 0.0;
	  } else {
	    V = 1.0 + xi * z;
	    if (V > 0.0) {
	      rval[i] = -log(rscale[iscale]) - (1.0 / xi + 1.0) * log(V);
	    } else {
	      rval[i] = R_NegInf;
	    }
	  }

	  // Rprintf("V =  %6.3f, val = %6.3f\n", V, rval[i]);
	  
	} /* non-exponential case */
	
      }   /* non-NA case     */

      if (logF == 0) {
	rval[i] = exp(rval[i]);
      }
     
      
    }
  
    UNPROTECT(4);
    return(val);
    
  }

}

 /* ==========================================================================
  * Distribution function
  * 
  * ========================================================================== */

SEXP Call_pGPD2(SEXP q,               /*  double                          */
		SEXP scale,           /*  double                          */
		SEXP shape,           /*  double                          */
		SEXP lowerTailFlag,   /*  integer                         */
		SEXP derivFlag,       /*  integer                         */
		SEXP hessianFlag) {   /*  integer                         */
 
  int n, nq, nscale, nshape, i, iq, iscale, ishape,
    lowerTail = INTEGER(lowerTailFlag)[0], 
    deriv = INTEGER(derivFlag)[0], hessian = INTEGER(hessianFlag)[0];
  
  double eps = 1e-4, z, V, A, B, S, sigma, xi, u, H, dHdsigma, dHdxi;
  
  SEXP val;
  
  PROTECT(q = coerceVector(q, REALSXP));
  PROTECT(scale = coerceVector(scale, REALSXP));
  PROTECT(shape = coerceVector(shape, REALSXP));

  double *rq = REAL(q), *rscale = REAL(scale), 
    *rshape = REAL(shape);

  nq = LENGTH(q);					       						
  nscale = LENGTH(scale);		
  nshape = LENGTH(shape);
  
  if ((nq == 0) || (nscale == 0) || (nshape == 0)) {
    UNPROTECT(3);
    return(allocVector(REALSXP, 0));
  }
  
  n = nq;						       						
  if (n < nscale) n = nscale;
  if (n < nshape) n = nshape;
  
  PROTECT(val = allocVector(REALSXP, n));
  double *rval = REAL(val);

  if (deriv) {

    SEXP grad, attrNm;

    PROTECT(grad = allocVector(REALSXP, n * 2));
    double *rgrad = REAL(grad);

    PROTECT(attrNm = NEW_CHARACTER(1)); 
    SET_STRING_ELT(attrNm, 0, mkChar("gradient"));
    
    // if (hessian) { 

    SEXP hess;

    PROTECT(hess = allocVector(REALSXP, n * 2 * 2));
    double *rhess = REAL(hess);
    double z1;
    
    // }
   
    for (i = iq = iscale = ishape = 0;  i < n; 
	 iq = (++iq == nq) ? 0 : iq, 	       
	   iscale = (++iscale == nscale) ? 0 : iscale,
	   ishape = (++ishape == nshape) ? 0 : ishape,
	   ++i) {


      // as a general rule, only 3 elements of the hessan can be non
      // zero
      if (hessian) {
	// row 'mu'
	rhess[i] = 0.0;
	rhess[i + n] = 0.0;
	rhess[i + 2 * n] = 0.0;
	rhess[i + 3 * n] = 0.0;
      }
      
      if (ISNA(rq[iq]) || (rscale[iscale] <= 0.0)) {

	rval[i] = NA_REAL;
	rgrad[i] = NA_REAL;
	rgrad[i + n] = NA_REAL;

	if (hessian) {
	  // row 'sigma'
	  rhess[i ] = NA_REAL;
	  rhess[i + 2 * n] = NA_REAL;
	  // row 'xi'
	  rhess[i + n] = NA_REAL;
	  rhess[i + 3 * n] = NA_REAL;
	}

      } else if (((rq[iq] == R_NegInf) && lowerTail) ||
		 ((rq[iq] == R_PosInf) && !lowerTail)) {
	
	rval[i] = 0.0;
	
	rgrad[i] = 0.0;
	rgrad[i + n] = 0.0;

	if (hessian) {
	  rhess[i ] = 0.0;
	  rhess[i + 2 * n] = 0.0;
	  rhess[i + n] = 0.0;
	  rhess[i + 3 * n] = 0.0;
	}
	
      } else if (((rq[iq] == R_PosInf) && lowerTail) ||
		 ((rq[iq] == R_NegInf) && !lowerTail)) {
	
	rval[i] = 1.0;

	rgrad[i] = 0.0;
	rgrad[i + n] = 0.0;

	if (hessian) {
	  rhess[i ] = 0.0;
	  rhess[i + 2 * n] = 0.0;
	  rhess[i + n] = 0.0;
	  rhess[i + 3 * n] = 0.0;
	}
	
      } else {
	
	z = rq[iq] / rscale[iscale];
	xi = rshape[ishape];
	sigma = rscale[iscale];
	
	if (fabs(xi) < eps) {
   
	  if (z < 0.0) {
	    S = 1.0;
	    rval[i] = S;
	    rgrad[i] = 0.0;
	    rgrad[i + n] = 0.0;
	  } else {
	    // improve precision for small xi
	    u = xi * z;
	    H = z * (1 - u * (1.0 / 2.0 - u  / 3.0)); 
	    S = exp(-H);
	    dHdsigma = - z * (1.0 - u) / sigma;
	    dHdxi = - z * z * (1.0 / 2.0 - 2.0 * u / 3.0);
	    rval[i] = S;
	    rgrad[i] = - S * dHdsigma;
	    rgrad[i + n] = - S * dHdxi;
	  }

	  // Rprintf("%4d, %4d, %6.3f, %7.2f, %6.3f, %6.3f", i, ishape, xi, z, Z, rscale[iscale]);
	  // Rprintf(" grad: %6.3f, %6.3f, %6.3f\n", rgrad[i], rgrad[i + n], rgrad[i + 2 * n]);

	  if (hessian) {
	    z1 = 2.0 - z;
	    // row 'sigma'
	    rhess[i] = - z * z1 * S / sigma / sigma;
	    rhess[i + 2 * n] = - z * z * z1 * S / 2.0 / sigma; 
	    // row 'xi'
	    rhess[i + n] = rhess[i + 2 * n];
	    rhess[i + 3 * n] = - z * z * z * (8.0 - 3.0 * z) * S / 12.0;
	  }
	 
	} else {      // non "quasi-exponential" case
	  
	  if (z < 0.0) {
	    S = 1.0;
	    rval[i] = S;

	    rgrad[i] = 0.0;
	    rgrad[i + n] = 0.0;

	    if (hessian) {
	      rhess[i] = 0.0;
	      rhess[i + 2 * n] = 0.0;
	      rhess[i + n] = 0.0;
	      rhess[i + 3 * n] = 0.0;
	    }
	    
	  } else {
	    
	    V = 1.0 + xi * z;
	    // Rprintf("%d, %d, %6.3f, %6.3f\n", i, ishape, xi, V);
	    
	    if (V > 0.0) {
	      
	      A = log(V);
	      B = z / V;
	      S = pow(V, - 1.0 / xi);
	      rval[i] = S;
	      
	      rgrad[i] =  S * B / sigma;
	      rgrad[i + n] = S * (A / xi - B) / xi;

	      if (hessian) {
		// row 'sigma'
		rhess[i] = - B * (2.0 - xi * B) * S / sigma / sigma +  rgrad[i] *  rgrad[i] / S;
		rhess[i + 2 * n] = - B * B * S / sigma +  rgrad[i] * rgrad[i + n] / S;
		// row 'xi'
		rhess[i + n] = rhess[i + 2 * n];
		z1 = (- 2.0 * A / xi / xi + B * (2.0 / xi + B)) / xi;
		rhess[i + 3 * n] = z1 * S + rgrad[i + n] *  rgrad[i + n] / S;
	      }
	      
	    } else {
	      
	      if (xi > 0.0) {
		rval[i] = 1.0;
	      } else {
		rval[i] = 0.0;
	      }
	      rgrad[i] = 0.0;
	      rgrad[i + n] = 0.0;   

	      if (hessian) {
		// row 'sigma'
		rhess[i] = 0.0;
		rhess[i + 2 * n] = 0.0;
		// row 'xi'
		rhess[i + n] = 0.0;
		rhess[i + 3 * n] = 0.0;
	      }	      
	    }

	  }

	} /* non-exponential case */
	
	if (INTEGER(lowerTailFlag)[0]) {
	  rval[i] = 1.0 - rval[i];
	  rgrad[i] = -rgrad[i];
	  rgrad[i + n] = -rgrad[i + n];
	  if (hessian) {
	    rhess[i] = -rhess[i];
	    rhess[i + n] = -rhess[i + n];
	    rhess[i + 2 * n] = -rhess[i + 2 * n];
	    rhess[i + 3 * n] = - rhess[i + 3 * n];
	  }
	}
	
      }   /* non-NA case     */
      
    }
  
    SET_ATTR(val, attrNm, grad);

    if (hessian) {
      SET_STRING_ELT(attrNm, 0, mkChar("hessian"));
      SET_ATTR(val, attrNm, hess);
    } 

    
    UNPROTECT(7);
    return(val);
    
  } else {

    for (i = iq = iscale = ishape = 0;  i < n; 
	 iq = (++iq == nq) ? 0 : iq, 	        
	   iscale = (++iscale == nscale) ? 0 : iscale,
	   ishape = (++ishape == nshape) ? 0 : ishape,
	   ++i) {
      
      if (ISNA(rq[iq]) || (rscale[iscale] <= 0.0)) {

	rval[i] = NA_REAL;
	
      } else if (((rq[iq] == R_NegInf) && lowerTail) ||
		 ((rq[iq] == R_PosInf) && !lowerTail)) {
	rval[i] = 0.0;
      } else if (((rq[iq] == R_PosInf) && lowerTail) ||
		 ((rq[iq] == R_NegInf) && !lowerTail)) {
	rval[i] = 1.0;
      } else {
	z = rq[iq] / rscale[iscale];
	xi = rshape[ishape];
	if (fabs(xi) < eps) {
	  if (z < 0.0) {
	    rval[i] = 1.0;
	  }	  
	  else {
	    // rval[i] = exp(-z);
	    // improved approx
	    u = xi * z;
	    H = z * (1 - u * (1.0 / 2.0 - u  / 3.0)); 
	    rval[i] = exp(-H);
	  }
	} else {
	  if (z < 0.0) {
	    rval[i] = 1.0;
	  }
	  else {
	    V = 1.0 + xi * z;
	    if (V > 0.0) { 
	      rval[i] = pow(V, - 1.0 / xi);
	    } else {
	      if (xi > 0.0) {
		rval[i] = 1.0;
	      } else {
		rval[i] = 0.0;
	      }
	    }
	  }
	} /* non-exponential case */
	
	if (INTEGER(lowerTailFlag)[0]) {
	  rval[i] = 1.0 - rval[i];
	}
	
      }   /* non-NA case     */
      
    }
  
    UNPROTECT(4);
    return(val);
    
  }

}

/* ==========================================================================
 * Quantile function
 * ========================================================================== */

SEXP Call_qGPD2(SEXP p,               /*  double                          */
		SEXP scale,           /*  double                          */
		SEXP shape,           /*  double                          */
		SEXP lowerTailFlag,   /*  integer                         */
		SEXP derivFlag,       /* integer                          */
		SEXP hessianFlag) {   /*  integer                         */
  
  int n, np, nscale, nshape, i, ip, iscale, ishape,
    lowerTail = INTEGER(lowerTailFlag)[0],
    deriv = INTEGER(derivFlag)[0], hessian = INTEGER(hessianFlag)[0] ;
  
  double eps = 1e-4, q, lq, q1, xi, sigma, V, W, rpi;
  
  SEXP val;
  
  PROTECT(p = coerceVector(p, REALSXP));
  PROTECT(scale = coerceVector(scale, REALSXP));
  PROTECT(shape = coerceVector(shape, REALSXP));

  double *rp = REAL(p), *rscale = REAL(scale), 
    *rshape = REAL(shape);

  np = LENGTH(p);	    	
  nscale = LENGTH(scale);		
  nshape = LENGTH(shape);
  
  if ((np == 0) || (nscale == 0) || (nshape == 0)) {
    UNPROTECT(3);
    return(allocVector(REALSXP, 0));				
  }
  
  n = np;		  			
  if (n < nscale) n = nscale;
  if (n < nshape) n = nshape;
  
  PROTECT(val = allocVector(REALSXP, n));
  double *rval = REAL(val);

  if (deriv) {

    SEXP grad, attrNm;

    PROTECT(grad = allocVector(REALSXP, n * 2));
    double *rgrad = REAL(grad);

    PROTECT(attrNm = NEW_CHARACTER(1)); 
    SET_STRING_ELT(attrNm, 0, mkChar("gradient"));

 // if (hessian) { 

      SEXP hess;

      PROTECT(hess = allocVector(REALSXP, n * 2 * 2));
      double *rhess = REAL(hess);
      // double dV, d2V;

      // }

    for (i = ip = iscale = ishape = 0;  i < n; 
	 ip = (++ip == np) ? 0 : ip, 	       
	   iscale = (++iscale == nscale) ? 0 : iscale,
	   ishape = (++ishape == nshape) ? 0 : ishape,
	   ++i) {
      
      // as a general rule, only 1 element of the hessian can be non
      // zero
      
      if (hessian) {
	
	// row 'sigma'
	rhess[i] = 0.0;
	rhess[i + n] = 0.0;
	rhess[i + 2 * n] = 0.0;
	// row 'xi'
	rhess[i + 3 * n] = 0.0;
	
      }

      if (ISNA(rp[ip]) || (rscale[iscale] <= 0.0)) {
	// Rprintf("NA case\n");

	rval[i] = NA_REAL;
	
	rgrad[i] = NA_REAL;
	rgrad[i + n] = NA_REAL;
	
	if (hessian) {
	  // row 'sigma'
	  rhess[i] = 0.0;
	  rhess[i + n] = NA_REAL;
	  rhess[i + 2 * n] = NA_REAL;
	  // row 'xi'
	  rhess[i + 3 * n] = NA_REAL;
	}

      } else if (((rp[ip] == 0.0) && lowerTail) ||
		 ((rp[ip] == 1.0) && !lowerTail)) {
	
	rval[i] = 0.0;
	
	rgrad[i] = 0.0;
	rgrad[i + n] = 0.0;
	
	if (hessian) {  
	  rhess[i] = 0.0;
	  rhess[i + n] = 0.0;
	  rhess[i + 2 * n] = 0.0;
	  rhess[i + 3 * n] = 0.0;
	}
	
      } else if (((rp[ip] == 1.0) && lowerTail) ||
		 ((rp[ip] == 0.0 && !lowerTail))) {
	
	xi = rshape[ishape];
	sigma = rscale[iscale];

	if (xi >= 0.0) {
	  
	  rval[i] = R_PosInf;
	
	  rgrad[i] =  NA_REAL;
	  rgrad[i + n] =  NA_REAL;

	  if (hessian) {  
	    rhess[i] = NA_REAL;
	    rhess[i + n] = NA_REAL;
	    rhess[i + 2 * n] = NA_REAL;
	    rhess[i + 3 * n] = NA_REAL;
	  } 

	} else {

	  rval[i] = - sigma / xi;
	  
	  rgrad[i] = lowerTail ? - 1.0 / xi : 1.0 / xi;
	  rgrad[i + n] = lowerTail? sigma / xi / xi : -sigma / xi / xi;
	  
	  if (hessian) {  
	    rhess[i] = 0.0;
	    rhess[i + n] = 1.0 / xi / xi;
	    rhess[i + 2 * n] = 0.0;
	    rhess[i + 3 * n] = -2.0 * sigma / xi / xi / xi;
	  } 
	  
	}
	
      } else {

	rpi = rp[ip];
	
	if (!INTEGER(lowerTailFlag)[0]) {
	  rpi = 1.0 - rpi;
	}

	xi = rshape[ishape];
	sigma = rscale[iscale];
	q = 1.0 - rpi;
	lq = log(q);

	if (fabs(xi) < eps) {

	  V = - lq;
	  W = lq * lq / 2.0;
	  
	  // improved approximation
	  rval[i] = rscale[iscale] * (-lq * (1.0 - lq * xi * (0.5 - lq * xi / 6.0))) ;	  

	  rgrad[i] =  lq * (-1.0 - 0.5 * lq * xi) ;
	  rgrad[i + n] = sigma * lq * lq * (0.5 - lq * xi / 3.0);

	  if (hessian) {
	    // row 'sigma'
	    rhess[i] = 0.0;
	    rhess[i + n] = W;
	    rhess[i + 2 * n] = W;
	    // row 'xi'
	    rhess[i + 3 * n] = - sigma * lq * lq * lq / 3.0;

	  }

	} else {

	  q1 =  pow(q, - xi);
	  V = (q1 - 1.0) / xi;
	  rval[i] = sigma * V;  
	  W = - (V + q1 * lq) / xi;
	  
	  rgrad[i] = V ;
	  rgrad[i + n] = sigma * W;

	  if (hessian) {
	    // row 'sigma'
	    rhess[i] = 0.0;
	    rhess[i + n] = W;
	    rhess[i + 2 * n] = W;
	    // row 'xi'
	    rhess[i + 3 * n] = sigma * (2 * V + q1 * lq * (2 + xi * lq)) / xi / xi;
	  }


	} /* non-exponential case */

      }   /* non-NA case     */
      
    }     /* loop            */

    SET_ATTR(val, attrNm, grad);
    
    if (hessian) {
      SET_STRING_ELT(attrNm, 0, mkChar("hessian"));
      SET_ATTR(val, attrNm, hess);
    } 

    UNPROTECT(7);
    return(val);
    
  } else {

    for (i = ip = iscale = ishape = 0;  i < n; 
	 ip = (++ip == np) ? 0 : ip, 	       
	   iscale = (++iscale == nscale) ? 0 : iscale,
	   ishape = (++ishape == nshape) ? 0 : ishape,
	   ++i) {
      
      if (ISNA(rp[ip]) || (rscale[iscale] <= 0.0)) {
	
	rval[i] = NA_REAL;

      } else if (((rp[ip] == 0.0) && lowerTail) ||
		 ((rp[ip] == 1.0) && !lowerTail)) {
	
	rval[i] = 0.0;
	
      } else if (((rp[ip] == 1.0) && lowerTail) ||
		 ((rp[ip] == 0.0 && !lowerTail))) {

	if (rshape[ishape] >= 0.0) {
	  rval[i] = R_PosInf;
	} else {
	  rval[i] = - rscale[iscale] / rshape[ishape];
	}
	
      } else {

	rpi = rp[ip];
	
	if (!INTEGER(lowerTailFlag)[0]) {
	  rpi = 1.0 - rpi;
	}

	xi = rshape[ishape];
	q = 1 - rpi;
	lq = log(q);
	
	if (fabs(xi) < eps) {

	  // rval[i] = -rscale[iscale] * log(q);
	  // better approx
	  rval[i] = rscale[iscale] * (-lq * (1.0 - lq * xi * (0.5 - lq * xi / 6.0))) ;	  
	  
	} else {

	  V = (pow(q, - xi) - 1.0) / xi;
	  rval[i] = rscale[iscale] * V;  
       	  	  
	} /* non-exp case */
	
      }   /* non-NA case     */
      
    }     /* loop            */
  
    UNPROTECT(4);
    return(val);
    
  }
}
