\name{warcolak}
\alias{warcolak}
\docType{data}
\title{
Pedigree and phenotypic values for a mythical population of Warcolaks}
\description{
A two trait example pedigree from the three generation breeding design of Fairbairn & Roff (2006) with two un-correlated traits.}

\usage{data(warcolak)}
\format{
  A data frame with 5400 observations on the following 13 variables.
  \describe{
    \item{\code{ID }}{a factor specifying 5400 unique individual identities}
    \item{\code{Dam }}{a factor specifying the unique Dam for each individual}
    \item{\code{Sire }}{a factor specifying the unique Sire for each individual}
    \item{\code{sex} }{a factor specifying \dQuote{M} if the individual is a male and \dQuote{F} if it is a female}
    \item{\code{trait1} }{a numeric vector of phenotypic values: see \sQuote{Details}}
    \item{\code{trait2} }{a numeric vector of phenotypic values: see \sQuote{Details}}
    \item{\code{t1_a} }{a numeric vector of the autosomal additive genetic effects underlying \sQuote{trait1}}
    \item{\code{t2_a} }{a numeric vector of the autosomal additive genetic effects underlying \sQuote{trait2}}
    \item{\code{t2_s} }{a numeric vector of the sex-chromosomal additive genetic effects underlying \sQuote{trait2}}
    \item{\code{t1_d} }{a numeric vector of the autosomal dominance genetic effects underlying \sQuote{trait1}}
    \item{\code{t2_d} }{a numeric vector of the autosomal dominance genetic effects underlying \sQuote{trait2}}
    \item{\code{t1_r} }{a numeric vector of the residual (environmental) effects underlying \sQuote{trait1}}
    \item{\code{t2_r} }{a numeric vector of the residual (environmental) effects underlying \sQuote{trait2}}
  }
}

\details{
Unique sets of relatives are specified for a three generation breeding design (Fairbairn & Roff, 2006).  Each set contains 72 individuals. This pedigree reflects an experiment which produces 75 of these basic sets from Fairbairn & Roff's design. The pedigree was created using \code{simPedDFC()}.

The dataset was simulated to have two un-correlated traits with different genetic architectures (see \code{examples} below). The trait means are both equal to 1 for males and 2 for females. The additive genetic, dominance genetic, and environmental (or residual) variances for both \code{trait1} and \code{trait2} are 0.4, 0.3, & 0.3, respectively. However, the additive genetic variance for \code{trait2} can be further decomposed to autosomal additive genetic variance (0.3) and X-linked additive genetic variance (0.1; assuming the \sQuote{no global dosage compensation} mechanism).

Females and males have equal variances (except for sex-chromosomal additive genetic variance, where by definition, males have half of the additive genetic variance as females; Wolak 2013) and a between-sex correlation of one for all genetic and residual effects (except the cross-sex residual covariance=0). All random effects were drawn from multivariate random normal distributions [e.g., autosomal additive genetic effects: N ~ (0, kronecker(A, G))] with means of zero and (co)variances equal to the product of the expected sex-specific (co)variances (e.g., G) and the relatedness (or incidence) matrix (e.g., A).

The actual variance in random effects will vary slightly from the amount specified in the simulation, because of Monte Carlo error. Thus, the random effects have been included as separate columns in the dataset. See \code{examples} below for the code that generated the dataset. 
}

\note{
Before nadiv version 2.14.0, the \code{warcolak} dataset used a 0/1 coding for \sQuote{sex} and did not contain the random effects.
}

\references{
Fairbairn, D.J. & Roff, D.A. 2006. The quantitative genetics of sexual dimorphism: assessing the importance of sex-linkage. Heredity 97, 319-328.

Wolak, M.E. 2013. The Quantitative Genetics of Sexual Differences: New Methodologies and an Empirical Investigation of Sex-Linked, Sex-Specific, Non-Additive, and Epigenetic Effects. Ph.D. Dissertation. University of California Riverside.
}

\examples{
 \dontrun{
  rm(list = ls())
  set.seed(101)
  library(nadiv)
  # create pedigree
  warcolak <- simPedDFC(F = 75, gpn = 4, fsn = 4, s = 2)
  names(warcolak)[1:3] <- c("ID", "Dam", "Sire")
  warcolak$trait2 <- warcolak$trait1 <- NA

  # Define covariance matrices for random effects:
  ## Autosomal additive genetic (trait1)
  Ga_t1 <- matrix(c(0.4, rep(0.399999, 2), 0.4), 2, 2)
  ## Autosomal additive genetic (trait2)
  Ga_t2 <- matrix(c(0.3, rep(0.299999, 2), 0.3), 2, 2)
  ## Sex-chromosomal additive genetic (trait2)
  Gs_t2 <- matrix(c(0.1, rep(0.099999, 2), 0.1), 2, 2)
  ## Autosomal dominance genetic
  Gd <- matrix(c(0.3, rep(0.299999, 2), 0.3), 2, 2)
  ## Environmental (or residual)
  ### Assumes no correlated environmental effects between sexes
  R <- diag(c(0.3, 0.3))

  ## define variables to be re-used
  pedn <- nrow(warcolak)
  # Female (homogametic sex chromosomes) in first column
  # Male (heterogametic sex chromosomes) in second column
  sexcol <- as.integer(warcolak$sex)

  # Create random effects
  ## Additive genetic
  ### trait1 autosomal
  tmp_a <- grfx(pedn, G = Ga_t1, incidence = makeA(warcolak[, 1:3]))
    var(tmp_a)
  warcolak$t1_a <- tmp_a[cbind(seq(pedn), sexcol)]
  ### trait2 autosomal
  tmp_a <- grfx(pedn, G = Ga_t2, incidence = makeA(warcolak[, 1:3]))
    var(tmp_a)
  warcolak$t2_a <- tmp_a[cbind(seq(pedn), sexcol)]
  ### trait2 sex-chromosomal
  tmp_s <- grfx(pedn, G = Gs_t2, incidence = makeS(warcolak[, 1:4],
	heterogametic = "M", DosageComp = "ngdc", returnS = TRUE)$S)
    matrix(c(var(tmp_s[which(sexcol == 1), 1]),
	rep(cov(tmp_s), 2), var(tmp_s[which(sexcol == 2), 2])), 2, 2)
    # NOTE above should be: covar = male var = 0.5* female var 
  warcolak$t2_s <- tmp_s[cbind(seq(pedn), sexcol)]

  ## Dominance genetic
  ### trait1 
  tmp_d <- grfx(pedn, G = Gd, incidence = makeD(warcolak[, 1:3], invertD = FALSE)$D)
    var(tmp_d)
  warcolak$t1_d <- tmp_d[cbind(seq(pedn), sexcol)]
  ### trait2
  tmp_d <- grfx(pedn, G = Gd, incidence = makeD(warcolak[, 1:3], invertD = FALSE)$D)
    var(tmp_d)
  warcolak$t2_d <- tmp_d[cbind(seq(pedn), sexcol)]

  ## Residual
  ### trait1
  tmp_r <- grfx(pedn, G = R, incidence = NULL) # warning of identity matrix
    var(tmp_r)
  warcolak$t1_r <- tmp_r[cbind(seq(pedn), sexcol)]
  ### trait2
  tmp_r <- grfx(pedn, G = R, incidence = NULL) # warning of identity matrix
    var(tmp_r)
  warcolak$t2_r <- tmp_r[cbind(seq(pedn), sexcol)]

  # Sum random effects and add sex-specific means to get phenotypes
  ## females have slightly greater mean trait values
  warcolak$trait1 <- 1 + (-1*sexcol + 2) + rowSums(warcolak[, c("t1_a", "t1_d", "t1_r")])
  warcolak$trait2 <- 1 + (-1*sexcol + 2) + rowSums(warcolak[, c("t2_a", "t2_s", "t2_d", "t2_r")])
 }
}

\keyword{datasets}
