% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mssm.R
\name{mssm}
\alias{mssm}
\title{Get Multivariate State Space Model Functions}
\usage{
mssm(fixed, family, data, random, weights, offsets, ti,
  control = mssm_control())
}
\arguments{
\item{fixed}{\code{\link{formula}} with outcome variable on the left hand
side and covariates with fixed effects on the right hand side.}

\item{family}{family for the observed outcome given the state variables
and covariates.}

\item{data}{\code{\link{data.frame}} or environment containing the variables
in \code{fixed} and \code{random}.}

\item{random}{\code{\link{formula}} for covariates with a random effect.
Left hand side is ignored.}

\item{weights}{optional prior weights.}

\item{offsets}{optional a priori known component in the linear predictor.}

\item{ti}{integer vector with time indices matching with each observation of
\code{fixed} and \code{random}.}

\item{control}{list with arguments passed to \code{\link{mssm_control}}.}
}
\value{
An object of class \code{mssmFunc} with the following elements
\item{pf_filter}{function to perform particle filtering. See
\link{mssm-pf}.}
\item{Laplace}{function to perform parameter estimation with a Laplace
approximation. See \link{mssm-Laplace}.}
\item{smoother}{function to compute smoothing weights for an \code{mssm}
object returned by the \code{pf_filter} function. See \link{mssm-smoother}.}
\item{terms_fixed}{\code{\link{terms.object}} for the covariates with
fixed effects.}
\item{terms_random}{\code{\link{terms.object}} for the covariates with
random effects.}
\item{y}{vector with outcomes.}
\item{X}{covariates with fixed effects.}
\item{Z}{covariates with random effects.}
\item{ti}{time indices for each observation.}
\item{weights}{prior weights for each observation.}
\item{offsets}{a priori known component in the linear predictor for
each observation.}
\item{call}{the matched call.}
\item{family}{character describing the conditional distribution of the
outcomes.}
}
\description{
Returns an object with a function that can be used to run a particle filter,
a function to perform parameter estimation using a Laplace approximation,
and a function to perform smoothing of particle weights.
}
\examples{
if(require(Ecdat)){
  # load data and fit glm to get starting values
  . <- print
  data("Gasoline", package = "Ecdat")
  glm_fit <- glm(lgaspcar ~ factor(country) + lincomep + lrpmg + lcarpcap,
                 Gamma("log"), Gasoline)

  # get object to perform estimation
  library(mssm)
  ll_func <- mssm(
  fixed = formula(glm_fit), random = ~ 1, family = Gamma("log"),
    data = Gasoline, ti = year, control = mssm_control(
      N_part = 1000L, n_threads = 1L))
  .(ll_func)

  # fit model with time-varying intercept with Laplace approximation
  disp <- summary(glm_fit)$dispersion
  laplace <- ll_func$Laplace(
    cfix = coef(glm_fit), disp = disp, F. = diag(.5, 1), Q = diag(1))
  .(laplace)

  # compare w/ glm
  .(logLik(laplace))
  .(logLik(glm_fit))
  .(rbind(laplace = laplace$cfix, glm = coef(glm_fit)))

  # run particle filter
  pf <- ll_func$pf_filter(
    cfix = laplace$cfix, disp = laplace$disp, F. = laplace$F., Q = laplace$Q)
  .(pf)

  # compare approximate log-likelihoods
  .(logLik(pf))
  .(logLik(laplace))

  # predicted values from filtering (does not appear random...)
  plot(pf)

  # plot predicted values from smoothing distribution
  pf <- ll_func$smoother(pf)
  plot(pf, which_weights = "smooth")
}

}
\seealso{
The README of the package contains examples of how to use this function.
See \url{https://github.com/boennecd/mssm}.
}
