% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gridfs.R
\name{gridfs}
\alias{gridfs}
\title{GridFS API}
\usage{
gridfs(
  db = "test",
  url = "mongodb://localhost",
  prefix = "fs",
  options = ssl_options()
)
}
\arguments{
\item{db}{name of database}

\item{url}{address of the mongodb server in mongo connection string
\href{https://docs.mongodb.com/manual/reference/connection-string}{URI format}}

\item{prefix}{string to prefix the collection name}

\item{options}{additional connection options such as SSL keys/certs.}
}
\description{
Connect to a GridFS database to search, read, write and delete files.
}
\details{
We support two interfaces for sending/receiving data from/to GridFS. The
\code{fs$read()} and \code{fs$write()} methods are the most flexible and can send data
from/to an R connection, such as a \link{file}, \link[=socketConnection]{socket}
or \link{url}. These methods support a progress counter and can be interrupted
if needed. These methods are recommended for reading or writing single files.

The \code{fs$upload()} and \code{fs$download()} methods on the other hand copy directly
between GridFS and your local disk. This API is vectorized so it can transfer
many files at once. However individual transfers cannot be interrupted and will
block R until completed. This API is only recommended to upload/download a large
number of small files.

Modifying files in GridFS is currently unsupported: uploading a file with the
same name will generate a new file.
}
\section{Methods}{

\describe{
\item{\code{find(filter = "{}", options = "{}")}}{Search and list files in the GridFS}
\item{\code{download(name, path = '.')}}{Download one or more files from GridFS to disk. Path may be an existing directory or vector of filenames equal to 'name'.}
\item{\code{upload(path, name = basename(path), content_type = NULL, metadata = NULL)}}{Upload one or more files from disk to GridFS. Metadata is an optional JSON string.}
\item{\code{read(name, con = NULL, progress = TRUE)}}{Reads a single file from GridFS into a writable R \link{connection}.
If \code{con} is a string it is treated as a filepath; if it is \code{NULL} then the output is buffered in memory and returned as a \link{raw} vector.}
\item{\code{write(con, name, content_type = NULL, metadata = NULL, progress = TRUE)}}{Stream write a single file into GridFS from a readable R \link{connection}.
If \code{con} is a string it is treated as a filepath; it may also be a \link{raw} vector containing the data to upload. Metadata is an optional JSON string.}
\item{\code{remove(name)}}{Remove a single file from the GridFS}
\item{\code{drop()}}{Removes the entire GridFS collection, including all files}
}
}

\examples{
# Upload a file to GridFS
fs <- gridfs(url = "mongodb+srv://readwrite:test@cluster0-84vdt.mongodb.net/test")
input <- file.path(R.home('doc'), "html/logo.jpg")
fs$upload(input, name = 'logo.jpg')

# Download the file back to disk
output <- file.path(tempdir(), 'logo1.jpg')
fs$download('logo.jpg', output)

# Or you can also stream it
con <- file(file.path(tempdir(), 'logo2.jpg'))
fs$read('logo.jpg', con)

# Delete the file on the server
fs$remove('logo.jpg')

files <- c(input, file.path(tempdir(), c('logo1.jpg', 'logo2.jpg')))
hashes <- tools::md5sum(files)
stopifnot(length(unique(hashes)) == 1)

\donttest{
# Insert Binary Data
fs <- gridfs()
buf <- serialize(nycflights13::flights, NULL)
fs$write(buf, 'flights')
out <- fs$read('flights')
flights <- unserialize(out$data)

tmp <- file.path(tempdir(), 'flights.rds')
fs$download('flights', tmp)
flights2 <- readRDS(tmp)
stopifnot(all.equal(flights, nycflights13::flights))
stopifnot(all.equal(flights2, nycflights13::flights))

# Show what we have
fs$find()
fs$drop()
}
}
