% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/TuningInstanceSingleCrit.R
\name{TuningInstanceSingleCrit}
\alias{TuningInstanceSingleCrit}
\title{Single Criterion Tuning Instance}
\description{
Specifies a general single-criteria tuning scenario, including objective
function and archive for Tuners to act upon. This class stores an
\code{ObjectiveTuning} object that encodes the black box objective function which
a \link{Tuner} has to optimize. It allows the basic operations of querying the
objective at design points (\verb{$eval_batch()}), storing the evaluations in the
internal \code{Archive} and accessing the final result (\verb{$result}).

Evaluations of hyperparameter configurations are performed in batches by
calling \code{\link[mlr3:benchmark]{mlr3::benchmark()}} internally. Before a batch is evaluated, the
\link[bbotk:Terminator]{bbotk::Terminator} is queried for the remaining budget. If the available
budget is exhausted, an exception is raised, and no further evaluations can
be performed from this point on.

The tuner is also supposed to store its final result, consisting of a
selected hyperparameter configuration and associated estimated performance
values, by calling the method \code{instance$assign_result}.
}
\examples{
library(data.table)
library(paradox)
library(mlr3)

# Objects required to define the performance evaluator:
task = tsk("iris")
learner = lrn("classif.rpart")
resampling = rsmp("holdout")
measure = msr("classif.ce")
param_set = ParamSet$new(list(
  ParamDbl$new("cp", lower = 0.001, upper = 0.1),
  ParamInt$new("minsplit", lower = 1, upper = 10))
)

terminator = trm("evals", n_evals = 5)
inst = TuningInstanceSingleCrit$new(
  task = task,
  learner = learner,
  resampling = resampling,
  measure = measure,
  search_space = param_set,
  terminator = terminator
)

# first 4 points as cross product
design = CJ(cp = c(0.05, 0.01), minsplit = c(5, 3))
inst$eval_batch(design)
inst$archive

# try more points, catch the raised terminated message
tryCatch(
  inst$eval_batch(data.table(cp = 0.01, minsplit = 7)),
  terminated_error = function(e) message(as.character(e))
)

# try another point although the budget is now exhausted
# -> no extra evaluations
tryCatch(
  inst$eval_batch(data.table(cp = 0.01, minsplit = 9)),
  terminated_error = function(e) message(as.character(e))
)

inst$archive

### Error handling
# get a learner which breaks with 50\% probability
# set encapsulation + fallback
learner = lrn("classif.debug", error_train = 0.5)
learner$encapsulate = c(train = "evaluate", predict = "evaluate")
learner$fallback = lrn("classif.featureless")

param_set = ParamSet$new(list(
  ParamDbl$new("x", lower = 0, upper = 1)
))

inst = TuningInstanceSingleCrit$new(
  task = tsk("wine"),
  learner = learner,
  resampling = rsmp("cv", folds = 3),
  measure = msr("classif.ce"),
  search_space = param_set,
  terminator = trm("evals", n_evals = 5)
)

tryCatch(
  inst$eval_batch(data.table(x = 1:5 / 5)),
  terminated_error = function(e) message(as.character(e))
)

archive = inst$archive$data()

# column errors: multiple errors recorded
print(archive)
}
\section{Super classes}{
\code{\link[bbotk:OptimInstance]{bbotk::OptimInstance}} -> \code{\link[bbotk:OptimInstanceSingleCrit]{bbotk::OptimInstanceSingleCrit}} -> \code{TuningInstanceSingleCrit}
}
\section{Active bindings}{
\if{html}{\out{<div class="r6-active-bindings">}}
\describe{
\item{\code{result_learner_param_vals}}{(\code{list()})\cr
Param values for the optimal learner call.}
}
\if{html}{\out{</div>}}
}
\section{Methods}{
\subsection{Public methods}{
\itemize{
\item \href{#method-new}{\code{TuningInstanceSingleCrit$new()}}
\item \href{#method-assign_result}{\code{TuningInstanceSingleCrit$assign_result()}}
\item \href{#method-clone}{\code{TuningInstanceSingleCrit$clone()}}
}
}
\if{html}{
\out{<details open ><summary>Inherited methods</summary>}
\itemize{
\item \out{<span class="pkg-link" data-pkg="bbotk" data-topic="OptimInstance" data-id="eval_batch">}\href{../../bbotk/html/OptimInstance.html#method-eval_batch}{\code{bbotk::OptimInstance$eval_batch()}}\out{</span>}
\item \out{<span class="pkg-link" data-pkg="bbotk" data-topic="OptimInstance" data-id="format">}\href{../../bbotk/html/OptimInstance.html#method-format}{\code{bbotk::OptimInstance$format()}}\out{</span>}
\item \out{<span class="pkg-link" data-pkg="bbotk" data-topic="OptimInstance" data-id="objective_function">}\href{../../bbotk/html/OptimInstance.html#method-objective_function}{\code{bbotk::OptimInstance$objective_function()}}\out{</span>}
\item \out{<span class="pkg-link" data-pkg="bbotk" data-topic="OptimInstance" data-id="print">}\href{../../bbotk/html/OptimInstance.html#method-print}{\code{bbotk::OptimInstance$print()}}\out{</span>}
}
\out{</details>}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-new"></a>}}
\if{latex}{\out{\hypertarget{method-new}{}}}
\subsection{Method \code{new()}}{
Creates a new instance of this \link[R6:R6Class]{R6} class.

This defines the resampled performance of a learner on a task, a
feasibility region for the parameters the tuner is supposed to optimize,
and a termination criterion.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{TuningInstanceSingleCrit$new(
  task,
  learner,
  resampling,
  measure,
  search_space,
  terminator,
  store_benchmark_result = TRUE,
  store_models = FALSE,
  check_values = FALSE
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{task}}{(\link[mlr3:Task]{mlr3::Task})\cr
Task to operate on.}

\item{\code{learner}}{(\link[mlr3:Learner]{mlr3::Learner}).}

\item{\code{resampling}}{(\link[mlr3:Resampling]{mlr3::Resampling})\cr
Uninstantiated resamplings are instantiated during construction
so that all configurations are evaluated on the same data splits.}

\item{\code{measure}}{(\link[mlr3:Measure]{mlr3::Measure})\cr
Measure to optimize.}

\item{\code{search_space}}{(\link[paradox:ParamSet]{paradox::ParamSet}).}

\item{\code{terminator}}{(\link{Terminator}).}

\item{\code{store_benchmark_result}}{(\code{logical(1)})\cr
Store benchmark result in archive?}

\item{\code{store_models}}{(\code{logical(1)})\cr
Store models in benchmark result?}

\item{\code{check_values}}{(\code{logical(1)})\cr
Should parameters before the evaluation and the results be checked for
validity?}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-assign_result"></a>}}
\if{latex}{\out{\hypertarget{method-assign_result}{}}}
\subsection{Method \code{assign_result()}}{
The \link{Tuner} object writes the best found point
and estimated performance value here. For internal use.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{TuningInstanceSingleCrit$assign_result(xdt, y, learner_param_vals = NULL)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{xdt}}{(\code{data.table::data.table()})\cr
x values as \code{data.table}. Each row is one point. Contains the value in
the \emph{search space} of the \link{TuningInstanceMultiCrit} object. Can contain
additional columns for extra information.}

\item{\code{y}}{(\code{numeric(1)})\cr
Optimal outcome.}

\item{\code{learner_param_vals}}{(\code{list()})\cr
Fixed parameter values of the learner that are neither part of the}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-clone"></a>}}
\if{latex}{\out{\hypertarget{method-clone}{}}}
\subsection{Method \code{clone()}}{
The objects of this class are cloneable with this method.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{TuningInstanceSingleCrit$clone(deep = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{deep}}{Whether to make a deep clone.}
}
\if{html}{\out{</div>}}
}
}
}
