\name{multiStep_dist-methods}
\docType{methods}
\alias{multiStep_dist}
\alias{multiStep_dist-methods}
\alias{multiStep_dist,MixAR,numeric,numeric,numeric-method}
\alias{multiStep_dist,MixARGaussian,numeric,missing,missing-method}
\alias{multiStep_dist,MixARGaussian,numeric,missing,ANY-method}
\title{Multi-step predictions for MixAR models}
\description{Multi-step predictions for MixAR models.}
\usage{
multiStep_dist(model, maxh, N, xcond, ...)
}
\arguments{
  \item{model}{a MixAR model.}
  \item{maxh}{maximal horizon, a positive integer.}
  \item{N}{
    an integer specifying the number of simulation samples to use, see
    'Details. This argument is used only by simulation based methods.
  }
  \item{xcond}{
    the past values needed for the conditional distribution, a numeric
    vector of length at least the maximal AR order of the components.
  }
  \item{\dots}{
    used only in some methods, see the details for the individual
    methods.
  }
}
\details{

  The function currently implements two methods: the exact method due to
  \insertCite{boshnakov2009mar;textual}{mixAR} and a simulation
  method described by \insertCite{WongLi2000}{mixAR} for Gaussian MixAR
  models but valid more generally.
  
  The simulation method is available for any MixAR model, while
  the exact method is currently implemented only for models with
  Gaussian components ("MixARGaussian" class).

  \code{multiStep_dist} returns a function which can be used to obtain
  various properties of the predictive distribution for lags up to
  \code{maxh}.

  If argument \code{N} is missing the exact method is tried. Currently
  an error will result if the exact method is not implemented for
  \code{model}.

  If argument \code{N} is given it must be a scalar numeric value, the
  number of simulations to be performed to construct an approximation
  for the predictive distributions.

  The simulation is done by \code{multiStep_dist}. The properties
  obtained later from the function returned by \code{multiStep_dist} use
  the samples generated by the call to \code{multiStep_dist}.
  To do a simulation with different parameters (e.g., with larger
  \code{N}) call \code{multiStep_dist} again.

  \subsection{Details on the returned function}{

  If \code{xcond} is missing \code{multiStep_dist} returns a function
  with arguments \code{h}, \code{what} and \code{xcond}.
  % (todo:) This case is currently available only for the exact method.
  If \code{xcond} is supplied, then it is fixed to that value and the
  arguments of the returned function are \code{h}, \code{what} and
  \code{'...'}. The dots argument is currently used in the case of the
  simulation method, see below.

  Let \code{f} be the function returned by \code{multiStep_dist}.
  Argument \code{h} is the required prediction horizon and can be a
  number in the interval \eqn{[1,maxh]}. Argument \code{what} is the
  required property of the predictive distribution for lag
  \code{h}. If \code{what} is a function, it is applied to the simulated
  sample for the requested horizon (currently available only for
  the simulation method). If \code{what} is a character string, the
  corresponding property of the predictive distribution for horizon
  \code{h} is returned.
  Currently possible values for \code{what} are:

  \describe{
    \item{"pdf"}{the probability density function.}
    \item{"cdf"}{the cumulative distribution function.}
    \item{"location"}{the location (conditional mean).}
    \item{"variance"}{the conditional variance, a.k.a (squared) volatility.}
    \item{"sd"}{the conditional standard deviation, a.k.a volatility.}
    \item{"skewness"}{the conditional skewness.}
    \item{"kurtosis"}{the conditional kurtosis.}
  }
  Note that \code{what = "pdf"} and \code{what = "cdf"} return functions
  even in the simulation case. For "pdf" the function is constructed
  using \code{density} and the \code{"..."} arguments passed to \code{f} will
  be passed on to \code{density} if finer control is needed.

  If \code{what} is none of the above, the raw object is returned
  currently (but this may change).

  }

}
\value{
  a function as described in sections \sQuote{Details} and \sQuote{Methods}
}
\section{Methods}{

  The Details section gives a rather detailed description of the
  function, so the descriptions below are brief.
  
  \describe{

    \item{\code{signature(model = "MixAR", maxh = "numeric", N = "numeric",
	xcond = "numeric")}}{ 
      Non-missing \code{N} requests the simulation method. The
      predictive distribution is approximated by simulating \code{N}
      of future paths up to horizon \code{maxh} and using a
      non-parametric estimate. Arguments \code{"..."} are passed to
      \code{density} to allow finer control.
    }

    \item{\code{signature(model = "MixARGaussian", maxh = "numeric", N = "missing", 
	xcond = "missing")}}{
      Computes the predictive distribution using the exact method.
      Returns a function with arguments \code{h}, \code{what} and \code{xcond}.
    }

    \item{\code{signature(model = "MixARGaussian", maxh = "numeric", N = "missing",
	xcond = "ANY")}}{
      Computes the predictive distribution using the exact method.
      Returns a function with arguments \code{h} and \code{what}.
      (i.e., \code{xcond} is fixed to the supplied argument \code{xcond}).
    }
  }
}
\references{
  \insertAllCited{}
}
\author{Georgi N. Boshnakov}
\seealso{
  \code{\link{predict_coef}}
}
\examples{
## exact method, without xcond
dist <- multiStep_dist(exampleModels$WL_ibm, maxh = 3)

tfpdf <- dist(3, "pdf", xcond = c(560, 600)) # xcond is argument to 'dist' here
tfcdf <- dist(3, "cdf", xcond = c(560, 600))
## plot the pdf (gbutils::plotpdf determines suitable range automatically)
gbutils::plotpdf(tfpdf, cdf = tfcdf)

args(dist(3, "pdf", xcond = c(500, 600)))  # x

## use a simulation method with N = 1000
tf  <- multiStep_dist(exampleModels$WL_ibm, maxh = 3, N = 1000, xcond = c(560, 600))
args(tf) # (h, what, ...)

## the exact method may also be used with fixed xcond:
tfe <- multiStep_dist(exampleModels$WL_ibm, maxh = 3, xcond = c(560, 600))

## get pdf and cdf for horizon 3
tfepdf <- tfe(3, "pdf")
tfecdf <- tfe(3, "cdf")
## plot the pdf
gbutils::plotpdf(tfepdf, cdf = tfecdf)

tf(3, "location")

tf(1, "location")
mix_location(exampleModels$WL_ibm, xcond = c(560, 600))

## larger simulation gives better approximation, in general
tf <- multiStep_dist(exampleModels$WL_ibm, maxh = 3, N = 10000, xcond = c(560, 600))
tf(1, "location")

tf1000pdf <- tf(3, "pdf")
tf1000cdf <- tf(3, "cdf")
gbutils::plotpdf(tf1000pdf, cdf = tf1000cdf)

## plot the exact and simulated pdf's together for comparison
gbutils::plotpdf(tfepdf, cdf = tfecdf)
curve(tf1000pdf, add = TRUE, col = "red")

## get the raw data
tfs <- tf(1, "sampled")
apply(tfs, 2, mean) # location for lags from 1 to maxh (here 3)

tf(1, "location")
tf(1, "variance")
tf(1, "sd")
mix_variance(exampleModels$WL_ibm, xcond = c(560, 600))
sqrt(mix_variance(exampleModels$WL_ibm, xcond = c(560, 600)))

mix_kurtosis(exampleModels$WL_ibm, xcond = c(359, 200))
mix_kurtosis(exampleModels$WL_ibm, xcond = c(359, 400))
}
\keyword{methods}
\keyword{MixAR}
\keyword{prediction}
