% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ci.mean.w.R
\name{ci.mean.w}
\alias{ci.mean.w}
\title{Within-Subject Confidence Interval for the Arithmetic Mean}
\usage{
ci.mean.w(..., data = NULL, adjust = TRUE,
          alternative = c("two.sided", "less", "greater"),
          conf.level = 0.95, na.omit = TRUE, digits = 2,
          as.na = NULL, write = NULL, append = TRUE, check = TRUE,
          output = TRUE)
}
\arguments{
\item{...}{a matrix or data frame with numeric variables representing
the levels of the within-subject factor, i.e., data are
specified in wide-format (i.e., multivariate person level
format). Alternatively, an expression indicating the variable
names in \code{data} e.g., \code{ci.mean.w(x1, x2, x3, data = dat)}.
Note that the operators
\code{.}, \code{+}, \code{-}, \code{~}, \code{:}, \code{::},
and \code{!} can also be used to select variables, see 'Details'
in the \code{\link{df.subset}} function.}

\item{data}{a data frame when specifying one or more variables in the
argument \code{...}. Note that the argument is \code{NULL}
when specifying a matrix or data frame
for the argument \code{...}.}

\item{adjust}{logical: if \code{TRUE} (default), difference-adjustment
for the Cousineau-Morey within-subject confidence intervals
is applied.}

\item{alternative}{a character string specifying the alternative hypothesis,
must be one of \code{"two.sided"} (default), \code{"greater"}
or \code{"less"}.}

\item{conf.level}{a numeric value between 0 and 1 indicating the confidence
level of the interval.}

\item{na.omit}{logical: if \code{TRUE} (default), incomplete cases are removed
before conducting the analysis (i.e., listwise deletion).}

\item{digits}{an integer value indicating the number of decimal places
to be used.}

\item{as.na}{a numeric vector indicating user-defined missing values,
i.e. these values are converted to \code{NA} before
conducting the analysis.}

\item{write}{a character string naming a text file with file extension
\code{".txt"} (e.g., \code{"Output.txt"}) for writing the
output into a text file.}

\item{append}{logical: if \code{TRUE} (default), output will be appended
to an existing text file with extension \code{.txt} specified
in \code{write}, if \code{FALSE} existing text file will be
overwritten.}

\item{check}{logical: if \code{TRUE} (default), argument specification is checked.}

\item{output}{logical: if \code{TRUE} (default), output is shown on the console.}
}
\value{
Returns an object of class \code{misty.object}, which is a list with following
entries:
  \item{\code{call}}{function call}
  \item{\code{type}}{type of analysis}
  \item{\code{data}}{data frame used for the current analysis}
  \item{\code{args}}{specification of function arguments}
  \item{\code{result}}{result table}
}
\description{
This function computes difference-adjusted Cousineau-Morey within-subject
confidence interval for the arithmetic mean.
}
\details{
The Cousineau within-subject confidence interval (CI, Cousineau, 2005) is an
alternative to the Loftus-Masson within-subject CI (Loftus & Masson, 1994)
that does not assume sphericity or homogeneity of covariances. This approach
removes individual differences by normalizing the raw scores using
participant-mean centering and adding the grand mean back to every score:

\deqn{Y_{ij}^{'} = Y_{ij} - \hat{\mu}_{i} + \hat{\mu}_{grand}}

where \eqn{Y_{ij}^{'}} is the score of the \eqn{i}th participant in condition
\eqn{j} (for \eqn{i = 1} to \eqn{n}), \eqn{\hat{\mu}_{i}} is the mean of
participant \eqn{i} across all \eqn{J} levels (for \eqn{j = 1} to \eqn{J}),
and \eqn{\hat{\mu}_{grand}} is the grand mean.

Morey (2008) pointed out that Cousineau's (2005) approach produces intervals
that are consistently too narrow due to inducing a positive covariance
between normalized scores within a condition introducing bias into the
estimate of the sample variances. The degree of bias is proportional to the
number of means and can be removed by rescaling the confidence interval by
a factor of \eqn{\sqrt{J - 1}/J}:

\deqn{\hat{\mu}_j \pm t_{n - 1, 1 - \alpha/2} \sqrt{\frac{J}{J - 1}} \hat{\sigma}^{'}_{{\hat{\mu}}_j}}

where \eqn{\hat{\sigma}^{'}_{{\mu}_j}} is the standard error of the mean computed
from the normalized scores of he \eqn{j}th factor level.

Baguley (2012) pointed out that the Cousineau-Morey interval is larger than
that for a difference in means by a factor of \eqn{\sqrt{2}} leading to a
misinterpretation of these intervals that overlap of 95\% confidence intervals
around individual means is indicates that a 95\% confidence interval for the
difference in means would include zero. Hence, following adjustment to the
Cousineau-Morey interval was proposed:

\deqn{\hat{\mu}_j \pm \frac{\sqrt{2}}{2} (t_{n - 1, 1 - \alpha/2} \sqrt{\frac{J}{J - 1}} \hat{\sigma}^{'}_{{\hat{\mu}}_j})}

The adjusted Cousineau-Morey interval is informative about the pattern of
differences between means and is computed by default (i.e., \code{adjust = TRUE}).
}
\examples{
dat <- data.frame(time1 = c(3, 2, 1, 4, 5, 2, 3, 5, 6, 7),
                  time2 = c(4, 3, 6, 5, 8, 6, 7, 3, 4, 5),
                  time3 = c(1, 2, 2, 3, 6, 5, 1, 2, 4, 6))

# Example 1: Difference-adjusted Cousineau-Morey confidence intervals
ci.mean.w(dat)

# Example 1: Alternative specification using the 'data' argument
ci.mean.w(., data = dat)

# Example 2: Cousineau-Morey confidence intervals
ci.mean.w(dat, adjust = FALSE)

\dontrun{
# Example 3: Write results into a text file
ci.mean.w(dat, write = "WS_Confidence_Interval.txt")}
}
\references{
Baguley, T. (2012). Calculating and graphing within-subject confidence intervals
for ANOVA. \emph{Behavior Research Methods, 44}, 158-175.
https://doi.org/10.3758/s13428-011-0123-7

Cousineau, D. (2005) Confidence intervals in within-subject designs: A simpler
solution to Loftus and Masson’s Method. \emph{Tutorials in Quantitative Methods
for Psychology, 1}, 42–45.  https://doi.org/10.20982/tqmp.01.1.p042

Loftus, G. R., and Masson, M. E. J. (1994). Using confidence intervals in
within-subject designs. \emph{Psychonomic Bulletin and Review, 1}, 476–90.
https://doi.org/10.3758/BF03210951

Morey, R. D. (2008). Confidence intervals from normalized data: A correction
to Cousineau. \emph{Tutorials in Quantitative Methods for Psychology, 4}, 61–4.
https://doi.org/10.20982/tqmp.01.1.p042
}
\seealso{
\code{\link{aov.w}}, \code{\link{test.z}}, \code{\link{test.t}},
\code{\link{ci.mean.diff}},' \code{\link{ci.median}}, \code{\link{ci.prop}},
\code{\link{ci.var}}, \code{\link{ci.sd}}, \code{\link{descript}}
}
\author{
Takuya Yanagida \email{takuya.yanagida@univie.ac.at}
}
