% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/AKMCS.R
\name{AKMCS}
\alias{AKMCS}
\title{Active learning reliability method combining Kriging and Monte Carlo
Simulation}
\usage{
AKMCS(
  dimension,
  lsf,
  N = 5e+05,
  N1 = 10 * dimension,
  Nmax = 200,
  Nmin = 2,
  X = NULL,
  y = NULL,
  failure = 0,
  precision = 0.05,
  bayesian = TRUE,
  compute.PPP = FALSE,
  meta_model = NULL,
  kernel = "matern5_2",
  learn_each_train = TRUE,
  crit_min = 2,
  lower.tail = TRUE,
  limit_fun_MH = NULL,
  failure_MH = 0,
  sampling_strategy = "MH",
  first_DOE = "Gaussian",
  seeds = NULL,
  seeds_eval = limit_fun_MH(seeds),
  burnin = 30,
  plot = FALSE,
  limited_plot = FALSE,
  add = FALSE,
  output_dir = NULL,
  verbose = 0
)
}
\arguments{
\item{dimension}{dimension of the input space.}

\item{lsf}{the function defining the failure/safety domain.}

\item{N}{Monte-Carlo population size.}

\item{N1}{size of the first DOE.}

\item{Nmax}{maximum number of calls to the LSF.}

\item{Nmin}{minimum number of calls during enrichment step.}

\item{X}{coordinates of already known points.}

\item{y}{value of the LSF on these points.}

\item{failure}{failure threshold.}

\item{precision}{maximum desired cov on the Monte-Carlo estimate.}

\item{bayesian}{estimate the conditional expectation E_X [ P[meta(X)<failure] ].}

\item{compute.PPP}{to simulate a Poisson process at each iteration to estimate
the conditional expectation and the SUR criteria based on the conditional
variance: h (average probability of misclassification at level \code{failure})
and I (integral of h over the whole interval [failure, infty))}

\item{meta_model}{provide here a kriging metamodel from km if wanted.}

\item{kernel}{specify the kernel to use for km.}

\item{learn_each_train}{specify if kernel parameters are re-estimated at each train.}

\item{crit_min}{minimum value of the criteria to be used for refinement.}

\item{lower.tail}{as for pxxxx functions, TRUE for estimating P(lsf(X) < failure), FALSE
for P(lsf(X) > failure)}

\item{limit_fun_MH}{define an area of exclusion with a limit function.}

\item{failure_MH}{the theshold for the limit_fun_MH function.}

\item{sampling_strategy}{either MH for Metropolis-Hastings of AR for accept-reject.}

\item{first_DOE}{either Gaussian or Uniform, to specify the population on which
clustering is done. Set to "No" for no initial DoE (use together with a first DoE
given in \code{X} for instance).}

\item{seeds}{if some points are already known to be in the appropriate subdomain.}

\item{seeds_eval}{value of the metamodel on these points.}

\item{burnin}{burnin parameter for MH.}

\item{plot}{set to TRUE for a full plot, ie refresh at each iteration.}

\item{limited_plot}{set to TRUE for a final plot with final DOE, metamodel and LSF.}

\item{add}{if plots are to be added to a current device.}

\item{output_dir}{if plots are to be saved in jpeg in a given directory.}

\item{verbose}{either 0 for almost no output, 1 for medium size output and 2 for all
outputs.}
}
\value{
An object of class \code{list} containing the failure probability and some
more outputs as described below:
\item{p}{the estimated failure probability.}
\item{cov}{the coefficient of variation of the Monte-Carlo probability estimate.}
\item{Ncall}{the total number of calls to the \code{lsf}.}
\item{X}{the final learning database, ie. all points where \code{lsf} has
been calculated.}
\item{y}{the value of the \code{lsf} on the learning database.}
\item{h}{the sequence of the estimated relative SUR criteria.}
\item{I}{the sequence of the estimated integrated SUR criteria.}
\item{meta_fun}{the metamodel approximation of the \code{lsf}. A call output is a
list containing the value and the standard deviation.}
\item{meta_model}{the final metamodel. An S4 object from \pkg{DiceKriging}. Note
that the algorithm enforces the problem to be the estimation of P[lsf(X)<failure]
and so using \sQuote{predict} with this object will return inverse values if
\code{lower.tail==FALSE}; in this scope prefer using directly \code{meta_fun} which
handles this possible issue.}
\item{points}{points in the failure domain according to the metamodel.}
\item{meta_eval}{evaluation of the metamodel on these points.}
\item{z_meta}{if \code{plot}==TRUE, the evaluation of the metamodel on the plot grid.}
}
\description{
Estimate a failure probability with the AKMCS method.
}
\details{
AKMCS strategy is based on a original Monte-Carlo population which
is classified
with a kriging-based metamodel. This means that no sampling is done during
refinements
steps. Indeed, it tries to classify this Monte-Carlo population with a
confidence greater
than a given value, for instance \sQuote{distance} to the failure should be
greater than
\code{crit_min} standard deviation.

Thus, while this criterion is not verified, the point minimizing it is added to
the learning database and then evaluated.

Finally, once all points are classified or when the maximum number of calls
has been reached, crude Monte-Carlo is performed. A final test controlling
the size of this population regarding the targeted coefficient of variation
is done; if it is too small then a new population of sufficient size
(considering ordre of magnitude of found probability) is generated, and
algorithm run again.
}
\note{
Problem is supposed to be defined in the standard space. If not,
use \code{\link{UtoX}} to do so. Furthermore, each time a set of vector
is defined as a matrix, \sQuote{nrow} = \code{dimension} and
\sQuote{ncol} = number of vector to be consistent with \code{as.matrix}
transformation of a vector.

Algorithm calls lsf(X) (where X is a matrix as defined previously) and
expects a vector in return. This allows the user to optimise the computation
of a batch of points, either by vectorial computation, or by the use of
external codes (optimised C or C++ codes for example) and/or parallel
computation; see examples in \link{MonteCarlo}.
}
\examples{
\dontrun{
res = AKMCS(dimension=2,lsf=kiureghian,plot=TRUE)

#Compare with crude Monte-Carlo reference value
N = 500000
dimension = 2
U = matrix(rnorm(dimension*N),dimension,N)
G = kiureghian(U)
P = mean(G<0)
cov = sqrt((1-P)/(N*P))
}

#See impact of kernel choice with serial function from Waarts:
waarts = function(u) {
  u = as.matrix(u)
  b1 = 3+(u[1,]-u[2,])^2/10 - sign(u[1,] + u[2,])*(u[1,]+u[2,])/sqrt(2)
  b2 = sign(u[2,]-u[1,])*(u[1,]-u[2,])+7/sqrt(2)
  val = apply(cbind(b1, b2), 1, min)
}

\dontrun{
res = list()
res$matern5_2 = AKMCS(2, waarts, plot=TRUE)
res$matern3_2 = AKMCS(2, waarts, kernel="matern3_2", plot=TRUE)
res$gaussian  = AKMCS(2, waarts, kernel="gauss", plot=TRUE)
res$exp       = AKMCS(2, waarts, kernel="exp", plot=TRUE)

#Compare with crude Monte-Carlo reference value
N = 500000
dimension = 2
U = matrix(rnorm(dimension*N),dimension,N)
G = waarts(U)
P = mean(G<0)
cov = sqrt((1-P)/(N*P))
}

}
\references{
\itemize{
\item
B. Echard, N. Gayton, M. Lemaire:\cr
\emph{AK-MCS : an Active learning reliability method combining Kriging and
Monte Carlo Simulation}\cr
Structural Safety, Elsevier, 2011.\cr

\item
B. Echard, N. Gayton, M. Lemaire and N. Relun:\cr
\emph{A combined Importance Sampling and Kriging reliability method for
small failure probabilities with time-demanding numerical models}\cr
Reliability Engineering \& System Safety,2012\cr

\item
B. Echard, N. Gayton and A. Bignonnet:\cr
\emph{A reliability analysis method for fatigue design}\cr
International Journal of Fatigue, 2014\cr
}
}
\seealso{
\code{\link{SubsetSimulation}}
\code{\link{MonteCarlo}}
\code{\link{MetaIS}}
\code{\link[DiceKriging]{km}} (in package \pkg{DiceKriging})
}
\author{
Clement WALTER \email{clementwalter@icloud.com}
}
