# mirai ------------------------------------------------------------------------

#' Daemon Instance
#'
#' Starts up an execution daemon to receive [mirai()] requests. Awaits data,
#' evaluates an expression in an environment containing the supplied data,
#' and returns the value to the host caller. Daemon settings may be controlled
#' by [daemons()] and this function should not need to be invoked directly,
#' unless deploying manually on remote resources.
#'
#' The network topology is such that daemons dial into the host or dispatcher,
#' which listens at the `url` address. In this way, network resources may be
#' added or removed dynamically and the host or dispatcher automatically
#' distributes tasks to all available daemons.
#'
#' @param url the character host or dispatcher URL to dial into, including the
#'   port to connect to, e.g. 'tcp://hostname:5555' or
#'   'tls+tcp://10.75.32.70:5555'.
#' @param dispatcher logical value, which should be set to TRUE if using
#'   dispatcher and FALSE otherwise.
#' @param ... reserved, but not currently used.
#' @param asyncdial whether to perform dials asynchronously. The default FALSE
#'   will error if a connection is not immediately possible (for instance if
#'   [daemons()] has yet to be called on the host, or the specified port is not
#'   open etc.). Specifying TRUE continues retrying (indefinitely) if not
#'   immediately successful, which is more resilient but can mask potential
#'   connection issues.
#' @param autoexit logical value, whether the daemon should exit automatically
#'   when its socket connection ends. By default, the process ends immediately
#'   when the host process ends. Supply `NA` to have a daemon complete any tasks
#'   in progress before exiting (see 'Persistence' section below).
#' @param cleanup logical value, whether to perform cleanup of the global
#'   environment and restore attached packages and options to an initial state
#'   after each evaluation.
#' @param output logical value, to output generated stdout / stderr if TRUE, or
#'   else discard if FALSE. Specify as TRUE in the `...` argument to [daemons()]
#'   or [launch_local()] to provide redirection of output to the host process
#'   (applicable only for local daemons).
#' @param idletime integer milliseconds maximum time to wait for a task (idle
#'   time) before exiting.
#' @param walltime integer milliseconds soft walltime (time limit) i.e. the
#'   minimum amount of real time elapsed before exiting.
#' @param maxtasks integer maximum number of tasks to execute (task limit)
#'   before exiting.
#' @param tlscert required for secure TLS connections over 'tls+tcp://'.
#'   **Either** the character path to a file containing X.509 certificate(s) in
#'   PEM format, comprising the certificate authority certificate chain starting
#'   with the TLS certificate and ending with the CA certificate, **or** a
#'   length 2 character vector comprising (i) the certificate authority
#'   certificate chain and (ii) the empty string `""`.
#' @param rs the initial value of .Random.seed. This is set automatically using
#'   L'Ecuyer-CMRG RNG streams generated by the host process if applicable, and
#'   should not be independently supplied.
#'
#' @return Invisibly, an integer exit code: 0L for normal termination, and a
#'   positive value if a self-imposed limit was reached: 1L (idletime), 2L
#'   (walltime), 3L (maxtasks).
#'
#' @section Persistence:
#'
#' The `autoexit` argument governs persistence settings for the daemon. The
#' default `TRUE` ensures that it exits as soon as its socket connection with
#' the host process drops. A 200ms grace period allows the daemon process to
#' exit normally, after which it will be forcefully terminated.
#'
#' Supplying `NA` ensures that a daemon always exits cleanly after its socket
#' connection with the host drops. This means that it can temporarily outlive
#' this connection, but only to complete any task that is currently in progress.
#' This can be useful if the daemon is performing a side effect such as writing
#' files to disk, with the result not being required back in the host process.
#'
#' Setting to `FALSE` allows the daemon to persist indefinitely even when there
#' is no longer a socket connection. This allows a host session to end and a new
#' session to connect at the URL where the daemon is dialed in. Daemons must be
#' terminated with `daemons(NULL)` in this case instead of `daemons(0)`. This
#' sends explicit exit signals to all connected daemons.
#'
#' @export
#'
daemon <- function(
  url,
  dispatcher = TRUE,
  ...,
  asyncdial = FALSE,
  autoexit = TRUE,
  cleanup = TRUE,
  output = FALSE,
  idletime = Inf,
  walltime = Inf,
  maxtasks = Inf,
  tlscert = NULL,
  rs = NULL
) {
  cv <- cv()
  sock <- socket(if (dispatcher) "poly" else "rep")
  on.exit({
    reap(sock)
    `[[<-`(., "sock", NULL)
    `[[<-`(., "otel_span", NULL)
  })
  `[[<-`(., "sock", sock)
  pipe_notify(sock, cv, remove = TRUE, flag = flag_value(autoexit))
  if (length(tlscert)) {
    tlscert <- tls_config(client = tlscert)
  }
  dial_sync_socket(sock, url, autostart = asyncdial || NA, tls = tlscert)
  `[[<-`(., "otel_span", otel_span("daemon connect", url))

  if (!output) {
    devnull <- file(nullfile(), open = "w", blocking = FALSE)
    sink(file = devnull)
    sink(file = devnull, type = "message")
  }
  xc <- 0L
  task <- 1L
  timeout <- if (idletime > walltime) {
    walltime
  } else if (is.finite(idletime)) {
    idletime
  }
  maxtime <- if (is.finite(walltime)) mclock() + walltime else FALSE

  if (dispatcher) {
    aio <- recv_aio(sock, mode = 1L, cv = cv)
    if (wait(cv)) {
      bundle <- collect_aio(aio)
      `[[<-`(globalenv(), ".Random.seed", if (is.numeric(rs)) as.integer(rs) else bundle[[1L]])
      if (is.list(bundle[[2L]])) {
        `opt<-`(sock, "serial", bundle[[2L]])
      }
      snapshot()
      repeat {
        aio <- recv_aio(sock, mode = 1L, timeout = timeout, cv = cv)
        wait(cv) || break
        m <- collect_aio(aio)
        is.integer(m) &&
          {
            m == 5L || next
            xc <- 1L
            break
          }
        (task >= maxtasks || maxtime && mclock() >= maxtime) &&
          {
            marked(send(sock, eval_mirai(m, sock), mode = 1L, block = TRUE))
            aio <- recv_aio(sock, mode = 8L, cv = cv)
            xc <- 2L + (task >= maxtasks)
            wait(cv)
            break
          }
        send(sock, eval_mirai(m, sock), mode = 1L, block = TRUE)
        if (cleanup) {
          do_cleanup()
        }
        task <- task + 1L
      }
    }
  } else {
    if (is.numeric(rs)) {
      `[[<-`(globalenv(), ".Random.seed", as.integer(rs))
    }
    snapshot()
    repeat {
      ctx <- .context(sock)
      aio <- recv_aio(ctx, mode = 1L, timeout = timeout, cv = cv)
      wait(cv) || break
      m <- collect_aio(aio)
      is.integer(m) &&
        {
          xc <- 1L
          break
        }
      (task >= maxtasks || maxtime && mclock() >= maxtime) &&
        {
          marked(send(ctx, eval_mirai(m), mode = 1L, block = TRUE))
          xc <- 2L + (task >= maxtasks)
          wait(cv)
          break
        }
      send(ctx, eval_mirai(m), mode = 1L, block = TRUE)
      if (cleanup) {
        do_cleanup()
      }
      task <- task + 1L
    }
  }

  if (!output) {
    sink(type = "message")
    sink()
    close.connection(devnull)
  }
  otel_span("daemon disconnect", url, links = list(.[["otel_span"]]))
  invisible(xc)
}

#' dot Daemon
#'
#' Ephemeral executor for the remote process. User code must not call this.
#' Consider `daemon(maxtasks = 1L)` instead.
#'
#' @inheritParams daemon
#'
#' @return Logical TRUE or FALSE.
#'
#' @noRd
#'
.daemon <- function(url) {
  cv <- cv()
  sock <- socket("rep")
  on.exit(reap(sock))
  pipe_notify(sock, cv, remove = TRUE, flag = tools::SIGTERM)
  dial(sock, url = url, autostart = NA, fail = 2L)
  `[[<-`(., "sock", sock)
  m <- recv(sock, mode = 1L, block = TRUE)
  marked(send(sock, eval_mirai(m), mode = 1L, block = TRUE)) || wait(cv)
}

# internals --------------------------------------------------------------------

eval_mirai <- function(._mirai_., sock = NULL) {
  withRestarts(
    withCallingHandlers(
      {
        if (length(sock)) {
          cancel <- recv_aio(sock, mode = 8L, cv = substitute())
          on.exit(stop_aio(cancel))
        }
        list2env(._mirai_.[["._globals_."]], envir = globalenv())
        sock <- otel_eval_span(._mirai_.[["._otel_."]])
        eval(._mirai_.[["._expr_."]], envir = ._mirai_., enclos = globalenv())
      },
      error = function(cnd) {
        otel_set_span_error(sock, "miraiError")
        invokeRestart("mirai_error", cnd, sys.calls())
      },
      interrupt = function(cnd) {
        otel_set_span_error(sock, "miraiInterrupt")
        invokeRestart("mirai_interrupt")
      }
    ),
    mirai_error = mk_mirai_error,
    mirai_interrupt = mk_interrupt_error
  )
}

dial_sync_socket <- function(sock, url, autostart = NA, tls = NULL) {
  cv <- cv()
  pipe_notify(sock, cv, add = TRUE)
  dial(sock, url = url, autostart = autostart, tls = tls, fail = 2L)
  wait(cv)
  pipe_notify(sock, NULL, add = TRUE)
}

do_cleanup <- function() {
  vars <- names(globalenv())
  rm(list = vars[!vars %in% .[["vars"]]], envir = globalenv())
  new <- search()
  lapply(new[!new %in% .[["se"]]], detach, character.only = TRUE)
  options(.[["op"]])
}

snapshot <- function() {
  `[[<-`(`[[<-`(`[[<-`(., "op", .Options), "se", search()), "vars", names(globalenv()))
}

flag_value <- function(autoexit) {
  is.na(autoexit) && return(TRUE)
  autoexit && return(tools::SIGTERM)
}

marked <- function(expr) {
  .mark()
  on.exit(.mark(FALSE))
  expr
}
