
mhde.grid<- function( Xmin, Xrange, MinScaEst, MaxScaEst, NGauss, Nsubs, WG, Xint, GpdfSqrt, Xdel2 ) {
  #-------------------------------------------------------------------------------------------------------------
  #
  #'mhde.grid
  #
  #'@description
  #'  This function evalutes the Hellinger distance on a 21 by 21 grid of location and scale values.  The
  #'  purpose is to find an initial (location,scale) pair where the Newton-Rhapson method will converge.
  #'  The grid is defined as equally spaced divisions of the reasonable range in location and scale
  #
  # History:
  #    Paul W. Eslinger : 17 Aug 2015 : Original source
  #
  # Parameters:
  #'   @param Xmin        Minimum sample value
  #'   @param Xrange      Range of the sample data
  #'   @param MinScaEst   Minimum reasonale scale estimate
  #'   @param MaxScaEst   Maximum reasonable scale estimate
  #'   @param NGauss      Number of Gauss integration intervals
  #'   @param Nsubs       Number of substeps in each Gauss integration interval
  #'   @param WG          Vector of weights for the Gauss-Legendre integration
  #'   @param Xint        Vector of X values for evaluating the Hellinger distance integral
  #'   @param GpdfSqrt    Vector of the square root of the data-based kernel density function
  #'   @param Xdel2       Delta step used in setting up the integration grid.  Needed for scaling purposes.
  #
  # Returned values:
  #'   @return List containing
  #'   \describe{
  #'   \item{BestLoc} {Location value for the minimum Hellinger distance on the grid}
  #'   \item{BestSca} {Scale value for the minimum Hellinger distance on the grid}
  #'   }
  #
  #-------------------------------------------------------------------------------------------------------------
  #
  # Define the number of grid points and the resulting spacing
  Nloc <- 21
  Nsca <- 21
  Dloc <- Xrange/(Nloc+1)
  Dsca <- (MaxScaEst-MinScaEst)/(Nsca+1)
  # Loop over grid points and calculate the minimum Hellinger distance
  MinHdis <- 9999.99
  for( Iloc in 1:Nloc ) {
  # Define a location
    Hloc <- Xmin + Iloc*Dloc
    for( Isca in 1:Nsca ) {
  #   Define a scale
      Hsca <- MinScaEst + Isca*Dsca
  #   Define the normal density for this grid location and scale
      Npdf <- dnorm(Xint,Hloc,Hsca)
      NpdfSqrt <- sqrt( Npdf )
  #   Calculate the Hellinger distance for this starting point
      Hdis <- mhde.hdis( NGauss, Nsubs, WG, GpdfSqrt, NpdfSqrt, Xdel2 )
  #   Set new initial values to the minimum distance over the grid search
      if( Hdis < MinHdis ) {
        BestLoc <- Hloc
        BestSca <- Hsca
        MinHdis <- Hdis
       }
    }
  }
  return(list(BestLoc,BestSca))
}

mhde.hdis <- function( NGauss, Nsubs, WG, GpdfSqrt, NpdfSqrt, Xdel2 ) {
  #-------------------------------------------------------------------------------------------------------------
  #
  #' mhde.hdis
  #
  #'@description
  #'   This function evaluates the Hellinger distance between the kernel-based density function and the normal
  #'   distribution function.  The normal distribution function is defined at a location and scale in the calling
  #'   function.  Both the kernel density and the normal density are provided to this function as square roots.
  #
  # History:
  #    Paul W. Eslinger : 17 Aug 2015 : Original source
  #
  # Parameters:
  #' @param   NGauss      Number of Gauss integration intervals
  #' @param   Nsubs       Number of substeps in each Gauss integration interval
  #' @param   WG          Vector of weights for the Gauss-Legendre integration
  #' @param   GpdfSqrt    Vector of the square root of the data-based kernel density function
  #' @param   NpdfSqrt    Vector of the normal distribution
  #' @param   Xdel2       Delta step used in setting up the integration grid.  Needed for scaling purposes.
  #
  # Returned values:
  #' @return   Hdis        Value of the Hellinger distance
  #
  #-------------------------------------------------------------------------------------------------------------
#
  Hdis <- 0.0
  idx <- 0
  for(i in 1:NGauss ){
    for(j in 1:Nsubs) {
      idx <- idx + 1
      Hdis <- Hdis + WG[j]*GpdfSqrt[idx]*NpdfSqrt[idx]
    }
  }
  Hdis <- Hdis * Xdel2
  Hdis <- sqrt( 1.0 - Hdis )
  return(Hdis)
}


mhde.checkparam <- function( EstLoc, EstSca, MinLocEst, MaxLocEst, MinScaEst, MaxScaEst ) {
  #-------------------------------------------------------------------------------------------------------------
  #
  #'mhde.checkparam
  #
  #'@description
  #'   This function returns a numeric error code that is nonzero if the location or scale estimates are
  #'   outside reasonable rnges.
  #
  # History:
  #   Paul W. Eslinger : 17 Aug 2015 : Original source
  #
  # Parameters:
  #' @param  EstLoc      Location estimate
  #' @param   EstSca      Scale estimate
  #' @param   MinLocEst   Minimum reasonable location estimate
  #' @param   MaxLocEst   Maximum reasonable Location estimate
  #' @param   MinScaEst   Minimum reasonable scale estimate
  #' @param   MaxScaEst   Maximim reasonable scale estimate
  #
  # Returned values:
  #' @return   ierr        Numeric error code (0=reasonable estimates)
  #
  #-------------------------------------------------------------------------------------------------------------
  ierr <- 0
  if( EstLoc < MinLocEst ) ierr <- 16
  if( EstLoc > MaxLocEst ) ierr <- 17
  if( EstSca < MinScaEst ) ierr <- 18
  if( EstSca > MaxScaEst ) ierr <- 19
  return(ierr)
}


mhde.calcest <- function(EstLoc, EstSca, MaxIter, NGauss, Nsubs, Xint, WG, GpdfSqrt, Xdel2, EpsLoc, EpsSca,
                         MinLocEst, MaxLocEst, MinScaEst, MaxScaEst ) {
  #-------------------------------------------------------------------------------------------------------------
  #
  #' mhde.calcest
  #
  #' @description
  #'   This function calculates the location and scale estimates corresponding to the minimum Hellinger
  #'   distance.  A Newton-Rhapson method with analytical derivatives is used.
  #
  # History:
  #    Paul W. Eslinger : 17 Aug 2015 : Original source
  #
  # Parameters:
  #' @param   EstLoc      Location estimate
  #' @param   EstSca      Scale estimate
  #' @param   MaxIter     The maximum number of iterations allowed
  #' @param   NGauss      Number of Gauss integration intervals
  #' @param   Nsubs       Number of substeps in each Gauss integration interval
  #' @param   Xint        Vector of X values for evaluating the Hellinger distance integral
  #' @param   WG          Vector of weights for the Gauss-Legendre integration
  #' @param   GpdfSqrt    Vector of the square root of the data-based kernel density function
  #' @param   Xdel2       Delta step used in setting up the integration grid.  Needed for scaling purposes.
  #' @param   EpsLoc      The location epsilon for convergence
  #' @param   EpsSca      The scale epsilon for convergence
  #' @param   MinLocEst   Minimum reasonable location estimate
  #' @param   MaxLocEst   Maximum reasonable Location estimate
  #' @param   MinScaEst   Minimum reasonable scale estimate
  #' @param   MaxScaEst   Maximim reasonable scale estimate
  #
  #' @return list containing
  #' \describe{
  #'  \item {Hloc}        {Location estimate for the minimized distance}
  #'   \item{Hsca}        {Scale estimate for the minimized distance}
  #'   \item{myiter}      {Number of iterations performed}
  #'   \item{ierr}        {Numeric error code (0=reasonable estimates)}
  #'   }
  #
  #-------------------------------------------------------------------------------------------------------------
  #
  ierr <- 0
  Hloc <- EstLoc
  Hsca <- EstSca
  #
  # Iterative loop on the estimates
  for( i in 1:MaxIter ) {
    myiter <- i
    # Zero the integration sums
    sdotl  <- 0.0
    sdots  <- 0.0
    sdotll <- 0.0
    sdotls <- 0.0
    sdotss <- 0.0
    # Integrate using a composite Gauss-Legendre rule
    Idx <- 0
    for( J in 1:NGauss ){
      for( K in 1:Nsubs ){
        Idx <- Idx + 1
        X   <- Xint[Idx]
        GX  <- GpdfSqrt[Idx]
        parts <- mhde.partial( X, Hloc, Hsca )
        sdotl  <- sdotl  + parts[[1]] * GX * WG[K]
        sdots  <- sdots  + parts[[2]] * GX * WG[K]
        sdotll <- sdotll + parts[[3]] * GX * WG[K]
        sdotls <- sdotls + parts[[4]] * GX * WG[K]
        sdotss <- sdotss + parts[[5]] * GX * WG[K]
      }
    }
    # The nominal multiplicative integration scaling of Xdel2 on sdotl, etc cancels out in update step
    # Start update procedure
    denom <- sdotss*sdotll - sdotls**2
    # Check for a singular Jacobian
    if( abs(denom) < 1.0E-25 ){
      ierr <- 21
      return(list(Hloc,Hsca,myiter,ierr))
    }
    # Compute the delta step for both location and scale
    dell <- -( sdotss*sdotl - sdotls*sdots ) / denom
    dels <- -( sdotll*sdots - sdotls*sdotl ) / denom
    # Negative scale is not allowed
    # Don't let any step move the scale estimate more than half the distance to zero
    delta <- 1.0
    if( (Hsca+dels) <= 0.0 ){ delta <- abs(Hsca/dels)/2.0 }
    # Always take a smaller first step
    if( i == 1 ) { delta <- min(delta,0.5) }
    # Update the estimates
    Hloc <- Hloc + delta * dell
    Hsca <- Hsca + delta * dels
    # Check if the values meet reasonable bounds
    ierr <- mhde.checkparam( Hloc, Hsca, MinLocEst, MaxLocEst, MinScaEst, MaxScaEst )
    if( ierr > 0 ) return(list(Hloc,Hsca,myiter,ierr))
    # If converged
    if( abs(dell)<EpsLoc & abs(dels)<EpsSca ) return(list(Hloc,Hsca,myiter,ierr))
  } # End of iteration loop
  # Iteration limit without convergence
  ierr <- 22
  return(list(Hloc,Hsca,myiter,ierr))
}

mhde.partial <- function(X,Mean,Sigma) {
  #-------------------------------------------------------------------------------------------------------------
  #
  #' mhde.partial
  #
  #' @description
  #'   This function evaluates the first and second partial derivatives of the square root of the normal density
  #'   function with respect to location and scale.
  #
  # History:
  #    Paul W. Eslinger : 17 Aug 2015 : Original source
  #
  # Parameters:
  #' @param   X       Location (X value) for evaluating the partial derivative
  #' @param   Mean    Mean of the normal density function
  #' @param   Sigma   Standard deviation of the normal density function
  #
  #' @return a list containing
  #' \describe{
  #'   \item{pl}    {Partial derivative with respect to location}
  #'   \item{ps}    {Partial derivative with respect to scale}
  #'   \item{pll}   {Second partial derivative with respect to location}
  #'   \item{pls}   {Second partial derivative with respect to location and scale}
  #'   \item{pss}   {Second partial derivative with respect to scale}
  #'   }
  #
  #-------------------------------------------------------------------------------------------------------------
  # const = 2**(-5/4) * PI**(-1/4)
  const <- 0.31580938887
  #
  z <- (X-Mean)/Sigma
  z2 <- z * z
  z3 <- z2 * z
  z4 <- z3 * z
  zexp <- exp( -0.25 * z2 )
  esm15 <- Sigma**(-1.5)
  esm25 <- esm15 / Sigma
  # Derivative with respect to location
  pl <- const * esm15 * z * zexp
  # Derivative with respect to scale
  ps <- const * esm15 * (z2-1.0) * zexp
  # Second derivative with respect to location
  pll <- const * esm25 * (0.5*z2-1.0) * zexp
  # Second derivative with respect to location and scale
  pls <- const * esm25 * (-0.25*z+0.5*z3 ) * zexp
  # Second derivative with respect to scale
  pss <- const * esm25 * (1.5-4.0*z2+0.5*z4) * zexp
  return(list(pl,ps,pll,pls,pss))
}

#-------------------------------------------------------------------------------------------------------------
#' Utility function for retrieving the critical value for any sample size and desired \eqn{\alpha} level
#
#' @description
#'   This function calculates the critical value for the minimized Hellinger distance for a given sample size
#'   and probability level for a goodness of fit test for normality.
#'
#'   P values below 0.0001 return the value for p=0.0001
#'   P values above 0.9999 return the value for p=0.9999
#'
#'   This utility function is not used by the other functions.  It is provided as a "curiosity" for users.
#
# History:
#   Paul W. Eslinger : 17 Aug 2015 : Original source
#
# Parameters:
#' @param   Nsize   The sample size is defined using \code{Nsize}.  No default value is set.
#' @param   Plevel  The probability level associated with the critical value is set using \code{Plevel}.  The default value is 0.95.  The \eqn{\alpha} level of the test is 1-\code{Plevel}.
#
# Returned values:
#' @return   pval   Critical value for the minimized Hellinger distance
#'
#'
#' @examples
#' ## example using default alpha level of .05
#' mhde.crit(45)
#'
#' ## example using alpha level of .01
#' mhde.crit(45, 0.99)
#'
#
#' @export
#
#-------------------------------------------------------------------------------------------------------------
mhde.crit <- function(Nsize,Plevel=0.95) {
  #
  # Load the coefficients from a data file
  pcoef <- mhde.loadpval()
  # Explicity define the alpha levels corresponding to the coefficients in the file
  nalpha <- 63
  alpha <- c(0.0001,0.001,0.005,0.01,0.015,0.02,0.025,0.03,0.035,0.04,
             0.045,0.05,0.055,0.06,0.065,0.07,0.075,0.08,0.085,0.09,
             0.095,0.1,0.125,0.15,0.175,0.2,0.25,0.3,0.35,0.4,0.45,0.5,
             0.55,0.6,0.65,0.7,0.75,0.8,0.825,0.85,0.875,0.9,0.905,0.91,
             0.915,0.92,0.925,0.93,0.935,0.94,0.945,0.95,0.955,0.96,
             0.965,0.97,0.975,0.98,0.985,0.99,0.995,0.999,0.9999)
  #
  pcrit <- numeric(nalpha)
  # If the sample size is 40 or smaller, use row Nsize-4 for the explicit coeffs
  if( Nsize <= 40 ) {
    nr <- Nsize-4
    for( i in 1:nalpha ){ pcrit[i] <- pcoef[nr,i] }
  } else {
    # Sample size greater than 40 pick the rows for the intercept and slope coefficients
    pintercept <- numeric(nalpha)
    pslope <- numeric(nalpha)
    # Pick the row for the intercept
    nr <- 0
    if( Nsize >=   41 & Nsize <=   60 ) nr <- 37
    if( Nsize >=   61 & Nsize <=   80 ) nr <- 38
    if( Nsize >=   81 & Nsize <=  100 ) nr <- 39
    if( Nsize >=  101 & Nsize <=  150 ) nr <- 40
    if( Nsize >=  151 & Nsize <=  200 ) nr <- 41
    if( Nsize >=  201 & Nsize <=  300 ) nr <- 42
    if( Nsize >=  301 & Nsize <=  400 ) nr <- 43
    if( Nsize >=  401 & Nsize <=  600 ) nr <- 44
    if( Nsize >=  601 & Nsize <=  800 ) nr <- 45
    if( Nsize >=  801 & Nsize <= 1000 ) nr <- 46
    if( Nsize >= 1001 & Nsize <= 1500 ) nr <- 47
    if( Nsize >= 1501 & Nsize <= 2000 ) nr <- 48
    if( Nsize >= 2001 & Nsize <= 3000 ) nr <- 49
    if( Nsize >= 3001 & Nsize <= 4000 ) nr <- 50
    if( Nsize >= 4001 & Nsize <= 5000 ) nr <- 51
    if( Nsize >= 5001 & Nsize <= 6000 ) nr <- 52
    if( Nsize >= 6001 & Nsize <= 8000 ) nr <- 53
    if( Nsize > 8001 ) nr <- 54
    for( i in 1:nalpha ) pintercept[i] <- pcoef[nr,i]
    # Pick the row for the slope (18 groups of sample sizes)
    nr <- nr + 18
    for( i in 1:nalpha ) pslope[i] <- pcoef[nr,i]
    # Calculate the critical values from intercept and slope (log-log fit)
    for( i in 1:nalpha ) pcrit[i] <- pintercept[i] * Nsize ** pslope[i]
  }
  # Have critical values for all alpha levels for this sample size
  # Now calculate the critical value for the p value using linear interpolation
  if( Plevel <= alpha[1] ) return(pcrit[1])
  if( Plevel >= alpha[nalpha] ) return(pcrit[nalpha])
  for( i in 2:nalpha){
    if( Plevel <= alpha[i] ) {
      Adel <- (Plevel-alpha[i-1])/(alpha[i]-alpha[i-1])
      Cdel <- pcrit[i] - pcrit[i-1]
      pval <- pcrit[i-1] + Cdel*Adel
      return(pval)
    }
  }
}

mhde.loadpval <- function() {
  #-------------------------------------------------------------------------------------------------------------
  #
  #' mhde.loadpval
  #
  #' @description
  #'   Load the coeficient data from which to calculate the p values
  #
  # History:
  #   Paul W. Eslinger :  1 Sep 2015 : Original source
  #
  # Parameters:
  #   None
  #
  #' @return
  #'   pcoef   A matrix of coefficients
  #
  #-------------------------------------------------------------------------------------------------------------
  ctmp <- c(
  0.068768,0.069260,0.070177,0.070900,0.071478,0.071973,0.072421,0.072828,0.073208,
  0.073563,0.073902,0.074223,0.074527,0.074820,0.075100,0.075368,0.075628,0.075881,
  0.076125,0.076359,0.076584,0.076805,0.077795,0.078768,0.079773,0.080808,0.082944,
  0.085148,0.087535,0.090375,0.094229,0.099106,0.105246,0.113042,0.123122,0.136439,
  0.154508,0.179753,0.196043,0.215781,0.240597,0.273833,0.281910,0.290574,0.300073,
  0.309949,0.319304,0.325324,0.329478,0.331887,0.333230,0.334382,0.336194,0.340463,
  0.347939,0.360425,0.382265,0.418823,0.476528,0.480221,0.481625,0.482073,0.484554,
  0.070946,0.071527,0.072608,0.073402,0.074003,0.074503,0.074938,0.075322,0.075669,
  0.075988,0.076281,0.076553,0.076813,0.077055,0.077286,0.077499,0.077707,0.077906,
  0.078100,0.078291,0.078479,0.078664,0.079577,0.080493,0.081433,0.082401,0.084496,
  0.086818,0.089417,0.092341,0.095702,0.099653,0.104484,0.110755,0.119086,0.130080,
  0.144813,0.165012,0.178135,0.194141,0.214123,0.240408,0.246722,0.253405,0.260501,
  0.268027,0.275886,0.283599,0.290491,0.296148,0.300432,0.302942,0.305067,0.307489,
  0.311029,0.317882,0.329682,0.349041,0.386603,0.430627,0.434181,0.436483,0.439434,
  0.059792,0.062558,0.065618,0.067300,0.068469,0.069394,0.070180,0.070877,0.071496,
  0.072063,0.072585,0.073068,0.073518,0.073940,0.074342,0.074724,0.075091,0.075444,
  0.075789,0.076119,0.076440,0.076757,0.078247,0.079655,0.081014,0.082356,0.085020,
  0.087758,0.090736,0.094153,0.098236,0.103268,0.109548,0.117418,0.127323,0.139909,
  0.156152,0.177560,0.191089,0.207456,0.227907,0.253312,0.258833,0.264303,0.269527,
  0.274307,0.278426,0.281339,0.284044,0.287503,0.292333,0.298541,0.306275,0.315878,
  0.327916,0.342948,0.363811,0.392288,0.399501,0.413997,0.491903,0.521524,0.796988,
  0.061364,0.063539,0.066358,0.068151,0.069430,0.070463,0.071337,0.072100,0.072791,
  0.073417,0.073990,0.074528,0.075034,0.075507,0.075961,0.076389,0.076800,0.077193,
  0.077569,0.077929,0.078278,0.078616,0.080146,0.081482,0.082723,0.083928,0.086311,
  0.088797,0.091527,0.094633,0.098256,0.102591,0.107890,0.114497,0.122854,0.133410,
  0.146985,0.164746,0.175919,0.189299,0.205808,0.226785,0.231575,0.236513,0.241581,
  0.246672,0.251681,0.256423,0.260669,0.263926,0.267266,0.271534,0.277112,0.284440,
  0.293702,0.305283,0.320284,0.340857,0.365852,0.374630,0.422481,0.467202,0.627102,
  0.055744,0.058644,0.062168,0.064360,0.065912,0.067148,0.068192,0.069105,0.069934,
  0.070679,0.071367,0.072004,0.072607,0.073173,0.073706,0.074216,0.074703,0.075167,
  0.075613,0.076044,0.076459,0.076859,0.078710,0.080395,0.081966,0.083476,0.086418,
  0.089397,0.092555,0.096046,0.100073,0.104871,0.110739,0.117933,0.126794,0.137757,
  0.151517,0.169258,0.180298,0.193474,0.209521,0.228958,0.233158,0.237318,0.241413,
  0.245333,0.248813,0.252237,0.256159,0.260896,0.266515,0.273105,0.280836,0.289934,
  0.300656,0.313443,0.329086,0.344995,0.355213,0.382787,0.433691,0.519531,0.776109,
  0.055855,0.058908,0.062557,0.064819,0.066434,0.067744,0.068852,0.069830,0.070705,
  0.071501,0.072237,0.072918,0.073555,0.074149,0.074712,0.075247,0.075756,0.076243,
  0.076706,0.077150,0.077580,0.077988,0.079880,0.081553,0.083086,0.084523,0.087263,
  0.089983,0.092832,0.095978,0.099590,0.103849,0.108979,0.115260,0.122972,0.132489,
  0.144414,0.159720,0.169187,0.180296,0.193884,0.210656,0.214412,0.218278,0.222251,
  0.226280,0.230260,0.234073,0.237604,0.241267,0.245609,0.250866,0.257129,0.264534,
  0.273382,0.284017,0.297077,0.313202,0.328836,0.344608,0.399421,0.478652,0.561384,
  0.052491,0.055794,0.059773,0.062258,0.064008,0.065413,0.066609,0.067655,0.068588,
  0.069440,0.070227,0.070955,0.071636,0.072281,0.072888,0.073464,0.074013,0.074543,
  0.075052,0.075541,0.076013,0.076471,0.078573,0.080463,0.082210,0.083867,0.087029,
  0.090138,0.093367,0.096825,0.100715,0.105203,0.110544,0.116984,0.124815,0.134361,
  0.146188,0.161198,0.170439,0.181339,0.194378,0.210017,0.213446,0.216913,0.220411,
  0.223874,0.227263,0.230785,0.234697,0.239149,0.244233,0.250066,0.256754,0.264453,
  0.273401,0.283928,0.296405,0.309841,0.322722,0.346593,0.390338,0.491213,0.669400,
  0.052647,0.055928,0.059889,0.062416,0.064218,0.065677,0.066907,0.067992,0.068959,
  0.069840,0.070654,0.071413,0.072118,0.072777,0.073402,0.073995,0.074562,0.075105,
  0.075623,0.076124,0.076604,0.077069,0.079206,0.081105,0.082841,0.084463,0.087494,
  0.090419,0.093388,0.096549,0.100064,0.104118,0.108918,0.114698,0.121712,0.130242,
  0.140787,0.154133,0.162313,0.171876,0.183326,0.197298,0.200418,0.203610,0.206878,
  0.210219,0.213610,0.216965,0.220418,0.224152,0.228445,0.233421,0.239097,0.245643,
  0.253301,0.262341,0.273223,0.286316,0.300487,0.318362,0.360308,0.442261,0.538721,
  0.050219,0.053694,0.057923,0.060573,0.062442,0.063944,0.065219,0.066345,0.067350,
  0.068258,0.069100,0.069882,0.070613,0.071306,0.071960,0.072577,0.073165,0.073733,
  0.074273,0.074803,0.075311,0.075806,0.078081,0.080119,0.081990,0.083751,0.087080,
  0.090298,0.093556,0.096973,0.100697,0.104907,0.109790,0.115591,0.122550,0.131002,
  0.141359,0.154390,0.162336,0.171624,0.182648,0.195830,0.198734,0.201704,0.204737,
  0.207799,0.210933,0.214242,0.217865,0.221896,0.226409,0.231494,0.237228,0.243758,
  0.251284,0.260073,0.270452,0.282441,0.295638,0.316271,0.354111,0.454418,0.561888,
  0.050376,0.053736,0.057921,0.060608,0.062536,0.064083,0.065389,0.066532,0.067553,
  0.068483,0.069343,0.070133,0.070873,0.071571,0.072233,0.072855,0.073454,0.074032,
  0.074584,0.075114,0.075633,0.076135,0.078414,0.080460,0.082323,0.084072,0.087317,
  0.090400,0.093464,0.096648,0.100081,0.103941,0.108420,0.113721,0.120079,0.127788,
  0.137208,0.149058,0.156262,0.164644,0.174597,0.186540,0.189208,0.191959,0.194793,
  0.197695,0.200666,0.203729,0.206989,0.210547,0.214530,0.218977,0.224038,0.229802,
  0.236388,0.244159,0.253356,0.264488,0.277555,0.295063,0.329523,0.408047,0.513906,
  0.048321,0.052019,0.056412,0.059202,0.061167,0.062737,0.064058,0.065229,0.066272,
  0.067216,0.068088,0.068900,0.069654,0.070369,0.071043,0.071691,0.072311,0.072907,
  0.073478,0.074031,0.074559,0.075078,0.077458,0.079584,0.081549,0.083387,0.086830,
  0.090125,0.093394,0.096768,0.100374,0.104360,0.108892,0.114183,0.120457,0.128047,
  0.137284,0.148815,0.155819,0.163967,0.173573,0.184988,0.187525,0.190108,0.192782,
  0.195518,0.198381,0.201393,0.204686,0.208319,0.212324,0.216751,0.221749,0.227413,
  0.233915,0.241413,0.250269,0.260908,0.273430,0.291779,0.324598,0.408703,0.526885,
  0.048448,0.051986,0.056349,0.059148,0.061137,0.062727,0.064068,0.065241,0.066289,
  0.067233,0.068108,0.068927,0.069691,0.070410,0.071089,0.071736,0.072355,0.072953,
  0.073528,0.074080,0.074620,0.075140,0.077521,0.079648,0.081594,0.083420,0.086798,
  0.089992,0.093122,0.096313,0.099693,0.103398,0.107596,0.112498,0.118305,0.125309,
  0.133874,0.144529,0.150982,0.158481,0.167286,0.177833,0.180163,0.182591,0.185113,
  0.187693,0.190371,0.193185,0.196195,0.199467,0.203081,0.207103,0.211619,0.216749,
  0.222579,0.229400,0.237452,0.247237,0.259224,0.275505,0.305204,0.374342,0.490921,
  0.046914,0.050657,0.055217,0.058054,0.060045,0.061638,0.062991,0.064169,0.065229,
  0.066194,0.067080,0.067894,0.068669,0.069400,0.070092,0.070754,0.071386,0.071993,
  0.072579,0.073147,0.073692,0.074221,0.076668,0.078859,0.080864,0.082743,0.086266,
  0.089605,0.092896,0.096236,0.099748,0.103552,0.107798,0.112693,0.118436,0.125318,
  0.133682,0.144086,0.150396,0.157664,0.166189,0.176284,0.178512,0.180834,0.183216,
  0.185726,0.188333,0.191124,0.194129,0.197393,0.201008,0.204964,0.209418,0.214475,
  0.220208,0.226849,0.234713,0.244133,0.255697,0.272071,0.301062,0.368628,0.494522,
  0.047092,0.050719,0.055196,0.058032,0.060025,0.061617,0.062954,0.064135,0.065188,
  0.066147,0.067035,0.067856,0.068639,0.069362,0.070057,0.070719,0.071352,0.071961,
  0.072548,0.073110,0.073658,0.074185,0.076628,0.078803,0.080803,0.082667,0.086144,
  0.089411,0.092602,0.095808,0.099151,0.102741,0.106724,0.111280,0.116637,0.123055,
  0.130858,0.140575,0.146451,0.153253,0.161193,0.170656,0.172759,0.174955,0.177207,
  0.179563,0.182023,0.184647,0.187443,0.190463,0.193748,0.197448,0.201553,0.206172,
  0.211427,0.217519,0.224690,0.233366,0.244214,0.259097,0.285546,0.345974,0.462188,
  0.045848,0.049690,0.054240,0.057096,0.059107,0.060710,0.062060,0.063241,0.064303,
  0.065264,0.066153,0.066986,0.067768,0.068507,0.069203,0.069868,0.070503,0.071113,
  0.071703,0.072270,0.072823,0.073358,0.075824,0.078041,0.080087,0.081998,0.085560,
  0.088924,0.092208,0.095513,0.098944,0.102613,0.106644,0.111188,0.116501,0.122794,
  0.130422,0.139922,0.145648,0.152240,0.159959,0.169060,0.171090,0.173189,0.175371,
  0.177656,0.180080,0.182663,0.185431,0.188448,0.191746,0.195377,0.199412,0.203917,
  0.209059,0.214963,0.221950,0.230392,0.240878,0.255561,0.281128,0.339934,0.461363,
  0.046126,0.049808,0.054285,0.057102,0.059087,0.060678,0.062026,0.063193,0.064251,
  0.065213,0.066104,0.066924,0.067698,0.068428,0.069123,0.069784,0.070415,0.071028,
  0.071614,0.072182,0.072735,0.073268,0.075729,0.077932,0.079962,0.081856,0.085375,
  0.088684,0.091883,0.095086,0.098384,0.101883,0.105710,0.110011,0.115013,0.120958,
  0.128151,0.137092,0.142478,0.148673,0.155894,0.164479,0.166383,0.168356,0.170432,
  0.172603,0.174899,0.177321,0.179911,0.182714,0.185781,0.189165,0.192908,0.197130,
  0.201944,0.207440,0.213978,0.221848,0.231739,0.245343,0.269089,0.323043,0.427154,
  0.045032,0.048903,0.053489,0.056346,0.058343,0.059931,0.061270,0.062449,0.063502,
  0.064458,0.065349,0.066173,0.066951,0.067686,0.068383,0.069050,0.069689,0.070300,
  0.070893,0.071463,0.072017,0.072552,0.075040,0.077262,0.079306,0.081231,0.084799,
  0.088166,0.091430,0.094718,0.098087,0.101650,0.105512,0.109818,0.114765,0.120590,
  0.127643,0.136385,0.141656,0.147713,0.154750,0.163078,0.164938,0.166873,0.168892,
  0.171013,0.173263,0.175658,0.178216,0.180997,0.184026,0.187358,0.191039,0.195165,
  0.199817,0.205245,0.211573,0.219220,0.228861,0.242361,0.265586,0.318723,0.423873,
  0.045404,0.049098,0.053559,0.056361,0.058323,0.059898,0.061217,0.062378,0.063421,
  0.064371,0.065250,0.066072,0.066844,0.067577,0.068269,0.068931,0.069566,0.070173,
  0.070756,0.071322,0.071873,0.072408,0.074864,0.077074,0.079105,0.081008,0.084543,
  0.087861,0.091068,0.094263,0.097531,0.100965,0.104664,0.108776,0.113472,0.119001,
  0.125693,0.134001,0.139000,0.144711,0.151394,0.159304,0.161073,0.162906,0.164825,
  0.166838,0.168957,0.171214,0.173636,0.176248,0.179094,0.182242,0.185701,0.189588,
  0.194008,0.199060,0.205019,0.212219,0.221315,0.233922,0.255608,0.305211,0.390247,
  0.044415,0.048284,0.052870,0.055679,0.057662,0.059223,0.060549,0.061713,0.062759,
  0.063716,0.064599,0.065423,0.066194,0.066928,0.067624,0.068290,0.068934,0.069544,
  0.070129,0.070697,0.071245,0.071777,0.074249,0.076482,0.078525,0.080448,0.084020,
  0.087374,0.090624,0.093872,0.097189,0.100655,0.104378,0.108483,0.113135,0.118577,
  0.125143,0.133280,0.138156,0.143750,0.150238,0.157913,0.159621,0.161405,0.163283,
  0.165263,0.167364,0.169598,0.171993,0.174559,0.177343,0.180405,0.183810,0.187606,
  0.191913,0.196888,0.202694,0.209755,0.218661,0.231058,0.252436,0.300823,0.383857,
  0.044885,0.048495,0.052970,0.055743,0.057684,0.059242,0.060558,0.061703,0.062738,
  0.063678,0.064541,0.065354,0.066119,0.066845,0.067533,0.068190,0.068815,0.069423,
  0.070003,0.070568,0.071117,0.071646,0.074104,0.076306,0.078333,0.080227,0.083763,
  0.087068,0.090269,0.093445,0.096662,0.100022,0.103607,0.107555,0.112017,0.117223,
  0.123479,0.131245,0.135888,0.141227,0.147428,0.154768,0.156411,0.158116,0.159915,
  0.161818,0.163822,0.165933,0.168208,0.170653,0.173327,0.176240,0.179463,0.183071,
  0.187168,0.191878,0.197405,0.204082,0.212523,0.224241,0.244361,0.290022,0.361150,
  0.043946,0.047773,0.052333,0.055122,0.057075,0.058624,0.059935,0.061088,0.062124,
  0.063069,0.063943,0.064758,0.065521,0.066243,0.066928,0.067587,0.068218,0.068823,
  0.069405,0.069973,0.070519,0.071048,0.073508,0.075718,0.077751,0.079661,0.083213,
  0.086546,0.089772,0.092978,0.096246,0.099639,0.103249,0.107194,0.111628,0.116758,
  0.122906,0.130512,0.135070,0.140291,0.146317,0.153472,0.155082,0.156762,0.158535,
  0.160383,0.162368,0.164450,0.166682,0.169099,0.171711,0.174572,0.177731,0.181293,
  0.185280,0.189899,0.195308,0.201861,0.210138,0.221643,0.241103,0.285728,0.353798,
  0.044264,0.047967,0.052410,0.055156,0.057084,0.058613,0.059905,0.061038,0.062057,
  0.062983,0.063848,0.064658,0.065413,0.066131,0.066811,0.067464,0.068087,0.068688,
  0.069268,0.069825,0.070369,0.070900,0.073334,0.075523,0.077542,0.079432,0.082952,
  0.086243,0.089417,0.092557,0.095744,0.099047,0.102544,0.106351,0.110629,0.115574,
  0.121481,0.128753,0.133118,0.138116,0.143913,0.150789,0.152323,0.153936,0.155628,
  0.157415,0.159307,0.161312,0.163468,0.165778,0.168288,0.171028,0.174079,0.177436,
  0.181248,0.185659,0.190803,0.197073,0.204995,0.216020,0.234659,0.276958,0.340438,
  0.043518,0.047315,0.051883,0.054666,0.056600,0.058140,0.059436,0.060578,0.061593,
  0.062523,0.063384,0.064192,0.064943,0.065658,0.066335,0.066985,0.067608,0.068205,
  0.068779,0.069335,0.069880,0.070406,0.072840,0.075034,0.077053,0.078943,0.082478,
  0.085784,0.088987,0.092149,0.095363,0.098676,0.102184,0.106000,0.110249,0.115128,
  0.120954,0.128075,0.132325,0.137193,0.142853,0.149577,0.151082,0.152659,0.154316,
  0.156068,0.157921,0.159892,0.161987,0.164254,0.166699,0.169391,0.172370,0.175710,
  0.179463,0.183797,0.188899,0.195002,0.202748,0.213536,0.231855,0.273290,0.334782,
  0.043771,0.047460,0.051926,0.054640,0.056546,0.058059,0.059346,0.060473,0.061483,
  0.062407,0.063251,0.064046,0.064789,0.065494,0.066166,0.066812,0.067429,0.068022,
  0.068591,0.069143,0.069678,0.070198,0.072607,0.074773,0.076772,0.078647,0.082141,
  0.085403,0.088558,0.091673,0.094817,0.098058,0.101478,0.105178,0.109292,0.113999,
  0.119584,0.126453,0.130564,0.135257,0.140712,0.147176,0.148633,0.150152,0.151750,
  0.153436,0.155230,0.157128,0.159156,0.161345,0.163717,0.166303,0.169167,0.172366,
  0.175941,0.180100,0.184920,0.190778,0.198213,0.208542,0.225882,0.265098,0.321104,
  0.043032,0.046886,0.051502,0.054239,0.056144,0.057664,0.058930,0.060053,0.061065,
  0.061986,0.062835,0.063623,0.064367,0.065069,0.065738,0.066383,0.066998,0.067593,
  0.068158,0.068709,0.069245,0.069763,0.072171,0.074345,0.076341,0.078218,0.081713,
  0.084994,0.088155,0.091281,0.094446,0.097698,0.101135,0.104833,0.108923,0.113593,
  0.119096,0.125848,0.129873,0.134463,0.139790,0.146127,0.147556,0.149053,0.150614,
  0.152278,0.154044,0.155935,0.157943,0.160096,0.162406,0.164941,0.167726,0.170846,
  0.174383,0.178396,0.183138,0.188826,0.196068,0.206249,0.223334,0.262055,0.316056,
  0.043188,0.047000,0.051473,0.054178,0.056067,0.057558,0.058825,0.059930,0.060923,
  0.061832,0.062673,0.063455,0.064191,0.064893,0.065558,0.066191,0.066802,0.067385,
  0.067947,0.068495,0.069025,0.069535,0.071923,0.074071,0.076056,0.077913,0.081366,
  0.084604,0.087723,0.090807,0.093903,0.097092,0.100448,0.104043,0.108020,0.112536,
  0.117862,0.124374,0.128262,0.132693,0.137838,0.143960,0.145348,0.146795,0.148315,
  0.149913,0.151610,0.153417,0.155348,0.157416,0.159674,0.162142,0.164830,0.167843,
  0.171238,0.175157,0.179730,0.185270,0.192307,0.202037,0.218472,0.255482,0.307943,
  0.042591,0.046512,0.051074,0.053814,0.055707,0.057206,0.058476,0.059584,0.060580,
  0.061487,0.062319,0.063101,0.063839,0.064533,0.065200,0.065829,0.066438,0.067018,
  0.067586,0.068127,0.068653,0.069167,0.071542,0.073683,0.075664,0.077517,0.080967,
  0.084211,0.087336,0.090428,0.093538,0.096739,0.100086,0.103690,0.107649,0.112139,
  0.117403,0.123799,0.127593,0.131954,0.137003,0.143028,0.144385,0.145819,0.147311,
  0.148883,0.150562,0.152334,0.154234,0.156279,0.158484,0.160894,0.163556,0.166518,
  0.169854,0.173670,0.178175,0.183644,0.190523,0.200095,0.216163,0.252462,0.304476,
  0.042762,0.046595,0.051089,0.053784,0.055645,0.057125,0.058378,0.059475,0.060464,
  0.061358,0.062190,0.062965,0.063688,0.064374,0.065021,0.065648,0.066247,0.066818,
  0.067375,0.067909,0.068427,0.068939,0.071297,0.073416,0.075368,0.077203,0.080617,
  0.083815,0.086901,0.089938,0.093002,0.096124,0.099408,0.102928,0.106795,0.111155,
  0.116256,0.122445,0.126124,0.130344,0.135223,0.141053,0.142370,0.143748,0.145195,
  0.146728,0.148337,0.150054,0.151901,0.153890,0.156028,0.158368,0.160932,0.163794,
  0.167028,0.170757,0.175071,0.180329,0.186987,0.196259,0.211807,0.246557,0.295435,
  0.042159,0.046135,0.050705,0.053413,0.055275,0.056753,0.058001,0.059098,0.060083,
  0.060974,0.061798,0.062570,0.063296,0.063985,0.064637,0.065258,0.065863,0.066443,
  0.066998,0.067531,0.068051,0.068558,0.070904,0.073023,0.074975,0.076808,0.080223,
  0.083418,0.086504,0.089543,0.092606,0.095748,0.099032,0.102549,0.106384,0.110715,
  0.115754,0.121862,0.125489,0.129616,0.134408,0.140115,0.141408,0.142773,0.144206,
  0.145718,0.147316,0.149007,0.150815,0.152767,0.154879,0.157195,0.159731,0.162538,
  0.165716,0.169367,0.173636,0.178763,0.185331,0.194409,0.209558,0.243751,0.292127,
  0.042381,0.046283,0.050748,0.053430,0.055283,0.056741,0.057981,0.059056,0.060026,
  0.060912,0.061722,0.062487,0.063202,0.063883,0.064529,0.065139,0.065735,0.066303,
  0.066853,0.067385,0.067900,0.068399,0.070713,0.072810,0.074733,0.076546,0.079918,
  0.083084,0.086136,0.089136,0.092158,0.095240,0.098466,0.101904,0.105673,0.109886,
  0.114806,0.120738,0.124257,0.128271,0.132938,0.138499,0.139759,0.141070,0.142452,
  0.143911,0.145467,0.147112,0.148878,0.150758,0.152797,0.155043,0.157473,0.160195,
  0.163279,0.166793,0.170923,0.175898,0.182242,0.191017,0.205695,0.239029,0.285179,
  0.041846,0.045774,0.050341,0.053029,0.054876,0.056332,0.057564,0.058642,0.059606,
  0.060493,0.061310,0.062070,0.062782,0.063461,0.064100,0.064714,0.065306,0.065876,
  0.066422,0.066952,0.067468,0.067963,0.070278,0.072365,0.074288,0.076094,0.079455,
  0.082610,0.085654,0.088654,0.091665,0.094753,0.097978,0.101404,0.105143,0.109341,
  0.114187,0.120018,0.123486,0.127428,0.132010,0.137466,0.138696,0.139993,0.141356,
  0.142795,0.144318,0.145954,0.147694,0.149558,0.151563,0.153760,0.156178,0.158865,
  0.161880,0.165333,0.169417,0.174321,0.180598,0.189262,0.203838,0.236377,0.282306,
  0.041927,0.045883,0.050376,0.053039,0.054864,0.056315,0.057544,0.058613,0.059568,
  0.060439,0.061250,0.061999,0.062713,0.063385,0.064023,0.064627,0.065210,0.065771,
  0.066315,0.066836,0.067340,0.067834,0.070112,0.072178,0.074080,0.075870,0.079206,
  0.082328,0.085339,0.088307,0.091285,0.094328,0.097489,0.100860,0.104521,0.108625,
  0.113370,0.119072,0.122439,0.126278,0.130749,0.136054,0.137263,0.138515,0.139849,
  0.141263,0.142753,0.144343,0.146037,0.147845,0.149815,0.151963,0.154324,0.156927,
  0.159867,0.163223,0.167198,0.171941,0.178010,0.186464,0.200572,0.231822,0.275092,
  0.041550,0.045535,0.050089,0.052735,0.054553,0.055994,0.057212,0.058293,0.059254,
  0.060128,0.060928,0.061681,0.062391,0.063056,0.063693,0.064302,0.064886,0.065445,
  0.065985,0.066511,0.067018,0.067508,0.069786,0.071839,0.073741,0.075526,0.078858,
  0.081968,0.084968,0.087922,0.090896,0.093933,0.097095,0.100451,0.104102,0.108181,
  0.112868,0.118498,0.121819,0.125600,0.129978,0.135205,0.136383,0.137628,0.138933,
  0.140305,0.141768,0.143316,0.144978,0.146774,0.148713,0.150804,0.153140,0.155719,
  0.158608,0.161930,0.165840,0.170529,0.176488,0.184762,0.198747,0.229676,0.272056,
  0.041572,0.045565,0.050047,0.052699,0.054503,0.055948,0.057149,0.058211,0.059168,
  0.060032,0.060825,0.061567,0.062266,0.062928,0.063558,0.064157,0.064730,0.065280,
  0.065813,0.066333,0.066833,0.067322,0.069578,0.071615,0.073497,0.075255,0.078544,
  0.081624,0.084592,0.087519,0.090453,0.093453,0.096565,0.099866,0.103453,0.107444,
  0.112038,0.117528,0.120756,0.124426,0.128698,0.133793,0.134949,0.136158,0.137427,
  0.138777,0.140202,0.141720,0.143337,0.145077,0.146948,0.148995,0.151261,0.153770,
  0.156611,0.159841,0.163616,0.168188,0.173978,0.182024,0.195507,0.225404,0.267508,
  0.041104,0.045243,0.049730,0.052370,0.054184,0.055618,0.056823,0.057878,0.058826,
  0.059688,0.060486,0.061228,0.061933,0.062589,0.063211,0.063812,0.064387,0.064942,
  0.065476,0.065993,0.066488,0.066973,0.069224,0.071253,0.073130,0.074890,0.078175,
  0.081243,0.084206,0.087123,0.090048,0.093033,0.096140,0.099438,0.103008,0.106974,
  0.111530,0.116949,0.120135,0.123768,0.127960,0.132995,0.134128,0.135323,0.136577,
  0.137898,0.139302,0.140802,0.142396,0.144111,0.145973,0.147994,0.150233,0.152706,
  0.155477,0.158664,0.162392,0.166898,0.172621,0.180617,0.193931,0.223430,0.265240,
  0.041382,0.045301,0.049766,0.052385,0.054180,0.055606,0.056796,0.057840,0.058773,
  0.059631,0.060423,0.061153,0.061839,0.062494,0.063115,0.063708,0.064278,0.064825,
  0.065350,0.065858,0.066352,0.066833,0.069063,0.071071,0.072929,0.074668,0.077921,
  0.080966,0.083897,0.086773,0.089663,0.092609,0.095673,0.098917,0.102429,0.106327,
  0.110788,0.116095,0.119207,0.122726,0.126840,0.131736,0.132851,0.134017,0.135247,
  0.136539,0.137911,0.139383,0.140953,0.142628,0.144438,0.146420,0.148593,0.150993,
  0.153715,0.156808,0.160418,0.164797,0.170404,0.178155,0.191068,0.220015,0.258941,
  0.071160,0.074950,0.081180,0.085870,0.089550,0.092530,0.095130,0.097390,0.099510,
  0.101560,0.103520,0.105240,0.106920,0.108550,0.110140,0.111620,0.113190,0.114600,
  0.115930,0.117280,0.118570,0.119840,0.125810,0.131220,0.136410,0.141260,0.150530,
  0.159360,0.167870,0.176440,0.185210,0.194420,0.204510,0.215870,0.229330,0.245540,
  0.266430,0.294320,0.312430,0.334010,0.360650,0.394240,0.402040,0.410510,0.419570,
  0.428970,0.439040,0.450150,0.462230,0.474980,0.489040,0.505000,0.522640,0.542320,
  0.564730,0.590950,0.622570,0.661770,0.712320,0.783970,0.914190,1.242390,1.817300,
  0.081540,0.081110,0.086740,0.091610,0.095310,0.098600,0.101270,0.103730,0.106010,
  0.108090,0.110110,0.111910,0.113740,0.115420,0.117090,0.118680,0.120140,0.121650,
  0.123050,0.124440,0.125780,0.127210,0.133620,0.139580,0.145080,0.150470,0.160550,
  0.170410,0.180120,0.189670,0.199720,0.210290,0.221460,0.233740,0.247360,0.263340,
  0.282750,0.306840,0.321490,0.338620,0.360110,0.387480,0.394130,0.400900,0.408260,
  0.416190,0.424590,0.433670,0.443460,0.454240,0.466240,0.479130,0.493900,0.510740,
  0.530130,0.552300,0.578750,0.612170,0.655610,0.720970,0.830600,1.090500,1.493320,
  0.083030,0.087240,0.093570,0.098850,0.102690,0.105600,0.108440,0.110970,0.113380,
  0.115480,0.117480,0.119400,0.121280,0.123120,0.124710,0.126270,0.127860,0.129380,
  0.131030,0.132560,0.133990,0.135300,0.141930,0.148180,0.154120,0.159680,0.170720,
  0.181160,0.191500,0.202090,0.212740,0.223820,0.235720,0.248440,0.262640,0.278500,
  0.296710,0.319060,0.332200,0.347860,0.366410,0.389650,0.394850,0.400490,0.406230,
  0.412510,0.419350,0.426920,0.435090,0.444200,0.453960,0.464590,0.476550,0.490610,
  0.506330,0.525690,0.548520,0.576630,0.613130,0.664700,0.756970,0.981300,1.293500,
  0.079130,0.087170,0.098490,0.105120,0.109820,0.113750,0.117060,0.120050,0.122590,
  0.125060,0.127270,0.129340,0.131320,0.133180,0.135000,0.136770,0.138420,0.140030,
  0.141610,0.143140,0.144640,0.146110,0.153180,0.159600,0.165740,0.171530,0.182760,
  0.193730,0.204530,0.215490,0.226790,0.238620,0.251070,0.264430,0.278920,0.295010,
  0.313540,0.335380,0.348210,0.362910,0.380080,0.401220,0.405980,0.411040,0.416530,
  0.422320,0.428480,0.434990,0.441990,0.449650,0.457810,0.467180,0.477550,0.489090,
  0.502290,0.517440,0.535630,0.557970,0.587970,0.631000,0.706310,0.891800,1.144190,
  0.079730,0.090760,0.103780,0.111600,0.117210,0.121700,0.125310,0.128540,0.131530,
  0.134130,0.136730,0.139030,0.141240,0.143290,0.145340,0.147280,0.149180,0.150960,
  0.152620,0.154270,0.155970,0.157600,0.164880,0.171510,0.177880,0.183930,0.195610,
  0.206870,0.218100,0.229450,0.241010,0.252990,0.265560,0.278980,0.293710,0.309730,
  0.327810,0.349240,0.361740,0.375970,0.392160,0.411530,0.415930,0.420800,0.425560,
  0.430890,0.436480,0.442350,0.448450,0.455540,0.463170,0.471220,0.480180,0.490590,
  0.501870,0.514680,0.530880,0.550210,0.574690,0.611980,0.675580,0.817960,1.044450,
  0.085290,0.096810,0.110440,0.118930,0.124740,0.129450,0.133400,0.136990,0.140040,
  0.142980,0.145640,0.148240,0.150590,0.152880,0.154960,0.156990,0.158920,0.160800,
  0.162700,0.164500,0.166200,0.167910,0.175930,0.183230,0.189970,0.196430,0.208670,
  0.220300,0.231880,0.243250,0.254900,0.266890,0.279640,0.293000,0.307340,0.323340,
  0.341390,0.362270,0.374360,0.388090,0.404020,0.423140,0.427430,0.431950,0.436740,
  0.441600,0.446840,0.452490,0.458580,0.464780,0.471480,0.478700,0.486590,0.495570,
  0.505620,0.517510,0.530850,0.547540,0.569500,0.599720,0.654130,0.785710,0.973680,
  0.090340,0.102920,0.118250,0.126880,0.133030,0.137760,0.142070,0.145700,0.149170,
  0.152330,0.155060,0.157540,0.160090,0.162570,0.164910,0.167020,0.169190,0.171170,
  0.173010,0.174890,0.176670,0.178420,0.186750,0.194450,0.201570,0.208300,0.220890,
  0.233030,0.244910,0.256740,0.268670,0.280990,0.293770,0.307160,0.321630,0.337290,
  0.355000,0.375750,0.387800,0.400970,0.415960,0.434110,0.438320,0.442600,0.446910,
  0.451820,0.456910,0.462360,0.467940,0.474130,0.480500,0.487780,0.496260,0.504920,
  0.515100,0.526400,0.540120,0.555580,0.576140,0.605470,0.650560,0.764060,0.952020,
  0.090530,0.106850,0.123890,0.133440,0.140100,0.145390,0.149760,0.153720,0.157300,
  0.160430,0.163600,0.166380,0.169000,0.171420,0.173720,0.176090,0.178130,0.180360,
  0.182420,0.184460,0.186370,0.188200,0.196930,0.204850,0.212260,0.219120,0.232230,
  0.244580,0.256740,0.268710,0.280640,0.292840,0.305460,0.318920,0.333360,0.349170,
  0.366900,0.387470,0.399430,0.412740,0.428080,0.445870,0.449850,0.453990,0.458590,
  0.463200,0.468000,0.473140,0.478880,0.484740,0.491080,0.498360,0.505870,0.514220,
  0.523730,0.534610,0.547340,0.562560,0.581630,0.607680,0.649600,0.738590,0.861090,
  0.096750,0.110790,0.129970,0.141110,0.148290,0.154060,0.158910,0.162950,0.166330,
  0.169700,0.172620,0.175420,0.178120,0.180940,0.183250,0.185450,0.187910,0.190040,
  0.192330,0.194180,0.196310,0.198090,0.207320,0.215520,0.223040,0.230350,0.243760,
  0.256520,0.268520,0.280630,0.293020,0.305420,0.318550,0.331880,0.346360,0.361900,
  0.379750,0.400130,0.411580,0.424410,0.439150,0.457090,0.461100,0.465300,0.469190,
  0.473970,0.478420,0.483560,0.488950,0.494910,0.501660,0.508330,0.515890,0.524360,
  0.533400,0.544270,0.555970,0.571100,0.588960,0.613020,0.655230,0.747700,0.857460,
  0.100900,0.119770,0.136050,0.146650,0.153650,0.159480,0.164320,0.168790,0.172760,
  0.176100,0.179800,0.183250,0.185930,0.188460,0.191040,0.193850,0.196080,0.198260,
  0.200360,0.202560,0.204580,0.206660,0.215490,0.223600,0.231430,0.238610,0.251590,
  0.264430,0.276840,0.288960,0.301090,0.313780,0.326350,0.340180,0.354150,0.369960,
  0.387300,0.407470,0.419170,0.432390,0.447790,0.465110,0.469070,0.473430,0.477890,
  0.482250,0.487310,0.492600,0.498050,0.503730,0.509690,0.516740,0.523720,0.531290,
  0.540110,0.550440,0.562180,0.575830,0.592920,0.615500,0.658440,0.756420,0.903070,
  0.104210,0.120630,0.142510,0.153970,0.161770,0.167740,0.172820,0.177080,0.180900,
  0.184370,0.187540,0.190480,0.193430,0.196100,0.198670,0.201060,0.203470,0.205700,
  0.207750,0.209860,0.211850,0.213890,0.222930,0.231490,0.239140,0.246370,0.260140,
  0.273070,0.285500,0.297670,0.310000,0.322230,0.335090,0.348750,0.363160,0.378680,
  0.396050,0.416600,0.428130,0.441180,0.456020,0.473550,0.477640,0.481660,0.486050,
  0.490530,0.495350,0.500230,0.505660,0.511520,0.517640,0.524360,0.532140,0.540100,
  0.549750,0.559940,0.572020,0.586690,0.605530,0.631430,0.672480,0.752690,0.864460,
  0.107490,0.126300,0.146530,0.159220,0.167520,0.173450,0.178590,0.183040,0.187340,
  0.191270,0.194550,0.197700,0.200690,0.203370,0.205890,0.208700,0.210860,0.213490,
  0.215980,0.217990,0.220300,0.222280,0.232270,0.240650,0.248500,0.256190,0.270220,
  0.282800,0.295230,0.307410,0.319510,0.331850,0.344580,0.357820,0.372320,0.387720,
  0.405800,0.425330,0.436500,0.449570,0.463760,0.480720,0.484080,0.488540,0.492790,
  0.497330,0.502550,0.508010,0.512860,0.519150,0.525470,0.532150,0.539650,0.548490,
  0.557620,0.568550,0.581400,0.595230,0.613110,0.636790,0.676140,0.769810,0.866300,
  0.112420,0.133620,0.155300,0.167320,0.175130,0.182030,0.187650,0.192170,0.196380,
  0.199980,0.203500,0.206810,0.209950,0.212900,0.215650,0.218260,0.220880,0.223210,
  0.225500,0.227840,0.229800,0.231810,0.241120,0.249570,0.257530,0.264760,0.278410,
  0.291760,0.304470,0.316910,0.329290,0.341900,0.355050,0.368990,0.383540,0.398840,
  0.415420,0.434780,0.446610,0.458760,0.473720,0.491370,0.495190,0.499060,0.503330,
  0.507790,0.512320,0.517160,0.522830,0.528270,0.534390,0.541440,0.548600,0.556070,
  0.564960,0.574640,0.585790,0.599780,0.617970,0.643870,0.687060,0.782320,0.958090,
  0.117630,0.141380,0.165020,0.177120,0.185310,0.191520,0.196940,0.201800,0.205670,
  0.209570,0.212900,0.215980,0.218610,0.221470,0.224230,0.226710,0.229070,0.231230,
  0.233360,0.235590,0.237680,0.239850,0.249700,0.258090,0.266240,0.273990,0.287690,
  0.300680,0.312770,0.324790,0.336930,0.349340,0.361910,0.375410,0.389580,0.405360,
  0.423800,0.443600,0.455040,0.468810,0.482890,0.500140,0.504450,0.508730,0.513170,
  0.517730,0.522540,0.527600,0.532670,0.538550,0.544960,0.551470,0.559440,0.566930,
  0.575820,0.585650,0.597360,0.612290,0.630510,0.654280,0.695630,0.784350,0.879910,
  0.119620,0.152540,0.174120,0.186630,0.195010,0.200570,0.205530,0.209870,0.214200,
  0.217460,0.220360,0.224190,0.226800,0.229730,0.232330,0.234570,0.237190,0.239780,
  0.242420,0.244480,0.245950,0.248840,0.258390,0.266630,0.274400,0.280300,0.294570,
  0.307050,0.319130,0.330850,0.343180,0.355640,0.368290,0.380450,0.393770,0.408890,
  0.425960,0.446750,0.458490,0.470550,0.485840,0.502710,0.506620,0.510900,0.515220,
  0.520230,0.524310,0.528880,0.534220,0.539770,0.546500,0.552960,0.559510,0.567780,
  0.576810,0.588760,0.601060,0.613010,0.629230,0.650440,0.692660,0.792230,0.898450,
  0.126040,0.152630,0.179260,0.189180,0.197560,0.204010,0.209380,0.214500,0.219100,
  0.223190,0.227030,0.229290,0.232220,0.235520,0.238110,0.241050,0.243290,0.246280,
  0.247600,0.249850,0.252830,0.254120,0.263000,0.273200,0.280080,0.288540,0.301580,
  0.314990,0.327790,0.339960,0.350410,0.361990,0.374280,0.386400,0.400260,0.416410,
  0.432800,0.450620,0.462280,0.474420,0.488280,0.504830,0.508340,0.512980,0.516130,
  0.520100,0.524530,0.530030,0.535920,0.540610,0.545580,0.553340,0.559870,0.568740,
  0.577650,0.585810,0.598300,0.613490,0.632110,0.658410,0.691850,0.745620,0.824450,
  0.168730,0.169710,0.184770,0.199200,0.207840,0.214150,0.219550,0.224030,0.227840,
  0.231740,0.235090,0.238600,0.242200,0.244690,0.247340,0.249470,0.252320,0.254160,
  0.257230,0.258280,0.260680,0.262480,0.272040,0.279360,0.287800,0.293840,0.307630,
  0.319370,0.331120,0.341170,0.353690,0.364990,0.378060,0.391130,0.404310,0.418230,
  0.435750,0.454100,0.465260,0.477040,0.488940,0.504510,0.508300,0.511840,0.515630,
  0.521310,0.524900,0.528440,0.533670,0.538720,0.545290,0.551000,0.558240,0.565590,
  0.574590,0.584670,0.596450,0.609920,0.626380,0.649480,0.689890,0.777690,0.885860,
  0.147580,0.176170,0.198580,0.207910,0.215930,0.221690,0.226780,0.231720,0.235580,
  0.238870,0.242740,0.245890,0.247900,0.250080,0.253370,0.254980,0.256850,0.259090,
  0.261470,0.264360,0.264390,0.268460,0.276610,0.284850,0.291220,0.298630,0.311480,
  0.323310,0.335610,0.349480,0.360600,0.371860,0.384560,0.395790,0.408370,0.424320,
  0.439590,0.459290,0.468490,0.480670,0.494460,0.512280,0.514740,0.518950,0.521590,
  0.525030,0.530450,0.535640,0.540320,0.546810,0.549480,0.555370,0.561650,0.569390,
  0.578750,0.588410,0.599300,0.614480,0.636190,0.661080,0.700940,0.775840,0.966820,
  -0.146860,-0.136320,-0.132550,-0.133910,-0.136090,-0.137940,-0.139700,-0.141130,-0.142630,
  -0.144220,-0.145830,-0.147050,-0.148290,-0.149540,-0.150800,-0.151890,-0.153230,-0.154300,
  -0.155240,-0.156260,-0.157200,-0.158140,-0.162410,-0.166050,-0.169570,-0.172630,-0.178290,
  -0.183330,-0.187780,-0.192110,-0.196370,-0.200750,-0.205650,-0.211260,-0.218220,-0.226650,
  -0.237690,-0.252100,-0.261180,-0.271450,-0.283420,-0.297330,-0.300360,-0.303650,-0.307100,
  -0.310530,-0.314120,-0.318040,-0.322200,-0.326380,-0.330880,-0.335900,-0.341220,-0.346900,
  -0.353080,-0.359980,-0.367910,-0.377160,-0.388110,-0.402100,-0.424810,-0.469910,-0.528110,
  -0.180250,-0.155670,-0.148730,-0.149670,-0.151310,-0.153440,-0.154960,-0.156530,-0.158060,
  -0.159440,-0.160890,-0.162060,-0.163390,-0.164540,-0.165760,-0.166880,-0.167810,-0.168890,
  -0.169820,-0.170750,-0.171640,-0.172740,-0.177140,-0.181160,-0.184650,-0.188090,-0.194070,
  -0.199750,-0.205030,-0.209830,-0.214850,-0.219980,-0.225160,-0.230760,-0.236760,-0.243790,
  -0.252240,-0.262270,-0.268130,-0.274740,-0.282960,-0.292990,-0.295390,-0.297740,-0.300300,
  -0.303010,-0.305810,-0.308790,-0.311940,-0.315330,-0.319070,-0.322910,-0.327270,-0.332100,
  -0.337480,-0.343310,-0.349950,-0.357980,-0.367690,-0.381480,-0.401210,-0.437800,-0.479860,
  -0.184580,-0.172390,-0.166120,-0.167100,-0.168380,-0.169150,-0.170630,-0.171970,-0.173450,
  -0.174580,-0.175730,-0.176890,-0.178100,-0.179330,-0.180190,-0.181070,-0.182070,-0.183010,
  -0.184200,-0.185240,-0.186140,-0.186860,-0.190970,-0.194860,-0.198500,-0.201720,-0.208160,
  -0.213780,-0.219090,-0.224380,-0.229350,-0.234290,-0.239470,-0.244760,-0.250520,-0.256640,
  -0.263300,-0.271240,-0.275660,-0.280930,-0.286950,-0.294270,-0.295810,-0.297500,-0.299150,
  -0.300970,-0.302960,-0.305190,-0.307570,-0.310210,-0.312960,-0.315860,-0.319080,-0.322900,
  -0.326970,-0.332020,-0.337680,-0.344300,-0.352400,-0.362900,-0.379980,-0.413670,-0.446890,
  -0.174290,-0.172190,-0.177190,-0.180390,-0.182880,-0.185220,-0.187140,-0.188960,-0.190310,
  -0.191800,-0.193020,-0.194160,-0.195270,-0.196280,-0.197310,-0.198330,-0.199220,-0.200090,
  -0.200980,-0.201810,-0.202650,-0.203460,-0.207440,-0.210900,-0.214180,-0.217170,-0.222860,
  -0.228250,-0.233290,-0.238230,-0.243140,-0.248100,-0.253100,-0.258220,-0.263500,-0.269070,
  -0.275210,-0.281990,-0.285820,-0.290070,-0.294850,-0.300570,-0.301800,-0.303110,-0.304550,
  -0.306050,-0.307610,-0.309240,-0.310970,-0.312850,-0.314780,-0.317060,-0.319530,-0.322220,
  -0.325230,-0.328580,-0.332520,-0.337180,-0.343330,-0.351660,-0.365000,-0.393040,-0.420640,
  -0.175800,-0.180270,-0.187670,-0.192380,-0.195910,-0.198740,-0.200770,-0.202630,-0.204390,
  -0.205800,-0.207360,-0.208610,-0.209830,-0.210920,-0.212080,-0.213140,-0.214190,-0.215130,
  -0.215950,-0.216800,-0.217720,-0.218590,-0.222160,-0.225280,-0.228320,-0.231120,-0.236450,
  -0.241370,-0.246130,-0.250770,-0.255300,-0.259800,-0.264310,-0.268930,-0.273820,-0.278810,
  -0.284100,-0.290090,-0.293430,-0.297130,-0.301100,-0.305640,-0.306640,-0.307790,-0.308830,
  -0.310050,-0.311300,-0.312580,-0.313850,-0.315430,-0.317090,-0.318760,-0.320610,-0.322800,
  -0.325040,-0.327490,-0.330710,-0.334360,-0.338740,-0.345520,-0.356100,-0.375770,-0.402320,
  -0.188550,-0.192380,-0.199360,-0.204350,-0.207620,-0.210340,-0.212540,-0.214600,-0.216190,
  -0.217820,-0.219240,-0.220670,-0.221900,-0.223100,-0.224130,-0.225140,-0.226080,-0.227010,
  -0.227980,-0.228880,-0.229680,-0.230520,-0.234370,-0.237740,-0.240710,-0.243500,-0.248610,
  -0.253210,-0.257660,-0.261760,-0.265840,-0.269860,-0.274030,-0.278150,-0.282360,-0.286890,
  -0.291740,-0.296980,-0.299880,-0.303100,-0.306700,-0.310870,-0.311770,-0.312710,-0.313710,
  -0.314670,-0.315720,-0.316850,-0.318060,-0.319210,-0.320440,-0.321730,-0.323110,-0.324700,
  -0.326440,-0.328530,-0.330700,-0.333440,-0.337030,-0.341690,-0.350000,-0.368210,-0.389170,
  -0.198640,-0.203130,-0.211340,-0.215690,-0.218940,-0.221270,-0.223600,-0.225440,-0.227300,
  -0.228960,-0.230260,-0.231380,-0.232650,-0.233920,-0.235080,-0.236040,-0.237110,-0.238000,
  -0.238790,-0.239650,-0.240430,-0.241200,-0.244870,-0.248190,-0.251130,-0.253820,-0.258620,
  -0.263090,-0.267270,-0.271250,-0.275090,-0.278910,-0.282690,-0.286440,-0.290350,-0.294320,
  -0.298620,-0.303410,-0.306090,-0.308840,-0.311820,-0.315360,-0.316190,-0.316990,-0.317750,
  -0.318690,-0.319630,-0.320630,-0.321600,-0.322710,-0.323770,-0.325030,-0.326560,-0.327980,
  -0.329700,-0.331510,-0.333740,-0.336000,-0.339060,-0.343360,-0.349030,-0.363280,-0.385110,
  -0.199050,-0.209370,-0.219090,-0.224080,-0.227550,-0.230240,-0.232370,-0.234360,-0.236120,
  -0.237580,-0.239170,-0.240450,-0.241660,-0.242720,-0.243720,-0.244830,-0.245660,-0.246690,
  -0.247600,-0.248500,-0.249310,-0.250070,-0.253680,-0.256840,-0.259710,-0.262230,-0.266940,
  -0.271130,-0.275110,-0.278830,-0.282340,-0.285780,-0.289180,-0.292700,-0.296300,-0.300080,
  -0.304100,-0.308520,-0.311000,-0.313650,-0.316600,-0.319820,-0.320510,-0.321220,-0.322050,
  -0.322830,-0.323620,-0.324470,-0.325450,-0.326390,-0.327400,-0.328600,-0.329760,-0.331030,
  -0.332470,-0.334090,-0.335940,-0.338070,-0.340630,-0.343950,-0.348790,-0.357610,-0.368350,
  -0.209490,-0.215080,-0.226610,-0.232840,-0.236460,-0.239320,-0.241680,-0.243500,-0.244880,
  -0.246390,-0.247590,-0.248750,-0.249900,-0.251200,-0.252100,-0.252950,-0.254050,-0.254890,
  -0.255890,-0.256560,-0.257460,-0.258100,-0.261740,-0.264790,-0.267470,-0.270070,-0.274540,
  -0.278600,-0.282140,-0.285630,-0.289100,-0.292370,-0.295750,-0.298940,-0.302300,-0.305690,
  -0.309490,-0.313550,-0.315690,-0.318020,-0.320600,-0.323710,-0.324380,-0.325070,-0.325630,
  -0.326430,-0.327070,-0.327880,-0.328710,-0.329640,-0.330730,-0.331700,-0.332820,-0.334080,
  -0.335320,-0.336880,-0.338390,-0.340430,-0.342590,-0.345330,-0.350150,-0.359510,-0.367640,
  -0.215660,-0.226720,-0.233470,-0.238610,-0.241770,-0.244500,-0.246670,-0.248740,-0.250540,
  -0.251920,-0.253660,-0.255270,-0.256300,-0.257280,-0.258320,-0.259560,-0.260400,-0.261220,
  -0.262000,-0.262870,-0.263620,-0.264420,-0.267520,-0.270300,-0.272990,-0.275330,-0.279260,
  -0.283140,-0.286700,-0.290000,-0.293160,-0.296400,-0.299370,-0.302630,-0.305620,-0.308980,
  -0.312430,-0.316270,-0.318420,-0.320800,-0.323510,-0.326300,-0.326930,-0.327660,-0.328370,
  -0.329010,-0.329820,-0.330650,-0.331460,-0.332280,-0.333100,-0.334150,-0.335070,-0.336030,
  -0.337190,-0.338560,-0.340040,-0.341650,-0.343590,-0.345940,-0.350880,-0.361260,-0.375370,
  -0.220310,-0.227740,-0.240130,-0.245620,-0.249190,-0.251780,-0.253950,-0.255660,-0.257180,
  -0.258540,-0.259750,-0.260850,-0.262010,-0.263010,-0.263970,-0.264830,-0.265730,-0.266530,
  -0.267230,-0.267970,-0.268660,-0.269390,-0.272420,-0.275310,-0.277720,-0.279950,-0.284090,
  -0.287770,-0.291140,-0.294290,-0.297370,-0.300230,-0.303180,-0.306210,-0.309240,-0.312340,
  -0.315650,-0.319470,-0.321470,-0.323700,-0.326130,-0.328890,-0.329540,-0.330140,-0.330800,
  -0.331460,-0.332180,-0.332860,-0.333650,-0.334490,-0.335330,-0.336260,-0.337370,-0.338410,
  -0.339740,-0.341020,-0.342540,-0.344350,-0.346630,-0.349620,-0.353920,-0.360540,-0.369000,
  -0.224580,-0.234070,-0.243960,-0.250200,-0.253950,-0.256350,-0.258430,-0.260190,-0.261960,
  -0.263560,-0.264770,-0.265940,-0.267050,-0.268000,-0.268860,-0.269940,-0.270620,-0.271620,
  -0.272540,-0.273180,-0.274020,-0.274660,-0.278040,-0.280620,-0.282980,-0.285310,-0.289290,
  -0.292570,-0.295730,-0.298690,-0.301500,-0.304260,-0.307000,-0.309720,-0.312650,-0.315570,
  -0.318980,-0.322300,-0.324120,-0.326280,-0.328440,-0.330960,-0.331380,-0.332090,-0.332700,
  -0.333350,-0.334160,-0.334980,-0.335590,-0.336520,-0.337390,-0.338280,-0.339290,-0.340520,
  -0.341680,-0.343110,-0.344760,-0.346320,-0.348330,-0.350780,-0.354670,-0.363610,-0.369280,
  -0.230510,-0.241440,-0.251560,-0.256690,-0.259770,-0.262670,-0.264920,-0.266580,-0.268140,
  -0.269400,-0.270670,-0.271850,-0.272970,-0.274000,-0.274940,-0.275820,-0.276720,-0.277460,
  -0.278200,-0.278980,-0.279550,-0.280170,-0.282940,-0.285400,-0.287670,-0.289630,-0.293220,
  -0.296670,-0.299790,-0.302700,-0.305470,-0.308190,-0.310930,-0.313760,-0.316540,-0.319270,
  -0.322050,-0.325190,-0.327130,-0.328940,-0.331230,-0.333830,-0.334370,-0.334890,-0.335480,
  -0.336100,-0.336690,-0.337320,-0.338120,-0.338800,-0.339600,-0.340550,-0.341440,-0.342310,
  -0.343390,-0.344500,-0.345740,-0.347320,-0.349360,-0.352240,-0.356790,-0.365740,-0.382540,
  -0.236250,-0.248540,-0.259170,-0.263820,-0.266850,-0.269040,-0.270970,-0.272700,-0.273940,
  -0.275270,-0.276330,-0.277290,-0.278050,-0.278950,-0.279830,-0.280580,-0.281280,-0.281890,
  -0.282490,-0.283170,-0.283780,-0.284440,-0.287320,-0.289610,-0.291840,-0.293920,-0.297320,
  -0.300440,-0.303150,-0.305770,-0.308340,-0.310880,-0.313340,-0.315920,-0.318500,-0.321310,
  -0.324560,-0.327710,-0.329480,-0.331650,-0.333630,-0.336050,-0.336680,-0.337290,-0.337900,
  -0.338510,-0.339160,-0.339820,-0.340450,-0.341220,-0.342050,-0.342840,-0.343890,-0.344730,
  -0.345780,-0.346880,-0.348190,-0.349900,-0.351880,-0.354250,-0.358320,-0.366060,-0.371880,
  -0.238210,-0.257690,-0.265660,-0.270150,-0.273030,-0.274630,-0.276140,-0.277440,-0.278850,
  -0.279730,-0.280490,-0.281800,-0.282480,-0.283370,-0.284110,-0.284690,-0.285480,-0.286270,
  -0.287090,-0.287640,-0.287910,-0.288870,-0.291450,-0.293530,-0.295480,-0.296660,-0.300180,
  -0.302970,-0.305580,-0.308000,-0.310550,-0.313030,-0.315440,-0.317530,-0.319790,-0.322360,
  -0.325170,-0.328560,-0.330390,-0.332100,-0.334370,-0.336670,-0.337200,-0.337800,-0.338380,
  -0.339100,-0.339570,-0.340120,-0.340800,-0.341490,-0.342400,-0.343170,-0.343910,-0.344920,
  -0.345990,-0.347520,-0.348940,-0.350050,-0.351630,-0.353540,-0.357800,-0.367280,-0.374490,
  -0.244500,-0.257770,-0.269060,-0.271740,-0.274540,-0.276630,-0.278320,-0.280000,-0.281500,
  -0.282790,-0.283990,-0.284440,-0.285260,-0.286300,-0.287000,-0.287900,-0.288460,-0.289410,
  -0.289570,-0.290190,-0.291150,-0.291340,-0.293530,-0.296390,-0.297880,-0.300070,-0.302940,
  -0.305970,-0.308720,-0.311190,-0.313000,-0.315120,-0.317330,-0.319350,-0.321710,-0.324510,
  -0.327040,-0.329570,-0.331350,-0.333060,-0.334950,-0.337160,-0.337600,-0.338270,-0.338590,
  -0.339060,-0.339610,-0.340370,-0.341170,-0.341670,-0.342190,-0.343250,-0.343980,-0.345110,
  -0.346150,-0.346930,-0.348390,-0.350130,-0.352170,-0.354970,-0.357660,-0.360150,-0.364330,
  -0.277810,-0.269910,-0.272530,-0.277660,-0.280360,-0.282190,-0.283760,-0.284990,-0.285990,
  -0.287100,-0.287990,-0.289000,-0.290080,-0.290680,-0.291360,-0.291830,-0.292640,-0.293020,
  -0.293950,-0.294000,-0.294660,-0.295050,-0.297400,-0.298940,-0.301000,-0.302160,-0.305220,
  -0.307550,-0.309880,-0.311600,-0.314070,-0.316060,-0.318480,-0.320750,-0.322870,-0.325000,
  -0.327810,-0.330450,-0.332080,-0.333690,-0.335110,-0.337080,-0.337590,-0.338020,-0.338480,
  -0.339330,-0.339690,-0.340030,-0.340690,-0.341270,-0.342130,-0.342760,-0.343650,-0.344480,
  -0.345550,-0.346700,-0.348030,-0.349460,-0.351110,-0.353390,-0.357330,-0.364990,-0.372600,
  -0.263000,-0.274090,-0.280560,-0.282420,-0.284620,-0.286040,-0.287360,-0.288750,-0.289710,
  -0.290470,-0.291560,-0.292350,-0.292680,-0.293100,-0.294040,-0.294270,-0.294630,-0.295160,
  -0.295770,-0.296590,-0.296240,-0.297570,-0.299260,-0.301110,-0.302320,-0.303960,-0.306610,
  -0.308920,-0.311380,-0.314280,-0.316230,-0.318140,-0.320380,-0.322070,-0.323990,-0.326610,
  -0.328790,-0.331720,-0.332860,-0.334530,-0.336360,-0.338790,-0.338990,-0.339550,-0.339760,
  -0.340120,-0.340860,-0.341530,-0.342060,-0.342930,-0.342980,-0.343640,-0.344320,-0.345220,
  -0.346340,-0.347410,-0.348570,-0.350300,-0.352840,-0.355370,-0.359100,-0.364730,-0.382340)
  pcoef <- matrix(ctmp,nrow=72,ncol=63,byrow=TRUE )
  return(pcoef)
}

mhde.pval <- function(pcoef,Nsize,Hdis) {

  #-------------------------------------------------------------------------------------------------------------
  #
  #'mhde.pval
  #
  #' @description
  #'   This function computes the upper tail probability level for a computed Hellinger distance given the
  #'   sample size.
  #
  # History:
  #    Paul W. Eslinger : 17 Aug 2015 : Original source
  #
  # Parameters:
  #' @param   pcoef   A matrix of coefficients used in functional fits for the tail probabilities
  #' @param   Nsize   Number of data values (sample size)
  #' @param   Hdis    The value of the Hellinger distance
  #
  # Returned values:
  #' @return   pval  Upper tail p value.
  #
  #-------------------------------------------------------------------------------------------------------------
#
  # Explicity define the alpha levels corresponding to the coefficients in the data file
  nalpha <- 63
  alpha <- c(0.0001,0.001,0.005,0.01,0.015,0.02,0.025,0.03,0.035,0.04,
             0.045,0.05,0.055,0.06,0.065,0.07,0.075,0.08,0.085,0.09,
             0.095,0.1,0.125,0.15,0.175,0.2,0.25,0.3,0.35,0.4,0.45,0.5,
             0.55,0.6,0.65,0.7,0.75,0.8,0.825,0.85,0.875,0.9,0.905,0.91,
             0.915,0.92,0.925,0.93,0.935,0.94,0.945,0.95,0.955,0.96,
             0.965,0.97,0.975,0.98,0.985,0.99,0.995,0.999,0.9999)
  #
  pcrit <- numeric(nalpha)
  # If the sample size is 40 or smaller, use row Nsize-4 for the explicit coeffs
  if( Nsize <= 40 ) {
    nr <- Nsize-4
    for( i in 1:nalpha ){ pcrit[i] <- pcoef[nr,i] }
  } else {
    pintercept <- numeric(nalpha)
    pslope <- numeric(nalpha)
    # Pick the row for the intercept
    nr <- 0
    if( Nsize >=   41 & Nsize <=   60 ) nr <- 37
    if( Nsize >=   61 & Nsize <=   80 ) nr <- 38
    if( Nsize >=   81 & Nsize <=  100 ) nr <- 39
    if( Nsize >=  101 & Nsize <=  150 ) nr <- 40
    if( Nsize >=  151 & Nsize <=  200 ) nr <- 41
    if( Nsize >=  201 & Nsize <=  300 ) nr <- 42
    if( Nsize >=  301 & Nsize <=  400 ) nr <- 43
    if( Nsize >=  401 & Nsize <=  600 ) nr <- 44
    if( Nsize >=  601 & Nsize <=  800 ) nr <- 45
    if( Nsize >=  801 & Nsize <= 1000 ) nr <- 46
    if( Nsize >= 1001 & Nsize <= 1500 ) nr <- 47
    if( Nsize >= 1501 & Nsize <= 2000 ) nr <- 48
    if( Nsize >= 2001 & Nsize <= 3000 ) nr <- 49
    if( Nsize >= 3001 & Nsize <= 4000 ) nr <- 50
    if( Nsize >= 4001 & Nsize <= 5000 ) nr <- 51
    if( Nsize >= 5001 & Nsize <= 6000 ) nr <- 52
    if( Nsize >= 6001 & Nsize <= 8000 ) nr <- 53
    if( Nsize > 8001 ) nr <- 54
    for( i in 1:nalpha ){ pintercept[i] <- pcoef[nr,i] }
    # Pick the row for the slope (18 groups of sample sizes)
    nr <- nr + 18
    for( i in 1:nalpha ){ pslope[i] <- pcoef[nr,i] }
    for( i in 1:nalpha ){ pcrit[i] <- pintercept[i] * Nsize ** pslope[i] }
  }
  # Have critical values for all alpha levels for this sample size
  # Now calculate the pvalue using linear interpolation
  if( Hdis <= pcrit[1] ) return(1.0-alpha[1])
  if( Hdis >= pcrit[nalpha] ) return(1.0-alpha[nalpha])
  for( i in 2:nalpha){
    if( Hdis <= pcrit[i] ) {
       Cdel <- (Hdis-pcrit[i-1])/(pcrit[i]-pcrit[i-1])
       Adel <- alpha[i] - alpha[i-1]
       pval <- alpha[i-1] + Cdel*Adel
       return(1.0-pval)
    }
  }
  return(1.0-pval)
}


#-------------------------------------------------------------------------------------------------------------
#' Utility function for retrieving \eqn{c_n}{Cn} value for any sample size
#
#'@description
#'   This function evaluates the \eqn{c_n}{Cn} bandwidth value for the Epanechnikov kernel based on the sample size.
#
# History:
#    Paul W. Eslinger : 17 Aug 2015 : Original source
#
# Parameters:
#' @param   Nsize   Number of values in the sample data set
#
# Returned value:
#' @return   The Epanechnikov kernel bandwidth parameter for the sample size.
#
#' @export
#
#-------------------------------------------------------------------------------------------------------------
mhde.cn <- function(Nsize) {

  if( Nsize ==  5 ) return(1.7738)
  if( Nsize ==  6 ) return(1.6583)
  if( Nsize ==  7 ) return(1.4884)
  if( Nsize ==  8 ) return(1.4221)
  if( Nsize ==  9 ) return(1.3273)
  if( Nsize == 10 ) return(1.2826)
  if( Nsize == 11 ) return(1.2213)
  if( Nsize == 12 ) return(1.1888)
  if( Nsize == 13 ) return(1.1443)
  if( Nsize == 14 ) return(1.1188)
  if( Nsize == 15 ) return(1.0848)
  if( Nsize == 16 ) return(1.0649)
  if( Nsize == 17 ) return(1.0380)
  if( Nsize == 18 ) return(1.0209)
  if( Nsize == 19 ) return(0.99949)
  if( Nsize == 20 ) return(0.98423)
  if( Nsize == 21 ) return(0.96641)
  if( Nsize == 22 ) return(0.95340)
  if( Nsize == 23 ) return(0.93794)
  if( Nsize == 24 ) return(0.92604)
  if( Nsize == 25 ) return(0.91282)
  if( Nsize == 26 ) return(0.90227)
  if( Nsize == 27 ) return(0.89025)
  if( Nsize == 28 ) return(0.88118)
  if( Nsize == 29 ) return(0.87021)
  if( Nsize == 30 ) return(0.86210)
  if( Nsize == 31 ) return(0.85202)
  if( Nsize == 32 ) return(0.84479)
  if( Nsize == 33 ) return(0.83597)
  if( Nsize == 34 ) return(0.82846)
  if( Nsize == 35 ) return(0.82136)
  if( Nsize == 36 ) return(0.81407)
  if( Nsize == 37 ) return(0.80666)
  if( Nsize == 38 ) return(0.80058)
  if( Nsize == 39 ) return(0.79402)
  if( Nsize == 40 ) return(0.78806)
  if( Nsize >=   41 & Nsize <=   70 ) return(2.3519*Nsize**(-0.29653))
  if( Nsize >=   71 & Nsize <=  100 ) return(2.2697*Nsize**(-0.28809))
  if( Nsize >=  101 & Nsize <=  150 ) return(2.2268*Nsize**(-0.28394))
  if( Nsize >=  151 & Nsize <=  200 ) return(2.1904*Nsize**(-0.28064))
  if( Nsize >=  201 & Nsize <=  300 ) return(2.1882*Nsize**(-0.28046))
  if( Nsize >=  301 & Nsize <=  400 ) return(2.1835*Nsize**(-0.28008))
  if( Nsize >=  401 & Nsize <=  600 ) return(2.1983*Nsize**(-0.28121))
  if( Nsize >=  601 & Nsize <=  800 ) return(2.2070*Nsize**(-0.28183))
  if( Nsize >=  801 & Nsize <= 1000 ) return(2.2267*Nsize**(-0.28317))
  if( Nsize >= 1001 & Nsize <= 1500 ) return(2.2501*Nsize**(-0.28467))
  if( Nsize >= 1501 & Nsize <= 2000 ) return(2.2822*Nsize**(-0.28662))
  if( Nsize >= 2001 & Nsize <= 3000 ) return(2.3029*Nsize**(-0.28780))
  if( Nsize >= 3001 & Nsize <= 4000 ) return(2.3330*Nsize**(-0.28942))
  if( Nsize >= 4001 & Nsize <= 5000 ) return(2.3502*Nsize**(-0.29031))
  if( Nsize >= 5001 & Nsize <= 6000 ) return(2.3596*Nsize**(-0.29078))
  if( Nsize >= 6001 & Nsize <= 8000 ) return(2.3929*Nsize**(-0.29239))
  if( Nsize > 8001 ) return(2.4130*Nsize**(-0.29332))
}

#-------------------------------------------------------------------------------------------------------------
#
#' Minimum Hellinger Distance Test for Normality
#
#' @description
#'   This function fits a normal distribution to a data set using a mimimum Hellinger distance approach and
#'   then performs a test of hypothesis that the data are from a normal distribution.
#
#' @details
#'   Let \emph{f(x)} and \emph{g(x)} be absolutely continuous probability density functions.  The square of the
#'   Hellinger distance can be written as \eqn{H^2 = 1 - \int\sqrt{f(x)g(x)}dx}.  For this package, \emph{f(x)}
#'   denotes the family of normal densities and \emph{g} is a data-based density obtained by using the Ephanechnikov
#'   kernel. The kernel has the form \eqn{w(z)=0.75(1-z^2 )} for \samp{-1<z<1} and 0 elsewhere.  Let the
#'   \emph{n} sample data be denoted by \eqn{x_1}{X1}, ..., \eqn{x_n}{Xn}.  The data-based kernel density at any
#'   point \emph{y} is calculated from \deqn{g_n(y) = \frac{1}{n s_n c_n }\sum\limits_{i=1}^n w(\frac{y-x_i}{s_n c_n})}
#'
#'   A Newton-Rhapson method with analytical derivatives is to determine the minimum Hellinger distance.
#'   Numerical integration is done using a 6-point composite Gauss-Legendre technique.
#
# Parameters:
#' @param   DataVec       The data are supplied by the user in a numeric vector.  The length of the vector determines the number of data values.
#
#' @param   NGauss        The number of subintervals for the Gauss-Legendre integration techniques is controlled by \code{NGauss}.  A default value of 100 is used.  A minimum of 25 is enforced.
#
#' @param   MaxIter       The maximum number of iterations that can occur in evaluating the minimum Hellinger distance is controlled by \code{MaxIter}.  A default of 25 is used.  A minimum of 1 is enforced.
#
#' @param   InitLocation  An optional initial location estimate can be defined using \code{InitLocation}.  The data median is the default value.
#
#' @param   InitScale     An optional initial scale estimate can be defined using \code{InitScale}. The data median absolute deviation is the default value.
#
#' @param   EpsLoc        The epsilon (in data units) below which the iterative minimization approach declares convergence in the location estimate is controlled by \code{EpsLoc}.  \code{EpsLoc} should be set to give approximately 5 digits of accuracy in the location estimate. A default value of 0.0001 is used.
#
#' @param   EpsSca        The epsilon (in data units) below which the iterative minimization approach declares convergence in the SCALE estimate is controlled by \code{EpsSca}.  \code{EpsSca} should be set to give approximately 5 digits of accuracy in the scale estimate. A default value of 0.0001 is used.
#
#' @param   Silent        A value of FALSE for \code{Silent} writes several results to the R console.  Use \code{Silent}=TRUE to eliminate the output.
#
#' @param   Small         A value of FALSE for \code{Small} returns a list of 11 objects.  Use \code{Small}=TRUE to return a shorter list containing only the Hellinger distance and the p-value.
#
# Returned values:
#' @return   Values returned in a list include the following items:
#' \itemize{
#'     \item {Minimized Hellinger distance}
#'     \item {p-value for the minimized distance}
#'     \item {Initial location used in the iterative solution}
#'     \item {Initial scale used in the iterative solution}
#'     \item {Final location estimate}
#'     \item {Final scale estimate}
#'     \item {Sample size}
#'     \item {Kernel density bandwidth parameter}
#'     \item {Vector of x values used in the integration for the Hellinger distance}
#'     \item {Vector of nonparametric density values at the x values used in the integration}
#'     \item {Vector of normal density values for the estimated location and scale at the x values used in integration}
#' }
#
# References:
#
#' @references
#'   Epanechnikov, VA. 1969. "Non-Parametric Estimation of a Multivariate Probability Density."
#'   Theory of Probability and its Applications 14(1):153-156. doi \url{http://dx.doi.org/10.1137/1114019}
#'
#'   Hellinger, E. 1909. "Neue Begrundung Der Theorie Quadratischer Formen Von Unendlichvielen
#'   Veranderlichen." Journal fur die reine und angewandte Mathematik 136:210-271.
#'   doi \url{http://dx.doi.org/10.1515/crll.1909.136.210}
#
#   Eslinger, Paul W, and Wayne A Woodward. 1991. "Minimum Hellinger Distance Estimation for Normal Models."
#   Journal of Statistical Computation and Simulation 39(1-2):95-114. doi http://dx.doi.org/10.1080/00949659108811342
#
#   Tjalling J. Ypma. 1995. "Historical development of the Newton-Raphson method", SIAM Review 37(4), 531-551.
#   doi:http://dx.doi.org/10.1137/1037125.
#
#   Mike "Pomax" Kamermans, 2011.  "Primer on Bezier curves"
#   http://pomax.github.io/bezierinfo/legendre-gauss.html
#
#' @author
#'   Paul W. Eslinger and Heather M. Orr
#
#' @examples
#' ## example with a normal data set
#' mhde.test(rnorm(20,0.0,1.0),Small=TRUE)
#'
#' ## example with a uniform data set including example plot
#' MyList <- mhde.test(runif(25,min=2,max=4))
#' mhde.plot(MyList)
#
#' @export
#' @importFrom stats dnorm mad median sd
#-------------------------------------------------------------------------------------------------------------
mhde.test <- function(DataVec,NGauss=100,MaxIter=25,InitLocation,InitScale,EpsLoc=0.0001,EpsSca=0.0001,
  Silent=FALSE,Small=FALSE ) {

  # Argument checking
  if(missing(DataVec)) {
    stop("The mhde.test function expects a numeric vector on input",call.=FALSE)
  }
  # Get the number of data values from the length of the input vector
  Nsize <- length(DataVec)
  # Error terminate if Nsize < 5
  if(Nsize < 5) stop("At least 5 data values are required",call.=FALSE)
  #
  # Check the maximum number of iterations
  if(!missing(MaxIter)) {
    if( MaxIter < 1 ) stop("At least one iteration must be performed",call.=FALSE)
  }
  #
  # Check values for number of integration subintervals
  if(!missing(NGauss)) {
    if( NGauss < 25 ) stop("At least 25 integration subintervals must be specified",call.=FALSE)
  }
  #
  # Any general checks on location and scale convergence epsilon must depend on scale of the inputs.
  #
  #--------------------------------------------------------------------------
  # Set up values for integration
  # Reference for coefficients is http://pomax.github.io/bezierinfo/legendre-gauss.html
  Nsubs <- 6
  XG <- c(-0.9324695142031521,-0.6612093864662645,-0.2386191860831969,0.2386191860831969,0.6612093864662645,0.9324695142031521)
  WG <- c( 0.1713244923791704, 0.3607615730481386, 0.4679139345726910,0.4679139345726910,0.3607615730481386,0.1713244923791704)
  Nsteps <- NGauss * Nsubs
  #
  # The coding cycles through indices on XG and WG in reverse order
  # Set up a vector of indices to use in "for loops"
  Jsubs <- sort(c(1:Nsubs),decreasing=TRUE)
  #
  #--------------------------------------------------------------------------
  # Sort the incoming data and store in a vector
  x <- sort(DataVec)
  #
  # Calculate summary statistics on the data
  Xmin <- x[1]
  Xmax <- x[Nsize]
  Xrange <- Xmax - Xmin
  Xmedian <- median(x)
  Xstdev <- sd(x)
  #
  # Determine reasonable parameter bounds
  # Constrain the location estimate to the data range
  MinLocEst <- Xmin
  MaxLocEst <- Xmax
  # Constrain the scale maximum to the data range
  MaxScaEst <- Xrange
  # Constrain the minimum scale to 2.5% of the 5% trimmed range
  Ntmp <- as.integer(0.025*Nsize + 1)
  Ns1 <- max( Ntmp, 2 )
  Ns2 <- Nsize - Ns1 + 1
  MinScaEst <- (x[Ns2]-x[Ns1]) / 40.0
  #
  # Set bandwidth values needed for the Epanechnikov kernel
  Sne <- mad(x)
  Cne <- mhde.cn(Nsize)
  Hne <- Cne * Sne
  #
  # Set up initial values
  if(missing(InitLocation)) { InitLocation <- Xmedian }
  if(missing(InitScale))    { InitScale <- mad(x) }
  #
  #--------------------------------------------------------------------------
  # Set up the Gauss-Legendre integration stuff
  # Don't need to look beyond data range adjusted for the kernel width
  Xmini <- Xmin - Hne
  Xmaxi <- Xmax + Hne
  Xdel <- ( Xmaxi - Xmini ) / NGauss
  Xdel2 <- 0.5 * Xdel
  Xint <- numeric(Nsteps)
  Idx <- 0
  for( i in 1:NGauss ){
    Xhalf <- Xmini + (i-1)*Xdel + Xdel2
    for ( j in Jsubs ){
      Idx <- Idx + 1
      Xint[Idx] <- Xhalf - Xdel2*XG[j]
    }
  }
  #
  #--------------------------------------------------------------------------
  # Generate the data-based kernel density function
  # Define the kernel density on the integration grid
  Gpdf <- numeric(Nsteps)
  #
  # Increment the density at every location for influence of every data value
  for( i in 1:Nsteps ){
    for( j in 1:Nsize ){
      Diff <- abs(Xint[i]-x[j])
      if( Diff < Hne ) Gpdf[i] <- Gpdf[i] + (1.0-(Diff/Hne)**2)
    }
  }
  # Final scaling for the density estimate
  for(i in 1:Nsteps ){ Gpdf[i] <- 0.75 * Gpdf[i] / (Hne*Nsize) }
  #
  # Need the square root of the density in the integration scheme
  GpdfSqrt <- sqrt( Gpdf )
  #
  #--------------------------------------------------------------------------
  # Do the parameter estimation using a Newton-Rhapson minimization method
  # with analytical derivatives
  #
  # Initial values
  EstLoc <- InitLocation
  EstSca <- InitScale
  #
  # Estimation via minimization of the Hellinger distance
  Estimate <- mhde.calcest(EstLoc, EstSca, MaxIter, NGauss, Nsubs, Xint, WG,
                           GpdfSqrt, Xdel2, EpsLoc, EpsSca, MinLocEst, MaxLocEst,
                           MinScaEst, MaxScaEst )
  #
  # Check for a valid estimation result. Valid if Estimate[[4]] is 0
  # If an invalid estimate, use a grid search to determine new initial values
  if( Estimate[[4]] != 0 ) {
    NewInitial <- mhde.grid( Xmin, Xrange, MinScaEst, MaxScaEst, NGauss, Nsubs,
                             WG, Xint, GpdfSqrt, Xdel2 )
    EstLoc <- NewInitial[[1]]
    EstSca <- NewInitial[[2]]
    Estimate <- mhde.calcest(EstLoc, EstSca, MaxIter, NGauss, Nsubs, Xint, WG,
                             GpdfSqrt, Xdel2, EpsLoc, EpsSca, MinLocEst, MaxLocEst,
                             MinScaEst, MaxScaEst )
  #
  # If didn't converge, use the solution for smallest Hellinger distance in the grid
    if( Estimate[[4]] != 0 ) {
      Estimate[[1]] <- NewInitial[[1]]
      Estimate[[2]] <- NewInitial[[2]]
    }
  }
  #
  #--------------------------------------------------------------------------
  # Use the location and scale estimates in a goodness of fit test for normality
  #
  # Evaluate the normal distribution at the estimated location and scale
  Npdf <- dnorm(Xint,Estimate[[1]],Estimate[[2]])
  NpdfSqrt <- sqrt( Npdf )
  #
  # Calculate the minimized Hellinger distance
  Hdis <- mhde.hdis( NGauss, Nsubs, WG, GpdfSqrt, NpdfSqrt, Xdel2 )
  #
  # Convert the minimum distance into a probability level (p value) for
  # the goodness of fit test for normality
  # Load the data from which to calculate the p values
  pcoef <- mhde.loadpval()
  pvalue <- mhde.pval(pcoef,Nsize,Hdis)
  #
  #--------------------------------------------------------------------------
  # Write the final summary information to the R console
  if( !Silent ) {
    cat("Minimum Hellinger Distance normality test","\n")
    cat("Data set          ",deparse(substitute(DataVec)),"\n")
    cat("Sample size       ",Nsize,"\n")
    cat("Location estimate ",Estimate[[1]],"\n")
    cat("Scale estimate    ",Estimate[[2]],"\n")
    cat("Hellinger Distance",Hdis,"\n")
    cat("p-value           ",pvalue,"\n")
  }
  #
  # Return the results in a list
  if( Small ) {return(list(Hdis,pvalue))}
  return(list(Hdis,pvalue,InitLocation,InitScale,Estimate[[1]],Estimate[[2]],Nsize,Cne,Xint,Gpdf,Npdf))
}

#-------------------------------------------------------------------------------------------------------------
#' Plot the non-parametric and normal densities
#
#' @description
#'   Simple plot function to display the data-based kernel density and the normal density for the final location and scale estimates.
#
# History:
#    Paul W. Eslinger : 17 Aug 2015 : Original source
#
# Parameters:
#' @param   ListOut    The output list from \code{mhde.test}.  The argument \code{Small} for \code{mhde.test} must have the value FALSE.
#
# Returned values:
#   None.
#' @export
#' @importFrom graphics legend lines plot
#-------------------------------------------------------------------------------------------------------------
mhde.plot <- function(ListOut) {
#
# Plot range
xrange <- range( ListOut[[9]] )
yrange <- range( c(ListOut[[10]]),(ListOut[[11]]) )
yrange[1] <- 0.0
plot( xrange, yrange, type="n", xlab="Sample Data Range", ylab="Proability Density",
  main="Kernel and Normal Densities", col.lab="black", col.main="black" )
lines(ListOut[[9]], ListOut[[10]], type="l", lwd=2.5, lty=1, pch=18)
lines(ListOut[[9]], ListOut[[11]], type="l", lwd=2.5, lty=1, pch=19, col="red")
legend(xrange[1], yrange[2], legend=c("normal","kernel"), cex=0.9, col=c("red","black"),
  pch=c(18,19), lty=c(1,1), title="Densities")
}
