% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/metaprop.R
\name{metaprop}
\alias{metaprop}
\title{Meta-analysis of single proportions}
\usage{
metaprop(event, n, studlab, data = NULL, subset = NULL,
  exclude = NULL, method, sm = gs("smprop"), incr = gs("incr"),
  allincr = gs("allincr"), addincr = gs("addincr"),
  method.ci = gs("method.ci"), level = gs("level"),
  level.comb = gs("level.comb"), comb.fixed = gs("comb.fixed"),
  comb.random = gs("comb.random"), hakn = gs("hakn"), method.tau,
  tau.preset = NULL, TE.tau = NULL, tau.common = gs("tau.common"),
  prediction = gs("prediction"), level.predict = gs("level.predict"),
  null.effect = NA, method.bias = gs("method.bias"),
  backtransf = gs("backtransf"), pscale = 1, title = gs("title"),
  complab = gs("complab"), outclab = "", byvar, bylab,
  print.byvar = gs("print.byvar"), byseparator = gs("byseparator"),
  keepdata = gs("keepdata"), warn = gs("warn"), control = NULL, ...)
}
\arguments{
\item{event}{Number of events.}

\item{n}{Number of observations.}

\item{studlab}{An optional vector with study labels.}

\item{data}{An optional data frame containing the study
information, i.e., event and n.}

\item{subset}{An optional vector specifying a subset of studies to
be used.}

\item{exclude}{An optional vector specifying studies to exclude
from meta-analysis, however, to include in printouts and forest
plots.}

\item{method}{A character string indicating which method is to be
used for pooling of studies. One of \code{"Inverse"} and
\code{"GLMM"}, can be abbreviated.}

\item{sm}{A character string indicating which summary measure
(\code{"PLOGIT"}, \code{"PAS"}, \code{"PFT"}, \code{"PLN"}, or
\code{"PRAW"}) is to be used for pooling of studies, see Details.}

\item{incr}{A numeric which is added to event number and sample
size of studies with zero or all events, i.e., studies with an
event probability of either 0 or 1.}

\item{allincr}{A logical indicating if \code{incr} is considered
for all studies if at least one study has either zero or all
events. If FALSE (default), \code{incr} is considered only in
studies with zero or all events.}

\item{addincr}{A logical indicating if \code{incr} is used for all
studies irrespective of number of events.}

\item{method.ci}{A character string indicating which method is used
to calculate confidence intervals for individual studies, see
Details.}

\item{level}{The level used to calculate confidence intervals for
individual studies.}

\item{level.comb}{The level used to calculate confidence intervals
for pooled estimates.}

\item{comb.fixed}{A logical indicating whether a fixed effect
meta-analysis should be conducted.}

\item{comb.random}{A logical indicating whether a random effects
meta-analysis should be conducted.}

\item{hakn}{A logical indicating whether the method by Hartung and
Knapp should be used to adjust test statistics and confidence
intervals.}

\item{method.tau}{A character string indicating which method is
used to estimate the between-study variance \eqn{\tau^2}, see
Details.}

\item{tau.preset}{Prespecified value for the square-root of the
between-study variance \eqn{\tau^2}.}

\item{TE.tau}{Overall treatment effect used to estimate the
between-study variance tau-squared.}

\item{tau.common}{A logical indicating whether tau-squared should
be the same across subgroups.}

\item{prediction}{A logical indicating whether a prediction
interval should be printed.}

\item{level.predict}{The level used to calculate prediction
interval for a new study.}

\item{null.effect}{A numeric value specifying the effect under the
null hypothesis.}

\item{method.bias}{A character string indicating which test is to
be used.  Either \code{"rank"}, \code{"linreg"}, or \code{"mm"},
can be abbreviated.  See function \code{\link{metabias}}.}

\item{backtransf}{A logical indicating whether results for
transformed proportions (argument \code{sm != "PRAW"}) should be
back transformed in printouts and plots. If TRUE (default),
results will be presented as proportions; otherwise transformed
proportions will be shown. See Details for presentation of
confidence intervals.}

\item{pscale}{A numeric defining a scaling factor for printing of
single event probabilities.}

\item{title}{Title of meta-analysis / systematic review.}

\item{complab}{Comparison label.}

\item{outclab}{Outcome label.}

\item{byvar}{An optional vector containing grouping information
(must be of same length as \code{event}).}

\item{bylab}{A character string with a label for the grouping
variable.}

\item{print.byvar}{A logical indicating whether the name of the
grouping variable should be printed in front of the group labels.}

\item{byseparator}{A character string defining the separator
between label and levels of grouping variable.}

\item{keepdata}{A logical indicating whether original data (set)
should be kept in meta object.}

\item{warn}{A logical indicating whether the addition of
\code{incr} to studies with zero or all events should result in a
warning.}

\item{control}{An optional list to control the iterative process to
estimate the between-study variance tau^2. This argument is
passed on to \code{\link[metafor]{rma.uni}} or
\code{\link[metafor]{rma.glmm}}, respectively.}

\item{\dots}{Additional arguments passed on to
\code{\link[metafor]{rma.glmm}} function.}
}
\value{
An object of class \code{c("metaprop", "meta")} with corresponding
\code{print}, \code{summary}, and \code{forest} functions. The
object is a list containing the following components:
\item{event, n, studlab, exclude,}{As defined above.}
\item{sm, incr, allincr, addincr, method.ci,}{As defined above.}
\item{level, level.comb,}{As defined above.}
\item{comb.fixed, comb.random,}{As defined above.}
\item{hakn, method.tau, tau.preset, TE.tau, null.hypothesis,}{As
  defined above.}
\item{method.bias, tau.common, title, complab, outclab,}{As defined
  above.}
\item{byvar, bylab, print.byvar, byseparator, warn}{As defined
  above.}
\item{TE, seTE}{Estimated (un)transformed proportion and its
  standard error for individual studies.}
\item{lower, upper}{Lower and upper confidence interval limits for
  individual studies.}
\item{zval, pval}{z-value and p-value for test of treatment effect
  for individual studies.}
\item{w.fixed, w.random}{Weight of individual studies (in fixed and
  random effects model).}
\item{TE.fixed, seTE.fixed}{Estimated overall (un)transformed
  proportion and standard error (fixed effect model).}
\item{lower.fixed, upper.fixed}{Lower and upper confidence interval
  limits (fixed effect model).}
\item{zval.fixed, pval.fixed}{z-value and p-value for test of
  overall effect (fixed effect model).}
\item{TE.random, seTE.random}{Estimated overall (un)transformed
  proportion and standard error (random effects model).}
\item{lower.random, upper.random}{Lower and upper confidence
  interval limits (random effects model).}
\item{zval.random, pval.random}{z-value or t-value and
  corresponding p-value for test of overall effect (random effects
  model).}
\item{prediction, level.predict}{As defined above.}
\item{seTE.predict}{Standard error utilised for prediction
  interval.}
\item{lower.predict, upper.predict}{Lower and upper limits of
  prediction interval.}
\item{k}{Number of studies combined in meta-analysis.}
\item{Q}{Heterogeneity statistic Q.}
\item{df.Q}{Degrees of freedom for heterogeneity statistic.}
\item{pval.Q}{P-value of heterogeneity test.}
\item{Q.LRT}{Heterogeneity statistic for likelihood-ratio test
  (only if \code{method = "GLMM"}).}
\item{df.Q.LRT}{Degrees of freedom for likelihood-ratio test}
\item{pval.Q.LRT}{P-value of likelihood-ratio test.}
\item{tau}{Square-root of between-study variance.}
\item{se.tau2}{Standard error of between-study variance.}
\item{C}{Scaling factor utilised internally to calculate common
  tau-squared across subgroups.}
\item{method}{A character string indicating method used for
  pooling: \code{"Inverse"}}
\item{df.hakn}{Degrees of freedom for test of treatment effect for
  Hartung-Knapp method (only if \code{hakn=TRUE}).}
\item{bylevs}{Levels of grouping variable - if \code{byvar} is not
  missing.}
\item{TE.fixed.w, seTE.fixed.w}{Estimated treatment effect and
  standard error in subgroups (fixed effect model) - if
  \code{byvar} is not missing.}
\item{lower.fixed.w, upper.fixed.w}{Lower and upper confidence
  interval limits in subgroups (fixed effect model) - if
  \code{byvar} is not missing.}
\item{zval.fixed.w, pval.fixed.w}{z-value and p-value for test of
  treatment effect in subgroups (fixed effect model) - if
  \code{byvar} is not missing.}
\item{TE.random.w, seTE.random.w}{Estimated treatment effect and
  standard error in subgroups (random effects model) - if
  \code{byvar} is not missing.}
\item{lower.random.w, upper.random.w}{Lower and upper confidence
  interval limits in subgroups (random effects model) - if
  \code{byvar} is not missing.}
\item{zval.random.w, pval.random.w}{z-value or t-value and
  corresponding p-value for test of treatment effect in subgroups
  (random effects model) - if \code{byvar} is not missing.}
\item{w.fixed.w, w.random.w}{Weight of subgroups (in fixed and
  random effects model) - if \code{byvar} is not missing.}
\item{df.hakn.w}{Degrees of freedom for test of treatment effect
  for Hartung-Knapp method in subgroups - if \code{byvar} is not
  missing and \code{hakn=TRUE}.}
\item{n.harmonic.mean.w}{Harmonic mean of number of observations in
  subgroups (for back transformation of Freeman-Tukey Double
  arcsine transformation) - if \code{byvar} is not missing.}
\item{event.w}{Number of events in subgroups - if \code{byvar} is
  not missing.}
\item{n.w}{Number of observations in subgroups - if \code{byvar} is
  not missing.}
\item{k.w}{Number of studies combined within subgroups - if
  \code{byvar} is not missing.}
\item{k.all.w}{Number of all studies in subgroups - if \code{byvar}
  is not missing.}
\item{Q.w.fixed}{Overall within subgroups heterogeneity statistic Q
  (based on fixed effect model) - if \code{byvar} is not missing.}
\item{Q.w.random}{Overall within subgroups heterogeneity statistic
  Q (based on random effects model) - if \code{byvar} is not
  missing (only calculated if argument \code{tau.common} is TRUE).}
\item{df.Q.w}{Degrees of freedom for test of overall within
  subgroups heterogeneity - if \code{byvar} is not missing.}
\item{pval.Q.w.fixed}{P-value of within subgroups heterogeneity
  statistic Q (based on fixed effect model) - if \code{byvar} is
  not missing.}
\item{pval.Q.w.random}{P-value of within subgroups heterogeneity
  statistic Q (based on random effects model) - if \code{byvar} is
  not missing.}
\item{Q.b.fixed}{Overall between subgroups heterogeneity statistic
  Q (based on fixed effect model) - if \code{byvar} is not
  missing.}
\item{Q.b.random}{Overall between subgroups heterogeneity statistic
  Q (based on random effects model) - if \code{byvar} is not
  missing.}
\item{df.Q.b}{Degrees of freedom for test of overall between
  subgroups heterogeneity - if \code{byvar} is not missing.}
\item{pval.Q.b.fixed}{P-value of between subgroups heterogeneity
  statistic Q (based on fixed effect model) - if \code{byvar} is
  not missing.}
\item{pval.Q.b.random}{P-value of between subgroups heterogeneity
  statistic Q (based on random effects model) - if \code{byvar} is
  not missing.}
\item{tau.w}{Square-root of between-study variance within subgroups
  - if \code{byvar} is not missing.}
\item{C.w}{Scaling factor utilised internally to calculate common
  tau-squared across subgroups - if \code{byvar} is not missing.}
\item{H.w}{Heterogeneity statistic H within subgroups - if
  \code{byvar} is not missing.}
\item{lower.H.w, upper.H.w}{Lower and upper confidence limti for
  heterogeneity statistic H within subgroups - if \code{byvar} is
  not missing.}
\item{I2.w}{Heterogeneity statistic I2 within subgroups - if
  \code{byvar} is not missing.}
\item{lower.I2.w, upper.I2.w}{Lower and upper confidence limti for
  heterogeneity statistic I2 within subgroups - if \code{byvar} is
  not missing.}
\item{incr.event}{Increment added to number of events.}
\item{keepdata}{As defined above.}
\item{data}{Original data (set) used in function call (if
  \code{keepdata=TRUE}).}
\item{subset}{Information on subset of original data used in
  meta-analysis (if \code{keepdata=TRUE}).}
\item{.glmm.fixed}{GLMM object generated by call of
  \code{\link[metafor]{rma.glmm}} function (fixed effect model).}
\item{.glmm.random}{GLMM object generated by call of
  \code{\link[metafor]{rma.glmm}} function (random effects model).}
\item{call}{Function call.}
\item{version}{Version of R package \bold{meta} used to create
  object.}
\item{version.metafor}{Version of R package \bold{metafor} used for
  GLMMs.}
}
\description{
Calculation of an overall proportion from studies reporting a
single proportion. Inverse variance method and generalised linear
mixed model (GLMM) are available for pooling. For GLMMs, the
\code{\link[metafor]{rma.glmm}} function from R package
\bold{metafor} (Viechtbauer 2010) is called internally.
}
\details{
This function provides methods for fixed effect and random effects
meta-analysis of single proportions to calculate an overall
proportion. Note, you should use R function \code{\link{metabin}}
to compare proportions of pairwise comparisons instead of using
\code{metaprop} for each treatment arm separately which will break
randomisation in randomised controlled trials.

The following transformations of proportions are
implemented to calculate an overall proportion:

\itemize{
\item Logit transformation (\code{sm = "PLOGIT"}, default)
\item Arcsine transformation (\code{sm = "PAS"})
\item Freeman-Tukey Double arcsine transformation (\code{sm = "PFT"})
\item Log transformation (\code{sm = "PLN"})
\item Raw, i.e. untransformed, proportions (\code{sm = "PRAW"})
}

A generalised linear mixed model (GLMM) - more specific, a random
intercept logistic regression model - can be utilised for the
meta-analysis of proportions (Stijnen et al., 2010). This is the
default method for the logit transformation (argument \code{sm =
"PLOGIT"}). Internally, the \code{\link[metafor]{rma.glmm}}
function from R package \bold{metafor} is called to fit a GLMM.

Classic meta-analysis (Borenstein et al., 2010) utilising the
(un)transformed proportions and corresponding standard errors in
the inverse variance method is conducted by calling the
\code{\link{metagen}} function internally. This is the only
available method for all transformations but the logit
transformation. The classic meta-analysis model with logit
transformed proportions is used by setting argument \code{method =
"Inverse"}.

Default settings are utilised for several arguments (assignments
using \code{\link{gs}} function). These defaults can be changed for
the current R session using the \code{\link{settings.meta}}
function.

Furthermore, R function \code{\link{update.meta}} can be used to
rerun a meta-analysis with different settings.

\subsection{Choice of transformation / meta-analysis method}{

Contradictory recommendations on the use of transformations of
proportions have been published in the literature. For example,
Barendregt et al. (2013) recommend the use of the Freeman-Tukey
double arcsine transformation instead of the logit transformation
whereas Warton & Hui (2011) strongly advise to use generalised
linear mixed models with the logit transformation instead of the
arcsine transformation.

Schwarzer et al. (2019) describe seriously misleading results in a
meta-analysis with very different sample sizes due to problems with
the back-transformation of the Freeman-Tukey transformation which
requires a single sample size (Miller, 1978). Accordingly,
Schwarzer et al. (2019) also recommend to use GLMMs for the
meta-analysis of single proportions, however, admit that individual
study weights are not available with this method. Meta-analysts
which require individual study weights should consider the inverse
variance method with the arcsine or logit transformation.

In order to prevent misleading conclusions for the Freeman-Tukey
double arcsine transformation, sensitivity analyses using other
transformations or using a range of sample sizes should be
conducted (Schwarzer et al., 2019).
}

\subsection{Continuity correction}{

If the summary measure is equal to "PLOGIT", "PLN", or "PRAW", a
continuity correction is applied if any study has either zero or
all events, i.e., an event probability of either 0 or 1.

By default, 0.5 is used as continuity correction (argument
\code{incr}). This continuity correction is used both to calculate
individual study results with confidence limits and to conduct
meta-analysis based on the inverse variance method. For GLMMs no
continuity correction is used.
}

\subsection{Confidence intervals for individual studies}{

Various methods are available to calculate confidence intervals for
individual study results (see Agresti & Coull 1998 and Newcombe
1988):
\itemize{
\item Clopper-Pearson interval also called 'exact' binomial
  interval (\code{method.ci = "CP"}, default)
\item Wilson Score interval (\code{method.ci = "WS"})
\item Wilson Score interval with continuity correction
  (\code{method.ci = "WSCC"})
\item Agresti-Coull interval (\code{method.ci = "AC"})
\item Simple approximation interval (\code{method.ci = "SA"})
\item Simple approximation interval with continuity correction
  (\code{method.ci = "SACC"})
\item Normal approximation interval based on summary measure,
  i.e. defined by argument \code{sm} (\code{method.ci = "NAsm"})
}

Note, with exception of the normal approximation based on the
summary measure, i.e. \code{method.ci = "NAsm"}, the same
confidence interval is calculated for individual studies for any
summary measure (argument \code{sm}) as only number of events and
observations are used in the calculation disregarding the chosen
transformation.

Results will be presented for transformed proportions if argument
\code{backtransf = FALSE} in the \code{\link{print.meta}},
\code{\link{print.summary.meta}}, or \code{\link{forest.meta}}
function. In this case, argument \code{method.ci = "NAsm"} is used,
i.e. confidence intervals based on the normal approximation based
on the summary measure.
}

\subsection{Estimation of between-study variance}{

The following methods to estimate the between-study variance
\eqn{\tau^2} (argument \code{method.tau}) are available for the
inverse variance method:
\itemize{
\item DerSimonian-Laird estimator (\code{method.tau = "DL"})
\item Paule-Mandel estimator (\code{method.tau = "PM"})
\item Restricted maximum-likelihood estimator (\code{method.tau =
  "REML"})
\item Maximum-likelihood estimator (\code{method.tau = "ML"})
\item Hunter-Schmidt estimator (\code{method.tau = "HS"})
\item Sidik-Jonkman estimator (\code{method.tau = "SJ"})
\item Hedges estimator (\code{method.tau = "HE"})
\item Empirical Bayes estimator (\code{method.tau = "EB"})
}
See \code{\link{metagen}} for more information on these
estimators. Note, the maximum-likelihood method is utilized for
GLMMs.
}

\subsection{Hartung-Knapp method}{

Hartung and Knapp (2001a,b) proposed an alternative method for
random effects meta-analysis based on a refined variance estimator
for the treatment estimate. Simulation studies (Hartung and Knapp,
2001a,b; IntHout et al., 2014; Langan et al., 2019) show improved
coverage probabilities compared to the classic random effects
method. However, in rare settings with very homogeneous treatment
estimates, the Hartung-Knapp method can be anti-conservative
(Wiksten et al., 2016). The Hartung-Knapp method is used if
argument \code{hakn = TRUE}.
}

\subsection{Prediction interval}{

A prediction interval for the proportion in a new study (Higgins et
al., 2009) is calculated if arguments \code{prediction} and
\code{comb.random} are \code{TRUE}. Note, the definition of
prediction intervals varies in the literature. This function
implements equation (12) of Higgins et al., (2009) which proposed a
\emph{t} distribution with \emph{K-2} degrees of freedom where
\emph{K} corresponds to the number of studies in the meta-analysis.
}

\subsection{Subgroup analysis}{

Argument \code{byvar} can be used to conduct subgroup analysis for
a categorical covariate. The \code{\link{metareg}} function can be
used instead for more than one categorical covariate or continuous
covariates.
}

\subsection{Specify the null hypothesis of test for an overall proportion}{

Argument \code{null.effect} can be used to specify the proportion
used under the null hypothesis in a test for an overall effect.

By default (\code{null.effect = NA}), no hypothesis test is
conducted as it is unclear which value is a sensible choice for the
data at hand.  An overall proportion of 50\%, for example, could be
tested by setting argument \code{null.effect = 0.5}.

Note, all tests for an overall effect are two-sided with the
alternative hypothesis that the effect is unequal to
\code{null.effect}.
}

\subsection{Presentation of meta-analysis results}{

Internally, both fixed effect and random effects models are
calculated regardless of values choosen for arguments
\code{comb.fixed} and \code{comb.random}. Accordingly, the estimate
for the random effects model can be extracted from component
\code{TE.random} of an object of class \code{"meta"} even if
argument \code{comb.random = FALSE}. However, all functions in R
package \bold{meta} will adequately consider the values for
\code{comb.fixed} and \code{comb.random}. E.g. function
\code{\link{print.meta}} will not print results for the random
effects model if \code{comb.random = FALSE}.

Argument \code{pscale} can be used to rescale proportions, e.g.
\code{pscale = 1000} means that proportions are expressed as events
per 1000 observations. This is useful in situations with (very) low
event probabilities.
}
}
\examples{
# Meta-analysis using generalised linear mixed model
#
metaprop(4:1, 10 * 1:4)

# Apply various classic meta-analysis methods to estimate
# proportions
#
m1 <- metaprop(4:1, 10 * 1:4, method = "Inverse")
m2 <- update(m1, sm = "PAS")
m3 <- update(m1, sm = "PRAW")
m4 <- update(m1, sm = "PLN")
m5 <- update(m1, sm = "PFT")
#
m1
m2
m3
m4
m5
#
forest(m1)
\dontrun{
forest(m2)
forest(m3)
forest(m3, pscale = 100)
forest(m4)
forest(m5)
}

# Do not back transform results, e.g. print logit transformed
# proportions if sm = "PLOGIT" and store old settings
#
oldset <- settings.meta(backtransf = FALSE)
#
m6  <- metaprop(4:1, c(10, 20, 30, 40), method = "Inverse")
m7  <- update(m6, sm = "PAS")
m8  <- update(m6, sm = "PRAW")
m9  <- update(m6, sm = "PLN")
m10 <- update(m6, sm = "PFT")
#
forest(m6)
\dontrun{
forest(m7)
forest(m8)
forest(m8, pscale = 100)
forest(m9)
forest(m10)
}

# Use old settings
#
settings.meta(oldset)

# Examples with zero events
#
m1 <- metaprop(c(0, 0, 10, 10), rep(100, 4), method = "Inverse")
m2 <- metaprop(c(0, 0, 10, 10), rep(100, 4), incr = 0.1, method = "Inverse")
#
summary(m1)
summary(m2)
#
\dontrun{
forest(m1)
forest(m2)
}

# Example from Miller (1978):
#
death <- c(3, 6, 10, 1)
animals <- c(11, 17, 21, 6)
#
m3 <- metaprop(death, animals, sm = "PFT")
forest(m3)

# Data examples from Newcombe (1998)
# - apply various methods to estimate confidence intervals for
#   individual studies
#
event <- c(81, 15, 0, 1)
n <- c(263, 148, 20, 29)
#
m1 <- metaprop(event, n, method.ci = "SA", method = "Inverse")
m2 <- update(m1, method.ci = "SACC")
m3 <- update(m1, method.ci = "WS")
m4 <- update(m1, method.ci = "WSCC")
m5 <- update(m1, method.ci = "CP")
#
lower <- round(rbind(NA, m1$lower, m2$lower, NA, m3$lower,
                     m4$lower, NA, m5$lower), 4)
upper <- round(rbind(NA, m1$upper, m2$upper, NA, m3$upper,
                     m4$upper, NA, m5$upper), 4)
#
tab1 <- data.frame(
  scen1 = meta:::formatCI(lower[, 1], upper[, 1]),
  scen2 = meta:::formatCI(lower[, 2], upper[, 2]),
  scen3 = meta:::formatCI(lower[, 3], upper[, 3]),
  scen4 = meta:::formatCI(lower[, 4], upper[, 4]),
  stringsAsFactors = FALSE
  )
names(tab1) <- c("r=81, n=263", "r=15, n=148",
                 "r=0, n=20", "r=1, n=29")
row.names(tab1) <- c("Simple", "- SA", "- SACC",
                     "Score", "- WS", "- WSCC",
                     "Binomial", "- CP")
tab1[is.na(tab1)] <- ""
# Newcombe (1998), Table I, methods 1-5:
tab1

# Same confidence interval, i.e. unaffected by choice of summary
# measure
#
print(metaprop(event, n, method.ci = "WS", method = "Inverse"), ma = FALSE)
print(metaprop(event, n, sm = "PLN", method.ci = "WS"), ma = FALSE)
print(metaprop(event, n, sm = "PFT", method.ci = "WS"), ma = FALSE)
print(metaprop(event, n, sm = "PAS", method.ci = "WS"), ma = FALSE)
print(metaprop(event, n, sm = "PRAW", method.ci = "WS"), ma = FALSE)

# Different confidence intervals as argument sm = "NAsm"
#
print(metaprop(event, n, method.ci = "NAsm", method = "Inverse"), ma = FALSE)
print(metaprop(event, n, sm = "PLN", method.ci = "NAsm"), ma = FALSE)
print(metaprop(event, n, sm = "PFT", method.ci = "NAsm"), ma = FALSE)
print(metaprop(event, n, sm = "PAS", method.ci = "NAsm"), ma = FALSE)
print(metaprop(event, n, sm = "PRAW", method.ci = "NAsm"), ma = FALSE)

# Different confidence intervals as argument backtransf = FALSE.
# Accordingly, method.ci = "NAsm" used internally.
#
print(metaprop(event, n, method.ci = "WS", method = "Inverse"),
      ma = FALSE, backtransf = FALSE)
print(metaprop(event, n, sm = "PLN", method.ci = "WS"),
      ma = FALSE, backtransf = FALSE)
print(metaprop(event, n, sm = "PFT", method.ci = "WS"),
      ma = FALSE, backtransf = FALSE)
print(metaprop(event, n, sm = "PAS", method.ci = "WS"),
      ma = FALSE, backtransf = FALSE)
print(metaprop(event, n, sm = "PRAW", method.ci = "WS"),
      ma = FALSE, backtransf = FALSE)

# Same results (printed on original and log scale, respectively)
#
print(metaprop(event, n, sm = "PLN", method.ci = "NAsm"), ma = FALSE)
print(metaprop(event, n, sm = "PLN"), ma = FALSE, backtransf = FALSE)
# Results for first study (on log scale)
round(log(c(0.3079848, 0.2569522, 0.3691529)), 4)

# Print results as events per 1000 observations
#
print(metaprop(6:8, c(100, 1200, 1000), method = "Inverse"),
      pscale = 1000, digits = 1)

}
\references{
Agresti A & Coull BA (1998):
Approximate is better than "exact" for interval estimation of
binomial proportions.
\emph{The American Statistician},
\bold{52}, 119--26

Barendregt JJ, Doi SA, Lee YY, Norman RE, Vos T (2013):
Meta-analysis of prevalence.
\emph{Journal of Epidemiology and Community Health},
\bold{67}, 974--8

Borenstein M, Hedges LV, Higgins JP, Rothstein HR (2010):
A basic introduction to fixed-effect and random-effects models for
meta-analysis.
\emph{Research Synthesis Methods},
\bold{1}, 97--111

DerSimonian R & Laird N (1986):
Meta-analysis in clinical trials.
\emph{Controlled Clinical Trials},
\bold{7}, 177--88

Edward JM et al. (2006):
Adherence to antiretroviral therapy in sub-saharan
Africa and North America - a meta-analysis.
\emph{Journal of the American Medical Association},
\bold{296}, 679--90

Freeman MF & Tukey JW (1950):
Transformations related to the angular and the square root.
\emph{Annals of Mathematical Statistics},
\bold{21}, 607--11

Higgins JPT, Thompson SG, Spiegelhalter DJ (2009):
A re-evaluation of random-effects meta-analysis.
\emph{Journal of the Royal Statistical Society: Series A},
\bold{172}, 137--59

Hartung J, Knapp G (2001a):
On tests of the overall treatment effect in meta-analysis with
normally distributed responses.
\emph{Statistics in Medicine},
\bold{20}, 1771--82

Hartung J, Knapp G (2001b):
A refined method for the meta-analysis of controlled clinical
trials with binary outcome.
\emph{Statistics in Medicine},
\bold{20}, 3875--89

IntHout J, Ioannidis JPA, Borm GF (2014):
The Hartung-Knapp-Sidik-Jonkman method for random effects
meta-analysis is straightforward and considerably outperforms the
standard DerSimonian-Laird method.
\emph{BMC Medical Research Methodology},
\bold{14}, 25

Langan D, Higgins JPT, Jackson D, Bowden J, Veroniki AA,
Kontopantelis E, et al. (2019):
A comparison of heterogeneity variance estimators in simulated
random-effects meta-analyses.
\emph{Research Synthesis Methods},
\bold{10}, 83--98

Miller JJ (1978):
The inverse of the Freeman-Tukey double arcsine transformation.
\emph{The American Statistician},
\bold{32}, 138

Newcombe RG (1998):
Two-sided confidence intervals for the single proportion:
comparison of seven methods.
\emph{Statistics in Medicine},
\bold{17}, 857--72

Pettigrew HM, Gart JJ, Thomas DG (1986):
The bias and higher cumulants of the logarithm of a binomial
variate.
\emph{Biometrika},
\bold{73}, 425--35

Schwarzer G, Chemaitelly H, Abu-Raddad LJ, Rücker G (2019):
Seriously misleading results using inverse of Freeman-Tukey double
arcsine transformation in meta-analysis of single proportions.
\emph{Research Synthesis Methods},
\bold{10}, 476--83

Stijnen T, Hamza TH, Ozdemir P (2010):
Random effects meta-analysis of event outcome in the framework of
the generalized linear mixed model with applications in sparse
data.
\emph{Statistics in Medicine},
\bold{29}, 3046--67

Viechtbauer W (2010):
Conducting meta-analyses in R with the metafor package.
\emph{Journal of Statistical Software},
\bold{36}, 1--48

Warton DI, Hui FKC (2011):
The arcsine is asinine: the analysis of proportions in ecology.
\emph{Ecology},
\bold{92}, 3--10

Wiksten A, Rücker G, Schwarzer G (2016):
Hartung-Knapp method is not always conservative compared with
fixed-effect meta-analysis.
\emph{Statistics in Medicine},
\bold{35}, 2503--15
}
\seealso{
\code{\link{update.meta}}, \code{\link{metacont}},
  \code{\link{metagen}}, \code{\link{print.meta}}
}
\author{
Guido Schwarzer \email{sc@imbi.uni-freiburg.de}
}
