/*
 *  Mathlib : A C Library of Special Functions
 *  Copyright (C) 1998	    Ross Ihaka
 *  Copyright (C) 2000-2013 The R Core Team
 *  Copyright (C) 2003	    The R Foundation
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, a copy is available at
 *  https://www.R-project.org/Licenses/
 */

#ifndef NORM_CDF_APPROX_H
#define NORM_CDF_APPROX_H
#include "config.h"
#include <cmath>

inline double fastncdf_pos(double const x) noexcept {
  constexpr double h = 0.0304370378105459,
                      hinv = 1. / h,
                      xmin = -6.36134090240406,
                      xinf = -37.5193;
  constexpr int n_pts = 210L;
  static constexpr double d_xmy[3L * n_pts] = {
    -6.36134090240406, 7.18354635949666e-10, 1e-10, -6.33090386459351, 7.94677529611875e-10, 1.21864587215781e-10, -6.30046682678296, 9.63052525768857e-10, 1.48375260031976e-10, -6.27002978897242, 1.16602215764173e-09, 1.80489519496518e-10, -6.23959275116187, 1.41046209439955e-09, 2.19355781032126e-10, -6.20915571335133, 1.70456600242307e-09, 2.66350095691678e-10, -6.17871867554078, 2.05808834094477e-09, 3.2311966076477e-10, -6.14828163773024, 2.48262994679732e-09, 3.91634320993238e-10, -6.11784459991969, 2.99197358062154e-09, 4.74247463885297e-10, -6.08740756210914, 3.60247774145568e-09, 5.73767946996302e-10, -6.0569705242986, 4.33353835871131e-09, 6.93544966341966e-10, -6.02653348648805, 5.20812946283561e-09, 8.37568088751392e-10, -5.99609644867751, 6.25343564391094e-09, 1.01058503310706e-09, -5.96565941086696, 7.50159105951519e-09, 1.21824020303046e-09, -5.93522237305642, 8.99054198115377e-09, 1.46723745454249e-09, -5.90478533524587, 1.07650524079096e-08, 1.76553113546581e-09, -5.87434829743532, 1.28778751660615e-08, 2.12255006888659e-09, -5.84391125962478, 1.53911141980619e-08, 2.5494598821636e-09, -5.81347422181423, 1.83778074719345e-08, 3.05946991847226e-09, -5.78303718400369, 2.19237641664728e-08, 3.66819192396e-09, -5.75260014619314, 2.61296945677588e-08, 4.3940587962411e-09, -5.7221631083826, 3.11136765136101e-08, 5.25881292703378e-09, -5.69172607057205, 3.70140083161658e-08, 6.28807509318079e-09, -5.6612890327615, 4.39925049573534e-08, 7.51200646831177e-09, -5.63085199495096, 5.22383020731982e-08, 8.96607816671593e-09, -5.60041495714041, 6.19722409143527e-08, 1.0691964819033e-08, -5.56997791932987, 7.345191719361e-08, 1.27385810465448e-08, -5.53954088151932, 8.69774876169069e-08, 1.5163282380791e-08, -5.50910384370878, 1.0289834006192e-07, 1.80332552050689e-08, -5.47866680589823, 1.21620746946713e-07, 2.14271237150052e-08, -5.44822976808768, 1.43616636458711e-07, 2.54368057517968e-08, -5.41779273027714, 1.69433633140358e-07, 3.01696537032394e-08, -5.38735569246659, 1.99706538011065e-07, 3.57509215483391e-08, -5.35691865465605, 2.35170439115035e-07, 4.23266046001514e-08, -5.3264816168455, 2.76675666300958e-07, 5.00667046428731e-08, -5.29604457903496, 3.25204829352496e-07, 5.91689800330718e-08, -5.26560754122441, 3.81892206498218e-07, 6.98632480172211e-08, -5.23517050341386, 4.48045781045799e-07, 8.24163150905499e-08, -5.20473346560332, 5.25172257626866e-07, 9.71376207743138e-08, -5.17429642779277, 6.15005426532805e-07, 1.14385690815427e-07, -5.14385938998223, 7.19538285102339e-07, 1.34575507616453e-07, -5.11342235217168, 8.41059369328824e-07, 1.58186918795017e-07, -5.08298531436114, 9.82193797033161e-07, 1.85774219266804e-07, -5.05254827655059, 1.14594957634333e-06, 2.1797705827018e-07, -5.02211123874004, 1.33576979008021e-06, 2.55532840435086e-07, -4.9916742009295, 1.55559132821499e-06, 2.99290809483893e-07, -4.96123716311895, 1.80991090707338e-06, 3.50228024584359e-07, -4.93080012530841, 2.10385918564399e-06, 4.09467462908516e-07, -4.90036308749786, 2.44328386641632e-06, 4.78298507747379e-07, -4.86992604968732, 2.83484275082393e-06, 5.58200109756537e-07, -4.83948901187677, 3.28610780777195e-06, 6.50866939734936e-07, -4.80905197406622, 3.80568140802842e-06, 7.58238884945907e-07, -4.77861493625568, 4.40332597756245e-06, 8.8253427755704e-07, -4.74817789844513, 5.09010842928394e-06, 1.02628728348836e-06, -4.71774086063459, 5.87856084509308e-06, 1.19238992300083e-06, -4.68730382282404, 6.78285899863951e-06, 1.38413924091574e-06, -4.6568667850135, 7.81902043360944e-06, 1.60529019461121e-06, -4.62642974720295, 9.00512394251629e-06, 1.86011488207414e-06, -4.5959927093924, 1.03615524265933e-05, 2.15346879046525e-06, -4.56555567158186, 1.19112612581026e-05, 2.49086480804248e-06, -4.53511863377131, 1.36800744117091e-05, 2.87855580903356e-06, -4.50468159596077, 1.56970107809175e-05, 3.32362669228302e-06, -4.47424455815022, 1.79946432482145e-05, 3.83409683033623e-06, -4.44380752033967, 2.06094932326192e-05, 4.4190339661494e-06, -4.41337048252913, 2.35824635948083e-05, 5.08868067989104e-06, -4.38293344471858, 2.69593129366472e-05, 5.85459463835138e-06, -4.35249640690804, 3.0791174487468e-05, 6.72980393428919e-06, -4.32205936909749, 3.51351229222269e-05, 7.72897892256374e-06, -4.29162233128695, 4.0054792605e-05, 8.86862206401318e-06, -4.2611852934764, 4.56210508931633e-05, 1.01672773965877e-05, -4.23074825566585, 5.19127302708832e-05, 1.16457613659973e-05, -4.20031121785531, 5.90174232028047e-05, 1.33274168647947e-05, -4.16987418004476, 6.7032343707431e-05, 1.52383924490068e-05, -4.13943714223422, 7.60652597417772e-05, 1.74079488246998e-05, -4.10900010442367, 8.62355005613409e-05, 1.98687948226656e-05, -4.07856306661313, 9.76750432689356e-05, 2.26574552070936e-05, -4.04812602880258, 0.0001105296827889, 2.58146727929121e-05, -4.01768899099203, 0.000124960289495848, 2.93858474755604e-05, -3.98725195318149, 0.000141144158685447, 3.3421514905316e-05, -3.95681491537094, 0.000159276455994531, 3.79778676648536e-05, -3.9263778775604, 0.000179571762754752, 4.31173219321865e-05, -3.89594083974985, 0.000202265725093469, 4.89091327301991e-05, -3.86550380193931, 0.000227616810372992, 5.54300609770812e-05, -3.83506676412876, 0.000255908174279877, 6.27650956574762e-05, -3.80462972631822, 0.000287449641534952, 7.10082345302499e-05, -3.77419268850767, 0.000322579802787325, 8.02633268735305e-05, -3.74375565069712, 0.000361668229777037, 9.06449818389622e-05, -3.71331861288658, 0.000405117810296592, 0.000102279546042724, -3.68288157507603, 0.000453367203846923, 0.000115306154058408, -3.65244453726549, 0.000506893418164093, 0.000129877855493825, -3.62200749945494, 0.000566214505985323, 0.000146162822327562, -3.5915704616444, 0.000631892380523166, 0.000164345640148934, -3.56113342383385, 0.000704535747121845, 0.000184628686883921, -3.5306963860233, 0.000784803147477258, 0.000207233602496991, -3.50025934821276, 0.000873406111610314, 0.000232402853031122, -3.46982231040221, 0.000971112411490767, 0.000260401392183081, -3.43938527259167, 0.00107874940881551, 0.000291518423404792, -3.40894823478112, 0.00119720748795193, 0.000326069265271524, -3.37851119697058, 0.00132744356346553, 0.000364397322560515, -3.34807415916003, 0.0014704846499638, 0.000406876165136656, -3.31763712134948, 0.00162743148021059, 0.000453911716342065, -3.28720008353894, 0.00179946215560148, 0.00050594455213114, -3.25676304572839, 0.00198783581114817, 0.000563452311679442, -3.22632600791785, 0.00219389627510803, 0.000626952219621287, -3.1958889701073, 0.00241907570132238, 0.000697003719435199, -3.16545193229675, 0.00266489815020669, 0.000774211216796731, -3.13501489448621, 0.00293298309218096, 0.000859226930953388, -3.10457785667566, 0.00322504880515377, 0.000952753851345537, -3.07414081886512, 0.00354291563549677, 0.00105554879580003, -3.04370378105457, 0.00388850908978577, 0.00116842556565992, -3.01326674324403, 0.00426386272246076, 0.00129225819218495, -2.98282970543348, 0.00467112078249316, 0.00142798426746494, -2.95239266762293, 0.0051125405801664, 0.00157660835193369, -2.92195562981239, 0.00559049453320347, 0.00173920544935789, -2.89151859200184, 0.00610747184973699, 0.00191692453890722, -2.8610815541913, 0.00666607980404313, 0.00211099215259247, -2.83064451638075, 0.0072690445595788, 0.00232271598499477, -2.80020747857021, 0.00791921149270265, 0.00255348852080535, -2.76977044075966, 0.00861954496955518, 0.00280479066426097, -2.73933340294911, 0.00937312752795234, 0.00307819535310145, -2.70889636513857, 0.010183158415842, 0.00337537113820367, -2.67845932732802, 0.0110529514379162, 0.00369808570856897, -2.64802228951748, 0.0119859320623907, 0.00404820933987163, -2.61758525170693, 0.0129856337407908, 0.00442771824332421, -2.58714821389639, 0.0140556933948445, 0.00483869779019632, -2.55671117608584, 0.0151998460263096, 0.00528334558594885, -2.52627413827529, 0.0164219184077667, 0.00576397436663085, -2.49583710046475, 0.0177258218151332, 0.00628301468894663, -2.4654000626542, 0.0191155437658921, 0.00684301738425125, -2.43496302484366, 0.020595138730814, 0.00744665574568984, -2.40452598703311, 0.0221687177912902, 0.00809672741677769, -2.37408894922257, 0.0238404372192851, 0.00879615594893947, -2.34365191141202, 0.0256144859623727, 0.00954799199490433, -2.31321487360148, 0.0274950720223337, 0.0103554141044083, -2.28277783579093, 0.0294864077223474, 0.0112217290883992, -2.25234079798038, 0.0315926938649058, 0.0121503719178928, -2.22190376016984, 0.0338181027901759, 0.0131449051238055, -2.19146672235929, 0.0361667603526257, 0.0142090176645039, -2.16102968454875, 0.038642726842262, 0.0153465232284812, -2.1305926467382, 0.0412499768857669, 0.0165613579405049, -2.10015560892765, 0.0439923783721264, 0.0178575774407936, -2.06971857111711, 0.0468736704569442, 0.0192393533082814, -2.03928153330656, 0.0498974407094835, 0.0207109688008277, -2.00884449549602, 0.0530671014765021, 0.0222768138873294, -1.97840745768547, 0.056385865547069, 0.0239413795491005, -1.94797041987493, 0.0598567212126841, 0.0257092513306023, -1.91753338206438, 0.0634824068271114, 0.027585102122632, -1.88709634425383, 0.0672653849802408, 0.0295736841644048, -1.85665930644329, 0.0712078164099615, 0.031679820254601, -1.82622226863274, 0.0753115337853523, 0.0339083941653576, -1.7957852308222, 0.0795780155033481, 0.0362643402573909, -1.76534819301165, 0.0840083596493505, 0.0387526322988849, -1.73491115520111, 0.0886032582799016, 0.0413782714954893, -1.70447411739056, 0.093362972192426, 0.0441462737436906, -1.67403707958001, 0.0982873063530628, 0.0470616561249409, -1.64360004176947, 0.103375586158663, 0.0501294226632204, -1.61316300395892, 0.108626634713026, 0.053354549374138, -1.58272596614838, 0.114038751300288, 0.0567419686392059, -1.55228892833783, 0.119609691239965, 0.0602965529445266, -1.52185189052729, 0.125336647308453, 0.0640230980287628, -1.49141485271674, 0.131216232910661, 0.0679263054908754, -1.46097781490619, 0.137244467182932, 0.0720107649136812, -1.43054077709565, 0.143416762204319, 0.0762809355647456, -1.4001037392851, 0.149727912487723, 0.080741127741439, -1.36966670147456, 0.156172086915268, 0.0853954838321115, -1.33922966366401, 0.162742823273516, 0.0902479591702228, -1.30879262585347, 0.169433025533907, 0.0953023027628533, -1.27835558804292, 0.176234964011928, 0.100562037979284, -1.24791855023237, 0.18314027852516, 0.106030443289196, -1.21748151242183, 0.190139984655556, 0.111710533143492, -1.18704447461128, 0.197224483205043, 0.117605039093711, -1.15660743680074, 0.204383572916003, 0.123716391248447, -1.12617039899019, 0.21160646650941, 0.130046700167109, -1.09573336117965, 0.218881810073482, 0.136597739292653, -1.0652963233691, 0.226197705814872, 0.143370928025603, -1.03485928555855, 0.233541738162624, 0.150367315541744, -1.00442224774801, 0.240901003192761, 0.157587565455196, -0.973985209937463, 0.248262141318368, 0.165031941427297, -0.943548172126917, 0.255611373166741, 0.172700293819665, -0.913111134316372, 0.262934538541711, 0.180592047487061, -0.882674096505826, 0.270217138345811, 0.188706190802249, -0.85223705869528, 0.277444379313691, 0.197041266000838, -0.821800020884734, 0.284601221385435, 0.205595360929238, -0.791362983074189, 0.291672427526274, 0.21436610227331, -0.760925945263643, 0.298642615777843, 0.22335065033906, -0.730488907453097, 0.305496313305857, 0.232545695449851, -0.700051869642552, 0.312218012190072, 0.241947456017205, -0.669614831832006, 0.31879222668473, 0.251551678334177, -0.63917779402146, 0.325203551661746, 0.261353638131827, -0.608740756210914, 0.331436721934659, 0.271348143930281, -0.578303718400369, 0.337476672149118, 0.281529542206485, -0.547866680589823, 0.343308596915468, 0.291891724391041, -0.517429642779277, 0.348918010851124, 0.302428135696488, -0.486992604968732, 0.35429080819472, 0.313131785769153, -0.456555567158186, 0.359413321650878, 0.32399526114639, -0.42611852934764, 0.364272380123711, 0.335010739490556, -0.395681491537094, 0.368855364998973, 0.346170005560716, -0.365244453726548, 0.373150264639264, 0.357464468872749, -0.334807415916003, 0.377145726763521, 0.368885182988397, -0.304370378105458, 0.380831108391634, 0.380422866363923, -0.273933340294912, 0.384196523046984, 0.392067924679493, -0.243496302484366, 0.387232884924108, 0.403810474561246, -0.21305926467382, 0.389931949745491, 0.415640368599337, -0.182622226863274, 0.392286352050426, 0.427547221557132, -0.152185189052728, 0.39428963867994, 0.439520437659177, -0.121748151242183, 0.395936298244767, 0.451549238838748, -0.0913111134316376, 0.397221786388035, 0.463622693819663, -0.0608740756210917, 0.398142546680622, 0.475729747901678, -0.0304370378105459, 0.398696027014707, 0.487859253314274, 0, 0.398880691389743, 0.5
  };

  int const i = static_cast<int>((x - xmin) * hinv),
          inc = i * 3L;

  // bounds checks
  if(i >= 0 and i < n_pts - 1){
    double const x0 = d_xmy[inc     ],
                 m0 = d_xmy[inc + 1L],
                 y0 = d_xmy[inc + 2L],
                 m1 = d_xmy[inc + 4L],
                 y1 = d_xmy[inc + 5L],
                  t = (x - x0) * hinv,
                 t1 = t - 1,
                h01 = t * t * (3 - 2 * t),
                h00 = 1 - h01,
                tt1 = t * t1,
                h10 = tt1 * t1,
                h11 = tt1 * t;

    return y0 * h00 + h * m0 * h10 + y1 * h01 + h * m1 * h11;
  }
  if(i >= n_pts - 1)
    return .5;
  else if(x < xinf)
    return 0.;

  // i < 0 and x >= xinf
  static constexpr double p[6] = {
    0.21589853405795699,
    0.1274011611602473639,
    0.022235277870649807,
    0.001421619193227893466,
    2.9112874951168792e-5,
    0.02307344176494017303
  };
  static constexpr double q[5] = {
    1.28426009614491121,
    0.468238212480865118,
    0.0659881378689285515,
    0.00378239633202758244,
    7.29751555083966205e-5
  };

  double const y = -x;
  double xsq = 1.0 / (x * x),
    xnum = p[5] * xsq,
    xden = xsq;
  for (int i = 0; i < 4; ++i) {
    xnum = (xnum + p[i]) * xsq;
    xden = (xden + q[i]) * xsq;
  }
  double temp = xsq * (xnum + p[4]) / (xden + q[4]);
  constexpr double M_1_SQRT_2PI_LOC =	0.398942280401432677939946059934;	/* 1/sqrt(2pi) */
  temp = (M_1_SQRT_2PI_LOC - temp) / y;

  xsq = std::trunc(x * 16) / 16;
  double const del = (x - xsq) * (x + xsq);
  return std::exp(-xsq * xsq * 0.5 -del * 0.5) * temp;
}

/** normal CDF approximations using monotone cubic interpolation. */
inline double pnorm_approx(double const x) noexcept {
  return x >= 0. ? .5 - fastncdf_pos(-x) + .5 : fastncdf_pos(x);
}

#endif
