% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/schafer_corr.R
\name{schafer_corr}
\alias{schafer_corr}
\alias{print.schafer_corr}
\alias{plot.schafer_corr}
\title{Schafer-Strimmer shrinkage correlation}
\usage{
schafer_corr(data)

\method{print}{schafer_corr}(x, digits = 4, max_rows = NULL, max_cols = NULL, ...)

\method{plot}{schafer_corr}(
  x,
  title = "Schafer-Strimmer shrinkage correlation",
  cluster = TRUE,
  hclust_method = "complete",
  triangle = "upper",
  show_values = FALSE,
  value_text_limit = 60,
  value_text_size = 3,
  palette = c("diverging", "viridis"),
  ...
)
}
\arguments{
\item{data}{A numeric matrix or a data frame with at least two numeric
columns. All non-numeric columns will be excluded. Columns must be numeric
and contain no \code{NA}s.}

\item{x}{An object of class \code{schafer_corr}.}

\item{digits}{Integer; number of decimal places to print.}

\item{max_rows}{Optional integer; maximum number of rows to display.
If \code{NULL}, all rows are shown.}

\item{max_cols}{Optional integer; maximum number of columns to display.
If \code{NULL}, all columns are shown.}

\item{...}{Additional arguments passed to \code{ggplot2::theme()}.}

\item{title}{Plot title.}

\item{cluster}{Logical; if TRUE, reorder rows/cols by hierarchical clustering
on distance \eqn{1 - r}.}

\item{hclust_method}{Linkage method for \code{hclust}; default \code{"complete"}.}

\item{triangle}{One of \code{"full"}, \code{"upper"}, \code{"lower"}.
Default to \code{upper}.}

\item{show_values}{Logical; print correlation values inside tiles (only if
matrix dimension \eqn{\le} \code{value_text_limit}).}

\item{value_text_limit}{Integer threshold controlling when values are drawn.}

\item{value_text_size}{Font size for values if shown.}

\item{palette}{Character; \code{"diverging"} (default) or \code{"viridis"}.}
}
\value{
A symmetric numeric matrix of class \code{schafer_corr} where entry
\code{(i, j)} is the shrunk correlation between the \code{i}-th and
\code{j}-th numeric columns. Attributes:
\itemize{
\item \code{method} = \code{"schafer_shrinkage"}
\item \code{description} = \code{"Schafer-Strimmer shrinkage correlation
  matrix"}
\item \code{package} = \code{"matrixCorr"}
}
Columns with zero variance are set to \code{NA} across row/column (including
the diagonal), matching \code{pearson_corr()} behaviour.

Invisibly returns \code{x}.

A \code{ggplot} object.
}
\description{
Computes a shrinkage correlation matrix using the Schafer-Strimmer approach
with an analytic, data-driven intensity \eqn{\hat\lambda}. The off-diagonals
of the sample Pearson correlation \eqn{R} are shrunk towards zero, yielding
\eqn{R_{\mathrm{shr}}=(1-\hat\lambda)R+\hat\lambda I} with
\eqn{\mathrm{diag}(R_{\mathrm{shr}})=1}, stabilising estimates when
\eqn{p \ge n}.

This function uses a high-performance 'C++' backend that forms
\eqn{X^\top X} via 'BLAS' 'SYRK', applies centring via a rank-1 update,
converts to Pearson correlation, estimates \eqn{\hat\lambda}, and shrinks
the off-diagonals:
\eqn{R_{\mathrm{shr}} = (1-\hat\lambda)R + \hat\lambda I}.

Prints a summary of the shrinkage correlation matrix with
optional truncation for large objects.

Heatmap of the shrinkage correlation matrix with optional
hierarchical clustering and triangular display. Uses \pkg{ggplot2} and
\code{geom_raster()} for speed on larger matrices.
}
\details{
Let \eqn{R} be the sample Pearson correlation matrix. The Schafer-Strimmer
shrinkage estimator targets the identity in correlation space and uses
\eqn{\hat\lambda = \frac{\sum_{i<j}\widehat{\mathrm{Var}}(r_{ij})}
{\sum_{i<j} r_{ij}^2}} (clamped to \eqn{[0,1]}), where
\eqn{\widehat{\mathrm{Var}}(r_{ij}) \approx \frac{(1-r_{ij}^2)^2}{n-1}}.
The returned estimator is \eqn{R_{\mathrm{shr}} = (1-\hat\lambda)R +
\hat\lambda I}.
}
\note{
No missing values are permitted. Columns with fewer than two observations
or zero variance are flagged as \code{NA} (row/column).
}
\examples{
## Multivariate normal with AR(1) dependence (Toeplitz correlation)
set.seed(1)
n <- 80; p <- 40; rho <- 0.6
d <- abs(outer(seq_len(p), seq_len(p), "-"))
Sigma <- rho^d

X <- MASS::mvrnorm(n, mu = rep(0, p), Sigma = Sigma)
colnames(X) <- paste0("V", seq_len(p))

Rshr <- schafer_corr(X)
print(Rshr, digits = 2, max_rows = 6, max_cols = 6)
plot(Rshr)

## Shrinkage typically moves the sample correlation closer to the truth
Rraw <- stats::cor(X)
off  <- upper.tri(Sigma, diag = FALSE)
mae_raw <- mean(abs(Rraw[off] - Sigma[off]))
mae_shr <- mean(abs(Rshr[off] - Sigma[off]))
print(c(MAE_raw = mae_raw, MAE_shrunk = mae_shr))
plot(Rshr, title = "Schafer-Strimmer shrinkage correlation")

}
\references{
Schafer, J. & Strimmer, K. (2005). A shrinkage approach to large-scale
covariance matrix estimation and implications for functional genomics.
\emph{Statistical Applications in Genetics and Molecular Biology}, 4(1).
}
\seealso{
\code{\link{print.schafer_corr}}, \code{\link{plot.schafer_corr}},
\code{\link{pearson_corr}}
}
\author{
Thiago de Paula Oliveira \email{toliveira@abacusbio.com}
}
