% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/character.R, R/dataframe.R, R/lime.R
\name{lime.character}
\alias{lime.character}
\alias{lime.data.frame}
\alias{lime}
\title{Create a model explanation function based on training data}
\usage{
\method{lime}{character}(x, model, preprocess,
  tokenization = default_tokenize, keep_word_position = FALSE, ...)

\method{lime}{data.frame}(x, model, bin_continuous = TRUE, n_bins = 4,
  quantile_bins = TRUE, ...)

lime(x, model, ...)
}
\arguments{
\item{x}{The training data used for training the model that should be
explained.}

\item{model}{The model whose output should be explained}

\item{preprocess}{Function to transform a \code{character} vector to the format
expected from the model.}

\item{tokenization}{function used to tokenize text for the permutations.}

\item{keep_word_position}{set to \code{TRUE} if to keep order of words. Warning:
each word will be replaced by \code{word_position}.}

\item{...}{Arguments passed on to methods}

\item{bin_continuous}{Should continuous variables be binned when making the explanation}

\item{n_bins}{The number of bins for continuous variables if \code{bin_continuous = TRUE}}

\item{quantile_bins}{Should the bins be based on \code{n_bins} quantiles or spread evenly over the range of the training data}
}
\value{
Return an explainer which can be used together with \code{\link[=explain]{explain()}} to
explain model predictions.
}
\description{
This is the main function of the \code{lime} package. It is a factory function
that returns a new function that can be used to explain the predictions made
by black box models. This is a generic with methods for the different data
types supported by lime.
}
\examples{
# Explaining a model based on text data

# Purpose is to classify sentences from scientific publications
# and find those where the team writes about their own work
# (category OWNX in the provided dataset).

library(text2vec)
library(xgboost)

data(train_sentences)
data(test_sentences)

get_matrix <- function(text) {
  it <- itoken(text, progressbar = FALSE)
  create_dtm(it, vectorizer = hash_vectorizer())
}

dtm_train = get_matrix(train_sentences$text)

xgb_model <- xgb.train(list(max_depth = 7, eta = 0.1, objective = "binary:logistic",
                 eval_metric = "error", nthread = 1),
                 xgb.DMatrix(dtm_train, label = train_sentences$class.text == "OWNX"),
                 nrounds = 50)

sentences <- head(test_sentences[test_sentences$class.text == "OWNX", "text"], 1)
explainer <- lime(train_sentences$text, xgb_model, get_matrix)
explanations <- explain(sentences, explainer, n_labels = 1, n_features = 2)

# We can see that many explanations are based
# on the presence of the word `we` in the sentences
# which makes sense regarding the task.
print(explanations)

# Explaining a model based on tabular data
library(MASS)
iris_test <- iris[1, 1:4]
iris_train <- iris[-1, 1:4]
iris_lab <- iris[[5]][-1]
# Create linear discriminant model on iris data
model <- lda(iris_train, iris_lab)
# Create explanation object
explanation <- lime(iris_train, model)

# This can now be used together with the explain method
explain(iris_test, explanation, n_labels = 1, n_features = 2)

}
