#' Switch column names
#' 
#' @description
#' The `switchcol_nfi()` function allows switching between the original Korean column names and English column names. 
#' If the input data has English column names, it changes them to the original Korean names, and vice versa.
#' 
#' @details
#' Important: Data with original Korean column names cannot be used with any \code{\link{read_nfi}} functions except `colchange_nfi()` itself. 
#' The option to revert to original Korean names is provided solely for users who wish to store or process the NFI data with original column names.
#' 
#' @param data : A `list` generated by \code{\link{read_nfi}}.
#' @return A `list` of dataframes with switched column names.
#'
#' @examples
#'
#' data("nfi_donghae")
#' 
#' #Switch column names from English to original Korean names
#' nfi_donghae_kor <- switchcol_nfi(nfi_donghae)
#' 
#' # Switch column names from original Korean to English names
#' nfi_donghae_eng <- switchcol_nfi(nfi_donghae_kor)
#' 
#' @export
# 

switchcol_nfi <- function(data){
  
  first_col_name <- colnames(data$plot)[1] 
  is_english <- grepl("^[A-Za-z_]+$", first_col_name)
  
  if(is_english) {
    
    for(i in 1:length(data)){
      
      data_colnames <- names(data[[i]])
      missing_names <- data_colnames[!data_colnames %in% col_name$Column_Name]
      
      for (name in missing_names) {
        col_name <- rbind(col_name, data.frame(Column_Name = name, English_Name = NA, Korean_Column_Name = name))
      }
      
      new_names <- stats::setNames(col_name$Korean_Column_Name, col_name$Column_Name)
      data[[i]] <- stats::setNames(data[[i]], new_names[names(data[[i]])])
    } 
    
  } else {
    
    for(i in 1:length(data)){
      
      data_colnames <- names(data[[i]])
      missing_names <- data_colnames[!data_colnames %in% col_name$Korean_Column_Name]
      
      for (name in missing_names) {
        col_name <- rbind(col_name, data.frame(Column_Name = name, English_Name = NA, Korean_Column_Name = name))
      }
      
      new_names <- stats::setNames(col_name$Column_Name, col_name$Korean_Column_Name)
      data[[i]] <- stats::setNames(data[[i]], new_names[names(data[[i]])])
    } 
  }
    
  return(data)
}

