% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/run_efa.R
\name{run_efa}
\alias{run_efa}
\title{Conducts exploratory factor analysis}
\usage{
run_efa(
  data,
  variables = names(data),
  m = floor(ncol(data)/4),
  rotation = "oblimin",
  simple = TRUE,
  min.loading = NA,
  single.item = c("keep", "drop", "none"),
  identified = TRUE,
  constrain0 = FALSE,
  ordered = FALSE,
  estimator = NULL,
  missing = "listwise",
  ...
)
}
\arguments{
\item{data}{a \code{data.frame} containing the variables (i.e., items) to factor analyze}

\item{variables}{character vector of column names in \code{data} indicating the variables to factor analyze. Default is to use all columns.}

\item{m}{integer; maximum number of factors to extract. Default is 4 items per factor.}

\item{rotation}{character (case-sensitive); any rotation method listed in
\code{\link[GPArotation]{rotations}} in the \code{GPArotation} package. Default is "oblimin".}

\item{simple}{logical; Should the simple structure be returned (default) when converting EFA results to CFA syntax?
If \code{FALSE}, items can cross-load on multiple factors.}

\item{min.loading}{numeric between 0 and 1 indicating the minimum (absolute) value of the loading for a variable on a factor
when converting EFA results to CFA syntax. Must be specified when \code{simple = FALSE}.}

\item{single.item}{character indicating how single-item factors should be treated.
Use \code{"keep"} (default) to keep them in the model when generating the CFA syntax, \code{"drop"}
to remove them, or \code{"none"} indicating the CFA syntax should not be generated for
this model and \code{""} is returned.}

\item{identified}{logical; Should identification check for rotational uniqueness a la Millsap (2001) be performed?
If the model is not identified \code{""} is returned.}

\item{constrain0}{logical; Should variable(s) with all loadings below \code{min.loading} still be included in model syntax?
If \code{TRUE}, variable(s) will load onto first factor with the loading constrained to 0.}

\item{ordered}{logical; Should items be treated as ordinal and the
polychoric correlations used in the factor analysis? When \code{FALSE} (default)
the Pearson correlation matrix is used. A character vector of item names is
also accepted to prompt estimation of the polychoric correlation matrix.}

\item{estimator}{if \code{ordered = FALSE}, the default is "MLMVS". If
\code{ordered = TRUE}, the default is "WLSMV". See \code{\link[lavaan]{lavOptions}} for other options.}

\item{missing}{default is "listwise". See \code{\link[lavaan]{lavOptions}} for other options.}

\item{...}{other arguments passed to \code{lavaan} functions. See \code{\link[lavaan]{lavOptions}}.}
}
\value{
A three-element \code{list}:
\itemize{
\item \strong{efas} \code{lavaan} object for each \emph{m} model
\item \strong{loadings} (rotated) factor loading matrix for each \emph{m} model
\item \strong{cfa.syntax} CFA syntax generated from loadings
}
}
\description{
This function is intended for use on independent samples rather than integrated
with k-fold cross-validation.
}
\details{
When converting EFA results to CFA syntax (via \code{\link[kfa]{efa_cfa_syntax}}), the simple structure is
defined as each variable loading onto a single factor. This is determined using the largest factor loading for each variable.
When \code{simple = FALSE}, variables are allowed to cross-load on multiple factors. In this case, all pathways with loadings
above the \code{min.loading} are retained. However, allowing cross-loading variables can result in model under-identification.
An identification check is run by default, but can be turned off by setting \code{identified = FALSE}.
}
\examples{

# simulate data based on a 3-factor model with standardized loadings
sim.mod <- "f1 =~ .7*x1 + .8*x2 + .3*x3 + .7*x4 + .6*x5 + .8*x6 + .4*x7
                f2 =~ .8*x8 + .7*x9 + .6*x10 + .5*x11 + .5*x12 + .7*x13 + .6*x14
                f3 =~ .6*x15 + .5*x16 + .9*x17 + .4*x18 + .7*x19 + .5*x20
                f1 ~~ .2*f2
                f2 ~~ .2*f3
                f1 ~~ .2*f3
                x9 ~~ .2*x10"
set.seed(1161)
sim.data <- simstandard::sim_standardized(sim.mod, n = 900,
                                          latent = FALSE,
                                          errors = FALSE)[c(2:9,1,10:20)]
# Run 1-, 2-, and 3-factor models
efas <- run_efa(sim.data, m = 3)

}
\references{
Millsap, R. E. (2001). When trivial constraints are not trivial: The choice of uniqueness constraints in confirmatory factor analysis. \emph{Structural Equation Modeling, 8}(1), 1-17. \doi{10.1207/S15328007SEM0801_1}
}
