% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/selection.R
\name{selection}
\alias{selection}
\title{Selecting a Sample from an Audit Population}
\usage{
selection(population, sampleSize, units = "records", algorithm = "random",
          bookValues = NULL, intervalStartingPoint = 1, ordered = TRUE, 
          ascending = TRUE, withReplacement = FALSE, seed = 1)
}
\arguments{
\item{population}{a data frame containing the population the auditor wishes to sample from.}

\item{sampleSize}{the number of observations that need to be selected from the population. Can also be an object of class \code{jfaPlanning}.}

\item{units}{can be either \code{records} (default) for record sampling, or \code{mus} for monetary unit sampling.}

\item{algorithm}{can be either one of \code{random} (default) for random sampling, \code{cell} for cell sampling, or \code{interval} for fixed interval sampling.}

\item{bookValues}{a character specifying the name of the column containing the book values (as in the population data).}

\item{intervalStartingPoint}{the starting point in the interval (used only in fixed interval sampling)}

\item{ordered}{if \code{TRUE} (default), the population is first ordered according to the value of their book values.}

\item{ascending}{if \code{TRUE} (default), order the population in ascending order.}

\item{withReplacement}{whether sampling should be performed with replacement. Defaults to \code{FALSE}.}

\item{seed}{seed to reproduce results. Default is 1.}
}
\value{
An object of class \code{jfaSelection} containing:

\item{population}{a data frame containing the input population.}
\item{sample}{a data frame containing the selected observations.}
\item{units}{the sampling units that were used for sampling.}
\item{algorithm}{the algorithm that was used for sampling.}
\item{bookValues}{if specified, the name of the specified book value column.}
}
\description{
This function takes a data frame and performs sampling according to one of three popular algorithms: random sampling, cell sampling, or fixed interval sampling. Sampling is done in combination with one of two sampling units: records or monetary units The returned object is of class \code{jfaSelection} and can be used with associated \code{print()} and \code{plot()} methods. 

For more details on how to use this function see the package vignette:
\code{vignette("jfa", package = "jfa")}
}
\details{
This first part of this section elaborates on the possible options for the \code{units} argument:

\itemize{
 \item{\code{records}:     In record sampling, each observation in the population is seen as a sampling unit. An observation of $5000 is therefore equally likely to be selected as an observation of $500.}
 \item{\code{mus}:         In monetary unit sampling, each monetary unit in the population is seen as a sampling unit. An observation of $5000 is therefore ten times more likely to be selected as an observation of $500.}
}

This second part of this section elaborates on the possible options for the \code{algorithm} argument:

\itemize{
 \item{\code{random}:      In random sampling each sampling unit in the population is drawn with equal probability.}
 \item{\code{cell}:        In cell sampling the sampling units in the population are divided into a number (equal to the sample size) of intervals. From each interval one sampling unit is selected with equal probability.}
 \item{\code{interval}:    In fixed interval sampling the sampling units in the population are divided into a number (equal to the sample size) of intervals. From each interval one sampling unit is selected according to a fixed starting point (\code{intervalStartingPoint}).}
}
}
\examples{
library(jfa)
set.seed(1)

# Generate some audit data (N = 1000).
population <- data.frame(ID = sample(1000:100000, size = 1000, replace = FALSE), 
                         bookValue = runif(n = 1000, min = 700, max = 1000))

# Draw a custom sample of 100 from the population (via random record sampling):

s1 <- selection(population = population, sampleSize = 100, algorithm = "random", 
                units = "records", seed = 1)
print(s1)

# ------------------------------------------------------------
#                  jfa Selection Summary
# ------------------------------------------------------------
# Input:
#       
# Population size:         1000 
# Requested sample size:   100 
# Sampling units:          Records 
# Algorithm:               Random sampling   
# ------------------------------------------------------------ 
# Output:
# 
# Obtained sample size:    100 
# ------------------------------------------------------------
# Statistics:
#
# Proportion n/N:          0.1 
# ------------------------------------------------------------  

# Use the result from the planning stage in the sampling stage:

p1 <- planning(materiality = 0.05, confidence = 0.95, expectedError = 0.025, 
               likelihood = "binomial")

# Draw a sample via random monetary unit sampling:
s2 <- selection(population = population, sampleSize = p1, algorithm = "random", 
                units = "mus", seed = 1, bookValues = "bookValue")
print(s2)

# ------------------------------------------------------------
#                  jfa Selection Summary
# ------------------------------------------------------------
# Input:
#      
# Population size:         1000 
# Requested sample size:   234 
# Sampling units:          Monetary units 
# Algorithm:               Random sampling   
# ------------------------------------------------------------ 
# Output:
#
# Obtained sample size:    234 
# ------------------------------------------------------------
# Statistics:
#
# Proportion n/N:          0.23 
# Percentage of value:     23.06\% 
# ------------------------------------------------------------ 

}
\references{
Wampler, B., & McEacharn, M. (2005). Monetary-unit sampling using Microsoft Excel. \emph{The CPA journal}, 75(5), 36.
}
\seealso{
\code{\link{auditPrior}} \code{\link{planning}} \code{\link{evaluation}}
}
\author{
Koen Derks, \email{k.derks@nyenrode.nl}
}
\keyword{audit}
\keyword{sample}
\keyword{selection}
