% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/GradienBased.R
\name{SmoothGrad}
\alias{SmoothGrad}
\title{SmoothGrad Method}
\description{
'SmoothGrad' was introduced by D. Smilkov et al. (2017) and is an extension
to the classical Vanilla \link{Gradient} method. It takes the mean of the
gradients for \code{n} perturbations of each data point, i.e. with
\eqn{\epsilon \sim N(0,\sigma)}
\deqn{1/n \sum_n d f(x+ \epsilon)_j / d x_j.}
}
\examples{
\dontshow{if (torch::torch_is_installed()) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
# ------------------------- Example 1: Torch -------------------------------
library(torch)

# Create nn_sequential model and data
model <- nn_sequential(
  nn_linear(5, 10),
  nn_relu(),
  nn_linear(10, 2),
  nn_sigmoid()
)
data <- torch_randn(25, 5)

# Create Converter
converter <- Converter$new(model, input_dim = c(5))

# Calculate the smoothed Gradients
smoothgrad <- SmoothGrad$new(converter, data)

# Print the result as a data.frame for first 5 rows
smoothgrad$get_result("data.frame")[1:5, ]

# Plot the result for both classes
plot(smoothgrad, output_idx = 1:2)

# Plot the boxplot of all datapoints
boxplot(smoothgrad, output_idx = 1:2)

# ------------------------- Example 2: Neuralnet ---------------------------
library(neuralnet)
data(iris)

# Train a neural network
nn <- neuralnet(Species ~ ., iris,
  linear.output = FALSE,
  hidden = c(10, 5),
  act.fct = "logistic",
  rep = 1
)

# Convert the trained model
converter <- Converter$new(nn)

# Calculate the smoothed gradients
smoothgrad <- SmoothGrad$new(converter, iris[, -5], times_input = FALSE)

# Plot the result for the first and 60th data point and all classes
plot(smoothgrad, data_idx = c(1, 60), output_idx = 1:3)

# Calculate SmoothGrad x Input and do not ignore the last activation
smoothgrad <- SmoothGrad$new(converter, iris[, -5], ignore_last_act = FALSE)

# Plot the result again
plot(smoothgrad, data_idx = c(1, 60), output_idx = 1:3)

# ------------------------- Example 3: Keras -------------------------------
library(keras)

if (is_keras_available()) {
  data <- array(rnorm(64 * 60 * 3), dim = c(64, 60, 3))

  model <- keras_model_sequential()
  model \%>\%
    layer_conv_1d(
      input_shape = c(60, 3), kernel_size = 8, filters = 8,
      activation = "softplus", padding = "valid"
    ) \%>\%
    layer_conv_1d(
      kernel_size = 8, filters = 4, activation = "tanh",
      padding = "same"
    ) \%>\%
    layer_conv_1d(
      kernel_size = 4, filters = 2, activation = "relu",
      padding = "valid"
    ) \%>\%
    layer_flatten() \%>\%
    layer_dense(units = 64, activation = "relu") \%>\%
    layer_dense(units = 16, activation = "relu") \%>\%
    layer_dense(units = 3, activation = "softmax")

  # Convert the model
  converter <- Converter$new(model)

  # Apply the SmoothGrad method
  smoothgrad <- SmoothGrad$new(converter, data, channels_first = FALSE)

  # Plot the result for the first datapoint and all classes
  plot(smoothgrad, output_idx = 1:3)

  # Plot the result as boxplots for first two classes
  boxplot(smoothgrad, output_idx = 1:2)

  # You can also create an interactive plot with plotly.
  # This is a suggested package, so make sure that it is installed
  library(plotly)

  # Result as boxplots
  boxplot(smoothgrad, as_plotly = TRUE)

  # Result of the second data point
  plot(smoothgrad, data_idx = 2, as_plotly = TRUE)
}

# ------------------------- Advanced: Plotly -------------------------------
# If you want to create an interactive plot of your results with custom
# changes, you can take use of the method plotly::ggplotly
library(ggplot2)
library(plotly)
library(neuralnet)
data(iris)

nn <- neuralnet(Species ~ .,
  iris,
  linear.output = FALSE,
  hidden = c(10, 8), act.fct = "tanh", rep = 1, threshold = 0.5
)
# create an converter for this model
converter <- Converter$new(nn)

# create new instance of 'SmoothGrad'
smoothgrad <- SmoothGrad$new(converter, iris[, -5])

library(plotly)

# Get the ggplot and add your changes
p <- plot(smoothgrad, output_idx = 1, data_idx = 1:2) +
  theme_bw() +
  scale_fill_gradient2(low = "green", mid = "black", high = "blue")

# Now apply the method plotly::ggplotly with argument tooltip = "text"
plotly::ggplotly(p, tooltip = "text")
\dontshow{\}) # examplesIf}
}
\references{
D. Smilkov et al. (2017) \emph{SmoothGrad: removing noise by adding noise.}
CoRR, abs/1706.03825
}
\section{Super classes}{
\code{\link[innsight:InterpretingMethod]{innsight::InterpretingMethod}} -> \code{\link[innsight:GradientBased]{innsight::GradientBased}} -> \code{SmoothGrad}
}
\section{Public fields}{
\if{html}{\out{<div class="r6-fields">}}
\describe{
\item{\code{n}}{Number of perturbations of the input data (default: \eqn{50}).}

\item{\code{noise_level}}{The standard deviation of the Gaussian
perturbation, i.e. \eqn{\sigma = (max(x) - min(x)) *} \code{noise_level}.}
}
\if{html}{\out{</div>}}
}
\section{Methods}{
\subsection{Public methods}{
\itemize{
\item \href{#method-new}{\code{SmoothGrad$new()}}
\item \href{#method-clone}{\code{SmoothGrad$clone()}}
}
}
\if{html}{
\out{<details open ><summary>Inherited methods</summary>}
\itemize{
\item \out{<span class="pkg-link" data-pkg="innsight" data-topic="InterpretingMethod" data-id="get_result">}\href{../../innsight/html/InterpretingMethod.html#method-get_result}{\code{innsight::InterpretingMethod$get_result()}}\out{</span>}
\item \out{<span class="pkg-link" data-pkg="innsight" data-topic="GradientBased" data-id="boxplot">}\href{../../innsight/html/GradientBased.html#method-boxplot}{\code{innsight::GradientBased$boxplot()}}\out{</span>}
\item \out{<span class="pkg-link" data-pkg="innsight" data-topic="GradientBased" data-id="plot">}\href{../../innsight/html/GradientBased.html#method-plot}{\code{innsight::GradientBased$plot()}}\out{</span>}
}
\out{</details>}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-new"></a>}}
\if{latex}{\out{\hypertarget{method-new}{}}}
\subsection{Method \code{new()}}{
Create a new instance of the SmoothGrad method.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{SmoothGrad$new(
  converter,
  data,
  channels_first = TRUE,
  output_idx = NULL,
  ignore_last_act = TRUE,
  times_input = FALSE,
  n = 50,
  noise_level = 0.1,
  dtype = "float"
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{converter}}{An instance of the R6 class \code{\link{Converter}}.}

\item{\code{data}}{The data for which the smoothed gradients are to be
calculated. It has to be an array or array-like format of size
\emph{(batch_size, dim_in)}.}

\item{\code{channels_first}}{The format of the given data, i.e. channels on
last dimension (\code{FALSE}) or after the batch dimension (\code{TRUE}). If the
data has no channels, use the default value \code{TRUE}.}

\item{\code{output_idx}}{This vector determines for which outputs the method
will be applied. By default (\code{NULL}), all outputs (but limited to the
first 10) are considered.}

\item{\code{ignore_last_act}}{A boolean value to include the last
activation into all the calculations, or not (default: \code{TRUE}). In
some cases, the last activation leads to a saturation problem.}

\item{\code{times_input}}{Multiplies the gradients with the input features.
This method is called 'SmoothGrad x Input'. Default: \code{FALSE}.}

\item{\code{n}}{Number of perturbations of the input data (default: \eqn{50}).}

\item{\code{noise_level}}{Determines the standard deviation of the gaussian
perturbation, i.e. \eqn{\sigma = (max(x) - min(x)) *} \code{noise_level}.}

\item{\code{dtype}}{The data type for the calculations. Use
either \code{'float'} for \link[torch:torch_dtype]{torch::torch_float} or \code{'double'} for
\link[torch:torch_dtype]{torch::torch_double}.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-clone"></a>}}
\if{latex}{\out{\hypertarget{method-clone}{}}}
\subsection{Method \code{clone()}}{
The objects of this class are cloneable with this method.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{SmoothGrad$clone(deep = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{deep}}{Whether to make a deep clone.}
}
\if{html}{\out{</div>}}
}
}
}
