% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sat_vapor_pressure.R
\encoding{UTF-8}
\name{dyn_visc_water}
\alias{dyn_visc_water}
\title{Absolute or Dynamic Viscosity for Liquid Water}
\usage{
dyn_visc_water(
  T,
  units = c("SI", "Eng", "Absolute"),
  Eng_units = c("slug/ft/s", "lbf*s/ft^2")
)
}
\arguments{
\item{T}{numeric vector that contains the temperature (degrees Celsius,
degrees Fahrenheit, or Kelvin)}

\item{units}{character vector that contains the system of units {options are
\code{SI} for International System of Units, \code{Eng} for English units
(United States Customary System in the United States and Imperial Units in
the United Kingdom), or \code{Absolute} for Absolute Units}}

\item{Eng_units}{character vector that contains the unit for the dynamic
viscosity of water in the English system {options are slug/ft/s or
lbf*s/ft^2}}
}
\value{
the absolute or dynamic viscosity as a numeric vector. The units are
  not returned.
}
\description{
This function solves for the absolute or dynamic viscosity of water using
only the temperature of the water in either units of degrees Celsius,
degrees Fahrenheit, or Kelvin.
}
\details{
The simplified equation is expressed as

\deqn{\\mu_s = \frac{1}{a + bT + cT^2 + dT^3}}

with

\deqn{a = 557.82468}
\deqn{b = 19.408782}
\deqn{c = 0.1360459}
\deqn{d = -3.1160832 * 10 ^ -4}

\describe{
  \item{\emph{\\mu_s}}{Water Absolute or Dynamic Viscosity (kg/m*s, slug/ft/s, or lbf*s/ft^2)}
  \item{\emph{T}}{the water temperature, degrees Celsius}
}
}
\examples{
# Example 1 (Compare to the tabulated values in the Reference paper)

install.load::load_package("iemisc", "data.table", "round")

T <- c(0, 0.01, 3.86, seq(5, 95, by = 5), 99.974, seq(100, 150, by = 5))

dynamic_viscosity <- data.table("Temperature (degrees C)" = T,
"mu (* 10 ^ 6, kg / m*s)" = round_r3(dyn_visc_water(T, units = "SI")
* 10^6, d = 1))
dynamic_viscosity





# Example 2 - Example from the hydraulics package

library(iemisc)

mu <- hydraulics::dvisc(T = 55, units = "Eng"); mu

mu2 <- dyn_visc_water(T = 55, units = "Eng", Eng_units = "lbf*s/ft^2"); mu2





}
\references{
C. O. Popiel & J. Wojtkowiak (1998). "Simple Formulas for Thermophysical Properties of Liquid Water for Heat Transfer Calculations (from 0C to 150C)". \emph{Heat Transfer Engineering}, 19:3, 87-101, \url{https://www.tandfonline.com/doi/abs/10.1080/01457639808939929}.
}
\author{
Irucka Embry
}
