% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/weighted_C.R
\encoding{UTF-8}
\name{weighted_C}
\alias{weighted_C}
\title{Calculate the Weighted C factor}
\usage{
weighted_C(
  C = NULL,
  area = NULL,
  area_pct = NULL,
  area_units = c("acre", "square feet", "square mile", "hectare", "square kilometers"),
  C_area_table = NULL,
  C_area_pct_table = NULL
)
}
\arguments{
\item{C}{numeric vector containing dimensionless C factor(s)}

\item{area}{numeric vector containing the surface land area}

\item{area_pct}{numeric vector containing the surface land area, as a
percent (decimal or whole number)}

\item{area_units}{character vector containing the units for area
(default = "acres"). The units should be consistent and not mixed.}

\item{C_area_table}{data.frame/data.table/tibble, list, or matrix
containing the C in column 1 and the area in column 2}

\item{C_area_pct_table}{data.frame/data.table/tibble, list, or matrix
containing the C in column 1 and the area_pct in column 2}
}
\value{
Weighted C factor as a single numeric vector, in the range [0, 1]
}
\description{
This function computes the weighted C factor using the user-supplied unit
or the default unit of an acre for use in the Rational Formula.
}
\examples{

# Note: the default area unit is acre

# Example 1

library("iemisc")

area1 <- c(220, 150, 30)
C1 <- c(75, 89, 80)
weighted_C(C = C1, area = area1)


# Example 2

library("iemisc")

area2 <- c(220, 150, 30)
area_pct2 <- area2 / sum(area2)
C2 <- c(80, 95, 80)
C_area_pct_table2 <- data.frame(C2, area_pct2)
weighted_C(C_area_pct_table = C_area_pct_table2)


# Example 3

install.load::load_package("iemisc", "data.table")

C_area_table3 <- data.table(C = c(98, 100, 45), area = c(2.53, 453.00, 0.21))
weighted_C(C_area_table = C_area_table3)


# Example 4

library("iemisc")

C4 <- c(98, 100, 45)
area_pct4 <- c(0.15, 0.23, 0.62)
weighted_C(C = C4, area_pct = area_pct4)


# Example 5

library("iemisc")

data_matrix5a <- matrix(c(98, 30, 40, 43, 57, 3.24, 1, 30, 50, 123), nrow = 5,
ncol = 2, dimnames = list(rep("", 5), c("C", "Area")))
weighted_C(C_area_table = data_matrix5a)


# using ramify to create the matrix

import::from(ramify, mat)

data_matrix5b <- mat("98 30 40 43 57;3.24 1 30 50 123", rows = FALSE, sep = " ",
dimnames = list(rep("", 5), c("C", "Area")))
weighted_C(C_area_table = data_matrix5b)


# Example 6 - using area in square feet

library("iemisc")

data_list6 <- list(C = c(77, 29, 68), Area = c(43560, 56893, 345329.32))
weighted_C(C_area_table = data_list6, area_units = "square feet")


# Example 7

install.load::load_package("iemisc", "data.table")

# Impervious area - 3.04 acre
# 45\% of total area
# 0.80 C Factor

# Pervious area - 4.67 acre
# 55\% of total area
# 0.20 C factor

C_area_table7 <- data.table(C = c(0.80, 0.20), area = c(3.04, 4.67))
weighted_C(C_area_table = C_area_table7)


# Example 8

# Impervious area - 2.44 acre
# 32\% of total area
# 0.80 C Factor

# Pervious area - 5.03 acre
# 68\% of total area
# 0.20 C factor

C8 <- c(0.80, 0.20)
area_pct8 <- c(0.32, 0.68)
weighted_C(C = C8, area_pct = area_pct8)


# Example 9

library("iemisc")

# Medium density residential area - 30 hectares (75.0\% of total area),
# 0.31 - 0.40 C factor
# High density residential area - 3 hectares (7.50\% of total area),
# 0.49 - 0.60 C factor
# Agricultural area - 7 hectares (17.5\% of total area), 0.15 - 0.21 C factor

C3 <- c(mean(seq(0.31, 0.40, by = 0.01)), mean(seq(0.49, 0.60, by = 0.01)),
mean(seq(0.15, 0.21, by = 0.01)))
area3 <- c(30, 3, 7)
weighted_C(C = C3, area = area3, area_units = "hectare")






}
\references{
Engineering Hydrology Training Series Module 104 - Runoff Curve Number Computations Study Guide, United States Department of Agriculture Soil Conservation Service National Employee Development Staff, September 1989, page 21 \url{https://web.archive.org/web/20210414043852/https://www.wcc.nrcs.usda.gov/ftpref/wntsc/H&H/training/runoff-curve-numbers1.pdf}. Retrieved thanks to the Internet Archive: Wayback Machine
}
\author{
Irucka Embry
}
