% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/req-error.R
\name{req_error}
\alias{req_error}
\title{Control handling of HTTP errors}
\usage{
req_error(req, is_error = NULL, body = NULL)
}
\arguments{
\item{req}{A \link{request}.}

\item{is_error}{A predicate function that takes a single argument (the
response) and returns \code{TRUE} or \code{FALSE} indicating whether or not an
R error should signalled.}

\item{body}{A callback function that takes a single argument (the response)
and returns a character vector of additional information to include in the
body of the error. This vector is passed along to the \code{message} argument
of \code{\link[rlang:abort]{rlang::abort()}} so you can use any formatting that it supports.}
}
\value{
A modified HTTP \link{request}.
}
\description{
\code{req_perform()} will automatically convert HTTP errors (i.e. any 4xx or 5xx
status code) into R errors. Use \code{req_error()} to either override the
defaults, or extract additional information from the response that would
be useful to expose to the user.
}
\section{Error handling}{
\code{req_perform()} is designed to succeed if and only if you get a valid HTTP
response. There are two ways a request can fail:
\itemize{
\item The HTTP request might fail, for example if the connection is dropped
or the server doesn't exist. This type of error will have class
\code{c("httr2_failure", "httr2_error")}.
\item The HTTP request might succeed, but return an HTTP status code that
represents a error, e.g. a \verb{404 Not Found} if the specified resource is
not found. This type of error will have (e.g.) class
\code{c("httr2_http_404", "httr2_http", "httr2_error")}.
}

These error classes are designed to be used in conjunction with R's
condition handling tools (\url{https://adv-r.hadley.nz/conditions.html}).
For example, if you want to return a default value when the server returns
a 404, use \code{tryCatch()}:

\if{html}{\out{<div class="sourceCode">}}\preformatted{tryCatch(
  req |> req_perform() |> resp_body_json(),
  httr2_http_404 = function(cnd) NULL
)
}\if{html}{\out{</div>}}

Or if you want to re-throw the error with some additional context, use
\code{withCallingHandlers()}, e.g.:

\if{html}{\out{<div class="sourceCode R">}}\preformatted{withCallingHandlers(
  req |> req_perform() |> resp_body_json(),
  httr2_http_404 = function(cnd) \{
    rlang::abort("Couldn't find user", parent = cnd)
  \}
)
}\if{html}{\out{</div>}}

Learn more about error chaining at \link[rlang:topic-error-chaining]{rlang::topic-error-chaining}.
}

\examples{
# Performing this request usually generates an error because httr2
# converts HTTP errors into R errors:
req <- request(example_url()) |>
  req_url_path("/status/404")
try(req |> req_perform())
# You can still retrieve it with last_response()
last_response()

# But you might want to suppress this behaviour:
resp <- req |>
  req_error(is_error = \(resp) FALSE) |>
  req_perform()
resp

# Or perhaps you're working with a server that routinely uses the
# wrong HTTP error codes only 500s are really errors
request("http://example.com") |>
  req_error(is_error = \(resp) resp_status(resp) == 500)

# Most typically you'll use req_error() to add additional information
# extracted from the response body (or sometimes header):
error_body <- function(resp) {
  resp_body_json(resp)$error
}
request("http://example.com") |>
  req_error(body = error_body)
# Learn more in https://httr2.r-lib.org/articles/wrapping-apis.html
}
\seealso{
\code{\link[=req_retry]{req_retry()}} to control when errors are automatically retried.
}
