% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/optim.R
\name{crit_optim}
\alias{crit_optim}
\title{Criterion optimization}
\usage{
crit_optim(model, crit, ..., h = 2, Xcand = NULL,
  control = list(multi.start = 10, maxit = 100), seed = NULL)
}
\arguments{
\item{model}{\code{homGP} or \code{hetGP} model}

\item{crit}{considered criterion, one of \code{"crit_cSUR"}, \code{"crit_EI"}, \code{"crit_ICU"},
\code{"crit_MCU"} and \code{"crit_tMSE"}. Note that \code{crit_IMSPE} has its dedicated method, see \code{\link[hetGP]{IMSPE_optim}}.}

\item{...}{additional parameters of the criterion}

\item{h}{horizon for multi-step ahead framework.
The decision is made between:
\itemize{
 \item sequential crit search starting by a new design (optimized first) then adding \code{h} replicates
 \item sequential crit searches starting by \code{1} to \code{h} replicates before adding a new point
}
Use \code{h = 0} for the myopic criterion, i.e., not looking ahead.}

\item{Xcand}{optional discrete set of candidates (otherwise a maximin LHS is used to initialise continuous search)}

\item{control}{list in case \code{Xcand == NULL}, with elements \code{multi.start},
to perform a multi-start optimization based on \code{\link[stats]{optim}}, with \code{maxit} iterations each.
Also, \code{tol_dist} defines the minimum distance to an existing design for a new point to be added, otherwise the closest existing design is chosen.
In a similar fashion, \code{tol_dist} is the minimum relative change of crit for adding a new design.}

\item{seed}{optional seed for the generation of LHS designs with \code{\link[DiceDesign]{maximinSA_LHS}}}
}
\value{
list with elements:
\itemize{
\item \code{par}: best first design,
\item \code{value}: criterion h-steps ahead starting from adding \code{par},
\item \code{path}: list of elements list(\code{par}, \code{value}, \code{new}) at each step \code{h}
}
}
\description{
Search for the best value of available criterion, possibly using a h-steps lookahead strategy to favor designs with replication
}
\details{
When looking ahead, the kriging believer heuristic is used,
 meaning that the non-observed value is replaced by the mean prediction in the update.
}
\examples{
###############################################################################
## Bi-variate example (myopic version)
###############################################################################

nvar <- 2 

set.seed(42)
ftest <- function(x, coef = 0.1) return(sin(2*pi*sum(x)) + rnorm(1, sd = coef))

n <- 25 # must be a square
xgrid0 <- seq(0.1, 0.9, length.out = sqrt(n))
designs <- as.matrix(expand.grid(xgrid0, xgrid0))
X <- designs[rep(1:n, sample(1:10, n, replace = TRUE)),]
Z <- apply(X, 1, ftest)

model <- mleHomGP(X, Z, lower = rep(0.1, nvar), upper = rep(1, nvar))

ngrid <- 51
xgrid <- seq(0,1, length.out = ngrid)
Xgrid <- as.matrix(expand.grid(xgrid, xgrid))

preds <- predict(x = Xgrid, object =  model)

## Initial plots
contour(x = xgrid,  y = xgrid, z = matrix(preds$mean, ngrid),
        main = "Predicted mean", nlevels = 20)
points(model$X0, col = 'blue', pch = 20)

crit <- "crit_EI"
crit_grid <- apply(Xgrid, 1, crit, model = model)
filled.contour(x = xgrid, y = xgrid, matrix(crit_grid, ngrid),
               nlevels = 20, color.palette = terrain.colors, 
               main = "Initial criterion landscape",
plot.axes = {axis(1); axis(2); points(model$X0, pch = 20)})

## Sequential crit search
nsteps <- 1 # Increase for better results

for(i in 1:nsteps){
  res <- crit_optim(model, crit = crit, h = 0, control = list(multi.start = 100, maxit = 50))
  newX <- res$par
  newZ <- ftest(newX)
  model <- update(object = model, Xnew = newX, Znew = newZ)
}

## Final plots
contour(x = xgrid,  y = xgrid, z = matrix(preds$mean, ngrid),
        main = "Predicted mean", nlevels = 20)
points(model$X0, col = 'blue', pch = 20)

crit_grid <- apply(Xgrid, 1, crit, model = model)
filled.contour(x = xgrid, y = xgrid, matrix(crit_grid, ngrid),
               nlevels = 20, color.palette = terrain.colors, 
               main = "Final criterion landscape",
plot.axes = {axis(1); axis(2); points(model$X0, pch = 20)})

###############################################################################
## Bi-variate example (look-ahead version)
###############################################################################
\dontrun{  
nvar <- 2 

set.seed(42)
ftest <- function(x, coef = 0.1) return(sin(2*pi*sum(x)) + rnorm(1, sd = coef))

n <- 25 # must be a square
xgrid0 <- seq(0.1, 0.9, length.out = sqrt(n))
designs <- as.matrix(expand.grid(xgrid0, xgrid0))
X <- designs[rep(1:n, sample(1:10, n, replace = TRUE)),]
Z <- apply(X, 1, ftest)

model <- mleHomGP(X, Z, lower = rep(0.1, nvar), upper = rep(1, nvar))

ngrid <- 51
xgrid <- seq(0,1, length.out = ngrid)
Xgrid <- as.matrix(expand.grid(xgrid, xgrid))

nsteps <- 5 # Increase for more steps

crit <- "crit_EI"

for(i in 1:nsteps){
  res <- crit_optim(model, h = 3, crit = crit, control = list(multi.start = 100, maxit = 50))
  
  # If a replicate is selected
  if(!res$path[[1]]$new) print("Add replicate")
  
  newX <- res$par
  newZ <- ftest(newX)
  model <- update(object = model, Xnew = newX, Znew = newZ)
  
  ## Plots 
  preds <- predict(x = Xgrid, object =  model)
  contour(x = xgrid,  y = xgrid, z = matrix(preds$mean, ngrid),
          main = "Predicted mean", nlevels = 20)
  points(model$X0, col = 'blue', pch = 20)
  points(newX, col = "red", pch = 20)
  
  crit_grid <- apply(Xgrid, 1, crit, model = model)
  filled.contour(x = xgrid, y = xgrid, matrix(crit_grid, ngrid),
                 nlevels = 20, color.palette = terrain.colors,
  plot.axes = {axis(1); axis(2); points(model$X0, pch = 20)})
}
}
}
\references{
M. Binois, J. Huang, R. B. Gramacy, M. Ludkovski (2018+), Replication or exploration? Sequential design for stochastic simulation experiments,
Technometrics (to appear).\cr 
Preprint available on arXiv:1710.03206.
}
