% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dsc.R
\name{dsc}
\alias{dsc}
\title{Generate dynamic subset forecast combinations}
\usage{
dsc(gamma_grid, psi_grid, y, mu_mat, var_mat, delta, n_cores)
}
\arguments{
\item{gamma_grid}{A numerical vector that contains discount factors
to exponentially down-weight the past predictive performance
of the candidate forecasts.}

\item{psi_grid}{An integer vector that controls
the (possible) sizes of the active subsets.}

\item{y}{A matrix of dimension `T * 1` or numeric vector of length `T`
containing the observations of the target variable.}

\item{mu_mat}{A matrix with `T` rows containing
the first moment of each predictive density in each column.}

\item{var_mat}{A matrix with `T` rows containing
the second moment of each predictive density in each column.}

\item{delta}{A numeric value denoting the discount factor used
to down-weight the past predictive performance of the subset combinations.}

\item{n_cores}{An integer that denotes the number of CPU-cores
used for the computational estimation.}
}
\value{
A list that contains:
* (1) a vector with the first moments (point forecasts) of the STSC-Model,
* (2) a vector with the the second moments (variance) of the STSC-Model,
* (3) a vector that contains the selected values for gamma,
* (4) a vector that contains the selected values for psi and
* (5) a matrix that indicates the selected signals for every point in time.
}
\description{
`dsc()` can be used to generate forecast combinations
from a set of candidate density forecasts. For each period,
`dsc()` selects a subset of predictive densities with highest ranks
regarding (local) predictive accuracy.
Both the identities of the candidate forecasts
that are used for building the combined forecast and
the subset sizes may vary over time based on the data.
If only one candidate forecast is picked, the approach (temporarily)
collapses to pure model selection.
}
\examples{
\donttest{

   #########################################################
   ######### Forecasting quarterly U.S. inflation ##########
   #### Please see Koop & Korobilis (2023) for further  ####
   #### details regarding the data & external forecasts ####
   #########################################################

   # Packages
   library("hdflex")

   ########## Get Data ##########
   # Load Data
   inflation_data   <-  inflation_data
   benchmark_ar2    <-  benchmark_ar2

   # Set Index for Target Variable
   i  <-  1   # (1 -> GDPCTPI; 2 -> PCECTPI; 3 -> CPIAUCSL; 4 -> CPILFESL)

   # Subset Data (keep only data relevant for target variable i)
   dataset  <-  inflation_data[, c(1+(i-1),                          # Target Variable
                                   5+(i-1),                          # Lag 1
                                   9+(i-1),                          # Lag 2
                                   (13:16)[-i],                      # Remaining Price Series
                                   17:452,                           # Exogenous Predictor Variables
                                   seq(453+(i-1)*16,468+(i-1)*16))]  # Ext. Point Forecasts

   ########## STSC ##########
   ### Part 1: TV-C Model ###
   # Set Target Variable
   y  <-  dataset[,  1, drop = FALSE]

   # Set 'Simple' Signals
   X  <-  dataset[, 2:442, drop = FALSE]

   # Set External Point Forecasts (Koop & Korobilis 2023)
   F  <-  dataset[, 443:458, drop = FALSE]

   # Set TV-C-Parameter
   sample_length  <-  4 * 5
   lambda_grid    <-  c(0.90, 0.95, 1)
   kappa_grid     <-  0.98
   n_cores        <-  1

   # Apply TV-C-Function
   results  <-  hdflex::tvc(y,
                            X,
                            F,
                            lambda_grid,
                            kappa_grid,
                            sample_length,
                            n_cores)

   # Assign TV-C-Results
   forecast_tvc      <-  results[[1]]
   variance_tvc      <-  results[[2]]

   # Define Burn-In Period
   sample_period_idx  <-  80:nrow(dataset)
   sub_forecast_tvc   <-  forecast_tvc[sample_period_idx, , drop = FALSE]
   sub_variance_tvc   <-  variance_tvc[sample_period_idx, , drop = FALSE]
   sub_y              <-  y[sample_period_idx, , drop = FALSE]
   sub_dates          <-  rownames(dataset)[sample_period_idx]

   ### Part 2: Dynamic Subset Combination ###
   # Set DSC-Parameter
   nr_mods     <-  ncol(sub_forecast_tvc)
   gamma_grid  <-  c(0.40, 0.50, 0.60, 0.70, 0.80, 0.90,
                     0.91, 0.92, 0.93, 0.94, 0.95, 0.96, 0.97, 0.98, 0.99, 1.00)
   psi_grid    <-  c(1:100)
   delta       <-  0.95
   n_cores     <-  1

   # Apply DSC-Function
   results  <-  hdflex::dsc(gamma_grid,
                            psi_grid,
                            sub_y,
                            sub_forecast_tvc,
                            sub_variance_tvc,
                            delta,
                            n_cores)

   # Assign DSC-Results
   sub_forecast_stsc    <-  results[[1]]
   sub_variance_stsc    <-  results[[2]]
   sub_chosen_gamma     <-  results[[3]]
   sub_chosen_psi       <-  results[[4]]
   sub_chosen_signals   <-  results[[5]]

   # Define Evaluation Period
   eval_date_start      <-  "1991-01-01"
   eval_date_end        <-  "2021-12-31"
   eval_period_idx      <-  which(sub_dates > eval_date_start & sub_dates <= eval_date_end)

   # Trim Objects
   oos_y                <-  sub_y[eval_period_idx, ]
   oos_forecast_stsc    <-  sub_forecast_stsc[eval_period_idx]
   oos_variance_stsc    <-  sub_variance_stsc[eval_period_idx]
   oos_chosen_gamma     <-  sub_chosen_gamma[eval_period_idx]
   oos_chosen_psi       <-  sub_chosen_psi[eval_period_idx]
   oos_chosen_signals   <-  sub_chosen_signals[eval_period_idx, , drop = FALSE]
   oos_dates            <-  sub_dates[eval_period_idx]

   # Add Dates
   names(oos_forecast_stsc)     <-  oos_dates
   names(oos_variance_stsc)     <-  oos_dates
   names(oos_chosen_gamma)      <-  oos_dates
   names(oos_chosen_psi)        <-  oos_dates
   rownames(oos_chosen_signals) <-  oos_dates

   ### Part 3: Evaluation ###
   # Apply Summary-Function
   summary_results  <-  summary_stsc(oos_y,
                                     benchmark_ar2[, i],
                                     oos_forecast_stsc)
   # Assign Summary-Results
   cssed  <-  summary_results[[3]]
   mse    <-  summary_results[[4]]

   ########## Results ##########
   # Relative MSE
   print(paste("Relative MSE:", round(mse[[1]] / mse[[2]], 4)))

   # Plot CSSED
   plot(x    = as.Date(oos_dates),
        y    = cssed,
        ylim = c(-0.0008, 0.0008),
        main = "Cumulated squared error differences",
        type = "l",
        lwd  = 1.5,
        xlab = "Date",
        ylab = "CSSED") + abline(h = 0, lty = 2, col = "darkgray")

   # Plot Predictive Signals
   vec  <-  seq_len(dim(oos_chosen_signals)[2])
   mat  <-  oos_chosen_signals \%*\% diag(vec)
   mat[mat == 0]  <- NA
   matplot(x    = as.Date(oos_dates),
           y    = mat,
           cex  = 0.4,
           pch  = 20,
           type = "p",
           main = "Evolution of selected signal(s)",
           xlab = "Date",
           ylab = "Predictive Signal")

   # Plot Psi
   plot(x    = as.Date(oos_dates),
        y    = oos_chosen_psi,
        ylim = c(1, 100),
        main = "Evolution of the subset size",
        type = "p",
        cex  = 0.75,
        pch  = 20,
        xlab = "Date",
        ylab = "Psi")
 }
}
\references{
Beckmann, J., Koop, G., Korobilis, D., and Schüssler, R. A. (2020) "Exchange rate predictability and dynamic bayesian learning."
\emph{Journal of Applied Econometrics}, 35 (4): 410–421.

Koop, G. and Korobilis, D. (2012) "Forecasting inflation using dynamic model averaging."
\emph{International Economic Review}, 53 (3): 867–886.

Koop, G. and Korobilis, D. (2023) "Bayesian dynamic variable selection in high dimensions."
\emph{International Economic Review}.

Raftery, A. E., Kárn`y, M., and Ettler, P. (2010) "Online prediction under model uncertainty via dynamic model averaging: Application to a cold rolling mill."
\emph{Technometrics}, 52 (1): 52–66.

Del Negro, M., Hasegawa, R. B., and Schorfheide, F. (2016) "Dynamic prediction pools: An investigation of financial frictions and forecasting performance."
\emph{Journal of Econometrics}, 192 (2): 391–405.

West, M. and Harrison, J. (1997) "Bayesian forecasting and dynamic models"
\emph{Springer}, 2nd edn.
}
\seealso{
\url{https://github.com/lehmasve/hdflex#readme}
}
\author{
Philipp Adämmer, Sven Lehmann, Rainer Schüssler
}
