/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Gmodule.h" 
#define H5O_FRIEND     

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5FLprivate.h" 
#include "H5Gpkg.h"      
#include "H5Iprivate.h"  
#include "H5Opkg.h"      
#include "H5VLprivate.h" 

static void      *H5O__group_get_copy_file_udata(void);
static void       H5O__group_free_copy_file_udata(void *udata);
static htri_t     H5O__group_isa(const H5O_t *loc);
static void      *H5O__group_open(const H5G_loc_t *obj_loc, H5I_type_t *opened_type);
static void      *H5O__group_create(H5F_t *f, void *_crt_info, H5G_loc_t *obj_loc);
static H5O_loc_t *H5O__group_get_oloc(hid_t obj_id);
static herr_t     H5O__group_bh_info(const H5O_loc_t *loc, H5O_t *oh, H5_ih_info_t *bh_info);

const H5O_obj_class_t H5O_OBJ_GROUP[1] = {{
    H5O_TYPE_GROUP,                  
    "group",                         
    H5O__group_get_copy_file_udata,  
    H5O__group_free_copy_file_udata, 
    H5O__group_isa,                  
    H5O__group_open,                 
    H5O__group_create,               
    H5O__group_get_oloc,             
    H5O__group_bh_info,              
    NULL                             
}};

H5FL_DEFINE(H5G_copy_file_ud_t);

static void *
H5O__group_get_copy_file_udata(void)
{
    void *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    
    if (NULL == (ret_value = H5FL_CALLOC(H5G_copy_file_ud_t)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static void
H5O__group_free_copy_file_udata(void *_udata)
{
    H5G_copy_file_ud_t *udata = (H5G_copy_file_ud_t *)_udata;

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(udata);

    
    H5O_msg_free(H5O_PLINE_ID, udata->common.src_pline);

    
    udata = H5FL_FREE(H5G_copy_file_ud_t, udata);

    FUNC_LEAVE_NOAPI_VOID
} 

static htri_t
H5O__group_isa(const H5O_t *oh)
{
    htri_t stab_exists;      
    htri_t linfo_exists;     
    htri_t ret_value = FAIL; 

    FUNC_ENTER_PACKAGE

    assert(oh);

    
    if ((stab_exists = H5O_msg_exists_oh(oh, H5O_STAB_ID)) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_NOTFOUND, FAIL, "unable to read object header");
    if ((linfo_exists = H5O_msg_exists_oh(oh, H5O_LINFO_ID)) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_NOTFOUND, FAIL, "unable to read object header");

    ret_value = (stab_exists > 0 || linfo_exists > 0);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static void *
H5O__group_open(const H5G_loc_t *obj_loc, H5I_type_t *opened_type)
{
    H5G_t *grp       = NULL; 
    void  *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    assert(obj_loc);

    *opened_type = H5I_GROUP;

    
    if (NULL == (grp = H5G_open(obj_loc)))
        HGOTO_ERROR(H5E_SYM, H5E_CANTOPENOBJ, NULL, "unable to open group");

    ret_value = (void *)grp;

done:
    if (NULL == ret_value)
        if (grp && H5G_close(grp) < 0)
            HDONE_ERROR(H5E_SYM, H5E_CLOSEERROR, NULL, "unable to release group");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static void *
H5O__group_create(H5F_t *f, void *_crt_info, H5G_loc_t *obj_loc)
{
    H5G_obj_create_t *crt_info  = (H5G_obj_create_t *)_crt_info; 
    H5G_t            *grp       = NULL;                          
    void             *ret_value = NULL;                          

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(crt_info);
    assert(obj_loc);

    
    if (NULL == (grp = H5G__create(f, crt_info)))
        HGOTO_ERROR(H5E_SYM, H5E_CANTINIT, NULL, "unable to create group");

    
    if (NULL == (obj_loc->oloc = H5G_oloc(grp)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, NULL, "unable to get object location of group");
    if (NULL == (obj_loc->path = H5G_nameof(grp)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, NULL, "unable to get path of group");

    
    ret_value = grp;

done:
    if (ret_value == NULL)
        if (grp && H5G_close(grp) < 0)
            HDONE_ERROR(H5E_SYM, H5E_CLOSEERROR, NULL, "unable to release group");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static H5O_loc_t *
H5O__group_get_oloc(hid_t obj_id)
{
    H5G_t     *grp;              
    H5O_loc_t *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    
    if (NULL == (grp = (H5G_t *)H5VL_object(obj_id)))
        HGOTO_ERROR(H5E_OHDR, H5E_BADID, NULL, "couldn't get object from ID");

    
    if (NULL == (ret_value = H5G_oloc(grp)))
        HGOTO_ERROR(H5E_OHDR, H5E_CANTGET, NULL, "unable to get object location from object");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__group_bh_info(const H5O_loc_t *loc, H5O_t *oh, H5_ih_info_t *bh_info)
{
    htri_t  exists;               
    H5HF_t *fheap      = NULL;    
    H5B2_t *bt2_name   = NULL;    
    H5B2_t *bt2_corder = NULL;    
    herr_t  ret_value  = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(loc);
    assert(loc->file);
    assert(H5_addr_defined(loc->addr));
    assert(oh);
    assert(bh_info);

    
    if ((exists = H5O_msg_exists_oh(oh, H5O_LINFO_ID)) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_NOTFOUND, FAIL, "unable to read object header");
    if (exists > 0) {
        H5O_linfo_t linfo; 

        
        if (NULL == H5O_msg_read_oh(loc->file, oh, H5O_LINFO_ID, &linfo))
            HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "can't read LINFO message");

        
        if (H5_addr_defined(linfo.name_bt2_addr)) {
            
            if (NULL == (bt2_name = H5B2_open(loc->file, linfo.name_bt2_addr, NULL)))
                HGOTO_ERROR(H5E_SYM, H5E_CANTOPENOBJ, FAIL, "unable to open v2 B-tree for name index");

            
            if (H5B2_size(bt2_name, &bh_info->index_size) < 0)
                HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "can't retrieve B-tree storage info for name index");
        } 

        
        if (H5_addr_defined(linfo.corder_bt2_addr)) {
            
            if (NULL == (bt2_corder = H5B2_open(loc->file, linfo.corder_bt2_addr, NULL)))
                HGOTO_ERROR(H5E_SYM, H5E_CANTOPENOBJ, FAIL,
                            "unable to open v2 B-tree for creation order index");

            
            if (H5B2_size(bt2_corder, &bh_info->index_size) < 0)
                HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL,
                            "can't retrieve B-tree storage info for creation order index");
        } 

        
        if (H5_addr_defined(linfo.fheap_addr)) {
            
            if (NULL == (fheap = H5HF_open(loc->file, linfo.fheap_addr)))
                HGOTO_ERROR(H5E_SYM, H5E_CANTOPENOBJ, FAIL, "unable to open fractal heap");

            
            if (H5HF_size(fheap, &bh_info->heap_size) < 0)
                HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "can't retrieve fractal heap storage info");
        } 
    }     
    else {
        H5O_stab_t stab; 

        
        if (NULL == H5O_msg_read_oh(loc->file, oh, H5O_STAB_ID, &stab))
            HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "can't find LINFO nor STAB messages");

        
        if (H5G__stab_bh_size(loc->file, &stab, bh_info) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "can't retrieve symbol table size info");
    } 

done:
    
    if (fheap && H5HF_close(fheap) < 0)
        HDONE_ERROR(H5E_SYM, H5E_CANTCLOSEOBJ, FAIL, "can't close fractal heap");
    if (bt2_name && H5B2_close(bt2_name) < 0)
        HDONE_ERROR(H5E_SYM, H5E_CANTCLOSEOBJ, FAIL, "can't close v2 B-tree for name index");
    if (bt2_corder && H5B2_close(bt2_corder) < 0)
        HDONE_ERROR(H5E_SYM, H5E_CANTCLOSEOBJ, FAIL, "can't close v2 B-tree for creation order index");

    FUNC_LEAVE_NOAPI(ret_value)
} 
