% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cols_merge.R
\name{cols_merge_n_pct}
\alias{cols_merge_n_pct}
\title{Merge two columns to combine counts and percentages}
\usage{
cols_merge_n_pct(data, col_n, col_pct, rows = everything(), autohide = TRUE)
}
\arguments{
\item{data}{\emph{The gt table data object}

\verb{obj:<gt_tbl>} // \strong{required}

This is the \strong{gt} table object that is commonly created through use of the
\code{\link[=gt]{gt()}} function.}

\item{col_n}{\emph{Column to target for counts}

\verb{<column-targeting expression>} // \strong{required}

The column that contains values for the count component. While select
helper functions such as \code{\link[=starts_with]{starts_with()}} and \code{\link[=ends_with]{ends_with()}} can be used for
column targeting, it's recommended that a single column name be used. This
is to ensure that exactly one column is provided here.}

\item{col_pct}{\emph{Column to target for percentages}

\verb{<column-targeting expression>} // \strong{required}

The column that contains values for the percentage component. While select
helper functions such as \code{\link[=starts_with]{starts_with()}} and \code{\link[=ends_with]{ends_with()}} can be used for
column targeting, it's recommended that a single column name be used. This
is to ensure that exactly one column is provided here. This column should
be formatted such that percentages are displayed (e.g., with
\code{fmt_percent()}).}

\item{rows}{\emph{Rows to target}

\verb{<row-targeting expression>} // \emph{default:} \code{everything()}

In conjunction with \code{columns}, we can specify which of their rows should
participate in the merging process. The default \code{\link[=everything]{everything()}} results in
all rows in \code{columns} being formatted. Alternatively, we can supply a
vector of row IDs within \code{c()}, a vector of row indices, or a select
helper function (e.g. \code{\link[=starts_with]{starts_with()}}, \code{\link[=ends_with]{ends_with()}}, \code{\link[=contains]{contains()}},
\code{\link[=matches]{matches()}}, \code{\link[=num_range]{num_range()}}, and \code{\link[=everything]{everything()}}). We can also use
expressions to filter down to the rows we need
(e.g., \verb{[colname_1] > 100 & [colname_2] < 50}).}

\item{autohide}{\emph{Automatic hiding of the \code{col_pct} column}

\verb{scalar<logical>} // \emph{default:} \code{TRUE}

An option to automatically hide the column specified as \code{col_pct}. Any
columns with their state changed to hidden will behave the same as before,
they just won't be displayed in the finalized table.}
}
\value{
An object of class \code{gt_tbl}.
}
\description{
\code{cols_merge_n_pct()} is a specialized variant of \code{\link[=cols_merge]{cols_merge()}},
It operates by taking two columns that constitute both a count (\code{col_n}) and
a fraction of the total population (\code{col_pct}) and merges them into a single
column. What results is a column containing both counts and their associated
percentages (e.g., \verb{12 (23.2\%)}). The column specified in \code{col_pct} is
dropped from the output table.
}
\section{Comparison with other column-merging functions}{


This function could be somewhat replicated using \code{\link[=cols_merge]{cols_merge()}}, however,
\code{cols_merge_n_pct()} employs the following specialized semantics for \code{NA}
and zero-value handling:
\enumerate{
\item \code{NA}s in \code{col_n} result in missing values for the merged
column (e.g., \code{NA} + \verb{10.2\%} = \code{NA})
\item \code{NA}s in \code{col_pct} (but not \code{col_n}) result in
base values only for the merged column (e.g., \code{13} + \code{NA} = \code{13})
\item \code{NA}s both \code{col_n} and \code{col_pct} result in
missing values for the merged column (e.g., \code{NA} + \code{NA} = \code{NA})
\item If a zero (\code{0}) value is in \code{col_n} then the formatted output will be
\code{"0"} (i.e., no percentage will be shown)
}

Any resulting \code{NA} values in the \code{col_n} column following the merge
operation can be easily formatted using \code{\link[=sub_missing]{sub_missing()}}.
Separate calls of \code{\link[=sub_missing]{sub_missing()}} can be used for the \code{col_n} and
\code{col_pct} columns for finer control of the replacement values. It is the
responsibility of the user to ensure that values are correct in both the
\code{col_n} and \code{col_pct} columns (this function neither generates nor
recalculates values in either). Formatting of each column can be done
independently in separate \code{\link[=fmt_number]{fmt_number()}} and \code{\link[=fmt_percent]{fmt_percent()}} calls.

This function is part of a set of four column-merging functions. The other
three are the general \code{\link[=cols_merge]{cols_merge()}} function and the specialized
\code{\link[=cols_merge_uncert]{cols_merge_uncert()}} and \code{\link[=cols_merge_range]{cols_merge_range()}} functions. These functions
operate similarly, where the non-target columns can be optionally hidden from
the output table through the \code{hide_columns} or \code{autohide} options.
}

\section{Examples}{


Using a summarized version of the \code{\link{pizzaplace}} dataset, let's create a
\strong{gt} table that displays the counts and percentages of the top 3 pizzas
sold by pizza category in 2015. The \code{cols_merge_n_pct()} function is used to
merge the \code{n} and \code{frac} columns (and the \code{frac} column is formatted using
\code{\link[=fmt_percent]{fmt_percent()}}).

\if{html}{\out{<div class="sourceCode r">}}\preformatted{pizzaplace |>
  dplyr::count(name, type, price, sort = TRUE) |>
  dplyr::mutate(frac = prop.table(n)) |>
  dplyr::slice_max(n, n = 3, by = type) |>
  dplyr::arrange(type) |>
  gt(
    rowname_col = "name",
    groupname_col = "type"
  ) |>
  fmt_currency(price) |>
  fmt_percent(frac) |>
  cols_merge_n_pct(
    col_n = n,
    col_pct = frac
  ) |>
  cols_label(
    n = md("*N* (\%)"),
    price = "Price"
  ) |>
  tab_style(
    style = cell_text(font = "monospace"),
    locations = cells_stub()
  ) |>
  tab_stubhead(md("Cat. and  \\nPizza Code")) |>
  tab_header(title = "Top 3 Pizzas Sold by Category in 2015") |>
  tab_options(table.width = px(512))
}\if{html}{\out{</div>}}

\if{html}{\out{
<img src="https://raw.githubusercontent.com/rstudio/gt/master/images/man_cols_merge_n_pct_1.png" alt="This image of a table was generated from the first code example in the `cols_merge_n_pct()` help file." style="width:100\%;">
}}
}

\section{Function ID}{

5-17
}

\section{Function Introduced}{

\code{v0.3.0} (May 12, 2021)
}

\seealso{
Other column modification functions: 
\code{\link{cols_add}()},
\code{\link{cols_align}()},
\code{\link{cols_align_decimal}()},
\code{\link{cols_hide}()},
\code{\link{cols_label}()},
\code{\link{cols_label_with}()},
\code{\link{cols_merge}()},
\code{\link{cols_merge_range}()},
\code{\link{cols_merge_uncert}()},
\code{\link{cols_move}()},
\code{\link{cols_move_to_end}()},
\code{\link{cols_move_to_start}()},
\code{\link{cols_nanoplot}()},
\code{\link{cols_unhide}()},
\code{\link{cols_units}()},
\code{\link{cols_width}()}
}
\concept{column modification functions}
