% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/format_vec.R
\name{vec_fmt_partsper}
\alias{vec_fmt_partsper}
\title{Format a vector as parts-per quantities}
\usage{
vec_fmt_partsper(
  x,
  to_units = c("per-mille", "per-myriad", "pcm", "ppm", "ppb", "ppt", "ppq"),
  symbol = "auto",
  decimals = 2,
  drop_trailing_zeros = FALSE,
  drop_trailing_dec_mark = TRUE,
  scale_values = TRUE,
  use_seps = TRUE,
  pattern = "{x}",
  sep_mark = ",",
  dec_mark = ".",
  force_sign = FALSE,
  incl_space = "auto",
  locale = NULL,
  output = c("auto", "plain", "html", "latex", "rtf", "word")
)
}
\arguments{
\item{x}{A numeric vector.}

\item{to_units}{A keyword that signifies the desired output quantity. This
can be any from the following set: \code{"per-mille"}, \code{"per-myriad"}, \code{"pcm"},
\code{"ppm"}, \code{"ppb"}, \code{"ppt"}, or \code{"ppq"}.}

\item{symbol}{The symbol/units to use for the quantity. By default, this is
set to \code{"auto"} and \strong{gt} will choose the appropriate symbol based on the
\code{to_units} keyword and the output context. However, this can be changed by
supplying a string (e.g, using \code{symbol = "ppbV"} when \code{to_units = "ppb"}).}

\item{decimals}{An option to specify the exact number of decimal places to
use. The default number of decimal places is \code{2}.}

\item{drop_trailing_zeros}{A logical value that allows for removal of
trailing zeros (those redundant zeros after the decimal mark).}

\item{drop_trailing_dec_mark}{A logical value that determines whether decimal
marks should always appear even if there are no decimal digits to display
after formatting (e.g, \code{23} becomes \code{23.}). The default for this is \code{TRUE},
which means that trailing decimal marks are not shown.}

\item{scale_values}{Should the values be scaled through multiplication
according to the keyword set in \code{to_units}? By default this is \code{TRUE} since
the expectation is that normally values are proportions. Setting to \code{FALSE}
signifies that the values are already scaled and require only the
appropriate symbol/units when formatted.}

\item{use_seps}{An option to use digit group separators. The type of digit
group separator is set by \code{sep_mark} and overridden if a locale ID is
provided to \code{locale}. This setting is \code{TRUE} by default.}

\item{pattern}{A formatting pattern that allows for decoration of the
formatted value. The value itself is represented by \code{{x}} and all other
characters are taken to be string literals.}

\item{sep_mark}{The mark to use as a separator between groups of digits
(e.g., using \code{sep_mark = ","} with \code{1000} would result in a formatted value
of \verb{1,000}).}

\item{dec_mark}{The character to use as a decimal mark (e.g., using
\code{dec_mark = ","} with \code{0.152} would result in a formatted value of
\verb{0,152}).}

\item{force_sign}{Should the positive sign be shown for positive values
(effectively showing a sign for all values except zero)? If so, use \code{TRUE}
for this option. The default is \code{FALSE}, where only negative numbers will
display a minus sign. This option is disregarded when using accounting
notation with \code{accounting = TRUE}.}

\item{incl_space}{An option for whether to include a space between the value
and the symbol/units. The default is \code{"auto"} which provides spacing
dependent on the mark itself. This can be directly controlled by using
either \code{TRUE} or \code{FALSE}.}

\item{locale}{An optional locale ID that can be used for formatting the value
according the locale's rules. Examples include \code{"en_US"} for English
(United States) and \code{"fr_FR"} for French (France). The use of a valid
locale ID will override any values provided in \code{sep_mark} and \code{dec_mark}.
We can use the \code{\link[=info_locales]{info_locales()}} function as a useful reference for all of
the locales that are supported.}

\item{output}{The output style of the resulting character vector. This can
either be \code{"auto"} (the default), \code{"plain"}, \code{"html"}, \code{"latex"}, \code{"rtf"},
or \code{"word"}. In \strong{knitr} rendering (i.e., Quarto or R Markdown), the
\code{"auto"} option will choose the correct \code{output} value}
}
\value{
A character vector.
}
\description{
With numeric values in a vector, we can format the values so that they
are rendered as \emph{per mille}, \emph{ppm}, \emph{ppb}, etc., quantities. The following
list of keywords (with associated naming and scaling factors) is available to
use within \code{vec_fmt_partsper()}:
\itemize{
\item \code{"per-mille"}: Per mille, (1 part in \verb{1,000})
\item \code{"per-myriad"}: Per myriad, (1 part in \verb{10,000})
\item \code{"pcm"}: Per cent mille (1 part in \verb{100,000})
\item \code{"ppm"}: Parts per million, (1 part in \verb{1,000,000})
\item \code{"ppb"}: Parts per billion, (1 part in \verb{1,000,000,000})
\item \code{"ppt"}: Parts per trillion, (1 part in \verb{1,000,000,000,000})
\item \code{"ppq"}: Parts per quadrillion, (1 part in \verb{1,000,000,000,000,000})
}

The function provides a lot of formatting control and we can use the
following options:
\itemize{
\item custom symbol/units: we can override the automatic symbol or units display
with our own choice as the situation warrants
\item decimals: choice of the number of decimal places, option to drop
trailing zeros, and a choice of the decimal symbol
\item digit grouping separators: options to enable/disable digit separators
and provide a choice of separator symbol
\item value scaling toggle: choose to disable automatic value scaling in the
situation that values are already scaled coming in (and just require the
appropriate symbol or unit display)
\item pattern: option to use a text pattern for decoration of the formatted
values
\item locale-based formatting: providing a locale ID will result in number
formatting specific to the chosen locale
}
}
\section{Examples}{


Let's create a numeric vector for the next few examples:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{num_vals <- c(10^(-3:-5), NA)
}\if{html}{\out{</div>}}

Using \code{vec_fmt_partsper()} with the default options will create a character
vector where the resultant per millle values have two decimal places and \code{NA}
values will render as \code{"NA"}. The rendering context will be autodetected
unless specified in the \code{output} argument (here, it is of the \code{"plain"}
output type).

\if{html}{\out{<div class="sourceCode r">}}\preformatted{vec_fmt_partsper(num_vals)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{#> [1] "1.00‰" "0.10‰" "0.01‰" "NA"
}\if{html}{\out{</div>}}

We can change the output units to a different measure. If ppm units are
desired then \code{to_units = "ppm"} can be used.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{vec_fmt_partsper(num_vals, to_units = "ppm")
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{#> [1] "1,000.00 ppm" "100.00 ppm" "10.00 ppm" "NA"
}\if{html}{\out{</div>}}

We can change the decimal mark to a comma, and we have to be sure to change
the digit separator mark from the default comma to something else (a period
works here):

\if{html}{\out{<div class="sourceCode r">}}\preformatted{vec_fmt_partsper(num_vals, to_units = "ppm", sep_mark = ".", dec_mark = ",")
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{#> [1] "1.000,00 ppm" "100,00 ppm" "10,00 ppm" "NA"
}\if{html}{\out{</div>}}

If we are formatting for a different locale, we could supply the locale ID
and let \strong{gt} handle these locale-specific formatting options:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{vec_fmt_partsper(num_vals, to_units = "ppm", locale = "es")
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{#> [1] "1.000,00 ppm" "100,00 ppm" "10,00 ppm" "NA"
}\if{html}{\out{</div>}}

As a last example, one can wrap the values in a pattern with the \code{pattern}
argument. Note here that \code{NA} values won't have the pattern applied.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{vec_fmt_partsper(num_vals, to_units = "ppm", pattern = "\{x\}V")
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{#> [1] "1,000.00 ppmV" "100.00 ppmV" "10.00 ppmV" "NA"
}\if{html}{\out{</div>}}
}

\section{Function ID}{

14-6
}

\seealso{
Other vector formatting functions: 
\code{\link{vec_fmt_bytes}()},
\code{\link{vec_fmt_currency}()},
\code{\link{vec_fmt_datetime}()},
\code{\link{vec_fmt_date}()},
\code{\link{vec_fmt_duration}()},
\code{\link{vec_fmt_engineering}()},
\code{\link{vec_fmt_fraction}()},
\code{\link{vec_fmt_integer}()},
\code{\link{vec_fmt_markdown}()},
\code{\link{vec_fmt_number}()},
\code{\link{vec_fmt_percent}()},
\code{\link{vec_fmt_scientific}()},
\code{\link{vec_fmt_time}()}
}
\concept{vector formatting functions}
