% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/expected_event.R
\name{expected_event}
\alias{expected_event}
\title{Expected events observed under piecewise exponential model}
\usage{
expected_event(
  enroll_rate = tibble::tibble(duration = c(2, 2, 10), rate = c(3, 6, 9)),
  fail_rate = tibble::tibble(duration = c(3, 100), fail_rate = log(2)/c(9, 18),
    dropout_rate = rep(0.001, 2)),
  total_duration = 25,
  simple = TRUE
)
}
\arguments{
\item{enroll_rate}{Enrollment rates; see details and examples.}

\item{fail_rate}{Failure rates and dropout rates by period.}

\item{total_duration}{Total follow-up from start of enrollment to data cutoff.}

\item{simple}{If default (\code{TRUE}), return numeric expected number of events,
otherwise a tibble as described below.}
}
\value{
The default when \code{simple = TRUE} is to return the total expected
number of events as a real number.
Otherwise, when \code{simple = FALSE}, a tibble is returned with
the following variables for each period specified in \code{fail_rate}:
\itemize{
\item \code{t}: start of period.
\item \code{fail_rate}: failure rate during the period.
\item \code{Events}: expected events during the period.
}

The records in the returned tibble correspond to the input tibble \code{fail_rate}.
}
\description{
Computes expected events over time and by strata
under the assumption of piecewise constant enrollment rates and piecewise
exponential failure and censoring rates.
The piecewise exponential distribution allows a simple method to specify a distribution
and enrollment pattern
where the enrollment, failure and dropout rates changes over time.
While the main purpose may be to generate a trial that can be analyzed at a single point in time or
using group sequential methods, the routine can also be used to simulate an adaptive trial design.
The intent is to enable sample size calculations under non-proportional hazards assumptions
for stratified populations.
}
\details{
More periods will generally be supplied in output than those that are input.
The intent is to enable expected event calculations in a tidy format to
maximize flexibility for a variety of purposes.
}
\section{Specification}{

\if{latex}{
 \itemize{
   \item Validate if input enrollment rate contains total duration column.
   \item Validate if input enrollment rate contains rate column.
   \item Validate if input failure rate contains duration column.
   \item Validate if input failure rate contains failure rate column.
   \item Validate if input failure rate contains dropout rate column.
   \item Validate if input trial total follow-up (total duration) is a non-empty vector of positive integers.
   \item Validate if input simple is logical.
   \item Define a tibble with the start opening for enrollment at zero and cumulative duration.
   Add the event (or failure) time corresponding to the start of the enrollment.
   Finally, add the enrollment rate to the tibble
   corresponding to the start and end (failure) time.
   This will be recursively used to calculate the expected
   number of events later. For details, see vignette/eEventsTheory.Rmd
   \item Define a tibble including the cumulative duration of failure rates, the corresponding start time of
   the enrollment, failure rate and dropout rates.  For details, see vignette/eEventsTheory.Rmd
   \item Only consider the failure rates in the interval of the end failure rate and total duration.
   \item Compute the failure rates over time using \code{stepfun} which is used
    to group rows by periods defined by fail_rate.
   \item Compute the dropout rate over time using \code{stepfun}.
   \item Compute the enrollment rate over time using \code{stepfun}. Details are
   available in vignette/eEventsTheory.Rmd.
   \item Compute expected events by interval at risk using the notations and descriptions in
   vignette/eEventsTheory.Rmd.
   \item Return \code{expected_event}
 }
}
}

\examples{
library(tibble)
library(gsDesign2)

# Default arguments, simple output (total event count only)
expected_event()

# Event count by time period
expected_event(simple = FALSE)

# Early cutoff
expected_event(total_duration = .5)

# Single time period example
expected_event(
  enroll_rate = tibble(duration = 10, rate = 10),
  fail_rate = tibble(duration = 100, fail_rate = log(2) / 6, dropout_rate = .01),
  total_duration = 22,
  simple = FALSE
)

# Single time period example, multiple enrollment periods
expected_event(
  enroll_rate = tibble(duration = c(5, 5), rate = c(10, 20)),
  fail_rate = tibble(duration = 100, fail_rate = log(2) / 6, dropout_rate = .01),
  total_duration = 22, simple = FALSE
)
}
