% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RcppExports.R
\name{dgkw}
\alias{dgkw}
\title{Density of the Generalized Kumaraswamy Distribution}
\usage{
dgkw(x, alpha, beta, gamma, delta, lambda, log_prob = FALSE)
}
\arguments{
\item{x}{Vector of quantiles (values between 0 and 1).}

\item{alpha}{Shape parameter \code{alpha} > 0. Can be a scalar or a vector.
Default: 1.0.}

\item{beta}{Shape parameter \code{beta} > 0. Can be a scalar or a vector.
Default: 1.0.}

\item{gamma}{Shape parameter \code{gamma} > 0. Can be a scalar or a vector.
Default: 1.0.}

\item{delta}{Shape parameter \code{delta} >= 0. Can be a scalar or a vector.
Default: 0.0.}

\item{lambda}{Shape parameter \code{lambda} > 0. Can be a scalar or a vector.
Default: 1.0.}

\item{log_prob}{Logical; if \code{TRUE}, the logarithm of the density is
returned. Default: \code{FALSE}.}
}
\value{
A vector of density values (\eqn{f(x)}) or log-density values
(\eqn{\log(f(x))}). The length of the result is determined by the recycling
rule applied to the arguments (\code{x}, \code{alpha}, \code{beta},
\code{gamma}, \code{delta}, \code{lambda}). Returns \code{0} (or \code{-Inf}
if \code{log_prob = TRUE}) for \code{x} outside the interval (0, 1), or
\code{NaN} if parameters are invalid.
}
\description{
Computes the probability density function (PDF) for the five-parameter
Generalized Kumaraswamy (GKw) distribution, defined on the interval (0, 1).
}
\details{
The probability density function of the Generalized Kumaraswamy (GKw)
distribution with parameters \code{alpha} (\eqn{\alpha}), \code{beta}
(\eqn{\beta}), \code{gamma} (\eqn{\gamma}), \code{delta} (\eqn{\delta}), and
\code{lambda} (\eqn{\lambda}) is given by:
\deqn{
f(x; \alpha, \beta, \gamma, \delta, \lambda) =
  \frac{\lambda \alpha \beta x^{\alpha-1}(1-x^{\alpha})^{\beta-1}}
       {B(\gamma, \delta+1)}
  [1-(1-x^{\alpha})^{\beta}]^{\gamma\lambda-1}
  [1-[1-(1-x^{\alpha})^{\beta}]^{\lambda}]^{\delta}
}
for \eqn{x \in (0,1)}, where \eqn{B(a, b)} is the Beta function
\code{\link[base]{beta}}.

This distribution was proposed by Cordeiro & de Castro (2011) and includes
several other distributions as special cases:
\itemize{
\item Kumaraswamy (Kw): \code{gamma = 1}, \code{delta = 0}, \code{lambda = 1}
\item Exponentiated Kumaraswamy (EKw): \code{gamma = 1}, \code{delta = 0}
\item Beta-Kumaraswamy (BKw): \code{lambda = 1}
\item Generalized Beta type 1 (GB1 - implies McDonald): \code{alpha = 1}, \code{beta = 1}
\item Beta distribution: \code{alpha = 1}, \code{beta = 1}, \code{lambda = 1}
}
The function includes checks for valid parameters and input values \code{x}.
It uses numerical stabilization for \code{x} close to 0 or 1.
}
\examples{
\donttest{
# Simple density evaluation at a point
dgkw(0.5, alpha = 2, beta = 3, gamma = 1, delta = 0, lambda = 1) # Kw case

# Plot the PDF for various parameter sets
x_vals <- seq(0.01, 0.99, by = 0.01)

# Standard Kumaraswamy (gamma=1, delta=0, lambda=1)
pdf_kw <- dgkw(x_vals, alpha = 2, beta = 3, gamma = 1, delta = 0, lambda = 1)

# Beta equivalent (alpha=1, beta=1, lambda=1) - Beta(gamma, delta+1)
pdf_beta <- dgkw(x_vals, alpha = 1, beta = 1, gamma = 2, delta = 3, lambda = 1)
# Compare with stats::dbeta
pdf_beta_check <- stats::dbeta(x_vals, shape1 = 2, shape2 = 3 + 1)
# max(abs(pdf_beta - pdf_beta_check)) # Should be close to zero

# Exponentiated Kumaraswamy (gamma=1, delta=0)
pdf_ekw <- dgkw(x_vals, alpha = 2, beta = 3, gamma = 1, delta = 0, lambda = 2)

plot(x_vals, pdf_kw, type = "l", ylim = range(c(pdf_kw, pdf_beta, pdf_ekw)),
     main = "GKw Densities Examples", ylab = "f(x)", xlab="x", col = "blue")
lines(x_vals, pdf_beta, col = "red")
lines(x_vals, pdf_ekw, col = "green")
legend("topright", legend = c("Kw(2,3)", "Beta(2,4) equivalent", "EKw(2,3, lambda=2)"),
       col = c("blue", "red", "green"), lty = 1, bty = "n")

# Log-density
log_pdf_val <- dgkw(0.5, 2, 3, 1, 0, 1, log_prob = TRUE)
print(log_pdf_val)
print(log(dgkw(0.5, 2, 3, 1, 0, 1))) # Should match

}

}
\references{
Cordeiro, G. M., & de Castro, M. (2011). A new family of generalized
distributions. \emph{Journal of Statistical Computation and Simulation},
\emph{81}(7), 883-898.

Kumaraswamy, P. (1980). A generalized probability density function for
double-bounded random processes. \emph{Journal of Hydrology}, \emph{46}(1-2), 79-88.
}
\seealso{
\code{\link{pgkw}}, \code{\link{qgkw}}, \code{\link{rgkw}} (if these exist),
\code{\link[stats]{dbeta}}, \code{\link[stats]{integrate}}
}
\author{
Lopes, J. E.
}
\keyword{density}
\keyword{distribution}
