#' Opens a full screen map after uploading a geojson file
#'
#' Takes a gist object and a input geojson file name and renders fullscreen map
#' 
#' @export
#' @importFrom assertthat assert_that has_extension
#' 
#' @param x An object of class \code{gist} generated by \code{\link{gist_create}}
#' or \code{\link{gist_create_obj}}
#' @param browse Default: \code{TRUE}. Set to \code{FALSE} if you don't want to 
#' automatically browse to the URL.
#' @examples \dontrun{
#' file <- system.file("examples", "ecoengine_eg.geojson", package = "gistr")
#' gist_id <- gist_create(file, browse = FALSE)
#' gist_map(gist_id)
#'}
gist_map <- function(x, browse = TRUE) {
  assertthat::assert_that(is(x, "gist"))
  file <- names(x$files)
  assertthat::assert_that(has_extension(file, "geojson"))
  render_url <- "https://render.githubusercontent.com/view/geojson/?url=https://gist.githubusercontent.com/"
  point <- length(x$history)
  full_path <- x$history[[point]]$url
  user <- x$history[[point]]$user$login
  commit <- gsub("https://api.github.com/gists/", "", full_path)
  actual_commit <- strsplit(commit, "/")[[1]][1]
  path <- paste(render_url, user, actual_commit, "raw", file, sep = "/")
  if (browse) {
    browseURL(path)
  } else {
    path
  }
}
