% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/curve_interval.R
\encoding{UTF-8}
\name{curve_interval}
\alias{curve_interval}
\alias{curve_interval.matrix}
\alias{curve_interval.rvar}
\alias{curve_interval.data.frame}
\title{Curvewise point and interval summaries for tidy data frames of draws from distributions}
\usage{
curve_interval(
  .data,
  ...,
  .along = NULL,
  .width = 0.5,
  na.rm = FALSE,
  .interval = c("mhd", "mbd", "bd", "bd-mbd")
)

\method{curve_interval}{matrix}(
  .data,
  ...,
  .along = NULL,
  .width = 0.5,
  na.rm = FALSE,
  .interval = c("mhd", "mbd", "bd", "bd-mbd")
)

\method{curve_interval}{rvar}(
  .data,
  ...,
  .along = NULL,
  .width = 0.5,
  na.rm = FALSE,
  .interval = c("mhd", "mbd", "bd", "bd-mbd")
)

\method{curve_interval}{data.frame}(
  .data,
  ...,
  .along = NULL,
  .width = 0.5,
  na.rm = FALSE,
  .interval = c("mhd", "mbd", "bd", "bd-mbd"),
  .simple_names = TRUE,
  .exclude = c(".chain", ".iteration", ".draw", ".row")
)
}
\arguments{
\item{.data}{One of:
\itemize{
\item A data frame (or grouped data frame as returned by \code{\link[=group_by]{group_by()}})
that contains draws to summarize.
\item A \link[posterior:rvar]{posterior::rvar} vector.
\item A matrix; in which case the first dimension should be draws and the second
dimension values of the curve.
}}

\item{...}{Bare column names or expressions that, when evaluated in the context of
\code{.data}, represent draws to summarize. If this is empty, then by default all
columns that are not group columns and which are not in \code{.exclude} (by default
\code{".chain"}, \code{".iteration"}, \code{".draw"}, and \code{".row"}) will be summarized.
This can be numeric columns, list columns containing numeric vectors, or
\code{\link[posterior:rvar]{posterior::rvar()}}s.}

\item{.along}{Which columns are the input values to the function describing the curve (e.g., the "x"
values). Supports tidyselect syntax, as in \code{\link[dplyr:select]{dplyr::select()}}. Intervals are calculated jointly with
respect to these variables, conditional on all other grouping variables in the data frame. The default
(\code{NULL}) causes \code{\link[=curve_interval]{curve_interval()}} to use all grouping variables in the input data frame as the value
for \code{.along}, which will generate the most conservative intervals. However, if you want to calculate
intervals for some function \code{y = f(x)} conditional on some other variable(s) (say, conditional on a
factor \code{g}), you would group by \code{g}, then use \code{.along = x} to calculate intervals jointly over \code{x}
conditional on \code{g}. To avoid selecting any variables as input values to the function describing the
curve, use \code{character()}; this will produce conditional intervals only (the result in this case should
be very similar to \code{median_qi()}). Currently only supported when \code{.data} is a data frame.}

\item{.width}{vector of probabilities to use that determine the widths of the resulting intervals.
If multiple probabilities are provided, multiple rows per group are generated, each with
a different probability interval (and value of the corresponding \code{.width} column).}

\item{na.rm}{logical value indicating whether \code{NA} values should be stripped before the computation proceeds.
If \code{FALSE} (the default), the presence of \code{NA} values in the columns to be summarized will generally
result in an error. If \code{TRUE}, \code{NA} values will be removed in the calculation of intervals so long
as \code{.interval} is \code{"mhd"}; other methods do not currently support \code{na.rm}. Be cautious in applying
this parameter: in general, it is unclear what a joint interval should be when any of the values
are missing!}

\item{.interval}{The method used to calculate the intervals. Currently, all methods rank the curves
using some measure of \emph{data depth}, then create envelopes containing the \code{.width}\% "deepest" curves.
Available methods are:
\itemize{
\item \code{"mhd"}: mean halfspace depth (Fraiman and Muniz 2001).
\item \code{"mbd"}: modified band depth (Sun and Genton 2011): calls \code{\link[fda:fbplot]{fda::fbplot()}} with \code{method = "MBD"}.
\item \code{"bd"}: band depth (Sun and Genton 2011): calls \code{\link[fda:fbplot]{fda::fbplot()}} with \code{method = "BD2"}.
\item \code{"bd-mbd"}: band depth, breaking ties with modified band depth (Sun and Genton 2011): calls
\code{\link[fda:fbplot]{fda::fbplot()}} with \code{method = "Both"}.
}}

\item{.simple_names}{When \code{TRUE} and only a single column / vector is to be summarized, use the
name \code{.lower} for the lower end of the interval and \code{.upper} for the
upper end. When \code{FALSE} and \code{.data} is a data frame,
names the lower and upper intervals for each column \code{x} \code{x.lower} and \code{x.upper}.}

\item{.exclude}{A character vector of names of columns to be excluded from summarization
if no column names are specified to be summarized. Default ignores several meta-data column
names used in \pkg{ggdist} and \pkg{tidybayes}.}
}
\value{
A data frame containing point summaries and intervals, with at least one column corresponding
to the point summary, one to the lower end of the interval, one to the upper end of the interval, the
width of the interval (\code{.width}), the type of point summary (\code{.point}), and the type of interval (\code{.interval}).
}
\description{
Translates draws from distributions in a grouped data frame into a set of point and
interval summaries using a curve boxplot-inspired approach.
}
\details{
Intervals are calculated by ranking the curves using some measure of \emph{data depth}, then
using binary search to find a cutoff \code{k} such that an envelope containing the \code{k}\% "deepest"
curves also contains \code{.width}\% of the curves, for each value of \code{.width} (note that \code{k}
and \code{.width} are not necessarily the same). This is in contrast to most functional boxplot
or curve boxplot approaches, which tend to simply take the \code{.width}\% deepest curves, and
are generally quite conservative (i.e. they may contain more than \code{.width}\% of the curves).

See Mirzargar \emph{et al.} (2014) or Juul \emph{et al.} (2020) for an accessible introduction
to data depth and curve boxplots / functional boxplots.
}
\examples{

library(dplyr)
library(ggplot2)

# generate a set of curves
k = 11 # number of curves
n = 201
df = tibble(
    .draw = rep(1:k, n),
    mean = rep(seq(-5,5, length.out = k), n),
    x = rep(seq(-15,15,length.out = n), each = k),
    y = dnorm(x, mean, 3)
  )

# see pointwise intervals...
df \%>\%
  group_by(x) \%>\%
  median_qi(y, .width = c(.5)) \%>\%
  ggplot(aes(x = x, y = y)) +
  geom_lineribbon(aes(ymin = .lower, ymax = .upper)) +
  geom_line(aes(group = .draw), alpha=0.15, data = df) +
  scale_fill_brewer() +
  ggtitle("50\% pointwise intervals with point_interval()") +
  theme_ggdist()

\dontshow{if (requireNamespace("posterior", quietly = TRUE)) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
# ... compare them to curvewise intervals
df \%>\%
  group_by(x) \%>\%
  curve_interval(y, .width = c(.5)) \%>\%
  ggplot(aes(x = x, y = y)) +
  geom_lineribbon(aes(ymin = .lower, ymax = .upper)) +
  geom_line(aes(group = .draw), alpha=0.15, data = df) +
  scale_fill_brewer() +
  ggtitle("50\% curvewise intervals with curve_interval()") +
  theme_ggdist()
\dontshow{\}) # examplesIf}
}
\references{
Fraiman, Ricardo and Graciela Muniz. (2001).
"Trimmed means for functional data".
\emph{Test} 10: 419–440.
\doi{10.1007/BF02595706}.

Sun, Ying and Marc G. Genton. (2011).
"Functional Boxplots".
\emph{Journal of Computational and Graphical Statistics}, 20(2): 316-334.
\doi{10.1198/jcgs.2011.09224}

Mirzargar, Mahsa, Ross T Whitaker, and Robert M Kirby. (2014).
"Curve Boxplot: Generalization of Boxplot for Ensembles of Curves".
\emph{IEEE Transactions on Visualization and Computer Graphics}. 20(12): 2654-2663.
\doi{10.1109/TVCG.2014.2346455}

Juul Jonas, Kaare Græsbøll, Lasse Engbo Christiansen, and Sune Lehmann. (2020).
"Fixed-time descriptive statistics underestimate extremes of epidemic curve ensembles".
\emph{arXiv e-print}.
\href{https://arxiv.org/abs/2007.05035}{arXiv:2007.05035}
}
\seealso{
\code{\link[=point_interval]{point_interval()}} for pointwise intervals. See \code{vignette("lineribbon")} for more examples
and discussion of the differences between pointwise and curvewise intervals.
}
\author{
Matthew Kay
}
