% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/binning_methods.R
\name{bin_dots}
\alias{bin_dots}
\title{Bin data values using a dotplot algorithm}
\usage{
bin_dots(
  x,
  y,
  binwidth,
  heightratio = 1,
  stackratio = 1,
  layout = c("bin", "weave", "hex", "swarm", "bar"),
  side = c("topright", "top", "right", "bottomleft", "bottom", "left", "topleft",
    "bottomright", "both"),
  orientation = c("horizontal", "vertical", "y", "x"),
  overlaps = "nudge"
)
}
\arguments{
\item{x}{numeric vector of x values}

\item{y}{numeric vector of y values}

\item{binwidth}{bin width}

\item{heightratio}{ratio of bin width to dot height}

\item{stackratio}{ratio of dot height to vertical distance between dot
centers}

\item{layout}{The layout method used
for the dots: \itemize{
\item \code{"bin"} (default): places dots on the off-axis at the midpoint of their bins as in the classic Wilkinson dotplot.
This maintains the alignment of rows and columns in the dotplot. This layout is slightly different from the
classic Wilkinson algorithm in that: (1) it nudges bins slightly to avoid overlapping bins and (2) if
the input data are symmetrical it will return a symmetrical layout.
\item \code{"weave"}: uses the same basic binning approach of \code{"bin"}, but places dots in the off-axis at their actual
positions (unless \code{overlaps = "nudge"}, in which case overlaps may be nudged out of the way). This maintains
the alignment of rows but does not align dots within columns.
\item \code{"hex"}: uses the same basic binning approach of \code{"bin"}, but alternates placing dots \code{+ binwidth/4} or
\code{- binwidth/4} in the off-axis from the bin center. This allows hexagonal packing by setting a \code{stackratio}
less than 1 (something like \code{0.9} tends to work).
\item \code{"swarm"}: uses the \code{"compactswarm"} layout from \code{\link[beeswarm:beeswarm]{beeswarm::beeswarm()}}. Does not maintain alignment of rows or
columns, but can be more compact and neat looking, especially for sample data (as opposed to quantile
dotplots of theoretical distributions, which may look better with \code{"bin"}, \code{"weave"}, or \code{"hex"}).
\item \code{"bar"}: for discrete distributions, lays out duplicate values in rectangular bars.
}}

\item{side}{Which side to place the slab on. \code{"topright"}, \code{"top"}, and \code{"right"} are synonyms
which cause the slab to be drawn on the top or the right depending on if \code{orientation} is \code{"horizontal"}
or \code{"vertical"}. \code{"bottomleft"}, \code{"bottom"}, and \code{"left"} are synonyms which cause the slab
to be drawn on the bottom or the left depending on if \code{orientation} is \code{"horizontal"} or
\code{"vertical"}. \code{"topleft"} causes the slab to be drawn on the top or the left, and \code{"bottomright"}
causes the slab to be drawn on the bottom or the right. \code{"both"} draws the slab mirrored on both
sides (as in a violin plot).}

\item{orientation}{Whether the dots are laid out horizontally or vertically.
Follows the naming scheme of \code{\link[=geom_slabinterval]{geom_slabinterval()}}:
\itemize{
\item \code{"horizontal"} assumes the data values for the dotplot are in the \code{x}
variable and that dots will be stacked up in the \code{y} direction.
\item \code{"vertical"} assumes the data values for the dotplot are in the \code{y}
variable and that dots will be stacked up in the \code{x} direction.
}

For compatibility with the base ggplot naming scheme for \code{orientation},
\code{"x"} can be used as an alias for \code{"vertical"} and \code{"y"} as an alias for
\code{"horizontal"}.}

\item{overlaps}{How to handle overlapping dots or bins in the \code{"bin"},
\code{"weave"}, and \code{"hex"} layouts (dots never overlap in the \code{"swarm"} or \code{"bar"} layouts).
For the purposes of this argument, dots are only considered to be overlapping
if they would be overlapping when \code{dotsize = 1} and \code{stackratio = 1}; i.e.
if you set those arguments to other values, overlaps may still occur.
One of: \itemize{
\item \code{"keep"}: leave overlapping dots as they are. Dots may overlap
(usually only slightly) in the \code{"bin"}, \code{"weave"}, and \code{"hex"} layouts.
\item \code{"nudge"}: nudge overlapping dots out of the way. Overlaps are avoided
using a constrained optimization which minimizes the squared distance of
dots to their desired positions, subject to the constraint that adjacent
dots do not overlap.
}}
}
\value{
A \code{data.frame} with three columns:
\itemize{
\item \code{x}: the x position of each dot
\item \code{y}: the y position of each dot
\item \code{bin}: a unique number associated with each bin
(supplied but not used when \code{layout = "swarm"})
}
}
\description{
Bins the provided data values using one of several dotplot algorithms.
}
\examples{

library(dplyr)
library(ggplot2)

x = qnorm(ppoints(20))
bin_df = bin_dots(x = x, y = 0, binwidth = 0.5, heightratio = 1)
bin_df

# we can manually plot the binning above, though this is only recommended
# if you are using find_dotplot_binwidth() and bin_dots() to build your own
# grob. For practical use it is much easier to use geom_dots(), which will
# automatically select good bin widths for you (and which uses
# find_dotplot_binwidth() and bin_dots() internally)
bin_df \%>\%
  ggplot(aes(x = x, y = y)) +
  geom_point(size = 4) +
  coord_fixed()

}
\seealso{
\code{\link[=find_dotplot_binwidth]{find_dotplot_binwidth()}} for an algorithm that finds good bin widths
to use with this function; \code{\link[=geom_dotsinterval]{geom_dotsinterval()}} for geometries that use
these algorithms to create dotplots.
}
