% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/morphol.disparity.r
\name{morphol.disparity}
\alias{morphol.disparity}
\title{Morphological disparity for one or more groups of specimens}
\usage{
morphol.disparity(
  f1,
  groups = NULL,
  partial = FALSE,
  iter = 999,
  seed = NULL,
  data = NULL,
  print.progress = TRUE,
  ...
)
}
\arguments{
\item{f1}{A formula describing the linear model used.  The left-hand portion of the formula should be
a 3D array (p x k x n) containing Procrustes shape variables for a set of specimens, or a matrix (n x variables). 
The right-hand portion of the formula should be " ~1" to use the overall mean, or "~ x1 + x2 + x3 +...", where each x is a 
covariate or factor.  (Interactions and nested terms also work.)  Alternatively, one can use an object of class "procD.lm" or
"lm.rrpp", which already has a formula defined.  This is especially helpful for analyses performed with \code{\link{procD.pgls}},
as residuals from PGLS will be used for analysis (see examples).}

\item{groups}{Either a formula designating groups, e.g., groups = ~ groups, or a factor, or 
a vector coercible to factor.  If NULL and if f1 is a procD.lm or lm.rrpp model fit, morphol.disparity
will attempt to define groups based on the terms of the model.  If there are no groups inherently
indicated in f1 and groups is NULL, a single Procrustes variance will be returned for the entire data set.}

\item{partial}{A logical value to indicate whether partial disparities should be calculated, sensu
Foote (1993).  If TRUE, the model formula should have only an intercept (e.g., coords ~ 1); otherwise an error 
will be returned.}

\item{iter}{Number of iterations for permutation test}

\item{seed}{An optional argument for setting the seed for random permutations of the resampling procedure.
If left NULL (the default), the exact same P-values will be found for repeated runs of the analysis (with the same number of iterations).
If seed = "random", a random seed will be used, and P-values will vary.  One can also specify an integer for specific seed values,
which might be of interest for advanced users.}

\item{data}{A data frame for the function environment, see \code{\link{geomorph.data.frame}}}

\item{print.progress}{A logical value to indicate whether a progress bar should be printed to the screen.
This is helpful for long-running analyses.}

\item{...}{Arguments passed on to procD.lm (typically associated with the lm function,
such as weights or offset, plus SS.type).}
}
\value{
Objects of class "morphol.disparity" return a list with the following components 
(if groups are specified): 
  \item{Procrustes.var}{Observed Procrustes variances.}
  \item{PV.dist}{Observed pairwise absolute differences (distances) 
  among group Procrustes variances.}
  \item{PV.dist.Pval}{P-values associated with pairwise differences.}
  \item{random.PV.dist}{Pairwise distance matrices produced in the resampling procedure.}
  \item{permutations}{Number of random permutations in resampling procedure.}
  \item{partial}{Logical value to indicate if partial disparities were calculated.}
  \item{call}{The match call}
}
\description{
Function estimates morphological disparity and performs pairwise comparisons among groups
}
\details{
The function estimates morphological disparity and performs pairwise comparisons to identify differences
among groups. Morphological disparity is estimated as the Procrustes variance, overall or for groups, 
using residuals of a linear model fit.  Procrustes variance is the same sum of the diagonal elements 
of the group covariance matrix divided by the number of observations in the group (e.g., Zelditch et al. 2012).
The function takes as input a formula to describe the linear model fit,
plus a formulaic indication of groups (e.g., ~ groups).  It is assumed that the formula describes shape data that 
have been GPA-aligned [e.g., \code{\link{gpagen}}], although the function can work with any multivariate data.

Partial disparities (Foote 1993) can also be calculated, but only for model formulas containing only an intercept 
(e.g., coords ~ 1).  Partial disparity has the same numerator as Procrustes variance (with respect to an overall mean)
but the denominator is N - 1 for all N observations, rather than n, the group size.  (The sum of all group n equals N.)
Partial disparities have the appeal that the sum of group partial disparities it the total disparitiy.

Absolute differences in Procrustes variances are test statistics that can be used to test differences
in morphological disparity among groups.  These differences are  statistically evaluated through permutation, 
where the vectors of residuals are randomized among groups. The function can be used to obtain disparity for the whole
dataset by using "a dummy group factor "~ 1" as the right-hand portion of the formula, in which case only Procrustes 
variance is returned.  Additionally, if the right-hand portion of the formula only contains (continuous) covariates, 
e.g., "~ Csize", Procrustes variance will be calculated for the whole data set or groups, after accounting for the
linear regression described.  Finally, different factors can be indicated in the formula and for groups, if one wishes to 
compare morphological disparities for groups comprising only a portion of or collapsing of the groups in a more complex model 
(see examples).

This function can be used with an object of class "procD.lm" or "lm.rrpp", if such analyses have already been performed.  
This is specially useful for analyses performed with  \code{\link{procD.pgls}}.  In this case, residuals obtained from PGLS estimation
of coefficients, rather than OLS estimation, will be used in the analysis.  Thus, one can account for phylogeny when comparing
morphological disparity among groups.  However, one should be aware that this approach only adjusts expected values because of phylogeny
and does not assert a null hypothesis of equal variances based on phylogenetic relatedness.  (For example, species means can be adjusted
because of phylogenetic relatedness, but the null hypothesis of equal variances is conditioned on the estimation of means.) 

\subsection{Notes for geomorph 3.1.0 and subsequent versions}{ 
 The function \code{\link{pairwise}} in the \code{RRPP} package can also be used to evaluate morphological 
 disparity, and will yield results identical to those of the current function. A simple example is shown below
 }
}
\examples{
data(plethodon)
Y.gpa<-gpagen(plethodon$land, print.progress = FALSE)    #GPA-alignment
gdf <- geomorph.data.frame(Y.gpa, species = plethodon$species, 
site = plethodon$site)

# Morphological disparity for entire data set
morphol.disparity(coords ~ 1, groups = NULL, data = gdf, 
iter = 999, print.progress = FALSE)

# Morphological disparity for entire data set, accounting for allometry
morphol.disparity(coords ~ Csize, groups= NULL, data = gdf, 
iter = 999, print.progress = FALSE)

# Morphological disparity without covariates, using overall mean
morphol.disparity(coords ~ 1, groups= ~ species*site, data = gdf, 
iter = 999, print.progress = FALSE)

# Morphological partial disparities for overal mean
morphol.disparity(coords ~ 1, groups= ~ species*site, partial = TRUE, 
data = gdf, iter = 999, print.progress = FALSE)

# Morphological disparity without covariates, using group means
morphol.disparity(coords ~ species*site, groups= ~species*site, 
data = gdf, iter = 999, print.progress = FALSE)

# Morphological disparity of different groups than those 
# described by the linear model
morphol.disparity(coords ~ Csize + species*site, groups= ~ species, 
data = gdf, iter = 999, print.progress = FALSE)

# Extracting components
MD <- morphol.disparity(coords ~ Csize + species*site, groups= ~ species, 
data = gdf, iter = 999, print.progress = FALSE)
MD$Procrustes.var # just the Procrustes variances


### Morphol.disparity can be used with previously-defined 
### procD.lm or lm.rrpp class objects

data(plethspecies)
Y.gpa<-gpagen(plethspecies$land)    #GPA-alignment
gp.end<-factor(c(0,0,1,0,0,1,1,0,0))  #endangered species vs. rest
names(gp.end)<-plethspecies$phy$tip

gdf <- geomorph.data.frame(Y.gpa, phy = plethspecies$phy, gp.end = gp.end)

pleth.ols <- procD.lm(coords ~ Csize + gp.end, 
data = gdf, iter = 999) # ordinary least squares
pleth.pgls <- procD.pgls(coords ~ Csize + gp.end, phy = phy, 
data = gdf, iter = 999) # phylogenetic generalized least squares

summary(pleth.ols)
summary(pleth.pgls)

morphol.disparity(f1 = pleth.ols, groups = ~ gp.end, data = gdf, 
iter = 999, print.progress = FALSE)
morphol.disparity(f1 = pleth.pgls, groups = ~ gp.end, data = gdf, 
iter = 999, print.progress = FALSE)

 ### Morphol.disparity using RRPP
 PW <- pairwise(pleth.ols, groups = gp.end)
 summary(PW, test.type = 'var')

}
\references{
Zelditch, M. L., D. L. Swiderski, H. D. Sheets, and W. L. Fink. 2012. Geometric morphometrics 
  for biologists: a primer. 2nd edition. Elsevier/Academic Press, Amsterdam.

Foote, M. 1993. Contributions of individual taxa to overall morphological disparity. 
Paleobiology, 19: 403-419.
}
\author{
Emma Sherratt and Michael Collyer
}
\keyword{analysis}
