% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/pls.pathmox.R
\name{pls.pathmox}
\alias{pls.pathmox}
\title{Pathmox Segmentation Trees in Partial Least Squares Structural Equation 
Modeling (PLS-SEM)}
\usage{
pls.pathmox(
  .model,
  .data,
  .catvar,
  .scheme = "path",
  .signif = 0.05,
  .deep = 2,
  .size = 0.1,
  .size_candidate = 50,
  .tree = TRUE
)
}
\arguments{
\item{.model}{A description of the user-specified model. The model is described using 
the \href{https://lavaan.ugent.be/tutorial/syntax1.html}{lavaan syntax}. 
Structural and measurement models are defined by enclosure between double quotes. 
The directional link between constructs is defined using the ("~") operator. 
On the left-hand side of the operator is the dependent construct and on the 
right-hand side are the explanatory constructs, separated by the ("+") operator. As for 
the outer model, constructs are defined by listing their corresponding MVs after 
the operator (“=~”) if mode A is chosen for computing the outer weights, 
or the operator(“<~”) if mode B is chosen. On the left-hand side of the operator, 
 is the construct and on the right-hand side are the MVs separated by the ("+") operator. 
Variable labels cannot contain (".").}

\item{.data}{A matrix or dataframe containing the manifest variables}

\item{.catvar}{A single factor or set of factors organized as a dataframe containing 
the categorical variables used as sources of heterogeneity}

\item{.scheme}{A string indicating the type of inner weighting scheme. Possible 
values are \code{"centroid"}, \code{"factorial"}, or \code{"path"}. By default
\code{.scheme} is equal to \code{"path"}}

\item{.signif}{A decimal value indicating the minimum threshold significance 
for the partitioning algorithm. By default \code{.signif} is equal to 0.05}

\item{.deep}{An integer indicating the maximum tree depth. Must be an integer equal to or 
greater than 1. By default \code{.deep} is equal to 2}

\item{.size}{A decimal value indicating the minimum admissible size for a node. It is 
equal to 0.10 by default}

\item{.size_candidate}{An integer indicating the minimum admissible size for a candidate 
node. It is equal to \code{50} by default}

\item{.tree}{A string indicating if the tree plot must be shown. By default, it is equal to 
\code{TRUE}}
}
\value{
An object of class \code{"plstree"}. This is a list with the
following results:

\item{MOX}{dataframe containing the results of the segmentation tree}

\item{terminal_paths}{dataframe containing PLS-SEM model terminal nodes 
(path coeff. & R^2)}

\item{var_imp}{dataframe containing the categorical variable ranking by importance}

\item{Fg.r}{dataframe containing the results of the F-global test for each 
node partition}

\item{Fc.r}{list of dataframes containing the results of the F-coefficient test 
for each node}

\item{hybrid}{list of datasets associated with the terminal node prepared for MGA with 
cSEM R package}

\item{other}{Other parameters and results}
}
\description{
\code{pls.pathmox} calculates a binary segmentation tree in the context 
of partial least squares structural equation modeling (PLS-SEM) following the 
PATHMOX algorithm. It detects heterogeneity  
in PLS-SEM models when the segmentation variables (categorical 
variables), external to the model, are available and when the objective 
of the research is exploratory. \code{pls.pathmox} function returns the 
most significant different models associated with the terminal nodes of the 
tree. It also returns a ranking according to importance of the categorical 
variables used in the split process. \code{pls.pathmox} also enables 
the hybrid multigroup analysis (Lamberti, 2021) to be run. In fact, the function 
also returns an object, \code{hybrid}, containing the datasets associated 
with the terminal nodes prepared to be treated with the \code{csem} package 
(Rademaker, 2020). It thus enables the MICOM and MGA procedures to be run 
(Hair et al. 2017; Henseler, 2016; Henseler, 2009).
}
\details{
\code{pls.pathmox} uses the classical lavaan syntax to define inner and outer 
models and mode (A and B). The user can choose to perform the analysis for a
single categorical variable or a set of factors. In the latter case, categorical 
variables must be organized as a dataframe. Scheme is equal to \code{path} by 
default following other PLS-SEM softwares (SMART-PLS (Ringle at al., 2015), 
cSEM R package (Rademaker, 2020)), but can be modified by the user to
\code{"centroid"} or \code{"factorial"}. Stops parameters (\code{.signif} (threshold 
significance for the partitioning algorithm), \code{.deep} (maximum tree depth),
and \code{.size} (minimum admissible size for a node)) are fixed respectively to 
signif=0.05, depth=2, and size = 0.10 according to Lamberti et al. (2016). 
However, again the user can modify these parameters according to the specific 
objectives of the analysis. \code{pls.pathmox} also defines a minimum 
admissible size for a candidate node that restricts the search of the optimal split 
to that partitions that produce nodes with an adequate number of observations. 
This parameter (\code{.size_candidate}) is fixed to 50 observations by default. 
PLS-SEM model coefficients used in the split process are estimated by applying the 
classical PLS-SEM algorithm improved by Lohmöller (1989). \code{pls.pathmox} also 
returns an object containing the datasets associated with the terminal nodes 
(\code{object$hybrid}) prepared to be treated with the \code{csem} package (Rademaker, 2020). 
This means that, once the pathmox finalizes the search for the most significant different 
groups associated with the most significant different PLS-SEM models, the user can perform a 
detailed analysis for each group using the \code{csem} package, or can 
run the MICOM and MGA procedures (Hair et al. 2017; Henseler et al. 2016) on the terminal nodes.
}
\examples{
 \dontrun{
# Example of PATHMOX approach in customer satisfaction analysis 
# (Spanish financial company).
# Model with 5 LVs (4 reflective: Image (IMAG), Value (VAL), 
# Satisfaction (SAT), and Loyalty (LOY); and 1 formative construct: 
# Quality (QUAL))

# Load library and dataset csibank
library(genpathmx)
data("csibank")

# Define the model using the laavan syntax. Use a set of regression formulas that define 
# first the structural model and then the measurement model

CSImodel <- "
# Structural model
VAL  ~ QUAL
SAT  ~ IMAG  + QUAL + VAL
LOY  ~ IMAG + SAT

# Measurement model
# Formative
QUAL <~ qual1 + qual2 + qual3 + qual4 + qual5 + qual6 + qual7 
     
# Reflective
IMAG <~ imag1 + imag2 + imag3 + imag4 + imag5 + imag6 
VAL  <~ val1  + val2  + val3  + val4
SAT  =~ sat1  + sat2  + sat3           
LOY  =~ loy1  + loy2  + loy3          

"

# Identify the categorical variable to be used as input variables 
# in the split process
CSIcatvar = csibank[,1:5]

# Check if variables are well specified (they have to be factors 
# and/or ordered factors)
str(CSIcatvar)

# Transform Age and Education into ordered factors
CSIcatvar$Age = factor(CSIcatvar$Age, levels=c("<=25", 
                                     "26-35", "36-45", "46-55", 
                                     "56-65", ">=66"),ordered=T)

CSIcatvar$Education = factor(CSIcatvar$Education, 
                            levels=c("Unfinished","Elementary", "Highschool",
                            "Undergrad", "Graduated"),ordered=T)
       
# Run Pathmox analysis (Lamberti et al., 2016; 2017)
csi.pathmox = pls.pathmox(
 .model = CSImodel ,
 .data  = csibank,
 .catvar= CSIcatvar,
 .signif = 0.05,
 .deep=2
)                     
                      
# Visualize results by summary
# summary(csi.pathmox)

# Run pathmox on one single variable
age = csibank[,2]

#' # Transform Age into an ordered factor
age = factor(age, levels=c("<=25", 
                                     "26-35", "36-45", "46-55", 
                                     "56-65", ">=66"),ordered=T)
csi.pathmox.age = pls.pathmox(
 .model = CSImodel ,
 .data  = csibank,
 .catvar= age,
 .signif = 0.05,
 .deep=1
)  

# Run hybrid multigroup analysis (Lamberti, 2021) using 
# the cSEM package (Rademaker, 2020)
# Install and load cSEM library
# Install.packages("cSEM")
# library(cSEM)

# Run cSEM Model for Pathmox terminal nodes

csilocalmodel = csem(
 .model = CSImodel,
 .data = csi.pathmox.age$hybrid)

# Check invariance and run MGA analysis (Hair et al., 2019)

testMICOM(csilocalmodel, .R =60)

to_compare <- "
#' # Structural model
VAL  ~ QUAL
SAT  ~ IMAG  + QUAL + VAL
LOY  ~ IMAG + SAT
"

testMGD(csilocalmodel, .parameters_to_compare=to_compare, 
.R_bootstrap= 60,.approach_mgd ="Henseler")

}

}
\references{
Hair, J. F. et al. (2019). When to use and how to report the results of PLS-SEM.  
\emph{European business review}, doi: 10.1108/EBR-11-2018-0203.

Hair Jr, J. F., et al. (2017). \emph{Advanced issues in partial least squares structural 
equation modeling}. SAGE publications.

Henseler, J., et al. (2016). Testing measurement invariance of composites using 
partial least squares. International marketing review. doi: 10.1108/IMR-09-2014-0304

Lamberti, G. (2021). Hybrid multigroup partial least squares structural equation
 modelling: an application to bank employee satisfaction and loyalty. \emph{Quality and Quantity},
 doi: 10.1007/s11135-021-01096-9.

Lamberti, G. et al. (2017). The Pathmox approach for PLS path modeling: discovering 
which constructs differentiate segments.\emph{Applied Stochastic Models in Business and Industry}, 
doi: 10.1002/asmb.2270.

Lamberti, G. et al. (2016). The Pathmox approach for PLS path modeling segmentation. 
\emph{Applied Stochastic Models in Business and Industry}, doi: 10.1002/asmb.2168.

Lamberti, G. (2015). \emph{Modeling with Heterogeneity}, PhD Dissertation.

Lohmoller, J.-B. (1989). \emph{Latent Variable Path Modeling with Partial Least 
Squares}, Physica: Heidelberg.

Rademaker M. E. (2020). cSEM: Composite-Based Structural Equation Modeling.
Available at \url{https://CRAN.R-project.org/package=cSEM}

Ringle, C. M. et al. (2015). SmartPLS 3. Boenningstedt: SmartPLS. Retrieved 
from  \url{https://www.smartpls.com}
}
\seealso{
\code{\link{print.plstree}}, \code{\link{summary.plstree}},  
\code{\link{bar_terminal}}, \code{\link{bar_impvar}} and \code{\link{plot.plstree}}
}
\author{
Giuseppe Lamberti
}
