#' Return a count of records
#'
#' Prior to downloading data it is often valuable to have some estimate of how
#' many records are available, both for deciding if the query is feasible,
#' and for estimating how long it will take to download. Alternatively, for some kinds
#' of reporting, the count of observations may be all that is required, for example
#' for understanding how observations are growing or shrinking in particular
#' locations, or for particular taxa. To this end, `atlas_counts()` takes
#' arguments in the same format as [atlas_occurrences()], and
#' provides either a total count of records matching the criteria, or a
#' `data.frame` of counts matching the criteria supplied to the `group_by`
#' argument.
#'
#' @param request optional `data_rquest` object: generated by a call to
#' [galah_call()].
#' @param identify `data.frame`: generated by a call to
#' [galah_identify()].
#' @param filter `data.frame`: generated by a call to
#' [galah_filter()]
#' @param geolocate `string`: generated by a call to
#' [galah_geolocate()]
#' @param group_by `data.frame`: An object of class `galah_group_by`,
#' as returned by [galah_group_by()]. Alternatively a vector of field
#' names (see [search_fields()] and [show_all_fields()].
#' @param limit `numeric`: maximum number of categories to return, defaulting to 100.
#' If limit is NULL, all results are returned. For some categories this will
#' take a while.
#' @param type `string`: one of `c("record", "species")`. Defaults to
#' "record". If "species", the number of species matching the criteria will be
#' returned, if "record", the number of records matching the criteria will be
#' returned.
#' @param refresh_cache `logical`: if set to `TRUE` and 
#' `galah_config(caching = TRUE)` then files cached from a previous query will 
#' be replaced by the current query
#' @return
#' 
#' An object of class `tbl_df` and `data.frame` (aka a tibble) returning: 
#'  * A single number, if `group_by` is not specified or,
#'  * A summary of counts grouped by field(s), if `group_by` is specified
#'
#' @section Examples:
#' ```{r, child = "man/rmd/setup.Rmd"}
#' ```
#' 
#' With no arguments, return the total number of records in the ALA
#' 
#' ```{r, comment = "#>", collapse = TRUE}
#' atlas_counts()
#' ```
#'
#' You can group counts by state and territory with `galah_group_by`
#' 
#' ```{r, comment = "#>", collapse = TRUE}
#' galah_call() |>
#'   galah_group_by(stateProvince) |>
#'   atlas_counts()
#' ```
#'
#' You can add a filter to narrow your search
#' 
#' ```{r, comment = "#>", collapse = TRUE}
#' galah_call() |>
#'   galah_filter(basisOfRecord == "FossilSpecimen")
#'   atlas_counts() 
#' ```
#' 
#' Specify `type = species` to count the number of species, and group record
#' counts by kingdom
#' 
#' ```{r, comment = "#>", collapse = TRUE, results = "hide"}
#' records <- galah_call() |>
#'   galah_group_by(kingdom) |>
#'   atlas_counts(type = "species")
#' ```
#' ```{r, comment = "#>", collapse = TRUE}
#' records
#' ```
#' 
#' Using `galah_group_by` allows you to cross-tabulate using two different 
#' variables, similar to using `dplyr::group_by() %>% dplyr::count()`
#' 
#' ```{r, comment = "#>", collapse = TRUE, results = "hide"}
#' records <- galah_call() |>
#'   galah_filter(year > 2015) |>
#'   galah_group_by(year, basisOfRecord) |>
#'   atlas_counts()
#' ```{r, comment = "#>", collapse = TRUE}
#' records
#' ```
#' 
#' @export
atlas_counts <- function(request = NULL, 
                         identify = NULL, 
                         filter = NULL, 
                         geolocate = NULL,
                         group_by = NULL, 
                         limit = 100,
                         type = c("record" ,"species"),
                         refresh_cache = FALSE
                         ) {
                           
  type <- match.arg(type)

  if(!is.null(request)){
    check_data_request(request)
    current_call <- update_galah_call(request, 
      identify = identify,
      filter = filter,
      geolocate = geolocate,
      group_by = group_by,
      limit = limit,
      type = type,
      refresh_cache = refresh_cache
    ) 

  }else{
    current_call <- galah_call(
      identify = identify,
      filter = filter,
      geolocate = geolocate,
      group_by = group_by,
      limit = limit,
      type = type,
      refresh_cache = refresh_cache
    )
  }

  # subset to available arguments
  custom_call <- current_call[
     names(current_call) %in% names(formals(atlas_counts_internal))]
  class(custom_call) <- "data_request"

  # check for caching
  caching <- getOption("galah_config")$caching
  cache_file <- cache_filename("counts", unlist(custom_call))
  if (caching && file.exists(cache_file) && !refresh_cache) {
    return(read_cache_file(cache_file))
  }
        
  # call using do.call
  result <- do.call(atlas_counts_internal, custom_call)
  attr(result, "data_type") <- "counts"
  attr(result, "data_request") <- custom_call

  # if caching requested, save
  if (caching) {
   write_cache_file(object = result, 
                    data_type = "counts",
                    cache_file = cache_file)
  }

  result                             
}


atlas_counts_internal <- function(identify = NULL, 
                                  filter = NULL, 
                                  geolocate = NULL,
                                  group_by = NULL, 
                                  limit = 100,
                                  type = "record",
                                  refresh_cache = FALSE
                                  ) {

  verbose <- getOption("galah_config")$verbose

  # if `group_by` is as a vector
  if(!is.null(group_by)){
    if(!inherits(group_by, "galah_group_by")){
      group_by <- galah_group_by(group_by)
    }
    if(nrow(group_by) < 1){
      group_by <- NULL
    }
  }
    
  # set options if group_by = NULL
  if(is.null(group_by)) {
    query <- list()
    profile <- extract_profile(filter)
    query <- build_query(identify, filter, geolocate, profile = profile)
    if (type == "species") {
      result <- species_count(query)
    }else{
      result <- record_count(query)
    }
    if(is.null(result)){
      bullets <- c(
        "Calling the API failed for `atlas_counts`.",
        i = "This might mean that the ALA system is down. Double check that your query is correct.",
        i = "If you continue to see this message, please email support@ala.org.au."
      )
      inform(bullets)
    }
    return(tibble(count = result))
  }                  
  
  # if all combinations of levels of `group_by` are needed (expand = TRUE)
  if(attr(group_by, "expand") & nrow(group_by) > 1){ 
    
    # get counts given the filter provided by the user
    field_values_df <- atlas_counts_lookup(
      identify = identify,
      filter = filter, 
      geolocate = geolocate,
      type = type,
      facets = group_by$name, 
      limit = NULL)
    n_fields_df <- data.frame(
      facets = group_by$name,
      n_fields = unlist(lapply(
        group_by$name, 
        function(a){length(which(!is.na(field_values_df[[a]])))})))

    if(sum(field_values_df$count) < 1){
      return(tibble(count = 0))
    }

    # work out which to pass as facets vs those we iterate over with lapply
    facets_large <- n_fields_df$facets[which.max(n_fields_df$n_fields)]
    facets_small <- n_fields_df$facets[n_fields_df$facets != facets_large]

    # work out what combinations of `group`s should be sent to atlas_counts_lookup
    levels_df <- expand.grid(
      lapply(
        field_values_df[, 
          which(names(field_values_df) %in% facets_small), 
          drop = FALSE], 
        function(a){a[!is.na(a)]}),
      stringsAsFactors = FALSE)
    levels_list <- split(levels_df, seq_len(nrow(levels_df)))
    filter_list <- lapply(levels_list, function(a){
      field <- colnames(a)
      value <- paste0("\'", a, "\'")
      paste(
        paste(field, value, sep = " == "),
        collapse = " & ")
    })
    
    # run `atlas_counts_lookup` the requisite number of times
    if (verbose) { pb <- txtProgressBar(max = 1, style = 3) } # start progressbar
    
    result_list <- lapply(seq_along(levels_list),
      function(a){
        if (verbose) {
          val <- (a / length(levels_list))
          setTxtProgressBar(pb, val)
        }
        filter_this_loop <- galah_filter(filter_list[[a]])    
        filter_final <- rbind(filter, filter_this_loop)
        counts_query <- atlas_counts_lookup(
          identify = identify,
          filter = filter_final,
          geolocate = geolocate,
          facets = n_fields_df$facets[which.max(n_fields_df$n_fields)],
          limit = limit,
          type = type)
        if(nrow(counts_query) > 0){   
          as.data.frame(list(levels_list[[a]], counts_query), row.names = NULL)
        }
      }) 
    if(verbose){close(pb)} # close progress bar
    if (all(unlist(lapply(result_list, is.null)))) {
      bullets <- c(
        "Calling the API failed for `atlas_counts`.",
        i = "This might mean that the ALA system is down. Double check that your query is correct.",
        i = "If you continue to see this message, please email support@ala.org.au."
      )
      inform(bullets)
      return(tibble())
    } else {
      return(as_tibble(rbindlist(result_list, fill = TRUE)))
    } 
     
  # if `groups` is of nrow == 1 (expand = FALSE)
  }else{
    result <- atlas_counts_lookup(
      identify, filter, geolocate, 
      facets = group_by$name, 
      limit, type, refresh_cache,
      verbose = verbose)
    if(is.null(result)){
      bullets <- c(
        "Calling the API failed for `atlas_counts`.",
        i = "This might mean that the ALA system is down. Double check that your query is correct.",
        i = "If you continue to see this message, please email support@ala.org.au."
      )
      inform(bullets)
      return(tibble())
    }else{
      result
    }
  } 
}

# workhorse function to do most of the actual processing
## NOTE: need to turn off caching for multiple runs
atlas_counts_lookup <- function(identify = NULL, 
                                  filter = NULL, 
                                  geolocate = NULL,
                                  facets, # NOTE: not `groups` as no multiply section here
                                  limit = NULL, 
                                  type = "record",
                                  refresh_cache = FALSE,
                                  verbose = FALSE # NOTE: internally `verbose` is manual, not from galah_config
                                  ) {
  
  page_size <- 100
  query <- list()
  profile <- extract_profile(filter)
  query <- build_query(identify, filter, geolocate, profile = profile)
  
  # add facets in a way that supports single or multiple queries 
  facets_temp <- as.list(facets)
  names(facets_temp) <- rep("facets", length(facets_temp))
  query <- c(query, facets_temp)

  # build url etc
  url <- server_config("records_base_url")
  path <- "occurrence/facets"

  total_cats <- total_categories(url, path, query)
  if(is.null(total_cats)) {
    return(NULL)
  }
  if(sum(total_cats) < 1){
    return(tibble(count = 0))
  }

  if (is.null(limit)) {
    limit <- sum(total_cats)
  }

  if (sum(total_cats) > limit && sum(total_cats) > page_size) {
    resp <- atlas_GET(url, path, params = query, paginate = TRUE, limit = limit,
                    page_size = page_size, offset_param = "foffset")
    if(is.null(resp)){return(NULL)}
    counts <- rbindlist(lapply(resp, function(a) {
      data.frame(jsonlite::fromJSON(a)$fieldResult)
      }))
  } else {
      query$flimit <- max(limit)
      resp <- atlas_GET(url, path, params = query)
      if(is.null(resp)){return(NULL)}
      counts <- rbindlist(resp$fieldResult)
  }

  if (sum(total_cats) > limit & galah_config()$verbose) {
    bullets <- c(
      glue("This field has {total_cats} values. {limit} will be returned."),
      i = "Increase `limit` to return more values, or decrease `limit` to return fewer."
    )
    inform(bullets) 
  }
  
  # parse out value
  value <- parse_fq(counts$fq)
  
  if (type == "species") {
    # this can take a while so add progress bar
    if (verbose) { pb <- txtProgressBar(max = 1, style = 3) }
    counts_final <- rbindlist(lapply(seq_along(value), function(x) {
      if (verbose) {
        val <- (x / length(value))
        setTxtProgressBar(pb, val)
      }
      species_query <- list()
      species_query$fq <- c(query$fq,
         # query_term(name = facets, value = value[[x]], include = TRUE))
         counts$fq[[x]])
      if(!is.null(geolocate)){
        species_query$wkt <- query$wkt
      }
      count <- species_count(species_query)
      if(is.null(count)){count <- NA}
      data.frame(name = value[[x]], count = count) |> as_tibble()
    }))
  } else {
    counts_final <- data.frame(
      name = value,
      count = counts$count)
  }
  
  if(length(facets) > 1){
    counts_final$field_name <- parse_field(counts$fq)
    counts_list <- split(counts_final, counts_final$field_name)
    counts_final <- as.data.frame(rbindlist(lapply(
      seq_along(facets), function(a){
        names(counts_list[[a]])[1] <- names(counts_list)[a]
        counts_list[[a]]
      }), 
      fill = TRUE))
     counts_final <- counts_final[, c(names(counts_list), "count")]
  }else{ # i.e. only one facet
    names(counts_final) <- c(facets, "count")
  }  
  
  as_tibble(counts_final)
}

# get just the record count for a query
# handle too long queries in here?
record_count <- function(query) {
  query$pageSize <- 0
  url <- server_config("records_base_url")
  resp <- atlas_GET(url, "occurrences/search", query)
  resp$totalRecords
}

species_count <- function(query) {
  query$flimit <- 1
  query$facets <- "speciesID"
  url <- server_config("records_base_url")
  total_categories(url, "occurrence/facets", query)
}

# Get number of categories of a filter
total_categories <- function(url, path, query) {
  query$flimit <- 1
  resp <- atlas_GET(url, path, params = query)
  if(is.null(resp)){
    NULL
  }else if(length(resp) < 1){
    0
  }else{
    resp$count
  }
}

# # Extract filter names and values returned from API
parse_field <- function(fq){
  str_extract(fq, "^[:alnum:]+")
}

parse_fq <- function(fq){
  gsub("\"", "", sub("^[[:alnum:]]+:",  "", fq))
}
