% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/notrend_test.R
\name{notrend_test}
\alias{notrend_test}
\title{Sieve Bootstrap Based Test for the Null Hypothesis of no Trend}
\usage{
notrend_test(
  x,
  B = 1000,
  test = c("t", "MK", "WAVK"),
  ar.method = "HVK",
  ar.order = NULL,
  ic = "BIC",
  factor.length = c("user.defined", "adaptive.selection"),
  Window = NULL,
  q = 3/4,
  j = c(8:11)
)
}
\arguments{
\item{x}{a vector containing a univariate time series. Missing values are not allowed.}

\item{B}{number of bootstrap simulations to obtain empirical critical values.
Default is 1000.}

\item{test}{trend test to implement: Student's t-test (\code{"t"}, default),
Mann--Kendall test (\code{"MK"}), or
WAVK test (\code{"WAVK"}, see \code{\link{WAVK}}).}

\item{ar.method}{method of estimating autoregression coefficients.
Default \code{"HVK"} delivers robust difference-based estimates by
\insertCite{Hall_VanKeilegom_2003;textual}{funtimes}. Alternatively,
options of \command{ar} function can be used, such as \code{"burg"},
\code{"ols"}, \code{"mle"}, and \code{"yw"}.}

\item{ar.order}{order of the autoregressive model when \code{ic = "none"}, or
the maximal order for IC-based filtering. Default is
\code{round(10*log10(length(x)))}, where \code{x} is the time series.}

\item{ic}{information criterion used to select the order of autoregressive filter (AIC of BIC),
considering models of orders \eqn{p=} 0,1,...,\code{ar.order}.
If \code{ic = "none"}, the AR(\eqn{p}) model with \eqn{p=} \code{ar.order} is used,
without order selection.}

\item{factor.length}{method to define the length of local windows (factors).
Used only if \code{test = "WAVK"}. Default option \code{"user.defined"} allows
to set only one value of the argument \code{Window}. The option
\code{"adaptive.selection"} sets \code{method = "boot"} and employs
heuristic \eqn{m}-out-of-\eqn{n} subsampling algorithm
\insertCite{Bickel_Sakov_2008}{funtimes} to select an optimal window from the set
of possible windows \code{length(x)*q^j} whose values are mapped to the largest
previous integer and greater than 2. Vector \code{x} is the time series tested.}

\item{Window}{length of the local window (factor), default is
\code{round(0.1*length(x))}. Used only if \code{test = "WAVK"}.
This argument is ignored if\cr \code{factor.length = "adaptive.selection"}.}

\item{q}{scalar from 0 to 1 to define the set of possible windows when
\code{factor.length =} \code{"adaptive.selection"}.
Used only if \code{test = "WAVK"}. Default is \eqn{3/4}.
This argument is ignored if \code{factor.length =} \code{"user.defined"}.}

\item{j}{numeric vector to define the set of possible windows when
\code{factor.length =} \code{"adaptive.selection"}.
Used only if \code{test = "WAVK"}. Default is \code{c(8:11)}.
This argument is ignored if \code{factor.length =} \code{"user.defined"}.}
}
\value{
A list with class \code{"htest"} containing the following components:
\item{method}{name of the method.}
\item{data.name}{name of the data.}
\item{statistic}{value of the test statistic.}
\item{p.value}{\eqn{p}-value of the test.}
\item{alternative}{alternative hypothesis.}
\item{estimate}{list with the following elements: employed AR order and estimated
AR coefficients.}
\item{parameter}{window that was used in WAVK test, included in the output only
if \code{test = "WAVK"}.}
}
\description{
A combination of time series trend tests for testing the null hypothesis of no trend,
versus the alternative hypothesis of a linear trend (Student's t-test),
or monotonic trend (Mann--Kendall test), or more general,
possibly non-monotonic trend (WAVK test).
}
\details{
This function tests the null hypothesis of no trend
versus different alternatives.
To set some other shape of trend as the null hypothesis, use \code{\link{wavk_test}}.
Note that \code{\link{wavk_test}} employs hybrid bootstrap, which is an alternative
to the sieve bootstrap employed by the current function.
}
\examples{
\dontrun{
# Fix seed for reproducible simulations:
set.seed(1)

#Simulate autoregressive time series of length n with smooth linear trend:
n <- 200
tsTrend <- 1 + 2*(1:n/n)
tsNoise <- arima.sim(n = n, list(order = c(2, 0, 0), ar = c(0.5, -0.1)))
U <- tsTrend + tsNoise
plot.ts(U)

#Use t-test
notrend_test(U)

#Use Mann--Kendall test and Yule-Walker estimates of the AR parameters
notrend_test(U, test = "MK", ar.method = "yw")

#Use WAVK test for the H0 of no trend, with m-out-of-n selection of the local window:
notrend_test(U, test = "WAVK", factor.length = "adaptive.selection")
# Sample output:
##	Sieve-bootstrap WAVK trend test
##
##data:  U
##WAVK test statistic = 21.654, moving window = 15, p-value < 2.2e-16
##alternative hypothesis: (non-)monotonic trend.
##sample estimates:
##$AR_order
##[1] 1
##
##$AR_coefficients
##    phi_1
##0.4041848
}

}
\references{
\insertAllCited{}
}
\seealso{
\code{\link[stats]{ar}}, \code{\link{HVK}}, \code{\link{WAVK}},
\code{\link{wavk_test}}, \code{vignette("trendtests", package = "funtimes")}
}
\author{
Vyacheslav Lyubchich
}
\keyword{htest}
\keyword{trend}
\keyword{ts}
