% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ibdEstim.R
\name{ibdEstim}
\alias{ibdEstim}
\title{Pairwise relatedness estimation}
\usage{
ibdEstim(
  x,
  ids = typedMembers(x),
  param = c("kappa", "delta"),
  markers = NULL,
  start = NULL,
  reltol = 1e-12,
  returnArgs = FALSE,
  ...
)
}
\arguments{
\item{x}{A \code{ped} object or a list of such.}

\item{ids}{Either a vector with ID labels, or a data frame/matrix with two
columns, where each row contains the ID labels of two individuals. The
entries are coerced to characters, and must match uniquely against the ID
labels of \code{x}. By default, all pairs of members of \code{x} are included.}

\item{param}{Either "kappa" (default) or "delta"; indicating which set of coefficients should be estimated.}

\item{markers}{A vector with names or indices of markers attached to x,
indicating which markers to include. If NULL (default), all markers are
used.}

\item{start}{Numeric of length 3 (if \code{param = "kappa"}) or 9 (if \code{param = "delta"}), indicating an initial value of for the optimisation. This must
be in the interior of the probability simplex, i.e., all elements must be
strictly positive and sum to 1. (The resulting estimate can still be on the border.)}

\item{reltol}{Relative convergence tolerance; passed on to \code{constrOptim()}.
Defaults to 1e-12,}

\item{returnArgs}{A logical; for debugging purposes.}

\item{...}{Further arguments.}
}
\value{
If \code{param = "kappa"}: A data frame with 6 columns: \code{id1}, \code{id2}, \code{N}
(the number of markers with no missing alleles), \code{k0}, \code{k1} and \code{k2}.

If \code{param = "delta"}: A data frame with 12 columns: \code{id1}, \code{id2}, \code{N} (the
number of markers with no missing alleles), \code{d1}, \code{d2}, ... \code{d9}.
}
\description{
Estimate the IBD coefficients \eqn{\kappa = (\kappa_0, \kappa_1,
\kappa_2)}{(k0, k1, k2)} or the condensed identity coefficients \eqn{\Delta =
(\Delta_1, ..., \Delta_9)}{(d1, ..., d9)} between a pair (or several pairs)
of pedigree members, using maximum likelihood methods.
}
\details{
It should be noted that this procedure estimates the \emph{realised} identity
coefficients of each pair, i.e., the actual fractions of the autosomes in
each IBD state. These may deviate substantially from the theoretical pedigree
coefficients.

Maximum likelihood estimation of relatedness coefficients originates with
Thompson (1975). Optimisation of \eqn{\kappa} is done in the \eqn{(\kappa_0,
\kappa_2)}{(k0, k2)}-plane and restricted to the triangle defined by
\deqn{\kappa_0 \ge 0, \kappa_2 \ge 0, \kappa_0 + \kappa_2 \le 1}{k0 >= 0, k2
>= 0, k0 + k2 <= 1}. Optimisation of \eqn{\Delta} is done in unit simplex of
R^8, using the first 8 coefficients.

The implementation uses \code{\link[=constrOptim]{constrOptim()}}, with the "BFGS" method.
}
\examples{

### Example 1: Siblings
x = nuclearPed(2)

# Simulate 100 markers
x = markerSim(x, N = 100, alleles = 1:4, seed = 123, verbose = FALSE)

# Estimate Delta (expectation = (0,0,0,0,0,0,1/4,1/2,1/4))
ibdEstim(x, ids = 3:4)


### Example 2: Full sib mating
y = fullSibMating(1)

# Simulate 200 SNP markers
y = markerSim(y, N = 1000, alleles = 1:10, seed = 123, verbose = FALSE)

# Estimate
ibdEstim(y, ids = 5:6, param = "delta")


}
\references{
\itemize{
\item E. A. Thompson (1975). \emph{The estimation of pairwise relationships.} Annals
of Human Genetics 39.
\item E. A. Thompson (2000). \emph{Statistical Inference from Genetic Data on
Pedigrees.} NSF-CBMS Regional Conference Series in Probability and
Statistics. Volume 6.
}
}
\seealso{
\code{\link[=constrOptim]{constrOptim()}}
}
\author{
Magnus Dehli Vigeland
}
