// adapted from <T>LAPACK implementation
//
// https://github.com/tlapack/tlapack
// https://github.com/tlapack/tlapack/blob/master/include/tlapack/lapack/lahqr.hpp

use crate::{
    linalg::{
        evd::hessenberg::{make_hessenberg_in_place, make_hessenberg_in_place_req},
        householder::{
            apply_block_householder_sequence_on_the_right_in_place_req,
            apply_block_householder_sequence_on_the_right_in_place_with_conj,
            apply_block_householder_sequence_transpose_on_the_left_in_place_req,
            apply_block_householder_sequence_transpose_on_the_left_in_place_with_conj,
            make_householder_in_place,
        },
        matmul::matmul,
        temp_mat_req,
        zip::Diag,
    },
    unzipped, zipped, ColMut, Conj, MatMut, MatRef, Parallelism,
};
use core::slice;
use dyn_stack::{PodStack, SizeOverflow, StackReq};
use faer_entity::*;
use reborrow::*;

fn max<T: PartialOrd>(a: T, b: T) -> T {
    if a > b {
        a
    } else {
        b
    }
}

fn min<T: PartialOrd>(a: T, b: T) -> T {
    if a < b {
        a
    } else {
        b
    }
}

fn abs1<E: ComplexField>(a: E) -> E::Real {
    a.faer_real().faer_abs().faer_add(a.faer_imag().faer_abs())
}

fn lahqr_shiftcolumn<E: ComplexField>(h: MatRef<'_, E>, mut v: ColMut<'_, E>, s1: E, s2: E) {
    let n = h.nrows();

    if n == 2 {
        let s = abs1(h.read(0, 0).faer_sub(s2)).faer_add(abs1(h.read(1, 0)));
        if s == E::Real::faer_zero() {
            v.write(0, E::faer_zero());
            v.write(1, E::faer_zero());
        } else {
            let s_inv = s.faer_inv();
            let h10s = h.read(1, 0).faer_scale_real(s_inv);
            v.write(
                0,
                (h10s.faer_mul(h.read(0, 1))).faer_add(
                    h.read(0, 0)
                        .faer_sub(s1)
                        .faer_mul((h.read(0, 0).faer_sub(s2)).faer_scale_real(s_inv)),
                ),
            );
            v.write(
                1,
                h10s.faer_mul(
                    h.read(0, 0)
                        .faer_add(h.read(1, 1))
                        .faer_sub(s1)
                        .faer_sub(s2),
                ),
            );
        }
    } else {
        let s = abs1(h.read(0, 0).faer_sub(s2))
            .faer_add(abs1(h.read(1, 0)))
            .faer_add(abs1(h.read(2, 0)));
        if s == E::Real::faer_zero() {
            v.write(0, E::faer_zero());
            v.write(1, E::faer_zero());
            v.write(2, E::faer_zero());
        } else {
            let s_inv = s.faer_inv();
            let h10s = h.read(1, 0).faer_scale_real(s_inv);
            let h20s = h.read(2, 0).faer_scale_real(s_inv);
            v.write(
                0,
                ((h.read(0, 0).faer_sub(s1))
                    .faer_mul((h.read(0, 0).faer_sub(s2)).faer_scale_real(s_inv)))
                .faer_add(h.read(0, 1).faer_mul(h10s))
                .faer_add(h.read(0, 2).faer_mul(h20s)),
            );
            v.write(
                1,
                (h10s.faer_mul(
                    h.read(0, 0)
                        .faer_add(h.read(1, 1).faer_sub(s1).faer_sub(s2)),
                ))
                .faer_add(h.read(1, 2).faer_mul(h20s)),
            );
            v.write(
                2,
                (h20s.faer_mul(
                    h.read(0, 0)
                        .faer_add(h.read(2, 2).faer_sub(s1).faer_sub(s2)),
                ))
                .faer_add(h10s.faer_mul(h.read(2, 1))),
            );
        }
    }
}

// ret: (eig1_re eig1_im) (eig2_re eig2_im)
fn lahqr_eig22<E: ComplexField>(mut a00: E, mut a01: E, mut a10: E, mut a11: E) -> (E, E) {
    let zero = E::Real::faer_zero();
    let half = E::Real::faer_from_f64(0.5);

    let s = abs1(a00)
        .faer_add(abs1(a01))
        .faer_add(abs1(a10))
        .faer_add(abs1(a11));
    if s == zero {
        return (E::faer_zero(), E::faer_zero());
    }

    let s_inv = s.faer_inv();

    a00 = a00.faer_scale_real(s_inv);
    a01 = a01.faer_scale_real(s_inv);
    a10 = a10.faer_scale_real(s_inv);
    a11 = a11.faer_scale_real(s_inv);

    let tr = (a00.faer_add(a11)).faer_scale_power_of_two(half);
    let det = ((a00.faer_sub(tr)).faer_mul(a00.faer_sub(tr))).faer_add(a01.faer_mul(a10));

    let rtdisc = det.faer_sqrt();
    (
        (tr.faer_add(rtdisc)).faer_scale_real(s),
        (tr.faer_sub(rtdisc)).faer_scale_real(s),
    )
}

fn rotg<E: ComplexField>(a: E, b: E, epsilon: E::Real, zero_threshold: E::Real) -> (E::Real, E, E) {
    let safmin = zero_threshold;
    let safmax = zero_threshold.faer_inv();
    let rtmin = zero_threshold.faer_div(epsilon).faer_sqrt();
    let rtmax = rtmin.faer_inv();

    // quick return
    if b == E::faer_zero() {
        return (E::Real::faer_one(), E::faer_zero(), E::faer_one());
    }

    let (c, s, r);
    if a == E::faer_zero() {
        c = E::Real::faer_zero();
        let g1 = max(b.faer_real().faer_abs(), b.faer_imag().faer_abs());
        if g1 > rtmin && g1 < rtmax {
            // Use unscaled algorithm
            let g2 = b
                .faer_real()
                .faer_abs2()
                .faer_add(b.faer_imag().faer_abs2());
            let d = g2.faer_sqrt();
            s = b.faer_conj().faer_scale_real(d.faer_inv());
            r = E::faer_from_real(d);
        } else {
            // Use scaled algorithm
            let u = min(safmax, max(safmin, g1));
            let uu = u.faer_inv();
            let gs = b.faer_scale_real(uu);
            let g2 = gs
                .faer_real()
                .faer_abs2()
                .faer_add(gs.faer_imag().faer_abs2());
            let d = g2.faer_sqrt();
            s = gs.faer_conj().faer_scale_real(d.faer_inv());
            r = E::faer_from_real(d.faer_mul(u));
        }
    } else {
        let f1 = max(E::faer_real(a).faer_abs(), E::faer_imag(a).faer_abs());
        let g1 = max(E::faer_real(b).faer_abs(), E::faer_imag(b).faer_abs());
        if f1 > rtmin && f1 < rtmax && g1 > rtmin && g1 < rtmax {
            // Use unscaled algorithm
            let f2 = a
                .faer_real()
                .faer_abs2()
                .faer_add(a.faer_imag().faer_abs2());
            let g2 = b
                .faer_real()
                .faer_abs2()
                .faer_add(b.faer_imag().faer_abs2());
            let h2 = f2.faer_add(g2);
            let d = if f2 > rtmin && h2 < rtmax {
                f2.faer_mul(h2).faer_sqrt()
            } else {
                f2.faer_sqrt().faer_mul(h2.faer_sqrt())
            };
            let p = d.faer_inv();
            c = f2.faer_scale_real(p);
            s = b.faer_conj().faer_mul(a.faer_scale_real(p));

            r = a.faer_scale_real(h2.faer_mul(p));
        } else {
            // Use scaled algorithm
            let u = min(safmax, max(safmin, max(f1, g1)));
            let uu = u.faer_inv();
            let gs = b.faer_scale_real(uu);
            let g2 = gs
                .faer_real()
                .faer_abs2()
                .faer_add(gs.faer_imag().faer_abs2());
            let (f2, h2, w);
            let fs;
            if f1.faer_scale_real(uu) < rtmin {
                // a is not well-scaled when scaled by g1.
                let v = min(safmax, max(safmin, f1));
                let vv = v.faer_inv();
                w = v.faer_mul(uu);
                fs = a.faer_scale_real(vv);
                f2 = fs
                    .faer_real()
                    .faer_abs2()
                    .faer_add(fs.faer_imag().faer_abs2());
                h2 = (f2.faer_mul(w).faer_mul(w)).faer_add(g2);
            } else {
                // Otherwise use the same scaling for a and b.
                w = E::Real::faer_one();
                fs = a.faer_scale_real(uu);
                f2 = fs
                    .faer_real()
                    .faer_abs2()
                    .faer_add(fs.faer_imag().faer_abs2());
                h2 = f2.faer_add(g2);
            }
            let d = if f2 > rtmin && h2 < rtmax {
                f2.faer_mul(h2).faer_sqrt()
            } else {
                f2.faer_sqrt().faer_mul(h2.faer_sqrt())
            };
            let p = d.faer_inv();
            c = (f2.faer_mul(p)).faer_mul(w);
            s = gs.faer_conj().faer_mul(fs.faer_scale_real(p));
            r = (fs.faer_scale_real(h2.faer_scale_real(p))).faer_scale_real(u);
        }
    }

    (c, s, r)
}

pub fn rot<E: ComplexField>(x: ColMut<'_, E>, y: ColMut<'_, E>, c: E::Real, s: E) {
    let n = x.nrows();
    if n == 0 || (c == E::Real::faer_one() && s == E::faer_zero()) {
        return;
    }

    zipped!(x, y).for_each(|unzipped!(mut x, mut y)| {
        let mut x_ = x.read();
        let mut y_ = y.read();

        (x_, y_) = (
            (x_.faer_scale_real(c)).faer_add(y_.faer_mul(s)),
            (y_.faer_scale_real(c)).faer_sub(x_.faer_mul(s.faer_conj())),
        );

        x.write(x_);
        y.write(y_);
    });
}

pub struct Rot<'a, E: ComplexField> {
    ai: ColMut<'a, E>,
    aj: ColMut<'a, E>,
    c: E::Real,
    s: E,
}

impl<E: ComplexField> pulp::WithSimd for Rot<'_, E> {
    type Output = ();

    #[inline(always)]
    fn with_simd<S: pulp::Simd>(self, simd: S) -> Self::Output {
        let Self { ai, aj, c, s } = self;
        debug_assert!(ai.nrows() == aj.nrows());

        let n = ai.nrows();
        let ai = ai.as_ptr_mut();
        let aj = aj.as_ptr_mut();

        let ai = unsafe {
            E::faer_map(
                ai,
                #[inline(always)]
                |ptr| slice::from_raw_parts_mut(ptr, n),
            )
        };
        let aj = unsafe {
            E::faer_map(
                aj,
                #[inline(always)]
                |ptr| slice::from_raw_parts_mut(ptr, n),
            )
        };

        let (ai_head, ai_tail) = faer_entity::slice_as_mut_simd::<E, S>(ai);
        let (aj_head, aj_tail) = faer_entity::slice_as_mut_simd::<E, S>(aj);

        let c = E::Real::faer_simd_splat(simd, c);
        let s = E::faer_simd_splat(simd, s);

        for (ai, aj) in E::faer_into_iter(ai_head).zip(E::faer_into_iter(aj_head)) {
            let mut ai_ = E::faer_deref(E::faer_rb(E::faer_as_ref(&ai)));
            let mut aj_ = E::faer_deref(E::faer_rb(E::faer_as_ref(&aj)));

            let tmp = E::faer_simd_conj_mul_adde(
                simd,
                s,
                into_copy::<E, _>(E::faer_copy(&aj_)),
                E::faer_simd_scale_real(simd, c, into_copy::<E, _>(E::faer_copy(&ai_))),
            );

            aj_ = from_copy::<E, _>(E::faer_simd_mul_adde(
                simd,
                E::faer_simd_neg(simd, s),
                into_copy::<E, _>(ai_),
                E::faer_simd_scale_real(simd, c, into_copy::<E, _>(aj_)),
            ));
            ai_ = from_copy::<E, _>(tmp);

            E::faer_map(
                E::faer_zip(ai, ai_),
                #[inline(always)]
                |(ai, ai_): (&mut _, _)| *ai = ai_,
            );
            E::faer_map(
                E::faer_zip(aj, aj_),
                #[inline(always)]
                |(aj, aj_): (&mut _, _)| *aj = aj_,
            );
        }

        let mut ai_ = E::faer_partial_load(simd, E::faer_rb(E::faer_as_ref(&ai_tail)));
        let mut aj_ = E::faer_partial_load(simd, E::faer_rb(E::faer_as_ref(&aj_tail)));

        let tmp = E::faer_simd_conj_mul_adde(simd, s, aj_, E::faer_simd_scale_real(simd, c, ai_));

        aj_ = E::faer_simd_mul_adde(
            simd,
            E::faer_simd_neg(simd, s),
            ai_,
            E::faer_simd_scale_real(simd, c, aj_),
        );
        ai_ = tmp;

        E::faer_partial_store(simd, ai_tail, ai_);
        E::faer_partial_store(simd, aj_tail, aj_);
    }
}

pub fn lahqr<E: ComplexField>(
    want_t: bool,
    a: MatMut<'_, E>,
    z: Option<MatMut<'_, E>>,
    w: ColMut<'_, E>,
    ilo: usize,
    ihi: usize,
    epsilon: E::Real,
    zero_threshold: E::Real,
) -> isize {
    assert!(a.nrows() == a.ncols());
    assert!(ilo <= ihi);

    let n = a.nrows();
    let nh = ihi - ilo;

    assert!(w.nrows() == n);

    if let Some(z) = z.rb() {
        assert!(z.nrows() == n);
        assert!(z.ncols() == n);
    }

    let mut a = a;
    let mut z = z;
    let mut w = w;

    let zero = E::Real::faer_zero();
    let eps = epsilon;
    let small_num = zero_threshold.faer_div(eps);
    let non_convergence_limit = 10;
    let dat1 = E::Real::faer_from_f64(0.75);
    let dat2 = E::Real::faer_from_f64(-0.4375);

    if nh == 0 {
        return 0;
    }

    if nh == 1 {
        w.write(ilo, a.read(ilo, ilo));
    }

    // itmax is the total number of QR iterations allowed.
    // For most matrices, 3 shifts per eigenvalue is enough, so
    // we set itmax to 30 times nh as a safe limit.
    let itmax = 30 * Ord::max(10, nh);

    // k_defl counts the number of iterations since a deflation
    let mut k_defl = 0usize;

    // istop is the end of the active subblock.
    // As more and more eigenvalues converge, it eventually
    // becomes ilo+1 and the loop ends.
    let mut istop = ihi;

    // istart is the start of the active subblock. Either
    // istart = ilo, or H(istart, istart-1) = 0. This means
    // that we can treat this subblock separately.
    let mut istart = ilo;

    let arch = E::Simd::default();
    for iter in 0..itmax + 1 {
        if iter == itmax {
            return istop as isize;
        }

        if istart + 1 >= istop {
            if istart + 1 == istop {
                w.write(istart, a.read(istart, istart));
            }
            // All eigenvalues have been found, exit and return 0.
            break;
        }

        // Determine range to apply rotations
        let istart_m;
        let istop_m;
        if !want_t {
            istart_m = istart;
            istop_m = istop;
        } else {
            istart_m = 0;
            istop_m = n;
        }

        // Check if active subblock has split
        for i in (istart + 1..istop).rev() {
            if abs1(a.read(i, i - 1)) < small_num {
                // A(i,i-1) is negligible, take i as new istart.
                a.write(i, i - 1, E::faer_zero());
                istart = i;
                break;
            }

            let mut tst = abs1(a.read(i - 1, i - 1)).faer_add(abs1(a.read(i, i)));
            if tst == zero {
                if i >= ilo + 2 {
                    tst = tst.faer_add(a.read(i - 1, i - 2).faer_abs());
                }
                if i + 1 < ihi {
                    tst = tst.faer_add(a.read(i + 1, i).faer_abs());
                }
            }

            if abs1(a.read(i, i - 1)) <= eps.faer_mul(tst) {
                //
                // The elementwise deflation test has passed
                // The following performs second deflation test due
                // to Ahues & Tisseur (LAWN 122, 1997). It has better
                // mathematical foundation and improves accuracy in some
                // examples.
                //
                // The test is |A(i,i-1)|*|A(i-1,i)| <=
                // eps*|A(i,i)|*|A(i-1,i-1)| The multiplications might overflow
                // so we do some scaling first.
                //

                let ab = max(abs1(a.read(i, i - 1)), abs1(a.read(i - 1, i)));
                let ba = min(abs1(a.read(i, i - 1)), abs1(a.read(i - 1, i)));
                let aa = max(
                    abs1(a.read(i, i)),
                    abs1(a.read(i, i).faer_sub(a.read(i - 1, i - 1))),
                );
                let bb = min(
                    abs1(a.read(i, i)),
                    abs1(a.read(i, i).faer_sub(a.read(i - 1, i - 1))),
                );
                let s = aa.faer_add(ab);
                if ba.faer_mul(ab.faer_div(s))
                    <= max(small_num, eps.faer_mul(bb.faer_mul(aa.faer_div(s))))
                {
                    // A(i,i-1) is negligible, take i as new istart.
                    a.write(i, i - 1, E::faer_zero());
                    istart = i;
                    break;
                }
            }
        }

        if istart + 1 >= istop {
            k_defl = 0;
            w.write(istart, a.read(istart, istart));
            istop = istart;
            istart = ilo;
            continue;
        }

        // Determine shift
        let (a00, a01, a10, a11);
        k_defl += 1;

        if k_defl % non_convergence_limit == 0 {
            // Exceptional shift
            let mut s = a.read(istop - 1, istop - 2).faer_abs();
            if istop > ilo + 2 {
                s = s.faer_add(a.read(istop - 2, istop - 3).faer_abs());
            };
            a00 = E::faer_from_real(dat1.faer_mul(s)).faer_add(a.read(istop - 1, istop - 1));
            a01 = E::faer_from_real(dat2.faer_mul(s));
            a10 = E::faer_from_real(s);
            a11 = a00;
        } else {
            // Wilkinson shift
            a00 = a.read(istop - 2, istop - 2);
            a10 = a.read(istop - 1, istop - 2);
            a01 = a.read(istop - 2, istop - 1);
            a11 = a.read(istop - 1, istop - 1);
        }

        let (mut s1, s2) = lahqr_eig22(a00, a01, a10, a11);

        if abs1(s1.faer_sub(a.read(istop - 1, istop - 1)))
            > abs1(s2.faer_sub(a.read(istop - 1, istop - 1)))
        {
            s1 = s2;
        }

        // We have already checked whether the subblock has split.
        // If it has split, we can introduce any shift at the top of the new
        // subblock. Now that we know the specific shift, we can also check
        // whether we can introduce that shift somewhere else in the subblock.
        let mut istart2 = istart;
        if istart + 2 < istop {
            for i in (istart + 1..istop - 1).rev() {
                let h00 = a.read(i, i).faer_sub(s1);
                let h10 = a.read(i + 1, i);

                let (_cs, sn, _r) = rotg(h00, h10, eps, zero_threshold);
                if abs1(sn.faer_conj().faer_mul(a.read(i, i - 1)))
                    <= eps.faer_mul(abs1(a.read(i, i - 1)).faer_add(abs1(a.read(i, i + 1))))
                {
                    istart2 = i;
                    break;
                }
            }
        }

        for i in istart2..istop - 1 {
            let (cs, sn, r);
            if i == istart2 {
                let h00 = a.read(i, i).faer_sub(s1);
                let h10 = a.read(i + 1, i);
                (cs, sn, _) = rotg(h00, h10, eps, zero_threshold);
                if i > istart {
                    a.write(i, i - 1, a.read(i, i - 1).faer_scale_real(cs));
                }
            } else {
                (cs, sn, r) = rotg(a.read(i, i - 1), a.read(i + 1, i - 1), eps, zero_threshold);
                a.write(i, i - 1, r);
                a.write(i + 1, i - 1, E::faer_zero());
            }

            // Apply G from the left to A
            let ai = unsafe { a.rb().row(i).subcols(i, istop_m - i).const_cast() };
            let aip1 = unsafe { a.rb().row(i + 1).subcols(i, istop_m - i).const_cast() };

            zipped!(ai, aip1).for_each(|unzipped!(mut ai, mut aip1)| {
                let ai_ = ai.read();
                let aip1_ = aip1.read();
                let tmp = (ai_.faer_scale_real(cs)).faer_add(aip1_.faer_mul(sn));
                aip1.write(
                    (ai_.faer_mul(sn.faer_conj().faer_neg())).faer_add(aip1_.faer_scale_real(cs)),
                );
                ai.write(tmp);
            });

            // Apply G**H from the right to A
            let nrows_ = Ord::min(i + 3, istop) - istart_m;
            let ai = unsafe { a.rb().col(i).subrows(istart_m, nrows_).const_cast() };
            let aip1 = unsafe { a.rb().col(i + 1).subrows(istart_m, nrows_).const_cast() };

            if a.row_stride() == 1 {
                arch.dispatch(Rot {
                    ai,
                    aj: aip1,
                    c: cs,
                    s: sn,
                });
            } else {
                zipped!(ai, aip1).for_each(|unzipped!(mut ai, mut aip1)| {
                    let ai_ = ai.read();
                    let aip1_ = aip1.read();
                    let tmp = (ai_.faer_scale_real(cs)).faer_add(aip1_.faer_mul(sn.faer_conj()));
                    aip1.write((ai_.faer_mul(sn.faer_neg())).faer_add(aip1_.faer_scale_real(cs)));
                    ai.write(tmp);
                });
            }
            if let Some(z) = z.rb_mut() {
                // Apply G**H to Z from the right
                let zi = unsafe { z.rb().col(i).const_cast() };
                let zip1 = unsafe { z.rb().col(i + 1).const_cast() };

                if z.row_stride() == 1 {
                    arch.dispatch(Rot {
                        ai: zi,
                        aj: zip1,
                        c: cs,
                        s: sn,
                    });
                } else {
                    zipped!(zi, zip1).for_each(|unzipped!(mut zi, mut zip1)| {
                        let zi_ = zi.read();
                        let zip1_ = zip1.read();
                        let tmp =
                            (zi_.faer_scale_real(cs)).faer_add(zip1_.faer_mul(sn.faer_conj()));
                        zip1.write(
                            (zi_.faer_mul(sn.faer_neg())).faer_add(zip1_.faer_scale_real(cs)),
                        );
                        zi.write(tmp);
                    });
                }
            }
        }
    }
    0
}

#[derive(Default, Clone, Copy, Debug)]
#[non_exhaustive]
pub struct EvdParams {
    /// Function that returns the number of shifts to use for a given matrix size
    pub recommended_shift_count:
        Option<fn(matrix_dimension: usize, active_block_dimension: usize) -> usize>,
    /// Function that returns the deflation window to use for a given matrix size
    pub recommended_deflation_window:
        Option<fn(matrix_dimension: usize, active_block_dimension: usize) -> usize>,
    /// Threshold to switch between blocked and unblocked code
    pub blocking_threshold: Option<usize>,
    /// Threshold of percent of aggressive-early-deflation window that must converge to skip a
    /// sweep
    pub nibble_threshold: Option<usize>,
}

pub fn default_recommended_shift_count(dim: usize, _active_block_dim: usize) -> usize {
    let n = dim;
    if n < 30 {
        2
    } else if n < 60 {
        4
    } else if n < 150 {
        12
    } else if n < 590 {
        32
    } else if n < 3000 {
        64
    } else if n < 6000 {
        128
    } else {
        256
    }
}

pub fn default_recommended_deflation_window(dim: usize, _active_block_dim: usize) -> usize {
    let n = dim;
    if n < 30 {
        2
    } else if n < 60 {
        4
    } else if n < 150 {
        10
    } else if n < 590 {
        #[cfg(feature = "std")]
        {
            (n as f64 / (n as f64).log2()) as usize
        }
        #[cfg(not(feature = "std"))]
        {
            libm::log2(n as f64 / (n as f64)) as usize
        }
    } else if n < 3000 {
        96
    } else if n < 6000 {
        192
    } else {
        384
    }
}

pub fn default_blocking_threshold() -> usize {
    75
}

pub fn default_nibble_threshold() -> usize {
    50
}

fn aggressive_early_deflation<E: ComplexField>(
    want_t: bool,
    mut a: MatMut<'_, E>,
    mut z: Option<MatMut<'_, E>>,
    mut s: ColMut<'_, E>,
    ilo: usize,
    ihi: usize,
    nw: usize,
    epsilon: E::Real,
    zero_threshold: E::Real,
    parallelism: Parallelism,
    mut stack: PodStack<'_>,
    params: EvdParams,
) -> (usize, usize) {
    let n = a.nrows();

    // Because we will use the lower triangular part of A as workspace,
    // We have a maximum window size
    let nw_max = (n - 3) / 3;
    let eps = epsilon;
    let small_num = zero_threshold
        .faer_div(eps)
        .faer_mul(E::Real::faer_from_f64(n as f64));

    // Size of the deflation window
    let jw = Ord::min(Ord::min(nw, ihi - ilo), nw_max);
    // First row index in the deflation window
    let kwtop = ihi - jw;

    // s is the value just outside the window. It determines the spike
    // together with the orthogonal schur factors.
    let mut s_spike = if kwtop == ilo {
        E::faer_zero()
    } else {
        a.read(kwtop, kwtop - 1)
    };

    if kwtop + 1 == ihi {
        // 1x1 deflation window, not much to do
        s.write(kwtop, a.read(kwtop, kwtop));
        let mut ns = 1;
        let mut nd = 0;
        if abs1(s_spike) <= max(small_num, eps.faer_mul(abs1(a.read(kwtop, kwtop)))) {
            ns = 0;
            nd = 1;
            if kwtop > ilo {
                a.write(kwtop, kwtop - 1, E::faer_zero());
            }
        }
        return (ns, nd);
    }

    // Define workspace matrices
    // We use the lower triangular part of A as workspace
    // TW and WH overlap, but WH is only used after we no longer need
    // TW so it is ok.
    let mut v = unsafe { a.rb().submatrix(n - jw, 0, jw, jw).const_cast() };
    let mut tw = unsafe { a.rb().submatrix(n - jw, jw, jw, jw).const_cast() };
    let mut wh = unsafe {
        a.rb()
            .submatrix(n - jw, jw, jw, n - 2 * jw - 3)
            .const_cast()
    };
    let mut wv = unsafe { a.rb().submatrix(jw + 3, 0, n - 2 * jw - 3, jw).const_cast() };
    let mut a = unsafe { a.rb().const_cast() };

    // Convert the window to spike-triangular form. i.e. calculate the
    // Schur form of the deflation window.
    // If the QR algorithm fails to convergence, it can still be
    // partially in Schur form. In that case we continue on a smaller
    // window (note the use of infqr later in the code).
    let a_window = a.rb().submatrix(kwtop, kwtop, ihi - kwtop, ihi - kwtop);
    let mut s_window = unsafe { s.rb().subrows(kwtop, ihi - kwtop).const_cast() };
    zipped!(tw.rb_mut())
        .for_each_triangular_lower(Diag::Include, |unzipped!(mut x)| x.write(E::faer_zero()));
    for j in 0..jw {
        for i in 0..Ord::min(j + 2, jw) {
            tw.write(i, j, a_window.read(i, j));
        }
    }
    v.fill_zero();
    v.rb_mut()
        .diagonal_mut()
        .column_vector_mut()
        .fill(E::faer_one());

    let infqr = if jw
        < params
            .blocking_threshold
            .unwrap_or(default_blocking_threshold())
    {
        lahqr(
            true,
            tw.rb_mut(),
            Some(v.rb_mut()),
            s_window.rb_mut(),
            0,
            jw,
            epsilon,
            zero_threshold,
        )
    } else {
        let infqr = multishift_qr(
            true,
            tw.rb_mut(),
            Some(v.rb_mut()),
            s_window.rb_mut(),
            0,
            jw,
            epsilon,
            zero_threshold,
            parallelism,
            stack.rb_mut(),
            params,
        )
        .0;
        for j in 0..jw {
            for i in j + 2..jw {
                tw.write(i, j, E::faer_zero());
            }
        }
        infqr
    };
    let infqr = infqr as usize;

    // Deflation detection loop
    // one eigenvalue block at a time, we will check if it is deflatable
    // by checking the bottom spike element. If it is not deflatable,
    // we move the block up. This moves other blocks down to check.
    let mut ns = jw;
    let nd;
    let mut ilst = infqr;
    while ilst < ns {
        // 1x1 eigenvalue block
        #[allow(clippy::disallowed_names)]
        let mut foo = abs1(tw.read(ns - 1, ns - 1));
        if foo == E::Real::faer_zero() {
            foo = abs1(s_spike);
        }
        if abs1(s_spike).faer_mul(abs1(v.read(0, ns - 1))) <= max(small_num, eps.faer_mul(foo)) {
            // Eigenvalue is deflatable
            ns -= 1;
        } else {
            // Eigenvalue is not deflatable.
            // Move it up out of the way.
            let ifst = ns - 1;
            schur_move(
                tw.rb_mut(),
                Some(v.rb_mut()),
                ifst,
                &mut ilst,
                epsilon,
                zero_threshold,
            );
            ilst += 1;
        }
    }

    if ns == 0 {
        s_spike = E::faer_zero();
    }

    if ns == jw {
        // Aggressive early deflation didn't deflate any eigenvalues
        // We don't need to apply the update to the rest of the matrix
        nd = jw - ns;
        ns -= infqr;
        return (ns, nd);
    }

    // sorting diagonal blocks of T improves accuracy for graded matrices.
    // Bubble sort deals well with exchange failures.
    let mut sorted = false;
    // Window to be checked (other eigenvalue are sorted)
    let mut sorting_window_size = jw;
    while !sorted {
        sorted = true;

        // Index of last eigenvalue that was swapped
        let mut ilst = 0;

        // Index of the first block
        let mut i1 = ns;

        while i1 + 1 < sorting_window_size {
            // Check if there is a next block
            if i1 + 1 == jw {
                ilst -= 1;
                break;
            }

            // Index of the second block
            let i2 = i1 + 1;

            // Size of the second block

            let ev1 = abs1(tw.read(i1, i1));
            let ev2 = abs1(tw.read(i2, i2));

            if ev1 > ev2 {
                i1 = i2;
            } else {
                sorted = false;
                let ierr = schur_swap(tw.rb_mut(), Some(v.rb_mut()), i1, epsilon, zero_threshold);
                if ierr == 0 {
                    i1 += 1;
                } else {
                    i1 = i2;
                }
                ilst = i1;
            }
        }
        sorting_window_size = ilst;
    }

    // Recalculate the eigenvalues
    let mut i = 0;
    while i < jw {
        s.write(kwtop + i, tw.read(i, i));
        i += 1;
    }

    // Reduce A back to Hessenberg form (if necessary)
    if s_spike != E::faer_zero() {
        // Reflect spike back
        {
            let mut vv = wv.rb_mut().col_mut(0).subrows_mut(0, ns);
            for i in 0..ns {
                vv.write(i, v.read(0, i).faer_conj());
            }
            let head = vv.read(0);
            let tail = vv.rb_mut().subrows_mut(1, ns - 1);
            let tail_norm = tail.rb().norm_l2();
            let (tau, beta) = make_householder_in_place(Some(tail), head, tail_norm);
            vv.write(0, E::faer_one());
            let tau = tau.faer_inv();

            {
                let mut tw_slice = tw.rb_mut().submatrix_mut(0, 0, ns, jw);
                let tmp = vv.rb().adjoint() * tw_slice.rb();
                matmul(
                    tw_slice.rb_mut(),
                    vv.rb(),
                    tmp.as_ref(),
                    Some(E::faer_one()),
                    tau.faer_neg(),
                    parallelism,
                );
            }

            {
                let mut tw_slice2 = tw.rb_mut().submatrix_mut(0, 0, jw, ns);
                let tmp = tw_slice2.rb() * vv.rb();
                matmul(
                    tw_slice2.rb_mut(),
                    tmp.as_ref(),
                    vv.rb().adjoint(),
                    Some(E::faer_one()),
                    tau.faer_neg(),
                    parallelism,
                );
            }

            {
                let mut v_slice = v.rb_mut().submatrix_mut(0, 0, jw, ns);
                let tmp = v_slice.rb() * vv.rb();
                matmul(
                    v_slice.rb_mut(),
                    tmp.as_ref(),
                    vv.rb().adjoint(),
                    Some(E::faer_one()),
                    tau.faer_neg(),
                    parallelism,
                );
            }
            vv.write(0, beta);
        }

        // Hessenberg reduction
        {
            let mut householder = wv.rb_mut().col_mut(0).subrows_mut(0, ns - 1);
            make_hessenberg_in_place(
                tw.rb_mut().submatrix_mut(0, 0, ns, ns),
                householder.rb_mut().as_2d_mut(),
                parallelism,
                stack.rb_mut(),
            );
            apply_block_householder_sequence_transpose_on_the_left_in_place_with_conj(
                tw.rb().submatrix(1, 0, ns - 1, ns - 1),
                householder.rb().transpose().as_2d(),
                Conj::Yes,
                unsafe { tw.rb().submatrix(1, ns, ns - 1, jw - ns).const_cast() },
                parallelism,
                stack.rb_mut(),
            );
            apply_block_householder_sequence_on_the_right_in_place_with_conj(
                tw.rb().submatrix(1, 0, ns - 1, ns - 1),
                householder.rb().transpose().as_2d(),
                Conj::No,
                v.rb_mut().submatrix_mut(0, 1, jw, ns - 1),
                parallelism,
                stack.rb_mut(),
            );
        }
    }

    // Copy the deflation window back into place
    if kwtop > 0 {
        a.write(kwtop, kwtop - 1, s_spike.faer_mul(v.read(0, 0).faer_conj()));
    }
    for j in 0..jw {
        for i in 0..Ord::min(j + 2, jw) {
            a.write(kwtop + i, kwtop + j, tw.read(i, j));
        }
    }

    // Store number of deflated eigenvalues
    nd = jw - ns;
    ns -= infqr;

    //
    // Update rest of the matrix using matrix matrix multiplication
    //
    let (istart_m, istop_m);
    if want_t {
        istart_m = 0;
        istop_m = n;
    } else {
        istart_m = ilo;
        istop_m = ihi;
    }

    // Horizontal multiply
    if ihi < istop_m {
        let mut i = ihi;
        while i < istop_m {
            let iblock = Ord::min(istop_m - i, wh.ncols());
            let mut a_slice = a.rb_mut().submatrix_mut(kwtop, i, ihi - kwtop, iblock);
            let mut wh_slice = wh
                .rb_mut()
                .submatrix_mut(0, 0, a_slice.nrows(), a_slice.ncols());
            matmul(
                wh_slice.rb_mut(),
                v.rb().adjoint(),
                a_slice.rb(),
                None,
                E::faer_one(),
                parallelism,
            );
            a_slice.copy_from(wh_slice.rb());
            i += iblock;
        }
    }

    // Vertical multiply
    if istart_m < kwtop {
        let mut i = istart_m;
        while i < kwtop {
            let iblock = Ord::min(kwtop - i, wv.nrows());
            let mut a_slice = a.rb_mut().submatrix_mut(i, kwtop, iblock, ihi - kwtop);
            let mut wv_slice = wv
                .rb_mut()
                .submatrix_mut(0, 0, a_slice.nrows(), a_slice.ncols());
            matmul(
                wv_slice.rb_mut(),
                a_slice.rb(),
                v.rb(),
                None,
                E::faer_one(),
                parallelism,
            );
            a_slice.copy_from(wv_slice.rb());
            i += iblock;
        }
    }
    // Update Z (also a vertical multiplication)
    if let Some(mut z) = z.rb_mut() {
        let mut i = 0;
        while i < n {
            let iblock = Ord::min(n - i, wv.nrows());
            let mut z_slice = z.rb_mut().submatrix_mut(i, kwtop, iblock, ihi - kwtop);
            let mut wv_slice = wv
                .rb_mut()
                .submatrix_mut(0, 0, z_slice.nrows(), z_slice.ncols());
            matmul(
                wv_slice.rb_mut(),
                z_slice.rb(),
                v.rb(),
                None,
                E::faer_one(),
                parallelism,
            );
            z_slice.copy_from(wv_slice.rb());
            i += iblock;
        }
    }

    (ns, nd)
}

fn schur_move<E: ComplexField>(
    mut a: MatMut<E>,
    mut q: Option<MatMut<E>>,
    ifst: usize,
    ilst: &mut usize,
    epsilon: E::Real,
    zero_threshold: E::Real,
) -> isize {
    let n = a.nrows();

    // Quick return
    if n == 0 {
        return 0;
    }

    let mut here = ifst;
    if ifst < *ilst {
        while here != *ilst {
            // Size of the next eigenvalue block
            let ierr = schur_swap(a.rb_mut(), q.rb_mut(), here, epsilon, zero_threshold);
            if ierr != 0 {
                // The swap failed, return with error
                *ilst = here;
                return 1;
            }
            here += 1;
        }
    } else {
        while here != *ilst {
            // Size of the next eigenvalue block
            let ierr = schur_swap(a.rb_mut(), q.rb_mut(), here - 1, epsilon, zero_threshold);
            if ierr != 0 {
                // The swap failed, return with error
                *ilst = here;
                return 1;
            }
            here -= 1;
        }
    }

    0
}

fn schur_swap<E: ComplexField>(
    mut a: MatMut<E>,
    q: Option<MatMut<E>>,
    j0: usize,
    epsilon: E::Real,
    zero_threshold: E::Real,
) -> isize {
    let n = a.nrows();

    let j1 = j0 + 1;
    let j2 = j0 + 2;

    //
    // In the complex case, there can only be 1x1 blocks to swap
    //
    let t00 = a.read(j0, j0);
    let t11 = a.read(j1, j1);
    //
    // Determine the transformation to perform the interchange
    //
    let (cs, sn, _) = rotg(a.read(j0, j1), t11.faer_sub(t00), epsilon, zero_threshold);

    a.write(j1, j1, t00);
    a.write(j0, j0, t11);

    // Apply transformation from the left
    if j2 < n {
        let row1 = unsafe { a.rb().row(j0).subcols(j2, n - j2).transpose().const_cast() };
        let row2 = unsafe { a.rb().row(j1).subcols(j2, n - j2).transpose().const_cast() };
        rot(row1, row2, cs, sn);
    }
    // Apply transformation from the right
    if j0 > 0 {
        let col1 = unsafe { a.rb().col(j0).subrows(0, j0).const_cast() };
        let col2 = unsafe { a.rb().col(j1).subrows(0, j0).const_cast() };

        rot(col1, col2, cs, sn.faer_conj());
    }
    if let Some(q) = q {
        let col1 = unsafe { q.rb().col(j0).const_cast() };
        let col2 = unsafe { q.rb().col(j1).const_cast() };
        rot(col1, col2, cs, sn.faer_conj());
    }

    0
}

pub fn multishift_qr_req<E: Entity>(
    n: usize,
    nh: usize,
    want_t: bool,
    want_z: bool,
    parallelism: Parallelism,
    params: EvdParams,
) -> Result<StackReq, SizeOverflow> {
    let nsr = (params
        .recommended_shift_count
        .unwrap_or(default_recommended_shift_count))(n, nh);

    let _ = want_t;
    let _ = want_z;

    if n <= 3 {
        return Ok(StackReq::empty());
    }

    let nw_max = (n - 3) / 3;

    StackReq::try_any_of([
        make_hessenberg_in_place_req::<E>(nw_max, 1, parallelism)?,
        apply_block_householder_sequence_transpose_on_the_left_in_place_req::<E>(
            nw_max, nw_max, nw_max,
        )?,
        apply_block_householder_sequence_on_the_right_in_place_req::<E>(nw_max, nw_max, nw_max)?,
        temp_mat_req::<E>(3, nsr)?,
    ])
}

/// returns err code, number of aggressive early deflations, number of qr sweeps
pub fn multishift_qr<E: ComplexField>(
    want_t: bool,
    a: MatMut<'_, E>,
    z: Option<MatMut<'_, E>>,
    w: ColMut<'_, E>,
    ilo: usize,
    ihi: usize,
    epsilon: E::Real,
    zero_threshold: E::Real,
    parallelism: Parallelism,
    stack: PodStack<'_>,
    params: EvdParams,
) -> (isize, usize, usize) {
    assert!(a.nrows() == a.ncols());
    assert!(ilo <= ihi);

    let n = a.nrows();
    let nh = ihi - ilo;

    assert!(w.nrows() == n);
    assert!(w.ncols() == 1);

    if let Some(z) = z.rb() {
        assert!(z.nrows() == n);
        assert!(z.ncols() == n);
    }

    let mut a = a;
    let mut z = z;
    let mut w = w;
    let mut stack = stack;

    let non_convergence_limit_window = 5;
    let non_convergence_limit_shift = 6;
    let dat1 = E::Real::faer_from_f64(0.75);
    let dat2 = E::Real::faer_from_f64(-0.4375);

    // This routine uses the space below the subdiagonal as workspace
    // For small matrices, this is not enough
    // if n < nmin, the matrix will be passed to lahqr
    let nmin = Ord::max(
        15,
        params
            .blocking_threshold
            .unwrap_or(default_blocking_threshold()),
    );
    let nibble = params
        .nibble_threshold
        .unwrap_or(default_nibble_threshold());

    // Recommended number of shifts
    let nsr = (params
        .recommended_shift_count
        .unwrap_or(default_recommended_shift_count))(n, nh);
    let nsr = Ord::min(Ord::min(nsr, (n.saturating_sub(3)) / 6), ihi - ilo - 1);
    let nsr = Ord::max(nsr / 2 * 2, 2);

    // Recommended deflation window size
    let nwr = (params
        .recommended_deflation_window
        .unwrap_or(default_recommended_deflation_window))(n, nh);
    let nwr = Ord::max(nwr, 2);
    let nwr = Ord::min(Ord::min(nwr, (n.saturating_sub(1)) / 3), ihi - ilo);

    // Tiny matrices must use lahqr
    if n < nmin {
        let err = lahqr(want_t, a, z, w, ilo, ihi, epsilon, zero_threshold);
        return (err, 0, 0);
    }
    if nh == 0 {
        return (0, 0, 0);
    }

    let nw_max = (n - 3) / 3;

    // itmax is the total number of QR iterations allowed.
    // For most matrices, 3 shifts per eigenvalue is enough, so
    // we set itmax to 30 times nh as a safe limit.
    let itmax = 30 * Ord::max(10, nh);

    // k_defl counts the number of iterations since a deflation
    let mut k_defl = 0;

    // istop is the end of the active subblock.
    // As more and more eigenvalues converge, it eventually
    // becomes ilo+1 and the loop ends.
    let mut istop = ihi;

    let mut info = 0;
    let mut nw = 0;

    let mut count_aed = 0;
    let mut count_sweep = 0;

    for iter in 0..itmax + 1 {
        if iter == itmax {
            // The QR algorithm failed to converge, return with error.
            info = istop as isize;
            break;
        }

        if ilo + 1 >= istop {
            if ilo + 1 == istop {
                w.write(ilo, a.read(ilo, ilo));
            }
            // All eigenvalues have been found, exit and return 0.
            break;
        }

        // istart is the start of the active subblock. Either
        // istart = ilo, or H(istart, istart-1) = 0. This means
        // that we can treat this subblock separately.
        let mut istart = ilo;

        // Find active block
        for i in (ilo + 1..istop).rev() {
            if a.read(i, i - 1) == E::faer_zero() {
                istart = i;
                break;
            }
        }

        //
        // Aggressive early deflation
        //
        let nh = istop - istart;
        let nwupbd = Ord::min(nh, nw_max);
        if k_defl < non_convergence_limit_window {
            nw = Ord::min(nwupbd, nwr);
        } else {
            // There have been no deflations in many iterations
            // Try to vary the deflation window size.
            nw = Ord::min(nwupbd, 2 * nw);
        }
        if nh <= 4 {
            // n >= nmin, so there is always enough space for a 4x4 window
            nw = nh;
        }
        if nw < nw_max {
            if nw + 1 >= nh {
                nw = nh
            };
            let kwtop = istop - nw;
            if (kwtop > istart + 2)
                && (abs1(a.read(kwtop, kwtop - 1)) > abs1(a.read(kwtop - 1, kwtop - 2)))
            {
                nw += 1;
            }
        }

        let (ls, ld) = aggressive_early_deflation(
            want_t,
            a.rb_mut(),
            z.rb_mut(),
            w.rb_mut(),
            istart,
            istop,
            nw,
            epsilon,
            zero_threshold,
            parallelism,
            stack.rb_mut(),
            params,
        );
        count_aed += 1;

        istop -= ld;

        if ld > 0 {
            k_defl = 0;
        }

        // Skip an expensive QR sweep if there is a (partly heuristic)
        // reason to expect that many eigenvalues will deflate without it.
        // Here, the QR sweep is skipped if many eigenvalues have just been
        // deflated or if the remaining active block is small.
        if ld > 0 && (100 * ld > nwr * nibble || (istop - istart) <= Ord::min(nmin, nw_max)) {
            continue;
        }

        k_defl += 1;
        let mut ns = Ord::min(nh - 1, Ord::min(Ord::max(2, ls), nsr));
        ns = ns / 2 * 2;
        let mut i_shifts = istop - ls;

        if k_defl % non_convergence_limit_shift == 0 {
            for i in (i_shifts + 1..istop).rev().step_by(2) {
                if i >= ilo + 2 {
                    let ss = abs1(a.read(i, i - 1)).faer_add(abs1(a.read(i - 1, i - 2)));
                    let aa = E::faer_from_real(dat1.faer_mul(ss)).faer_add(a.read(i, i));
                    let bb = E::faer_from_real(ss);
                    let cc = E::faer_from_real(dat2.faer_mul(ss));
                    let dd = aa;
                    let (s1, s2) = lahqr_eig22(aa, bb, cc, dd);
                    w.write(i - 1, s1);
                    w.write(i, s2);
                } else {
                    w.write(i - 1, a.read(i, i));
                    w.write(i, a.read(i, i));
                }
            }
        } else {
            if ls <= ns / 2 {
                // Got ns/2 or fewer shifts? Then use multi/double shift qr to
                // get more
                let mut temp = a.rb_mut().submatrix_mut(n - ns, 0, ns, ns);
                let mut shifts = w.rb_mut().subrows_mut(istop - ns, ns);
                let ierr = lahqr(
                    false,
                    temp.rb_mut(),
                    None,
                    shifts.rb_mut(),
                    0,
                    ns,
                    epsilon,
                    zero_threshold,
                ) as usize;

                ns = ns - ierr;

                if ns < 2 {
                    // In case of a rare QR failure, use eigenvalues
                    // of the trailing 2x2 submatrix
                    let aa = a.read(istop - 2, istop - 2);
                    let bb = a.read(istop - 2, istop - 1);
                    let cc = a.read(istop - 1, istop - 2);
                    let dd = a.read(istop - 1, istop - 1);
                    let (s1, s2) = lahqr_eig22(aa, bb, cc, dd);
                    w.write(istop - 2, s1);
                    w.write(istop - 1, s2);
                    ns = 2;
                }

                i_shifts = istop - ns;
            }

            // Sort the shifts (helps a little)
            // Bubble sort keeps complex conjugate pairs together
            let mut sorted = false;
            let mut k = istop;
            while !sorted && k > i_shifts {
                sorted = true;
                for i in i_shifts..k - 1 {
                    if abs1(w.read(i)) < abs1(w.read(i + 1)) {
                        sorted = false;
                        let wi = w.read(i);
                        let wip1 = w.read(i + 1);
                        w.write(i, wip1);
                        w.write(i + 1, wi);
                    }
                }
                k -= 1;
            }

            // Shuffle shifts into pairs of real shifts
            // and pairs of complex conjugate shifts
            // assuming complex conjugate shifts are
            // already adjacent to one another. (Yes,
            // they are.)
            for i in (i_shifts + 2..istop).rev().step_by(2) {
                if w.read(i).faer_imag() != w.read(i - 1).faer_imag().faer_neg() {
                    let tmp = w.read(i);
                    w.write(i, w.read(i - 1));
                    w.write(i - 1, w.read(i - 2));
                    w.write(i - 2, tmp);
                }
            }

            // Since we shuffled the shifts, we will only drop
            // Real shifts
            if ns % 2 == 1 {
                ns -= 1;
            }
            i_shifts = istop - ns;
        }

        let mut shifts = w.rb_mut().subrows_mut(i_shifts, ns);
        multishift_qr_sweep(
            want_t,
            a.rb_mut(),
            z.rb_mut(),
            shifts.rb_mut(),
            istart,
            istop,
            epsilon,
            zero_threshold,
            parallelism,
            stack.rb_mut(),
        );
        count_sweep += 1;
    }

    (info, count_aed, count_sweep)
}

fn move_bulge<E: ComplexField>(
    mut h: MatMut<'_, E>,
    mut v: ColMut<'_, E>,
    s1: E,
    s2: E,
    epsilon: E::Real,
) {
    // Perform delayed update of row below the bulge
    // Assumes the first two elements of the row are zero
    let v0 = v.read(0).faer_real();
    let v1 = v.read(1);
    let v2 = v.read(2);
    let refsum = v2.faer_scale_real(v0).faer_mul(h.read(3, 2));

    h.write(3, 0, refsum.faer_neg());
    h.write(3, 1, refsum.faer_neg().faer_mul(v1.faer_conj()));
    h.write(3, 2, h.read(3, 2).faer_sub(refsum.faer_mul(v2.faer_conj())));

    // Generate reflector to move bulge down
    v.write(0, h.read(1, 0));
    v.write(1, h.read(2, 0));
    v.write(2, h.read(3, 0));

    let head = v.read(0);
    let tail = v.rb_mut().subrows_mut(1, 2);
    let tail_norm = tail.rb().norm_l2();
    let (tau, beta) = make_householder_in_place(Some(tail), head, tail_norm);
    v.write(0, tau.faer_inv());

    // Check for bulge collapse
    if h.read(3, 0) != E::faer_zero()
        || h.read(3, 1) != E::faer_zero()
        || h.read(3, 2) != E::faer_zero()
    {
        // The bulge hasn't collapsed, typical case
        h.write(1, 0, beta);
        h.write(2, 0, E::faer_zero());
        h.write(3, 0, E::faer_zero());
    } else {
        // The bulge has collapsed, attempt to reintroduce using
        // 2-small-subdiagonals trick
        let mut vt_storage = E::faer_map(E::faer_zero().faer_into_units(), |zero_unit| {
            [zero_unit, zero_unit, zero_unit]
        });
        let vt_ptr = E::faer_map(E::faer_as_mut(&mut vt_storage), |array| array.as_mut_ptr());
        let mut vt = unsafe { crate::col::from_raw_parts_mut::<'_, E>(vt_ptr, 3, 1) };

        let h2 = h.rb().submatrix(1, 1, 3, 3);
        lahqr_shiftcolumn(h2, vt.rb_mut(), s1, s2);

        let head = vt.read(0);
        let tail = vt.rb_mut().subrows_mut(1, 2);
        let tail_norm = tail.rb().norm_l2();
        let (tau, _) = make_householder_in_place(Some(tail), head, tail_norm);
        vt.write(0, tau.faer_inv());
        let vt0 = vt.read(0);
        let vt1 = vt.read(1);
        let vt2 = vt.read(2);

        let refsum = (vt0.faer_conj().faer_mul(h.read(1, 0)))
            .faer_add(vt1.faer_conj().faer_mul(h.read(2, 0)));

        if abs1(h.read(2, 0).faer_sub(refsum.faer_mul(vt1))).faer_add(abs1(refsum.faer_mul(vt2)))
            > epsilon.faer_mul(
                abs1(h.read(0, 0))
                    .faer_add(abs1(h.read(1, 1)))
                    .faer_add(abs1(h.read(2, 2))),
            )
        {
            // Starting a new bulge here would create non-negligible fill. Use
            // the old one.
            h.write(1, 0, beta);
            h.write(2, 0, E::faer_zero());
            h.write(3, 0, E::faer_zero());
        } else {
            // Fill-in is negligible, use the new reflector.
            h.write(1, 0, h.read(1, 0).faer_sub(refsum));
            h.write(2, 0, E::faer_zero());
            h.write(3, 0, E::faer_zero());
            v.write(0, vt.read(0));
            v.write(1, vt.read(1));
            v.write(2, vt.read(2));
        }
    }
}

fn multishift_qr_sweep<E: ComplexField>(
    want_t: bool,
    a: MatMut<E>,
    mut z: Option<MatMut<E>>,
    s: ColMut<E>,
    ilo: usize,
    ihi: usize,
    epsilon: E::Real,
    zero_threshold: E::Real,
    parallelism: Parallelism,
    stack: PodStack<'_>,
) {
    let n = a.nrows();

    let eps = epsilon;
    let small_num = zero_threshold
        .faer_div(eps)
        .faer_mul(E::Real::faer_from_f64(n as f64));
    assert!(n >= 12);

    let (mut v, _stack) = crate::linalg::temp_mat_zeroed::<E>(3, s.nrows() / 2, stack);
    let mut v = v.as_mut();

    let n_block_max = (n - 3) / 3;
    let n_shifts_max = Ord::min(ihi - ilo - 1, Ord::max(2, 3 * (n_block_max / 4)));

    let mut n_shifts = Ord::min(s.nrows(), n_shifts_max);
    if n_shifts % 2 == 1 {
        n_shifts -= 1;
    }
    let n_bulges = n_shifts / 2;

    let n_block_desired = Ord::min(2 * n_shifts, n_block_max);

    // Define workspace matrices
    // We use the lower triangular part of A as workspace

    // U stores the orthogonal transformations
    let mut u = unsafe {
        a.rb()
            .submatrix(n - n_block_desired, 0, n_block_desired, n_block_desired)
            .const_cast()
    };
    // Workspace for horizontal multiplications
    let mut wh = unsafe {
        a.rb()
            .submatrix(
                n - n_block_desired,
                n_block_desired,
                n_block_desired,
                n - 2 * n_block_desired - 3,
            )
            .const_cast()
    };
    // Workspace for vertical multiplications
    let mut wv = unsafe {
        a.rb()
            .submatrix(
                n_block_desired + 3,
                0,
                n - 2 * n_block_desired - 3,
                n_block_desired,
            )
            .const_cast()
    };
    let mut a = unsafe { a.rb().const_cast() };

    // i_pos_block points to the start of the block of bulges
    let mut i_pos_block;

    //
    // The following code block introduces the bulges into the matrix
    //
    {
        // Near-the-diagonal bulge introduction
        // The calculations are initially limited to the window:
        // A(ilo:ilo+n_block,ilo:ilo+n_block) The rest is updated later via
        // level 3 BLAS

        let n_block = Ord::min(n_block_desired, ihi - ilo);
        let mut istart_m = ilo;
        let mut istop_m = ilo + n_block;
        let mut u2 = u.rb_mut().submatrix_mut(0, 0, n_block, n_block);
        u2.fill_zero();
        u2.rb_mut()
            .diagonal_mut()
            .column_vector_mut()
            .fill(E::faer_one());

        for i_pos_last in ilo..ilo + n_block - 2 {
            // The number of bulges that are in the pencil
            let n_active_bulges = Ord::min(n_bulges, ((i_pos_last - ilo) / 2) + 1);

            for i_bulge in 0..n_active_bulges {
                let i_pos = i_pos_last - 2 * i_bulge;
                let mut v = v.rb_mut().col_mut(i_bulge);
                if i_pos == ilo {
                    // Introduce bulge
                    let h = a.rb().submatrix(ilo, ilo, 3, 3);

                    let s1 = s.read(s.nrows() - 1 - 2 * i_bulge);
                    let s2 = s.read(s.nrows() - 1 - 2 * i_bulge - 1);
                    lahqr_shiftcolumn(h, v.rb_mut(), s1, s2);

                    debug_assert!(v.nrows() == 3);
                    let head = v.read(0);
                    let tail = v.rb_mut().subrows_mut(1, 2);
                    let tail_norm = tail.rb().norm_l2();
                    let (tau, _) = make_householder_in_place(Some(tail), head, tail_norm);
                    v.write(0, tau.faer_inv());
                } else {
                    // Chase bulge down
                    let mut h = a.rb_mut().submatrix_mut(i_pos - 1, i_pos - 1, 4, 4);
                    let s1 = s.read(s.nrows() - 1 - 2 * i_bulge);
                    let s2 = s.read(s.nrows() - 1 - 2 * i_bulge - 1);
                    move_bulge(h.rb_mut(), v.rb_mut(), s1, s2, epsilon);
                }

                // Apply the reflector we just calculated from the right
                // We leave the last row for later (it interferes with the
                // optimally packed bulges)

                let v0 = v.read(0).faer_real();
                let v1 = v.read(1);
                let v2 = v.read(2);

                for j in istart_m..i_pos + 3 {
                    let sum = a
                        .read(j, i_pos)
                        .faer_add(v1.faer_mul(a.read(j, i_pos + 1)))
                        .faer_add(v2.faer_mul(a.read(j, i_pos + 2)));
                    a.write(j, i_pos, a.read(j, i_pos).faer_sub(sum.faer_scale_real(v0)));
                    a.write(
                        j,
                        i_pos + 1,
                        a.read(j, i_pos + 1)
                            .faer_sub(sum.faer_scale_real(v0).faer_mul(v1.faer_conj())),
                    );
                    a.write(
                        j,
                        i_pos + 2,
                        a.read(j, i_pos + 2)
                            .faer_sub(sum.faer_scale_real(v0).faer_mul(v2.faer_conj())),
                    );
                }

                // Apply the reflector we just calculated from the left
                // We only update a single column, the rest is updated later
                let sum = a
                    .read(i_pos, i_pos)
                    .faer_add(v1.faer_conj().faer_mul(a.read(i_pos + 1, i_pos)))
                    .faer_add(v2.faer_conj().faer_mul(a.read(i_pos + 2, i_pos)));
                a.write(
                    i_pos,
                    i_pos,
                    a.read(i_pos, i_pos).faer_sub(sum.faer_scale_real(v0)),
                );
                a.write(
                    i_pos + 1,
                    i_pos,
                    a.read(i_pos + 1, i_pos)
                        .faer_sub(sum.faer_scale_real(v0).faer_mul(v1)),
                );
                a.write(
                    i_pos + 2,
                    i_pos,
                    a.read(i_pos + 2, i_pos)
                        .faer_sub(sum.faer_scale_real(v0).faer_mul(v2)),
                );

                // Test for deflation.
                if (i_pos > ilo) && (a.read(i_pos, i_pos - 1) != E::faer_zero()) {
                    let mut tst1 =
                        abs1(a.read(i_pos - 1, i_pos - 1)).faer_add(abs1(a.read(i_pos, i_pos)));
                    if tst1 == E::Real::faer_zero() {
                        if i_pos > ilo + 1 {
                            tst1 = tst1.faer_add(abs1(a.read(i_pos - 1, i_pos - 2)));
                        }
                        if i_pos > ilo + 2 {
                            tst1 = tst1.faer_add(abs1(a.read(i_pos - 1, i_pos - 3)));
                        }
                        if i_pos > ilo + 3 {
                            tst1 = tst1.faer_add(abs1(a.read(i_pos - 1, i_pos - 4)));
                        }
                        if i_pos < ihi - 1 {
                            tst1 = tst1.faer_add(abs1(a.read(i_pos + 1, i_pos)));
                        }
                        if i_pos < ihi - 2 {
                            tst1 = tst1.faer_add(abs1(a.read(i_pos + 2, i_pos)));
                        }
                        if i_pos < ihi - 3 {
                            tst1 = tst1.faer_add(abs1(a.read(i_pos + 3, i_pos)));
                        }
                    }
                    if abs1(a.read(i_pos, i_pos - 1)) < max(small_num, eps.faer_mul(tst1)) {
                        let ab = max(
                            abs1(a.read(i_pos, i_pos - 1)),
                            abs1(a.read(i_pos - 1, i_pos)),
                        );
                        let ba = min(
                            abs1(a.read(i_pos, i_pos - 1)),
                            abs1(a.read(i_pos - 1, i_pos)),
                        );
                        let aa = max(
                            abs1(a.read(i_pos, i_pos)),
                            abs1(a.read(i_pos, i_pos).faer_sub(a.read(i_pos - 1, i_pos - 1))),
                        );
                        let bb = min(
                            abs1(a.read(i_pos, i_pos)),
                            abs1(a.read(i_pos, i_pos).faer_sub(a.read(i_pos - 1, i_pos - 1))),
                        );
                        let s = aa.faer_add(ab);
                        if ba.faer_mul(ab.faer_div(s))
                            <= max(small_num, eps.faer_mul(bb.faer_mul(aa.faer_div(s))))
                        {
                            a.write(i_pos, i_pos - 1, E::faer_zero());
                        }
                    }
                }
            }

            // Delayed update from the left
            for i_bulge in 0..n_active_bulges {
                let i_pos = i_pos_last - 2 * i_bulge;
                let v = v.rb_mut().col_mut(i_bulge);

                let v0 = v.read(0).faer_real();
                let v1 = v.read(1);
                let v2 = v.read(2);

                for j in i_pos + 1..istop_m {
                    let sum = a
                        .read(i_pos, j)
                        .faer_add(v1.faer_conj().faer_mul(a.read(i_pos + 1, j)))
                        .faer_add(v2.faer_conj().faer_mul(a.read(i_pos + 2, j)));
                    a.write(i_pos, j, a.read(i_pos, j).faer_sub(sum.faer_scale_real(v0)));
                    a.write(
                        i_pos + 1,
                        j,
                        a.read(i_pos + 1, j)
                            .faer_sub(sum.faer_scale_real(v0).faer_mul(v1)),
                    );
                    a.write(
                        i_pos + 2,
                        j,
                        a.read(i_pos + 2, j)
                            .faer_sub(sum.faer_scale_real(v0).faer_mul(v2)),
                    );
                }
            }

            // Accumulate the reflectors into U
            for i_bulge in 0..n_active_bulges {
                let i_pos = i_pos_last - 2 * i_bulge;
                let v = v.rb_mut().col_mut(i_bulge);

                let v0 = v.read(0).faer_real();
                let v1 = v.read(1);
                let v2 = v.read(2);

                let i1 = 0;
                let i2 = Ord::min(u2.nrows(), (i_pos_last - ilo) + (i_pos_last - ilo) + 3);

                for j in i1..i2 {
                    let sum = u2
                        .read(j, i_pos - ilo)
                        .faer_add(v1.faer_mul(u2.read(j, i_pos - ilo + 1)))
                        .faer_add(v2.faer_mul(u2.read(j, i_pos - ilo + 2)));

                    u2.write(
                        j,
                        i_pos - ilo,
                        u2.read(j, i_pos - ilo).faer_sub(sum.faer_scale_real(v0)),
                    );
                    u2.write(
                        j,
                        i_pos - ilo + 1,
                        u2.read(j, i_pos - ilo + 1)
                            .faer_sub(sum.faer_scale_real(v0).faer_mul(v1.faer_conj())),
                    );
                    u2.write(
                        j,
                        i_pos - ilo + 2,
                        u2.read(j, i_pos - ilo + 2)
                            .faer_sub(sum.faer_scale_real(v0).faer_mul(v2.faer_conj())),
                    );
                }
            }
        }

        // Update rest of the matrix
        if want_t {
            istart_m = 0;
            istop_m = n;
        } else {
            istart_m = ilo;
            istop_m = ihi;
        }

        // Horizontal multiply
        if ilo + n_shifts + 1 < istop_m {
            let mut i = ilo + n_block;
            while i < istop_m {
                let iblock = Ord::min(istop_m - i, wh.ncols());
                let mut a_slice = a.rb_mut().submatrix_mut(ilo, i, n_block, iblock);
                let mut wh_slice =
                    wh.rb_mut()
                        .submatrix_mut(0, 0, a_slice.nrows(), a_slice.ncols());
                matmul(
                    wh_slice.rb_mut(),
                    u2.rb().adjoint(),
                    a_slice.rb(),
                    None,
                    E::faer_one(),
                    parallelism,
                );
                a_slice.copy_from(wh_slice.rb());
                i += iblock;
            }
        }
        // Vertical multiply
        if istart_m < ilo {
            let mut i = istart_m;
            while i < ilo {
                let iblock = Ord::min(ilo - i, wv.nrows());
                let mut a_slice = a.rb_mut().submatrix_mut(i, ilo, iblock, n_block);
                let mut wv_slice =
                    wv.rb_mut()
                        .submatrix_mut(0, 0, a_slice.nrows(), a_slice.ncols());
                matmul(
                    wv_slice.rb_mut(),
                    a_slice.rb(),
                    u2.rb(),
                    None,
                    E::faer_one(),
                    parallelism,
                );
                a_slice.copy_from(wv_slice.rb());
                i += iblock;
            }
        }
        // Update Z (also a vertical multiplication)
        if let Some(mut z) = z.rb_mut() {
            let mut i = 0;
            while i < n {
                let iblock = Ord::min(n - i, wv.nrows());
                let mut z_slice = z.rb_mut().submatrix_mut(i, ilo, iblock, n_block);
                let mut wv_slice =
                    wv.rb_mut()
                        .submatrix_mut(0, 0, z_slice.nrows(), z_slice.ncols());
                matmul(
                    wv_slice.rb_mut(),
                    z_slice.rb(),
                    u2.rb(),
                    None,
                    E::faer_one(),
                    parallelism,
                );
                z_slice.copy_from(wv_slice.rb());
                i += iblock;
            }
        }

        i_pos_block = ilo + n_block - n_shifts;
    }

    //
    // The following code block moves the bulges down until they are low enough
    // to be removed
    //
    while i_pos_block + n_block_desired < ihi {
        // Number of positions each bulge will be moved down
        let n_pos = Ord::min(n_block_desired - n_shifts, ihi - n_shifts - 1 - i_pos_block);
        // Actual blocksize
        let n_block = n_shifts + n_pos;

        let mut u2 = u.rb_mut().submatrix_mut(0, 0, n_block, n_block);
        u2.fill_zero();
        u2.rb_mut()
            .diagonal_mut()
            .column_vector_mut()
            .fill(E::faer_one());

        // Near-the-diagonal bulge chase
        // The calculations are initially limited to the window:
        // A(i_pos_block-1:i_pos_block+n_block,i_pos_block:i_pos_block+n_block)
        // The rest is updated later via level 3 BLAS
        let mut istart_m = i_pos_block;
        let mut istop_m = i_pos_block + n_block;

        for i_pos_last in i_pos_block + n_shifts - 2..i_pos_block + n_shifts - 2 + n_pos {
            for i_bulge in 0..n_bulges {
                let i_pos = i_pos_last - 2 * i_bulge;
                let mut v = v.rb_mut().col_mut(i_bulge);

                // Chase bulge down
                let mut h = a.rb_mut().submatrix_mut(i_pos - 1, i_pos - 1, 4, 4);
                let s1 = s.read(s.nrows() - 1 - 2 * i_bulge);
                let s2 = s.read(s.nrows() - 1 - 2 * i_bulge - 1);
                move_bulge(h.rb_mut(), v.rb_mut(), s1, s2, epsilon);

                // Apply the reflector we just calculated from the right
                // We leave the last row for later (it interferes with the
                // optimally packed bulges)

                let v0 = v.read(0).faer_real();
                let v1 = v.read(1);
                let v2 = v.read(2);

                for j in istart_m..i_pos + 3 {
                    let sum = a
                        .read(j, i_pos)
                        .faer_add(v1.faer_mul(a.read(j, i_pos + 1)))
                        .faer_add(v2.faer_mul(a.read(j, i_pos + 2)));
                    a.write(j, i_pos, a.read(j, i_pos).faer_sub(sum.faer_scale_real(v0)));
                    a.write(
                        j,
                        i_pos + 1,
                        a.read(j, i_pos + 1)
                            .faer_sub(sum.faer_scale_real(v0).faer_mul(v1.faer_conj())),
                    );
                    a.write(
                        j,
                        i_pos + 2,
                        a.read(j, i_pos + 2)
                            .faer_sub(sum.faer_scale_real(v0).faer_mul(v2.faer_conj())),
                    );
                }

                // Apply the reflector we just calculated from the left
                // We only update a single column, the rest is updated later
                let sum = a
                    .read(i_pos, i_pos)
                    .faer_add(v1.faer_conj().faer_mul(a.read(i_pos + 1, i_pos)))
                    .faer_add(v2.faer_conj().faer_mul(a.read(i_pos + 2, i_pos)));
                a.write(
                    i_pos,
                    i_pos,
                    a.read(i_pos, i_pos).faer_sub(sum.faer_scale_real(v0)),
                );
                a.write(
                    i_pos + 1,
                    i_pos,
                    a.read(i_pos + 1, i_pos)
                        .faer_sub(sum.faer_scale_real(v0).faer_mul(v1)),
                );
                a.write(
                    i_pos + 2,
                    i_pos,
                    a.read(i_pos + 2, i_pos)
                        .faer_sub(sum.faer_scale_real(v0).faer_mul(v2)),
                );

                // Test for deflation.
                if (i_pos > ilo) && (a.read(i_pos, i_pos - 1) != E::faer_zero()) {
                    let mut tst1 =
                        abs1(a.read(i_pos - 1, i_pos - 1)).faer_add(abs1(a.read(i_pos, i_pos)));
                    if tst1 == E::Real::faer_zero() {
                        if i_pos > ilo + 1 {
                            tst1 = tst1.faer_add(abs1(a.read(i_pos - 1, i_pos - 2)));
                        }
                        if i_pos > ilo + 2 {
                            tst1 = tst1.faer_add(abs1(a.read(i_pos - 1, i_pos - 3)));
                        }
                        if i_pos > ilo + 3 {
                            tst1 = tst1.faer_add(abs1(a.read(i_pos - 1, i_pos - 4)));
                        }
                        if i_pos < ihi - 1 {
                            tst1 = tst1.faer_add(abs1(a.read(i_pos + 1, i_pos)));
                        }
                        if i_pos < ihi - 2 {
                            tst1 = tst1.faer_add(abs1(a.read(i_pos + 2, i_pos)));
                        }
                        if i_pos < ihi - 3 {
                            tst1 = tst1.faer_add(abs1(a.read(i_pos + 3, i_pos)));
                        }
                    }
                    if abs1(a.read(i_pos, i_pos - 1)) < max(small_num, eps.faer_mul(tst1)) {
                        let ab = max(
                            abs1(a.read(i_pos, i_pos - 1)),
                            abs1(a.read(i_pos - 1, i_pos)),
                        );
                        let ba = min(
                            abs1(a.read(i_pos, i_pos - 1)),
                            abs1(a.read(i_pos - 1, i_pos)),
                        );
                        let aa = max(
                            abs1(a.read(i_pos, i_pos)),
                            abs1(a.read(i_pos, i_pos).faer_sub(a.read(i_pos - 1, i_pos - 1))),
                        );
                        let bb = min(
                            abs1(a.read(i_pos, i_pos)),
                            abs1(a.read(i_pos, i_pos).faer_sub(a.read(i_pos - 1, i_pos - 1))),
                        );
                        let s = aa.faer_add(ab);
                        if ba.faer_mul(ab.faer_div(s))
                            <= max(small_num, eps.faer_mul(bb.faer_mul(aa.faer_div(s))))
                        {
                            a.write(i_pos, i_pos - 1, E::faer_zero());
                        }
                    }
                }
            }

            // Delayed update from the left
            for i_bulge in 0..n_bulges {
                let i_pos = i_pos_last - 2 * i_bulge;
                let v = v.rb_mut().col_mut(i_bulge);

                let v0 = v.read(0).faer_real();
                let v1 = v.read(1);
                let v2 = v.read(2);

                for j in i_pos + 1..istop_m {
                    let sum = a
                        .read(i_pos, j)
                        .faer_add(v1.faer_conj().faer_mul(a.read(i_pos + 1, j)))
                        .faer_add(v2.faer_conj().faer_mul(a.read(i_pos + 2, j)));
                    a.write(i_pos, j, a.read(i_pos, j).faer_sub(sum.faer_scale_real(v0)));
                    a.write(
                        i_pos + 1,
                        j,
                        a.read(i_pos + 1, j)
                            .faer_sub(sum.faer_scale_real(v0).faer_mul(v1)),
                    );
                    a.write(
                        i_pos + 2,
                        j,
                        a.read(i_pos + 2, j)
                            .faer_sub(sum.faer_scale_real(v0).faer_mul(v2)),
                    );
                }
            }

            // Accumulate the reflectors into U
            for i_bulge in 0..n_bulges {
                let i_pos = i_pos_last - 2 * i_bulge;
                let v = v.rb_mut().col_mut(i_bulge);

                let v0 = v.read(0).faer_real();
                let v1 = v.read(1);
                let v2 = v.read(2);

                let i1 = (i_pos - i_pos_block) - (i_pos_last + 2 - i_pos_block - n_shifts);
                let i2 = Ord::min(
                    u2.nrows(),
                    (i_pos_last - i_pos_block) + (i_pos_last + 2 - i_pos_block - n_shifts) + 3,
                );

                for j in i1..i2 {
                    let sum = u2
                        .read(j, i_pos - i_pos_block)
                        .faer_add(v1.faer_mul(u2.read(j, i_pos - i_pos_block + 1)))
                        .faer_add(v2.faer_mul(u2.read(j, i_pos - i_pos_block + 2)));

                    u2.write(
                        j,
                        i_pos - i_pos_block,
                        u2.read(j, i_pos - i_pos_block)
                            .faer_sub(sum.faer_scale_real(v0)),
                    );
                    u2.write(
                        j,
                        i_pos - i_pos_block + 1,
                        u2.read(j, i_pos - i_pos_block + 1)
                            .faer_sub(sum.faer_scale_real(v0).faer_mul(v1.faer_conj())),
                    );
                    u2.write(
                        j,
                        i_pos - i_pos_block + 2,
                        u2.read(j, i_pos - i_pos_block + 2)
                            .faer_sub(sum.faer_scale_real(v0).faer_mul(v2.faer_conj())),
                    );
                }
            }
        }

        // Update rest of the matrix
        if want_t {
            istart_m = 0;
            istop_m = n;
        } else {
            istart_m = ilo;
            istop_m = ihi;
        }

        // Horizontal multiply
        if i_pos_block + n_block < istop_m {
            let mut i = i_pos_block + n_block;
            while i < istop_m {
                let iblock = Ord::min(istop_m - i, wh.ncols());
                let mut a_slice = a.rb_mut().submatrix_mut(i_pos_block, i, n_block, iblock);
                let mut wh_slice =
                    wh.rb_mut()
                        .submatrix_mut(0, 0, a_slice.nrows(), a_slice.ncols());
                matmul(
                    wh_slice.rb_mut(),
                    u2.rb().adjoint(),
                    a_slice.rb(),
                    None,
                    E::faer_one(),
                    parallelism,
                );
                a_slice.copy_from(wh_slice.rb());
                i += iblock;
            }
        }

        // Vertical multiply
        if istart_m < i_pos_block {
            let mut i = istart_m;
            while i < i_pos_block {
                let iblock = Ord::min(i_pos_block - i, wv.nrows());
                let mut a_slice = a.rb_mut().submatrix_mut(i, i_pos_block, iblock, n_block);
                let mut wv_slice =
                    wv.rb_mut()
                        .submatrix_mut(0, 0, a_slice.nrows(), a_slice.ncols());
                matmul(
                    wv_slice.rb_mut(),
                    a_slice.rb(),
                    u2.rb(),
                    None,
                    E::faer_one(),
                    parallelism,
                );
                a_slice.copy_from(wv_slice.rb());
                i += iblock;
            }
        }
        // Update Z (also a vertical multiplication)
        if let Some(mut z) = z.rb_mut() {
            let mut i = 0;
            while i < n {
                let iblock = Ord::min(n - i, wv.nrows());
                let mut z_slice = z.rb_mut().submatrix_mut(i, i_pos_block, iblock, n_block);
                let mut wv_slice =
                    wv.rb_mut()
                        .submatrix_mut(0, 0, z_slice.nrows(), z_slice.ncols());
                matmul(
                    wv_slice.rb_mut(),
                    z_slice.rb(),
                    u2.rb(),
                    None,
                    E::faer_one(),
                    parallelism,
                );
                z_slice.copy_from(wv_slice.rb());
                i += iblock;
            }
        }

        i_pos_block += n_pos;
    }

    //
    // The following code removes the bulges from the matrix
    //
    {
        let n_block = ihi - i_pos_block;

        let mut u2 = u.rb_mut().submatrix_mut(0, 0, n_block, n_block);
        u2.fill_zero();
        u2.rb_mut()
            .diagonal_mut()
            .column_vector_mut()
            .fill(E::faer_one());

        // Near-the-diagonal bulge chase
        // The calculations are initially limited to the window:
        // A(i_pos_block-1:ihi,i_pos_block:ihi) The rest is updated later via
        // level 3 BLAS
        let mut istart_m = i_pos_block;
        let mut istop_m = ihi;

        for i_pos_last in i_pos_block + n_shifts - 2..ihi + n_shifts - 1 {
            let mut i_bulge_start = if i_pos_last + 3 > ihi {
                (i_pos_last + 3 - ihi) / 2
            } else {
                0
            };

            for i_bulge in i_bulge_start..n_bulges {
                let i_pos = i_pos_last - 2 * i_bulge;
                if i_pos == ihi - 2 {
                    // Special case, the bulge is at the bottom, needs a smaller
                    // reflector (order 2)
                    let mut v = v.rb_mut().subrows_mut(0, 2).col_mut(i_bulge);
                    let mut h = a.rb_mut().subrows_mut(i_pos, 2).col_mut(i_pos - 1);
                    let head = h.read(0);
                    let tail = h.rb_mut().subrows_mut(1, 1);
                    let tail_norm = tail.rb().norm_l2();
                    let (tau, beta) = make_householder_in_place(Some(tail), head, tail_norm);
                    v.write(0, tau.faer_inv());
                    v.write(1, h.read(1));
                    h.write(0, beta);
                    h.write(1, E::faer_zero());

                    let t0 = v.read(0).faer_conj();
                    let v1 = v.read(1);
                    let t1 = t0.faer_mul(v1);
                    // Apply the reflector we just calculated from the right
                    for j in istart_m..i_pos + 2 {
                        let sum = a.read(j, i_pos).faer_add(v1.faer_mul(a.read(j, i_pos + 1)));
                        a.write(
                            j,
                            i_pos,
                            a.read(j, i_pos).faer_sub(sum.faer_mul(t0.faer_conj())),
                        );
                        a.write(
                            j,
                            i_pos + 1,
                            a.read(j, i_pos + 1).faer_sub(sum.faer_mul(t1.faer_conj())),
                        );
                    }
                    // Apply the reflector we just calculated from the left
                    for j in i_pos..istop_m {
                        let sum = a
                            .read(i_pos, j)
                            .faer_add(v1.faer_conj().faer_mul(a.read(i_pos + 1, j)));
                        a.write(i_pos, j, a.read(i_pos, j).faer_sub(sum.faer_mul(t0)));
                        a.write(
                            i_pos + 1,
                            j,
                            a.read(i_pos + 1, j).faer_sub(sum.faer_mul(t1)),
                        );
                    }
                    // Accumulate the reflector into U
                    // The loop bounds should be changed to reflect the fact
                    // that U2 starts off as diagonal
                    for j in 0..u2.nrows() {
                        let sum = u2
                            .read(j, i_pos - i_pos_block)
                            .faer_add(v1.faer_mul(u2.read(j, i_pos - i_pos_block + 1)));
                        u2.write(
                            j,
                            i_pos - i_pos_block,
                            u2.read(j, i_pos - i_pos_block)
                                .faer_sub(sum.faer_mul(t0.faer_conj())),
                        );
                        u2.write(
                            j,
                            i_pos - i_pos_block + 1,
                            u2.read(j, i_pos - i_pos_block + 1)
                                .faer_sub(sum.faer_mul(t1.faer_conj())),
                        );
                    }
                } else {
                    let mut v = v.rb_mut().col_mut(i_bulge);
                    let mut h = a.rb_mut().submatrix_mut(i_pos - 1, i_pos - 1, 4, 4);
                    let s1 = s.read(s.nrows() - 1 - 2 * i_bulge);
                    let s2 = s.read(s.nrows() - 1 - 2 * i_bulge - 1);
                    move_bulge(h.rb_mut(), v.rb_mut(), s1, s2, epsilon);

                    {
                        let t0 = v.read(0).faer_conj();
                        let v1 = v.read(1);
                        let t1 = t0.faer_mul(v1);
                        let v2 = v.read(2);
                        let t2 = t0.faer_mul(v2);
                        // Apply the reflector we just calculated from the right
                        // (but leave the last row for later)
                        for j in istart_m..i_pos + 3 {
                            let sum = a
                                .read(j, i_pos)
                                .faer_add(v1.faer_mul(a.read(j, i_pos + 1)))
                                .faer_add(v2.faer_mul(a.read(j, i_pos + 2)));
                            a.write(
                                j,
                                i_pos,
                                a.read(j, i_pos).faer_sub(sum.faer_mul(t0.faer_conj())),
                            );
                            a.write(
                                j,
                                i_pos + 1,
                                a.read(j, i_pos + 1).faer_sub(sum.faer_mul(t1.faer_conj())),
                            );
                            a.write(
                                j,
                                i_pos + 2,
                                a.read(j, i_pos + 2).faer_sub(sum.faer_mul(t2.faer_conj())),
                            );
                        }
                    }

                    let v0 = v.read(0).faer_real();
                    let v1 = v.read(1);
                    let v2 = v.read(2);
                    // Apply the reflector we just calculated from the left
                    // We only update a single column, the rest is updated later
                    let sum = a
                        .read(i_pos, i_pos)
                        .faer_add(v1.faer_conj().faer_mul(a.read(i_pos + 1, i_pos)))
                        .faer_add(v2.faer_conj().faer_mul(a.read(i_pos + 2, i_pos)));
                    a.write(
                        i_pos,
                        i_pos,
                        a.read(i_pos, i_pos).faer_sub(sum.faer_scale_real(v0)),
                    );
                    a.write(
                        i_pos + 1,
                        i_pos,
                        a.read(i_pos + 1, i_pos)
                            .faer_sub(sum.faer_scale_real(v0).faer_mul(v1)),
                    );
                    a.write(
                        i_pos + 2,
                        i_pos,
                        a.read(i_pos + 2, i_pos)
                            .faer_sub(sum.faer_scale_real(v0).faer_mul(v2)),
                    );

                    // Test for deflation.
                    if (i_pos > ilo) && (a.read(i_pos, i_pos - 1) != E::faer_zero()) {
                        let mut tst1 =
                            abs1(a.read(i_pos - 1, i_pos - 1)).faer_add(abs1(a.read(i_pos, i_pos)));
                        if tst1 == E::Real::faer_zero() {
                            if i_pos > ilo + 1 {
                                tst1 = tst1.faer_add(abs1(a.read(i_pos - 1, i_pos - 2)));
                            }
                            if i_pos > ilo + 2 {
                                tst1 = tst1.faer_add(abs1(a.read(i_pos - 1, i_pos - 3)));
                            }
                            if i_pos > ilo + 3 {
                                tst1 = tst1.faer_add(abs1(a.read(i_pos - 1, i_pos - 4)));
                            }
                            if i_pos < ihi - 1 {
                                tst1 = tst1.faer_add(abs1(a.read(i_pos + 1, i_pos)));
                            }
                            if i_pos < ihi - 2 {
                                tst1 = tst1.faer_add(abs1(a.read(i_pos + 2, i_pos)));
                            }
                            if i_pos < ihi - 3 {
                                tst1 = tst1.faer_add(abs1(a.read(i_pos + 3, i_pos)));
                            }
                        }
                        if abs1(a.read(i_pos, i_pos - 1)) < max(small_num, eps.faer_mul(tst1)) {
                            let ab = max(
                                abs1(a.read(i_pos, i_pos - 1)),
                                abs1(a.read(i_pos - 1, i_pos)),
                            );
                            let ba = min(
                                abs1(a.read(i_pos, i_pos - 1)),
                                abs1(a.read(i_pos - 1, i_pos)),
                            );
                            let aa = max(
                                abs1(a.read(i_pos, i_pos)),
                                abs1(a.read(i_pos, i_pos).faer_sub(a.read(i_pos - 1, i_pos - 1))),
                            );
                            let bb = min(
                                abs1(a.read(i_pos, i_pos)),
                                abs1(a.read(i_pos, i_pos).faer_sub(a.read(i_pos - 1, i_pos - 1))),
                            );
                            let s = aa.faer_add(ab);
                            if ba.faer_mul(ab.faer_div(s))
                                <= max(small_num, eps.faer_mul(bb.faer_mul(aa.faer_div(s))))
                            {
                                a.write(i_pos, i_pos - 1, E::faer_zero());
                            }
                        }
                    }
                }
            }

            i_bulge_start = if i_pos_last + 4 > ihi {
                (i_pos_last + 4 - ihi) / 2
            } else {
                0
            };

            // Delayed update from the left
            for i_bulge in i_bulge_start..n_bulges {
                let i_pos = i_pos_last - 2 * i_bulge;
                let v = v.rb_mut().col_mut(i_bulge);

                let v0 = v.read(0).faer_real();
                let v1 = v.read(1);
                let v2 = v.read(2);

                for j in i_pos + 1..istop_m {
                    let sum = a
                        .read(i_pos, j)
                        .faer_add(v1.faer_conj().faer_mul(a.read(i_pos + 1, j)))
                        .faer_add(v2.faer_conj().faer_mul(a.read(i_pos + 2, j)));
                    a.write(i_pos, j, a.read(i_pos, j).faer_sub(sum.faer_scale_real(v0)));
                    a.write(
                        i_pos + 1,
                        j,
                        a.read(i_pos + 1, j)
                            .faer_sub(sum.faer_scale_real(v0).faer_mul(v1)),
                    );
                    a.write(
                        i_pos + 2,
                        j,
                        a.read(i_pos + 2, j)
                            .faer_sub(sum.faer_scale_real(v0).faer_mul(v2)),
                    );
                }
            }

            // Accumulate the reflectors into U
            for i_bulge in i_bulge_start..n_bulges {
                let i_pos = i_pos_last - 2 * i_bulge;
                let v = v.rb_mut().col_mut(i_bulge);

                let v0 = v.read(0).faer_real();
                let v1 = v.read(1);
                let v2 = v.read(2);

                let i1 = (i_pos - i_pos_block) - (i_pos_last + 2 - i_pos_block - n_shifts);
                let i2 = Ord::min(
                    u2.nrows(),
                    (i_pos_last - i_pos_block) + (i_pos_last + 2 - i_pos_block - n_shifts) + 3,
                );

                for j in i1..i2 {
                    let sum = u2
                        .read(j, i_pos - i_pos_block)
                        .faer_add(v1.faer_mul(u2.read(j, i_pos - i_pos_block + 1)))
                        .faer_add(v2.faer_mul(u2.read(j, i_pos - i_pos_block + 2)));

                    u2.write(
                        j,
                        i_pos - i_pos_block,
                        u2.read(j, i_pos - i_pos_block)
                            .faer_sub(sum.faer_scale_real(v0)),
                    );
                    u2.write(
                        j,
                        i_pos - i_pos_block + 1,
                        u2.read(j, i_pos - i_pos_block + 1)
                            .faer_sub(sum.faer_scale_real(v0).faer_mul(v1.faer_conj())),
                    );
                    u2.write(
                        j,
                        i_pos - i_pos_block + 2,
                        u2.read(j, i_pos - i_pos_block + 2)
                            .faer_sub(sum.faer_scale_real(v0).faer_mul(v2.faer_conj())),
                    );
                }
            }
        }

        // Update rest of the matrix
        if want_t {
            istart_m = 0;
            istop_m = n;
        } else {
            istart_m = ilo;
            istop_m = ihi;
        }

        debug_assert!(i_pos_block + n_block == ihi);

        // Horizontal multiply
        if ihi < istop_m {
            let mut i = ihi;
            while i < istop_m {
                let iblock = Ord::min(istop_m - i, wh.ncols());
                let mut a_slice = a.rb_mut().submatrix_mut(i_pos_block, i, n_block, iblock);
                let mut wh_slice =
                    wh.rb_mut()
                        .submatrix_mut(0, 0, a_slice.nrows(), a_slice.ncols());
                matmul(
                    wh_slice.rb_mut(),
                    u2.rb().adjoint(),
                    a_slice.rb(),
                    None,
                    E::faer_one(),
                    parallelism,
                );
                a_slice.copy_from(wh_slice.rb());
                i += iblock;
            }
        }

        // Vertical multiply
        if istart_m < i_pos_block {
            let mut i = istart_m;
            while i < i_pos_block {
                let iblock = Ord::min(i_pos_block - i, wv.nrows());
                let mut a_slice = a.rb_mut().submatrix_mut(i, i_pos_block, iblock, n_block);
                let mut wv_slice =
                    wv.rb_mut()
                        .submatrix_mut(0, 0, a_slice.nrows(), a_slice.ncols());
                matmul(
                    wv_slice.rb_mut(),
                    a_slice.rb(),
                    u2.rb(),
                    None,
                    E::faer_one(),
                    parallelism,
                );
                a_slice.copy_from(wv_slice.rb());
                i += iblock;
            }
        }
        // Update Z (also a vertical multiplication)
        if let Some(mut z) = z.rb_mut() {
            let mut i = 0;
            while i < n {
                let iblock = Ord::min(n - i, wv.nrows());
                let mut z_slice = z.rb_mut().submatrix_mut(i, i_pos_block, iblock, n_block);
                let mut wv_slice =
                    wv.rb_mut()
                        .submatrix_mut(0, 0, z_slice.nrows(), z_slice.ncols());
                matmul(
                    wv_slice.rb_mut(),
                    z_slice.rb(),
                    u2.rb(),
                    None,
                    E::faer_one(),
                    parallelism,
                );
                z_slice.copy_from(wv_slice.rb());
                i += iblock;
            }
        }
    }
}

#[cfg(test)]
mod tests {
    use super::*;
    use crate::{assert, complex_native::c64, mat, Col, Mat};
    use assert_approx_eq::assert_approx_eq;

    macro_rules! make_stack {
        ($req: expr $(,)?) => {
            ::dyn_stack::PodStack::new(&mut ::dyn_stack::GlobalPodBuffer::new($req.unwrap()))
        };
    }

    #[test]
    fn test_3() {
        let n = 3;
        let h = mat![
            [
                c64::new(0.997386, 0.677592),
                c64::new(0.646064, 0.936948),
                c64::new(0.090948, 0.674011),
            ],
            [
                c64::new(0.212396, 0.976794),
                c64::new(0.460270, 0.926436),
                c64::new(0.494441, 0.888187),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.616652, 0.840012),
                c64::new(0.768245, 0.349193),
            ],
        ];

        let mut q = Mat::from_fn(n, n, |i, j| {
            if i == j {
                c64::faer_one()
            } else {
                c64::faer_zero()
            }
        });
        let mut w = Col::zeros(n);
        let mut t = h.clone();
        lahqr(
            true,
            t.as_mut(),
            Some(q.as_mut()),
            w.as_mut(),
            0,
            n,
            f64::EPSILON,
            f64::MIN_POSITIVE,
        );

        let h_reconstructed = &q * &t * q.adjoint();

        for i in 0..n {
            for j in 0..n {
                assert_approx_eq!(h_reconstructed.read(i, j), h.read(i, j));
            }
        }
    }

    #[test]
    fn test_n() {
        for n in [1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 16, 32, 128] {
            for _ in 0..10 {
                let mut h = Mat::<c64>::zeros(n, n);
                for j in 0..n {
                    for i in 0..n {
                        if i <= j + 1 {
                            h.write(i, j, c64::new(rand::random(), rand::random()));
                        }
                    }
                }

                let mut q = Mat::from_fn(n, n, |i, j| {
                    if i == j {
                        c64::faer_one()
                    } else {
                        c64::faer_zero()
                    }
                });

                let mut w = Col::zeros(n);

                let mut t = h.clone();
                lahqr(
                    true,
                    t.as_mut(),
                    Some(q.as_mut()),
                    w.as_mut(),
                    0,
                    n,
                    f64::EPSILON,
                    f64::MIN_POSITIVE,
                );
                dbgf::dbgf!("6.6?", &t, &h);

                let h_reconstructed = &q * &t * q.adjoint();

                for i in 0..n {
                    for j in 0..n {
                        assert_approx_eq!(h_reconstructed.read(i, j), h.read(i, j));
                    }
                }
            }
        }
    }

    #[test]
    fn test_multi_n() {
        for n in [
            1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 16, 32, 63, 64, 65, 128, 256,
        ] {
            for _ in 0..10 {
                let mut h = Mat::<c64>::zeros(n, n);
                for j in 0..n {
                    for i in 0..n {
                        if i <= j + 1 {
                            h.write(i, j, c64::new(rand::random(), rand::random()));
                        }
                    }
                }

                let mut q = Mat::from_fn(n, n, |i, j| {
                    if i == j {
                        c64::faer_one()
                    } else {
                        c64::faer_zero()
                    }
                });

                let mut w = Col::zeros(n);

                let mut t = h.clone();
                let params = EvdParams {
                    recommended_shift_count: None,
                    recommended_deflation_window: None,
                    blocking_threshold: Some(15),
                    nibble_threshold: Some(14),
                };
                multishift_qr(
                    true,
                    t.as_mut(),
                    Some(q.as_mut()),
                    w.as_mut(),
                    0,
                    n,
                    f64::EPSILON,
                    f64::MIN_POSITIVE,
                    Parallelism::None,
                    make_stack!(multishift_qr_req::<c64>(
                        n,
                        n,
                        true,
                        true,
                        Parallelism::None,
                        params,
                    )),
                    params,
                );
                for j in 0..n {
                    for i in j + 1..n {
                        t.write(i, j, c64::faer_zero());
                    }
                }
                dbgf::dbgf!("6.6?", &t, &h);

                let h_reconstructed = &q * &t * q.adjoint();

                for i in 0..n {
                    for j in 0..n {
                        assert_approx_eq!(h_reconstructed.read(i, j), h.read(i, j));
                    }
                }
            }
        }
    }

    #[test]
    fn test_multi_100() {
        let n = 100;

        let h = [
            [
                c64::new(0.247362, 0.915457),
                c64::new(0.729822, 0.217719),
                c64::new(0.557421, 0.288464),
                c64::new(0.328797, 0.239523),
                c64::new(0.777550, 0.831544),
                c64::new(0.987836, 0.172338),
                c64::new(0.398255, 0.340496),
                c64::new(0.895207, 0.209341),
                c64::new(0.514276, 0.133539),
                c64::new(0.112494, 0.772690),
                c64::new(0.794636, 0.624804),
                c64::new(0.764424, 0.189969),
                c64::new(0.448585, 0.845465),
                c64::new(0.648511, 0.998682),
                c64::new(0.761988, 0.891177),
                c64::new(0.248977, 0.098031),
                c64::new(0.251652, 0.303864),
                c64::new(0.341391, 0.737074),
                c64::new(0.724391, 0.632790),
                c64::new(0.644896, 0.231057),
                c64::new(0.824051, 0.613854),
                c64::new(0.716392, 0.677729),
                c64::new(0.098836, 0.500676),
                c64::new(0.011666, 0.426799),
                c64::new(0.879832, 0.626437),
                c64::new(0.898294, 0.204360),
                c64::new(0.305867, 0.437079),
                c64::new(0.335694, 0.013785),
                c64::new(0.963588, 0.221925),
                c64::new(0.572279, 0.201790),
                c64::new(0.157022, 0.485770),
                c64::new(0.002149, 0.028072),
                c64::new(0.464055, 0.223266),
                c64::new(0.483257, 0.789599),
                c64::new(0.443072, 0.964070),
                c64::new(0.231826, 0.472352),
                c64::new(0.059165, 0.681934),
                c64::new(0.295935, 0.702026),
                c64::new(0.049729, 0.483953),
                c64::new(0.442269, 0.661843),
                c64::new(0.774934, 0.140070),
                c64::new(0.102210, 0.834111),
                c64::new(0.326798, 0.629013),
                c64::new(0.724024, 0.877097),
                c64::new(0.631407, 0.953489),
                c64::new(0.108545, 0.371405),
                c64::new(0.937458, 0.968946),
                c64::new(0.160439, 0.886438),
                c64::new(0.347411, 0.151744),
                c64::new(0.108679, 0.783657),
                c64::new(0.862349, 0.605088),
                c64::new(0.071644, 0.888313),
                c64::new(0.167642, 0.853602),
                c64::new(0.971924, 0.432988),
                c64::new(0.265114, 0.244212),
                c64::new(0.165003, 0.412567),
                c64::new(0.930197, 0.307039),
                c64::new(0.013761, 0.187137),
                c64::new(0.439960, 0.553093),
                c64::new(0.909387, 0.800205),
                c64::new(0.301502, 0.869243),
                c64::new(0.347811, 0.384708),
                c64::new(0.941104, 0.925138),
                c64::new(0.576176, 0.184279),
                c64::new(0.568036, 0.388525),
                c64::new(0.566769, 0.536556),
                c64::new(0.058159, 0.243945),
                c64::new(0.221479, 0.495539),
                c64::new(0.111847, 0.004560),
                c64::new(0.781100, 0.170008),
                c64::new(0.039433, 0.479257),
                c64::new(0.850582, 0.896525),
                c64::new(0.219405, 0.923765),
                c64::new(0.725664, 0.321608),
                c64::new(0.252339, 0.069854),
                c64::new(0.674924, 0.327177),
                c64::new(0.735469, 0.565386),
                c64::new(0.568203, 0.177555),
                c64::new(0.842474, 0.661287),
                c64::new(0.060288, 0.816681),
                c64::new(0.255325, 0.462335),
                c64::new(0.856666, 0.234204),
                c64::new(0.157914, 0.342594),
                c64::new(0.191756, 0.858922),
                c64::new(0.761656, 0.116901),
                c64::new(0.199292, 0.477377),
                c64::new(0.435394, 0.823778),
                c64::new(0.363253, 0.843141),
                c64::new(0.618471, 0.196208),
                c64::new(0.571452, 0.087156),
                c64::new(0.967526, 0.131176),
                c64::new(0.053565, 0.173195),
                c64::new(0.514817, 0.215551),
                c64::new(0.197349, 0.887501),
                c64::new(0.678657, 0.991695),
                c64::new(0.081140, 0.604964),
                c64::new(0.675069, 0.545161),
                c64::new(0.322368, 0.150800),
                c64::new(0.272257, 0.873348),
                c64::new(0.735019, 0.653493),
            ],
            [
                c64::new(0.315191, 0.478662),
                c64::new(0.833807, 0.697483),
                c64::new(0.310534, 0.396302),
                c64::new(0.839271, 0.949712),
                c64::new(0.325344, 0.235542),
                c64::new(0.693700, 0.247816),
                c64::new(0.759020, 0.118982),
                c64::new(0.160333, 0.765951),
                c64::new(0.631740, 0.843597),
                c64::new(0.554191, 0.633453),
                c64::new(0.742897, 0.788577),
                c64::new(0.840315, 0.054435),
                c64::new(0.089497, 0.589866),
                c64::new(0.563293, 0.695071),
                c64::new(0.916185, 0.138849),
                c64::new(0.067603, 0.481400),
                c64::new(0.906483, 0.980935),
                c64::new(0.715502, 0.313255),
                c64::new(0.738656, 0.268801),
                c64::new(0.839109, 0.956635),
                c64::new(0.690422, 0.335274),
                c64::new(0.488201, 0.501880),
                c64::new(0.328307, 0.260638),
                c64::new(0.291425, 0.432487),
                c64::new(0.638943, 0.328813),
                c64::new(0.984401, 0.711157),
                c64::new(0.806059, 0.405321),
                c64::new(0.599686, 0.547603),
                c64::new(0.381971, 0.484660),
                c64::new(0.992921, 0.606005),
                c64::new(0.961329, 0.893073),
                c64::new(0.694046, 0.744479),
                c64::new(0.517949, 0.141120),
                c64::new(0.666088, 0.116443),
                c64::new(0.590009, 0.707290),
                c64::new(0.397481, 0.500323),
                c64::new(0.027404, 0.757967),
                c64::new(0.624940, 0.042651),
                c64::new(0.632875, 0.116292),
                c64::new(0.224254, 0.776686),
                c64::new(0.910262, 0.616570),
                c64::new(0.277812, 0.737948),
                c64::new(0.057866, 0.784957),
                c64::new(0.152657, 0.911839),
                c64::new(0.674222, 0.041607),
                c64::new(0.511282, 0.147960),
                c64::new(0.494807, 0.331465),
                c64::new(0.228156, 0.943136),
                c64::new(0.439864, 0.301055),
                c64::new(0.272690, 0.238139),
                c64::new(0.893550, 0.363394),
                c64::new(0.272024, 0.665481),
                c64::new(0.051683, 0.055642),
                c64::new(0.059980, 0.933988),
                c64::new(0.312177, 0.931784),
                c64::new(0.692342, 0.669074),
                c64::new(0.708301, 0.746008),
                c64::new(0.654037, 0.048102),
                c64::new(0.805826, 0.440089),
                c64::new(0.269123, 0.844255),
                c64::new(0.189183, 0.907510),
                c64::new(0.422676, 0.220691),
                c64::new(0.730443, 0.605872),
                c64::new(0.922972, 0.193306),
                c64::new(0.932751, 0.640147),
                c64::new(0.415496, 0.541532),
                c64::new(0.284173, 0.456411),
                c64::new(0.112323, 0.847687),
                c64::new(0.906880, 0.909230),
                c64::new(0.629705, 0.631650),
                c64::new(0.022350, 0.197708),
                c64::new(0.716942, 0.390986),
                c64::new(0.945447, 0.269241),
                c64::new(0.745904, 0.211792),
                c64::new(0.818865, 0.295856),
                c64::new(0.657553, 0.479682),
                c64::new(0.824392, 0.485086),
                c64::new(0.506792, 0.486432),
                c64::new(0.048943, 0.563696),
                c64::new(0.917321, 0.741947),
                c64::new(0.011085, 0.625994),
                c64::new(0.511457, 0.815998),
                c64::new(0.618929, 0.165593),
                c64::new(0.200062, 0.607617),
                c64::new(0.335587, 0.993400),
                c64::new(0.088285, 0.474581),
                c64::new(0.977839, 0.850574),
                c64::new(0.080987, 0.751683),
                c64::new(0.488594, 0.731679),
                c64::new(0.489062, 0.434918),
                c64::new(0.147371, 0.659253),
                c64::new(0.585042, 0.742962),
                c64::new(0.914979, 0.911878),
                c64::new(0.938842, 0.912843),
                c64::new(0.685735, 0.695321),
                c64::new(0.335607, 0.733544),
                c64::new(0.835737, 0.400742),
                c64::new(0.250515, 0.714784),
                c64::new(0.877781, 0.755097),
                c64::new(0.446200, 0.130484),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.331657, 0.262359),
                c64::new(0.244727, 0.638396),
                c64::new(0.581313, 0.528013),
                c64::new(0.946119, 0.256675),
                c64::new(0.555156, 0.078901),
                c64::new(0.524032, 0.703333),
                c64::new(0.993889, 0.129179),
                c64::new(0.237493, 0.327509),
                c64::new(0.055600, 0.847578),
                c64::new(0.088908, 0.274087),
                c64::new(0.149218, 0.359951),
                c64::new(0.701262, 0.528096),
                c64::new(0.718525, 0.572322),
                c64::new(0.990188, 0.336295),
                c64::new(0.535049, 0.076191),
                c64::new(0.186912, 0.608159),
                c64::new(0.977768, 0.997542),
                c64::new(0.872098, 0.900481),
                c64::new(0.450666, 0.425147),
                c64::new(0.621391, 0.445795),
                c64::new(0.613905, 0.690011),
                c64::new(0.163519, 0.248689),
                c64::new(0.205288, 0.506597),
                c64::new(0.744933, 0.297656),
                c64::new(0.138310, 0.293064),
                c64::new(0.874282, 0.170779),
                c64::new(0.740816, 0.825517),
                c64::new(0.045324, 0.683082),
                c64::new(0.339080, 0.033500),
                c64::new(0.077581, 0.066504),
                c64::new(0.220183, 0.032719),
                c64::new(0.163831, 0.811177),
                c64::new(0.524874, 0.049641),
                c64::new(0.036958, 0.062986),
                c64::new(0.568491, 0.884327),
                c64::new(0.914892, 0.844849),
                c64::new(0.656459, 0.432534),
                c64::new(0.981313, 0.996293),
                c64::new(0.156926, 0.438136),
                c64::new(0.518909, 0.711744),
                c64::new(0.812142, 0.201055),
                c64::new(0.385009, 0.749246),
                c64::new(0.834279, 0.256328),
                c64::new(0.378963, 0.761068),
                c64::new(0.037043, 0.091670),
                c64::new(0.091428, 0.886068),
                c64::new(0.321316, 0.060991),
                c64::new(0.197774, 0.109478),
                c64::new(0.013886, 0.152236),
                c64::new(0.733118, 0.678769),
                c64::new(0.349739, 0.922512),
                c64::new(0.437076, 0.985890),
                c64::new(0.789399, 0.863788),
                c64::new(0.098481, 0.433876),
                c64::new(0.140539, 0.363167),
                c64::new(0.624408, 0.333908),
                c64::new(0.149382, 0.749237),
                c64::new(0.102141, 0.391356),
                c64::new(0.870619, 0.233448),
                c64::new(0.276125, 0.033807),
                c64::new(0.788045, 0.012740),
                c64::new(0.140971, 0.337287),
                c64::new(0.666063, 0.203683),
                c64::new(0.045933, 0.211809),
                c64::new(0.034878, 0.716547),
                c64::new(0.989171, 0.077564),
                c64::new(0.664159, 0.717327),
                c64::new(0.285891, 0.286707),
                c64::new(0.501182, 0.174097),
                c64::new(0.013578, 0.152712),
                c64::new(0.973179, 0.909657),
                c64::new(0.412261, 0.975651),
                c64::new(0.657502, 0.195360),
                c64::new(0.080765, 0.520424),
                c64::new(0.740716, 0.938439),
                c64::new(0.960119, 0.588442),
                c64::new(0.037048, 0.442824),
                c64::new(0.858198, 0.148410),
                c64::new(0.844994, 0.632135),
                c64::new(0.189434, 0.551046),
                c64::new(0.906756, 0.418544),
                c64::new(0.145521, 0.241500),
                c64::new(0.896957, 0.102364),
                c64::new(0.963494, 0.481932),
                c64::new(0.949465, 0.949494),
                c64::new(0.698473, 0.304047),
                c64::new(0.496960, 0.740081),
                c64::new(0.624386, 0.863860),
                c64::new(0.186898, 0.679316),
                c64::new(0.883993, 0.641905),
                c64::new(0.843153, 0.417276),
                c64::new(0.011741, 0.892577),
                c64::new(0.911255, 0.720138),
                c64::new(0.924570, 0.168352),
                c64::new(0.742747, 0.637457),
                c64::new(0.499916, 0.954243),
                c64::new(0.895616, 0.098343),
                c64::new(0.204341, 0.314443),
                c64::new(0.649155, 0.065885),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.279330, 0.200421),
                c64::new(0.656860, 0.846191),
                c64::new(0.078371, 0.908207),
                c64::new(0.118789, 0.598040),
                c64::new(0.886693, 0.984756),
                c64::new(0.289293, 0.632213),
                c64::new(0.326069, 0.667493),
                c64::new(0.243075, 0.405723),
                c64::new(0.681192, 0.365090),
                c64::new(0.456548, 0.942152),
                c64::new(0.117724, 0.254845),
                c64::new(0.812516, 0.102193),
                c64::new(0.298388, 0.097790),
                c64::new(0.109248, 0.641105),
                c64::new(0.247543, 0.655247),
                c64::new(0.504984, 0.259295),
                c64::new(0.256707, 0.472077),
                c64::new(0.958231, 0.153130),
                c64::new(0.564108, 0.333260),
                c64::new(0.112267, 0.546179),
                c64::new(0.495952, 0.773444),
                c64::new(0.883847, 0.653666),
                c64::new(0.873270, 0.196981),
                c64::new(0.605119, 0.587676),
                c64::new(0.876861, 0.514449),
                c64::new(0.155908, 0.881484),
                c64::new(0.010313, 0.234397),
                c64::new(0.613524, 0.210318),
                c64::new(0.030855, 0.530377),
                c64::new(0.244104, 0.266909),
                c64::new(0.672875, 0.549270),
                c64::new(0.967687, 0.385814),
                c64::new(0.900165, 0.647687),
                c64::new(0.647362, 0.854106),
                c64::new(0.990863, 0.388665),
                c64::new(0.918010, 0.833200),
                c64::new(0.847446, 0.274499),
                c64::new(0.505008, 0.219954),
                c64::new(0.980723, 0.813352),
                c64::new(0.184564, 0.191929),
                c64::new(0.915982, 0.669395),
                c64::new(0.383139, 0.251168),
                c64::new(0.968762, 0.283487),
                c64::new(0.136034, 0.468001),
                c64::new(0.366363, 0.938617),
                c64::new(0.141509, 0.305781),
                c64::new(0.138932, 0.223510),
                c64::new(0.899131, 0.970246),
                c64::new(0.623190, 0.972601),
                c64::new(0.349562, 0.841147),
                c64::new(0.652153, 0.166931),
                c64::new(0.059863, 0.071986),
                c64::new(0.352602, 0.831250),
                c64::new(0.142682, 0.133041),
                c64::new(0.779363, 0.792894),
                c64::new(0.939586, 0.146436),
                c64::new(0.260930, 0.196435),
                c64::new(0.317724, 0.443361),
                c64::new(0.494313, 0.275325),
                c64::new(0.785212, 0.941097),
                c64::new(0.731804, 0.996094),
                c64::new(0.288562, 0.574506),
                c64::new(0.779104, 0.312999),
                c64::new(0.128480, 0.230822),
                c64::new(0.783211, 0.157697),
                c64::new(0.273168, 0.139638),
                c64::new(0.224522, 0.779218),
                c64::new(0.173166, 0.909800),
                c64::new(0.552759, 0.029199),
                c64::new(0.771181, 0.145833),
                c64::new(0.027655, 0.256837),
                c64::new(0.867614, 0.251680),
                c64::new(0.386665, 0.315398),
                c64::new(0.360899, 0.573859),
                c64::new(0.668708, 0.526350),
                c64::new(0.689177, 0.334519),
                c64::new(0.025291, 0.307119),
                c64::new(0.345042, 0.019471),
                c64::new(0.789760, 0.719525),
                c64::new(0.793157, 0.421004),
                c64::new(0.776176, 0.044362),
                c64::new(0.564112, 0.295169),
                c64::new(0.516398, 0.071471),
                c64::new(0.551167, 0.817572),
                c64::new(0.609354, 0.816551),
                c64::new(0.608546, 0.325273),
                c64::new(0.095880, 0.876590),
                c64::new(0.469386, 0.723095),
                c64::new(0.807517, 0.460735),
                c64::new(0.040890, 0.258258),
                c64::new(0.373461, 0.525255),
                c64::new(0.803529, 0.278010),
                c64::new(0.938321, 0.507100),
                c64::new(0.750302, 0.795066),
                c64::new(0.383077, 0.660575),
                c64::new(0.851240, 0.537057),
                c64::new(0.922193, 0.088407),
                c64::new(0.132955, 0.896962),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.541176, 0.213800),
                c64::new(0.055289, 0.866253),
                c64::new(0.793814, 0.673114),
                c64::new(0.388335, 0.847877),
                c64::new(0.467342, 0.668064),
                c64::new(0.872978, 0.926393),
                c64::new(0.368032, 0.874631),
                c64::new(0.985142, 0.569661),
                c64::new(0.223288, 0.829772),
                c64::new(0.036679, 0.431764),
                c64::new(0.966409, 0.964462),
                c64::new(0.234639, 0.036226),
                c64::new(0.410713, 0.485268),
                c64::new(0.657409, 0.247961),
                c64::new(0.556006, 0.889590),
                c64::new(0.492743, 0.259281),
                c64::new(0.853356, 0.664594),
                c64::new(0.442558, 0.360879),
                c64::new(0.379378, 0.212513),
                c64::new(0.375497, 0.327946),
                c64::new(0.457485, 0.759649),
                c64::new(0.767964, 0.593202),
                c64::new(0.703972, 0.787452),
                c64::new(0.637206, 0.047649),
                c64::new(0.295419, 0.112890),
                c64::new(0.359455, 0.753014),
                c64::new(0.416821, 0.755433),
                c64::new(0.679133, 0.276088),
                c64::new(0.201199, 0.331420),
                c64::new(0.145993, 0.396517),
                c64::new(0.790180, 0.010070),
                c64::new(0.423530, 0.343241),
                c64::new(0.067135, 0.459603),
                c64::new(0.359921, 0.827177),
                c64::new(0.393071, 0.568616),
                c64::new(0.739557, 0.078482),
                c64::new(0.025703, 0.338871),
                c64::new(0.289627, 0.516078),
                c64::new(0.280563, 0.212957),
                c64::new(0.353660, 0.809065),
                c64::new(0.868511, 0.285249),
                c64::new(0.212131, 0.536091),
                c64::new(0.090353, 0.508409),
                c64::new(0.297210, 0.088947),
                c64::new(0.000443, 0.271580),
                c64::new(0.676747, 0.840645),
                c64::new(0.564976, 0.627044),
                c64::new(0.422724, 0.270323),
                c64::new(0.168837, 0.495277),
                c64::new(0.267089, 0.115740),
                c64::new(0.772203, 0.429281),
                c64::new(0.583360, 0.802402),
                c64::new(0.431460, 0.095637),
                c64::new(0.912250, 0.626725),
                c64::new(0.609026, 0.245385),
                c64::new(0.778113, 0.808350),
                c64::new(0.518982, 0.291490),
                c64::new(0.511992, 0.170120),
                c64::new(0.226763, 0.547021),
                c64::new(0.614790, 0.228372),
                c64::new(0.762392, 0.403270),
                c64::new(0.787686, 0.059468),
                c64::new(0.887644, 0.329107),
                c64::new(0.431230, 0.251256),
                c64::new(0.640605, 0.132503),
                c64::new(0.313584, 0.443068),
                c64::new(0.978586, 0.350226),
                c64::new(0.191324, 0.165691),
                c64::new(0.541977, 0.776232),
                c64::new(0.400233, 0.987371),
                c64::new(0.552699, 0.248859),
                c64::new(0.435885, 0.537325),
                c64::new(0.764246, 0.506262),
                c64::new(0.708169, 0.319242),
                c64::new(0.867256, 0.874209),
                c64::new(0.292619, 0.272509),
                c64::new(0.643935, 0.918770),
                c64::new(0.912011, 0.822607),
                c64::new(0.743979, 0.706268),
                c64::new(0.948011, 0.314097),
                c64::new(0.773267, 0.683963),
                c64::new(0.829798, 0.179651),
                c64::new(0.784784, 0.855000),
                c64::new(0.210917, 0.444375),
                c64::new(0.962698, 0.930504),
                c64::new(0.038745, 0.086622),
                c64::new(0.238235, 0.824315),
                c64::new(0.554350, 0.363664),
                c64::new(0.216626, 0.900797),
                c64::new(0.795121, 0.855483),
                c64::new(0.585897, 0.640472),
                c64::new(0.402094, 0.439532),
                c64::new(0.186325, 0.058003),
                c64::new(0.468742, 0.474751),
                c64::new(0.385064, 0.116919),
                c64::new(0.490516, 0.811192),
                c64::new(0.705007, 0.936418),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.504525, 0.275602),
                c64::new(0.581714, 0.923980),
                c64::new(0.784429, 0.517392),
                c64::new(0.694721, 0.918042),
                c64::new(0.736435, 0.451541),
                c64::new(0.478398, 0.458670),
                c64::new(0.917999, 0.681193),
                c64::new(0.924837, 0.649548),
                c64::new(0.789458, 0.972121),
                c64::new(0.478846, 0.104062),
                c64::new(0.592473, 0.333113),
                c64::new(0.831848, 0.191320),
                c64::new(0.796934, 0.352247),
                c64::new(0.646904, 0.005961),
                c64::new(0.842753, 0.826201),
                c64::new(0.199294, 0.490205),
                c64::new(0.769102, 0.438923),
                c64::new(0.446627, 0.964947),
                c64::new(0.745186, 0.958642),
                c64::new(0.860051, 0.508763),
                c64::new(0.252206, 0.550260),
                c64::new(0.135175, 0.280995),
                c64::new(0.768068, 0.897398),
                c64::new(0.939665, 0.869227),
                c64::new(0.578220, 0.867439),
                c64::new(0.725137, 0.232940),
                c64::new(0.450222, 0.126592),
                c64::new(0.908854, 0.760924),
                c64::new(0.796657, 0.647183),
                c64::new(0.783444, 0.801277),
                c64::new(0.996766, 0.844475),
                c64::new(0.377512, 0.744496),
                c64::new(0.002539, 0.885059),
                c64::new(0.967171, 0.156495),
                c64::new(0.295415, 0.282586),
                c64::new(0.340317, 0.373176),
                c64::new(0.837608, 0.723693),
                c64::new(0.870025, 0.318869),
                c64::new(0.505777, 0.190773),
                c64::new(0.964125, 0.705819),
                c64::new(0.817387, 0.016462),
                c64::new(0.569007, 0.820041),
                c64::new(0.109563, 0.607906),
                c64::new(0.937891, 0.696292),
                c64::new(0.453103, 0.014623),
                c64::new(0.508613, 0.509688),
                c64::new(0.075323, 0.758417),
                c64::new(0.529915, 0.527384),
                c64::new(0.917754, 0.536743),
                c64::new(0.043143, 0.769651),
                c64::new(0.019817, 0.689931),
                c64::new(0.398399, 0.149912),
                c64::new(0.564852, 0.510680),
                c64::new(0.470166, 0.940782),
                c64::new(0.215438, 0.794656),
                c64::new(0.020560, 0.209655),
                c64::new(0.430041, 0.923772),
                c64::new(0.967261, 0.173571),
                c64::new(0.952390, 0.661186),
                c64::new(0.941806, 0.394577),
                c64::new(0.187801, 0.378212),
                c64::new(0.739727, 0.010822),
                c64::new(0.484095, 0.645754),
                c64::new(0.363152, 0.019576),
                c64::new(0.177671, 0.306288),
                c64::new(0.229200, 0.200683),
                c64::new(0.590264, 0.771698),
                c64::new(0.275760, 0.962651),
                c64::new(0.954762, 0.762074),
                c64::new(0.793666, 0.941696),
                c64::new(0.820121, 0.218360),
                c64::new(0.201549, 0.948204),
                c64::new(0.837364, 0.291420),
                c64::new(0.840401, 0.058099),
                c64::new(0.424386, 0.511376),
                c64::new(0.006618, 0.460554),
                c64::new(0.948996, 0.476161),
                c64::new(0.716407, 0.014602),
                c64::new(0.637296, 0.096833),
                c64::new(0.553516, 0.776390),
                c64::new(0.045962, 0.394421),
                c64::new(0.373100, 0.260907),
                c64::new(0.147335, 0.875636),
                c64::new(0.433242, 0.945208),
                c64::new(0.619115, 0.117876),
                c64::new(0.470043, 0.215531),
                c64::new(0.192970, 0.943398),
                c64::new(0.141771, 0.458706),
                c64::new(0.361972, 0.984832),
                c64::new(0.732352, 0.408699),
                c64::new(0.723612, 0.322045),
                c64::new(0.662175, 0.938809),
                c64::new(0.332571, 0.676057),
                c64::new(0.909820, 0.215665),
                c64::new(0.925620, 0.022299),
                c64::new(0.424417, 0.558210),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.664100, 0.715876),
                c64::new(0.306003, 0.918067),
                c64::new(0.738031, 0.856596),
                c64::new(0.519573, 0.955956),
                c64::new(0.538104, 0.878495),
                c64::new(0.343654, 0.405434),
                c64::new(0.274486, 0.384978),
                c64::new(0.288923, 0.444287),
                c64::new(0.897896, 0.952822),
                c64::new(0.777598, 0.968945),
                c64::new(0.334201, 0.160386),
                c64::new(0.209868, 0.106069),
                c64::new(0.867320, 0.093870),
                c64::new(0.679453, 0.542672),
                c64::new(0.876689, 0.249522),
                c64::new(0.274410, 0.272696),
                c64::new(0.805269, 0.057745),
                c64::new(0.424354, 0.893543),
                c64::new(0.710866, 0.255001),
                c64::new(0.112813, 0.891013),
                c64::new(0.293643, 0.728290),
                c64::new(0.424476, 0.782843),
                c64::new(0.416680, 0.959461),
                c64::new(0.024328, 0.048543),
                c64::new(0.726901, 0.467630),
                c64::new(0.818641, 0.719145),
                c64::new(0.930312, 0.196630),
                c64::new(0.118224, 0.461610),
                c64::new(0.850370, 0.982100),
                c64::new(0.733812, 0.188636),
                c64::new(0.834908, 0.686902),
                c64::new(0.800755, 0.776162),
                c64::new(0.353491, 0.147317),
                c64::new(0.574225, 0.544286),
                c64::new(0.458991, 0.652634),
                c64::new(0.900823, 0.036654),
                c64::new(0.983832, 0.302139),
                c64::new(0.077913, 0.505631),
                c64::new(0.822623, 0.472645),
                c64::new(0.507072, 0.264395),
                c64::new(0.627892, 0.677008),
                c64::new(0.492069, 0.720020),
                c64::new(0.836802, 0.719225),
                c64::new(0.517493, 0.841356),
                c64::new(0.953892, 0.508602),
                c64::new(0.002379, 0.731934),
                c64::new(0.574351, 0.417829),
                c64::new(0.755134, 0.174647),
                c64::new(0.727789, 0.949780),
                c64::new(0.957562, 0.992268),
                c64::new(0.633580, 0.243574),
                c64::new(0.756835, 0.492746),
                c64::new(0.240099, 0.997078),
                c64::new(0.703756, 0.559459),
                c64::new(0.072492, 0.968219),
                c64::new(0.959788, 0.277995),
                c64::new(0.661461, 0.941819),
                c64::new(0.111737, 0.763205),
                c64::new(0.295152, 0.882562),
                c64::new(0.492568, 0.480833),
                c64::new(0.115088, 0.431051),
                c64::new(0.568129, 0.587202),
                c64::new(0.264254, 0.725769),
                c64::new(0.298884, 0.147189),
                c64::new(0.110293, 0.504983),
                c64::new(0.012780, 0.109612),
                c64::new(0.437704, 0.690835),
                c64::new(0.571051, 0.552427),
                c64::new(0.308212, 0.133583),
                c64::new(0.318483, 0.097583),
                c64::new(0.520854, 0.452190),
                c64::new(0.018061, 0.525337),
                c64::new(0.819254, 0.873038),
                c64::new(0.474728, 0.805593),
                c64::new(0.486643, 0.697993),
                c64::new(0.649597, 0.697582),
                c64::new(0.782052, 0.940335),
                c64::new(0.587743, 0.635511),
                c64::new(0.279865, 0.674201),
                c64::new(0.174687, 0.805640),
                c64::new(0.871215, 0.334582),
                c64::new(0.944343, 0.317877),
                c64::new(0.457410, 0.589672),
                c64::new(0.345857, 0.800753),
                c64::new(0.068360, 0.572089),
                c64::new(0.043464, 0.303757),
                c64::new(0.130664, 0.842130),
                c64::new(0.797796, 0.301784),
                c64::new(0.520515, 0.784033),
                c64::new(0.964881, 0.220223),
                c64::new(0.002487, 0.838886),
                c64::new(0.828365, 0.679851),
                c64::new(0.924351, 0.186967),
                c64::new(0.222602, 0.797951),
                c64::new(0.547503, 0.745845),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.761329, 0.474137),
                c64::new(0.057140, 0.021554),
                c64::new(0.341875, 0.030452),
                c64::new(0.454658, 0.195647),
                c64::new(0.108317, 0.305343),
                c64::new(0.482688, 0.024172),
                c64::new(0.110135, 0.606529),
                c64::new(0.872017, 0.832805),
                c64::new(0.210274, 0.058658),
                c64::new(0.546038, 0.781291),
                c64::new(0.784601, 0.228759),
                c64::new(0.822710, 0.819986),
                c64::new(0.415218, 0.292593),
                c64::new(0.982274, 0.258480),
                c64::new(0.629042, 0.821195),
                c64::new(0.406773, 0.549015),
                c64::new(0.995871, 0.778788),
                c64::new(0.039236, 0.899381),
                c64::new(0.368055, 0.090768),
                c64::new(0.691680, 0.570506),
                c64::new(0.511426, 0.769842),
                c64::new(0.386492, 0.077254),
                c64::new(0.913777, 0.999235),
                c64::new(0.154210, 0.038114),
                c64::new(0.363958, 0.490296),
                c64::new(0.423887, 0.404663),
                c64::new(0.134981, 0.171248),
                c64::new(0.944690, 0.857868),
                c64::new(0.069656, 0.436922),
                c64::new(0.568224, 0.060069),
                c64::new(0.847804, 0.872848),
                c64::new(0.663877, 0.141819),
                c64::new(0.474788, 0.158607),
                c64::new(0.002091, 0.393863),
                c64::new(0.077639, 0.893645),
                c64::new(0.504144, 0.324619),
                c64::new(0.763920, 0.787534),
                c64::new(0.980880, 0.593065),
                c64::new(0.616328, 0.025537),
                c64::new(0.944112, 0.244000),
                c64::new(0.010341, 0.924079),
                c64::new(0.671194, 0.453097),
                c64::new(0.104769, 0.007300),
                c64::new(0.506511, 0.578985),
                c64::new(0.216464, 0.030540),
                c64::new(0.034870, 0.981198),
                c64::new(0.601744, 0.846565),
                c64::new(0.053338, 0.648381),
                c64::new(0.450082, 0.240713),
                c64::new(0.321715, 0.996102),
                c64::new(0.240902, 0.755036),
                c64::new(0.687335, 0.907080),
                c64::new(0.442760, 0.608330),
                c64::new(0.402324, 0.986593),
                c64::new(0.348717, 0.219912),
                c64::new(0.424926, 0.766619),
                c64::new(0.795066, 0.263490),
                c64::new(0.785027, 0.276597),
                c64::new(0.456680, 0.848079),
                c64::new(0.196293, 0.095246),
                c64::new(0.751354, 0.249836),
                c64::new(0.429790, 0.180219),
                c64::new(0.267194, 0.440343),
                c64::new(0.566302, 0.481341),
                c64::new(0.557362, 0.015110),
                c64::new(0.623771, 0.689169),
                c64::new(0.190737, 0.984802),
                c64::new(0.172350, 0.900894),
                c64::new(0.251772, 0.633124),
                c64::new(0.513337, 0.444512),
                c64::new(0.753186, 0.930539),
                c64::new(0.964207, 0.410712),
                c64::new(0.180238, 0.113712),
                c64::new(0.731805, 0.732117),
                c64::new(0.100688, 0.376921),
                c64::new(0.615828, 0.734217),
                c64::new(0.749916, 0.497537),
                c64::new(0.002625, 0.591863),
                c64::new(0.644224, 0.882223),
                c64::new(0.690282, 0.190992),
                c64::new(0.893521, 0.928840),
                c64::new(0.466513, 0.312486),
                c64::new(0.140925, 0.221800),
                c64::new(0.259923, 0.144308),
                c64::new(0.725702, 0.223996),
                c64::new(0.305583, 0.808398),
                c64::new(0.817229, 0.205274),
                c64::new(0.288736, 0.620168),
                c64::new(0.253590, 0.779970),
                c64::new(0.438128, 0.662336),
                c64::new(0.255910, 0.969372),
                c64::new(0.539765, 0.703620),
                c64::new(0.379613, 0.338690),
                c64::new(0.130420, 0.870391),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.281250, 0.457196),
                c64::new(0.553758, 0.503325),
                c64::new(0.490258, 0.124320),
                c64::new(0.786502, 0.325276),
                c64::new(0.127180, 0.366602),
                c64::new(0.134249, 0.774897),
                c64::new(0.315897, 0.239912),
                c64::new(0.034332, 0.330397),
                c64::new(0.271820, 0.908755),
                c64::new(0.687357, 0.275991),
                c64::new(0.715441, 0.446821),
                c64::new(0.266875, 0.854911),
                c64::new(0.708890, 0.545354),
                c64::new(0.077122, 0.151865),
                c64::new(0.253644, 0.730437),
                c64::new(0.190611, 0.119619),
                c64::new(0.067786, 0.904203),
                c64::new(0.396297, 0.524168),
                c64::new(0.324399, 0.938615),
                c64::new(0.048354, 0.398907),
                c64::new(0.601310, 0.480633),
                c64::new(0.901911, 0.555953),
                c64::new(0.224240, 0.654160),
                c64::new(0.190490, 0.899168),
                c64::new(0.996899, 0.230060),
                c64::new(0.506326, 0.931737),
                c64::new(0.835350, 0.132773),
                c64::new(0.689645, 0.958794),
                c64::new(0.537812, 0.661072),
                c64::new(0.911060, 0.590255),
                c64::new(0.067496, 0.713671),
                c64::new(0.694958, 0.262810),
                c64::new(0.844676, 0.839687),
                c64::new(0.989296, 0.082079),
                c64::new(0.995364, 0.483861),
                c64::new(0.577952, 0.896696),
                c64::new(0.022898, 0.855596),
                c64::new(0.463523, 0.774210),
                c64::new(0.761949, 0.334033),
                c64::new(0.826583, 0.941408),
                c64::new(0.030058, 0.171974),
                c64::new(0.751235, 0.336407),
                c64::new(0.447569, 0.001303),
                c64::new(0.814994, 0.837680),
                c64::new(0.312454, 0.238153),
                c64::new(0.148646, 0.980932),
                c64::new(0.065710, 0.485325),
                c64::new(0.841912, 0.541389),
                c64::new(0.296602, 0.955343),
                c64::new(0.427599, 0.065198),
                c64::new(0.910705, 0.806317),
                c64::new(0.956512, 0.913340),
                c64::new(0.314531, 0.754588),
                c64::new(0.029930, 0.928508),
                c64::new(0.826030, 0.758061),
                c64::new(0.798540, 0.011416),
                c64::new(0.849948, 0.503950),
                c64::new(0.703862, 0.250694),
                c64::new(0.334658, 0.358087),
                c64::new(0.343169, 0.214128),
                c64::new(0.403816, 0.896222),
                c64::new(0.004652, 0.337637),
                c64::new(0.639194, 0.527597),
                c64::new(0.562645, 0.200108),
                c64::new(0.130442, 0.647892),
                c64::new(0.074370, 0.373034),
                c64::new(0.741777, 0.929961),
                c64::new(0.850655, 0.070431),
                c64::new(0.448984, 0.194008),
                c64::new(0.976091, 0.990299),
                c64::new(0.625688, 0.263407),
                c64::new(0.628268, 0.807432),
                c64::new(0.930037, 0.003414),
                c64::new(0.456355, 0.206976),
                c64::new(0.011826, 0.414906),
                c64::new(0.696273, 0.572816),
                c64::new(0.983242, 0.712490),
                c64::new(0.925181, 0.895583),
                c64::new(0.360160, 0.878410),
                c64::new(0.504108, 0.111352),
                c64::new(0.160686, 0.140605),
                c64::new(0.570738, 0.191626),
                c64::new(0.676459, 0.665012),
                c64::new(0.784552, 0.680081),
                c64::new(0.098649, 0.516121),
                c64::new(0.205273, 0.778610),
                c64::new(0.434327, 0.310256),
                c64::new(0.866206, 0.582293),
                c64::new(0.719858, 0.584182),
                c64::new(0.211123, 0.642164),
                c64::new(0.448459, 0.938724),
                c64::new(0.685214, 0.675736),
                c64::new(0.821699, 0.104142),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.762233, 0.254906),
                c64::new(0.282796, 0.821377),
                c64::new(0.230042, 0.121300),
                c64::new(0.814784, 0.653078),
                c64::new(0.773767, 0.078697),
                c64::new(0.571516, 0.766801),
                c64::new(0.755685, 0.938336),
                c64::new(0.269764, 0.992994),
                c64::new(0.598369, 0.670755),
                c64::new(0.040979, 0.219360),
                c64::new(0.960505, 0.980348),
                c64::new(0.793943, 0.728118),
                c64::new(0.583504, 0.425626),
                c64::new(0.314256, 0.965120),
                c64::new(0.927737, 0.557444),
                c64::new(0.536900, 0.329226),
                c64::new(0.173988, 0.289097),
                c64::new(0.785395, 0.759280),
                c64::new(0.645432, 0.698285),
                c64::new(0.026213, 0.015188),
                c64::new(0.137592, 0.438188),
                c64::new(0.894743, 0.738587),
                c64::new(0.005395, 0.229090),
                c64::new(0.960169, 0.807781),
                c64::new(0.253019, 0.104073),
                c64::new(0.951972, 0.148383),
                c64::new(0.655662, 0.937177),
                c64::new(0.221910, 0.362053),
                c64::new(0.492987, 0.050088),
                c64::new(0.363505, 0.347282),
                c64::new(0.584538, 0.947470),
                c64::new(0.507813, 0.848104),
                c64::new(0.766702, 0.274560),
                c64::new(0.458391, 0.266606),
                c64::new(0.189173, 0.949957),
                c64::new(0.253112, 0.984897),
                c64::new(0.571023, 0.622702),
                c64::new(0.827870, 0.970213),
                c64::new(0.280466, 0.714010),
                c64::new(0.933515, 0.539319),
                c64::new(0.951287, 0.664505),
                c64::new(0.221803, 0.846934),
                c64::new(0.681162, 0.828269),
                c64::new(0.241963, 0.516494),
                c64::new(0.197164, 0.194033),
                c64::new(0.051560, 0.547219),
                c64::new(0.078404, 0.830762),
                c64::new(0.845807, 0.508960),
                c64::new(0.370348, 0.845713),
                c64::new(0.143998, 0.835709),
                c64::new(0.183469, 0.332996),
                c64::new(0.095398, 0.488281),
                c64::new(0.981171, 0.233496),
                c64::new(0.939250, 0.533890),
                c64::new(0.599947, 0.383013),
                c64::new(0.888847, 0.016008),
                c64::new(0.281500, 0.849602),
                c64::new(0.263634, 0.727870),
                c64::new(0.318140, 0.910614),
                c64::new(0.545748, 0.076173),
                c64::new(0.668332, 0.206571),
                c64::new(0.388080, 0.347205),
                c64::new(0.611075, 0.114643),
                c64::new(0.694929, 0.146425),
                c64::new(0.975727, 0.368355),
                c64::new(0.747484, 0.416643),
                c64::new(0.340646, 0.883942),
                c64::new(0.300997, 0.366858),
                c64::new(0.523824, 0.453834),
                c64::new(0.025292, 0.232514),
                c64::new(0.780472, 0.027187),
                c64::new(0.388570, 0.768596),
                c64::new(0.439242, 0.783692),
                c64::new(0.584253, 0.547184),
                c64::new(0.447356, 0.450026),
                c64::new(0.604316, 0.604147),
                c64::new(0.269517, 0.226360),
                c64::new(0.350350, 0.532746),
                c64::new(0.031654, 0.993903),
                c64::new(0.938172, 0.983360),
                c64::new(0.859667, 0.724463),
                c64::new(0.517305, 0.595401),
                c64::new(0.507892, 0.282801),
                c64::new(0.093921, 0.939854),
                c64::new(0.864624, 0.789211),
                c64::new(0.460581, 0.505156),
                c64::new(0.948532, 0.232594),
                c64::new(0.330381, 0.713507),
                c64::new(0.866430, 0.158657),
                c64::new(0.251104, 0.265277),
                c64::new(0.821879, 0.790737),
                c64::new(0.504309, 0.638158),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.929779, 0.734148),
                c64::new(0.482822, 0.136545),
                c64::new(0.822025, 0.810775),
                c64::new(0.968682, 0.759398),
                c64::new(0.824234, 0.018718),
                c64::new(0.279519, 0.637786),
                c64::new(0.864062, 0.754173),
                c64::new(0.883063, 0.947834),
                c64::new(0.978932, 0.832647),
                c64::new(0.108744, 0.977103),
                c64::new(0.121843, 0.087693),
                c64::new(0.170359, 0.068747),
                c64::new(0.581379, 0.488896),
                c64::new(0.934925, 0.499277),
                c64::new(0.942417, 0.183928),
                c64::new(0.393964, 0.638999),
                c64::new(0.322281, 0.246808),
                c64::new(0.733407, 0.206246),
                c64::new(0.193888, 0.167587),
                c64::new(0.912600, 0.619939),
                c64::new(0.372085, 0.349843),
                c64::new(0.258421, 0.739553),
                c64::new(0.593035, 0.195997),
                c64::new(0.260812, 0.005966),
                c64::new(0.234602, 0.900769),
                c64::new(0.528731, 0.229335),
                c64::new(0.424740, 0.529025),
                c64::new(0.355056, 0.592621),
                c64::new(0.567516, 0.480549),
                c64::new(0.942477, 0.628082),
                c64::new(0.545550, 0.708920),
                c64::new(0.416123, 0.057100),
                c64::new(0.287489, 0.370781),
                c64::new(0.059555, 0.037338),
                c64::new(0.166931, 0.193747),
                c64::new(0.160329, 0.825578),
                c64::new(0.686360, 0.480028),
                c64::new(0.649297, 0.788636),
                c64::new(0.622213, 0.335760),
                c64::new(0.717951, 0.819263),
                c64::new(0.379302, 0.294782),
                c64::new(0.616378, 0.255408),
                c64::new(0.108894, 0.473673),
                c64::new(0.635840, 0.859348),
                c64::new(0.893123, 0.059023),
                c64::new(0.439904, 0.510039),
                c64::new(0.879963, 0.316777),
                c64::new(0.411363, 0.358371),
                c64::new(0.092588, 0.564482),
                c64::new(0.485318, 0.772833),
                c64::new(0.754468, 0.582057),
                c64::new(0.348919, 0.228684),
                c64::new(0.861125, 0.724930),
                c64::new(0.771415, 0.568626),
                c64::new(0.648784, 0.670444),
                c64::new(0.139906, 0.123516),
                c64::new(0.798978, 0.029723),
                c64::new(0.429955, 0.586904),
                c64::new(0.139709, 0.848911),
                c64::new(0.822904, 0.593750),
                c64::new(0.373495, 0.981987),
                c64::new(0.137763, 0.722706),
                c64::new(0.378900, 0.295119),
                c64::new(0.733349, 0.028802),
                c64::new(0.754157, 0.493659),
                c64::new(0.910687, 0.979098),
                c64::new(0.139389, 0.177713),
                c64::new(0.872771, 0.265941),
                c64::new(0.134669, 0.762190),
                c64::new(0.505089, 0.148699),
                c64::new(0.491902, 0.859694),
                c64::new(0.846171, 0.933060),
                c64::new(0.845493, 0.605066),
                c64::new(0.380912, 0.080237),
                c64::new(0.942480, 0.495345),
                c64::new(0.799450, 0.765162),
                c64::new(0.309441, 0.158130),
                c64::new(0.653779, 0.939343),
                c64::new(0.281960, 0.331618),
                c64::new(0.406322, 0.469164),
                c64::new(0.957345, 0.432556),
                c64::new(0.197621, 0.267563),
                c64::new(0.067700, 0.960559),
                c64::new(0.802247, 0.963885),
                c64::new(0.326979, 0.345922),
                c64::new(0.313192, 0.746932),
                c64::new(0.997904, 0.429886),
                c64::new(0.237066, 0.845585),
                c64::new(0.171762, 0.632153),
                c64::new(0.467327, 0.482712),
                c64::new(0.807306, 0.039551),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.039169, 0.055922),
                c64::new(0.590870, 0.793592),
                c64::new(0.748642, 0.363946),
                c64::new(0.768724, 0.280382),
                c64::new(0.164883, 0.027646),
                c64::new(0.598699, 0.267952),
                c64::new(0.945008, 0.743813),
                c64::new(0.721549, 0.238672),
                c64::new(0.900901, 0.455882),
                c64::new(0.152397, 0.400620),
                c64::new(0.478101, 0.225782),
                c64::new(0.486973, 0.438368),
                c64::new(0.022831, 0.775907),
                c64::new(0.178928, 0.792794),
                c64::new(0.878480, 0.403492),
                c64::new(0.733611, 0.536099),
                c64::new(0.146996, 0.059703),
                c64::new(0.152945, 0.958547),
                c64::new(0.169830, 0.260092),
                c64::new(0.612048, 0.646116),
                c64::new(0.411249, 0.379732),
                c64::new(0.621532, 0.628702),
                c64::new(0.396252, 0.283108),
                c64::new(0.737276, 0.512216),
                c64::new(0.394047, 0.355116),
                c64::new(0.407149, 0.168881),
                c64::new(0.833917, 0.637144),
                c64::new(0.544205, 0.092340),
                c64::new(0.564519, 0.106643),
                c64::new(0.745555, 0.865885),
                c64::new(0.800167, 0.206768),
                c64::new(0.825834, 0.207477),
                c64::new(0.539784, 0.053009),
                c64::new(0.151371, 0.605693),
                c64::new(0.932419, 0.888430),
                c64::new(0.829147, 0.689129),
                c64::new(0.548113, 0.651277),
                c64::new(0.606840, 0.125448),
                c64::new(0.673024, 0.573973),
                c64::new(0.009648, 0.574117),
                c64::new(0.199809, 0.257471),
                c64::new(0.076281, 0.766166),
                c64::new(0.191524, 0.039707),
                c64::new(0.351735, 0.275420),
                c64::new(0.750760, 0.933292),
                c64::new(0.125400, 0.303908),
                c64::new(0.829364, 0.512639),
                c64::new(0.509435, 0.242217),
                c64::new(0.076935, 0.365274),
                c64::new(0.992927, 0.465988),
                c64::new(0.721725, 0.524211),
                c64::new(0.353123, 0.659698),
                c64::new(0.234704, 0.223865),
                c64::new(0.747632, 0.558185),
                c64::new(0.199184, 0.547359),
                c64::new(0.680125, 0.579314),
                c64::new(0.292683, 0.150177),
                c64::new(0.520877, 0.665286),
                c64::new(0.181009, 0.200895),
                c64::new(0.300164, 0.774376),
                c64::new(0.801306, 0.423175),
                c64::new(0.311131, 0.263651),
                c64::new(0.070988, 0.432310),
                c64::new(0.385315, 0.684269),
                c64::new(0.578428, 0.318080),
                c64::new(0.990976, 0.648917),
                c64::new(0.055406, 0.141073),
                c64::new(0.019972, 0.754003),
                c64::new(0.294881, 0.208275),
                c64::new(0.123810, 0.455336),
                c64::new(0.099450, 0.194655),
                c64::new(0.444951, 0.332398),
                c64::new(0.420025, 0.209846),
                c64::new(0.339397, 0.849301),
                c64::new(0.706286, 0.070328),
                c64::new(0.552971, 0.400951),
                c64::new(0.382099, 0.483153),
                c64::new(0.132283, 0.988319),
                c64::new(0.884262, 0.065757),
                c64::new(0.798132, 0.658707),
                c64::new(0.113247, 0.857510),
                c64::new(0.406661, 0.685097),
                c64::new(0.631435, 0.935519),
                c64::new(0.119072, 0.097054),
                c64::new(0.197303, 0.492401),
                c64::new(0.527270, 0.929175),
                c64::new(0.236637, 0.528237),
                c64::new(0.152955, 0.498006),
                c64::new(0.162884, 0.625559),
                c64::new(0.907460, 0.018217),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.483965, 0.800964),
                c64::new(0.667206, 0.386462),
                c64::new(0.796224, 0.661389),
                c64::new(0.388711, 0.635504),
                c64::new(0.980319, 0.880237),
                c64::new(0.346505, 0.727447),
                c64::new(0.325753, 0.313542),
                c64::new(0.672522, 0.050786),
                c64::new(0.764330, 0.488853),
                c64::new(0.171476, 0.843934),
                c64::new(0.250870, 0.634494),
                c64::new(0.494856, 0.535428),
                c64::new(0.981350, 0.293585),
                c64::new(0.902472, 0.331473),
                c64::new(0.760460, 0.929733),
                c64::new(0.560599, 0.110265),
                c64::new(0.063759, 0.787113),
                c64::new(0.892287, 0.642687),
                c64::new(0.454077, 0.544921),
                c64::new(0.679872, 0.714917),
                c64::new(0.162272, 0.683906),
                c64::new(0.934624, 0.052388),
                c64::new(0.166343, 0.979865),
                c64::new(0.651687, 0.810362),
                c64::new(0.821362, 0.835319),
                c64::new(0.639555, 0.453515),
                c64::new(0.307161, 0.782103),
                c64::new(0.090268, 0.559933),
                c64::new(0.028995, 0.081449),
                c64::new(0.686509, 0.972855),
                c64::new(0.517467, 0.824392),
                c64::new(0.250160, 0.981315),
                c64::new(0.745133, 0.491498),
                c64::new(0.111877, 0.325447),
                c64::new(0.394458, 0.442360),
                c64::new(0.085827, 0.153762),
                c64::new(0.036105, 0.324350),
                c64::new(0.011247, 0.985951),
                c64::new(0.542966, 0.402390),
                c64::new(0.592787, 0.013654),
                c64::new(0.570449, 0.193372),
                c64::new(0.532274, 0.581500),
                c64::new(0.138512, 0.587447),
                c64::new(0.555635, 0.065408),
                c64::new(0.489196, 0.993022),
                c64::new(0.149267, 0.091271),
                c64::new(0.370291, 0.868358),
                c64::new(0.548562, 0.896719),
                c64::new(0.135477, 0.314728),
                c64::new(0.173887, 0.650567),
                c64::new(0.003519, 0.739062),
                c64::new(0.008450, 0.728390),
                c64::new(0.445266, 0.949088),
                c64::new(0.489544, 0.537289),
                c64::new(0.500328, 0.928926),
                c64::new(0.131734, 0.056522),
                c64::new(0.701701, 0.867558),
                c64::new(0.101820, 0.405368),
                c64::new(0.450557, 0.031509),
                c64::new(0.122419, 0.975564),
                c64::new(0.365257, 0.534415),
                c64::new(0.421144, 0.857562),
                c64::new(0.440903, 0.147258),
                c64::new(0.972227, 0.117599),
                c64::new(0.236920, 0.070007),
                c64::new(0.456752, 0.822350),
                c64::new(0.790688, 0.853018),
                c64::new(0.183244, 0.662004),
                c64::new(0.902872, 0.047397),
                c64::new(0.264838, 0.835089),
                c64::new(0.435313, 0.010885),
                c64::new(0.799335, 0.025390),
                c64::new(0.209371, 0.507826),
                c64::new(0.610896, 0.569685),
                c64::new(0.729988, 0.644768),
                c64::new(0.933647, 0.133526),
                c64::new(0.368506, 0.083344),
                c64::new(0.914414, 0.814043),
                c64::new(0.495673, 0.497073),
                c64::new(0.824076, 0.079490),
                c64::new(0.441706, 0.119901),
                c64::new(0.770456, 0.101977),
                c64::new(0.135753, 0.369606),
                c64::new(0.665904, 0.776744),
                c64::new(0.450603, 0.328408),
                c64::new(0.029306, 0.506737),
                c64::new(0.958892, 0.115982),
                c64::new(0.230828, 0.493861),
                c64::new(0.039050, 0.348988),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.563969, 0.215230),
                c64::new(0.200860, 0.558926),
                c64::new(0.017842, 0.386949),
                c64::new(0.501512, 0.783509),
                c64::new(0.004253, 0.267862),
                c64::new(0.304512, 0.787357),
                c64::new(0.593255, 0.854689),
                c64::new(0.914290, 0.554721),
                c64::new(0.684163, 0.138557),
                c64::new(0.172521, 0.132677),
                c64::new(0.480114, 0.605295),
                c64::new(0.275018, 0.552696),
                c64::new(0.516759, 0.012574),
                c64::new(0.514930, 0.721953),
                c64::new(0.855787, 0.878680),
                c64::new(0.477766, 0.750273),
                c64::new(0.619594, 0.269752),
                c64::new(0.767082, 0.848524),
                c64::new(0.480979, 0.334259),
                c64::new(0.864691, 0.584055),
                c64::new(0.630097, 0.186060),
                c64::new(0.265688, 0.300823),
                c64::new(0.660976, 0.281635),
                c64::new(0.501005, 0.734647),
                c64::new(0.623826, 0.608322),
                c64::new(0.110859, 0.601506),
                c64::new(0.698188, 0.758936),
                c64::new(0.835168, 0.029124),
                c64::new(0.004484, 0.381556),
                c64::new(0.343663, 0.607716),
                c64::new(0.720473, 0.103217),
                c64::new(0.383911, 0.149037),
                c64::new(0.018212, 0.580847),
                c64::new(0.533480, 0.231564),
                c64::new(0.430386, 0.586856),
                c64::new(0.419748, 0.841936),
                c64::new(0.113503, 0.172716),
                c64::new(0.716379, 0.093537),
                c64::new(0.659701, 0.775674),
                c64::new(0.452776, 0.982651),
                c64::new(0.997484, 0.275631),
                c64::new(0.368194, 0.567956),
                c64::new(0.587550, 0.672910),
                c64::new(0.730038, 0.369474),
                c64::new(0.221113, 0.635320),
                c64::new(0.319171, 0.096274),
                c64::new(0.705871, 0.129025),
                c64::new(0.873987, 0.382525),
                c64::new(0.098808, 0.639908),
                c64::new(0.984377, 0.470093),
                c64::new(0.844070, 0.918781),
                c64::new(0.346543, 0.426029),
                c64::new(0.661011, 0.562758),
                c64::new(0.044625, 0.467252),
                c64::new(0.009241, 0.515385),
                c64::new(0.565023, 0.136493),
                c64::new(0.051247, 0.140898),
                c64::new(0.197681, 0.736551),
                c64::new(0.699107, 0.516610),
                c64::new(0.299135, 0.938436),
                c64::new(0.489430, 0.804633),
                c64::new(0.046672, 0.074616),
                c64::new(0.225431, 0.157307),
                c64::new(0.281284, 0.665819),
                c64::new(0.758063, 0.371717),
                c64::new(0.339990, 0.229678),
                c64::new(0.111099, 0.021764),
                c64::new(0.300737, 0.363682),
                c64::new(0.127251, 0.471653),
                c64::new(0.420396, 0.493507),
                c64::new(0.799774, 0.039209),
                c64::new(0.498668, 0.250861),
                c64::new(0.119869, 0.934243),
                c64::new(0.728724, 0.995256),
                c64::new(0.055502, 0.932074),
                c64::new(0.321959, 0.314326),
                c64::new(0.426325, 0.522986),
                c64::new(0.016314, 0.209950),
                c64::new(0.445037, 0.057052),
                c64::new(0.214954, 0.670359),
                c64::new(0.062921, 0.893370),
                c64::new(0.310017, 0.443929),
                c64::new(0.033505, 0.215010),
                c64::new(0.433145, 0.836948),
                c64::new(0.321161, 0.434003),
                c64::new(0.335604, 0.753146),
                c64::new(0.232997, 0.051175),
                c64::new(0.449838, 0.822531),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.381605, 0.334681),
                c64::new(0.560213, 0.536739),
                c64::new(0.414463, 0.861030),
                c64::new(0.861617, 0.331007),
                c64::new(0.412024, 0.304047),
                c64::new(0.398990, 0.577441),
                c64::new(0.490449, 0.342271),
                c64::new(0.941907, 0.886512),
                c64::new(0.996036, 0.321067),
                c64::new(0.477521, 0.630002),
                c64::new(0.325834, 0.063338),
                c64::new(0.717430, 0.927620),
                c64::new(0.346095, 0.095020),
                c64::new(0.298347, 0.690078),
                c64::new(0.006054, 0.705601),
                c64::new(0.367199, 0.170317),
                c64::new(0.131273, 0.020678),
                c64::new(0.435624, 0.672278),
                c64::new(0.122184, 0.367257),
                c64::new(0.813396, 0.983972),
                c64::new(0.042951, 0.649093),
                c64::new(0.084975, 0.420433),
                c64::new(0.501455, 0.109287),
                c64::new(0.959981, 0.473198),
                c64::new(0.474251, 0.624029),
                c64::new(0.793972, 0.619738),
                c64::new(0.258283, 0.608500),
                c64::new(0.820788, 0.050570),
                c64::new(0.131801, 0.704912),
                c64::new(0.242098, 0.781407),
                c64::new(0.472412, 0.476462),
                c64::new(0.172876, 0.420547),
                c64::new(0.534554, 0.855458),
                c64::new(0.864444, 0.549792),
                c64::new(0.566688, 0.398758),
                c64::new(0.721648, 0.089724),
                c64::new(0.202547, 0.189317),
                c64::new(0.385434, 0.614045),
                c64::new(0.563256, 0.491295),
                c64::new(0.713063, 0.525299),
                c64::new(0.541740, 0.523136),
                c64::new(0.817027, 0.799198),
                c64::new(0.466757, 0.657703),
                c64::new(0.602946, 0.934538),
                c64::new(0.439136, 0.291484),
                c64::new(0.573826, 0.471973),
                c64::new(0.332450, 0.203145),
                c64::new(0.756586, 0.814000),
                c64::new(0.485232, 0.913642),
                c64::new(0.395099, 0.910118),
                c64::new(0.603470, 0.775366),
                c64::new(0.479518, 0.616871),
                c64::new(0.117789, 0.250279),
                c64::new(0.316659, 0.390211),
                c64::new(0.961454, 0.668674),
                c64::new(0.866251, 0.725367),
                c64::new(0.057382, 0.186860),
                c64::new(0.492004, 0.176278),
                c64::new(0.801255, 0.562411),
                c64::new(0.443016, 0.801791),
                c64::new(0.142786, 0.022347),
                c64::new(0.123558, 0.500740),
                c64::new(0.951937, 0.370175),
                c64::new(0.913617, 0.017096),
                c64::new(0.111700, 0.861513),
                c64::new(0.454589, 0.404850),
                c64::new(0.173981, 0.124860),
                c64::new(0.357380, 0.877563),
                c64::new(0.269623, 0.694489),
                c64::new(0.173250, 0.465384),
                c64::new(0.264800, 0.287655),
                c64::new(0.975610, 0.046665),
                c64::new(0.305468, 0.866713),
                c64::new(0.426472, 0.708351),
                c64::new(0.058702, 0.194481),
                c64::new(0.607929, 0.521343),
                c64::new(0.436087, 0.566151),
                c64::new(0.295087, 0.658905),
                c64::new(0.189262, 0.890021),
                c64::new(0.651342, 0.561763),
                c64::new(0.013591, 0.667646),
                c64::new(0.490111, 0.807117),
                c64::new(0.049202, 0.688385),
                c64::new(0.722299, 0.201393),
                c64::new(0.015583, 0.436442),
                c64::new(0.217194, 0.073205),
                c64::new(0.408862, 0.686188),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.573542, 0.388167),
                c64::new(0.933630, 0.351839),
                c64::new(0.703146, 0.847474),
                c64::new(0.248465, 0.132849),
                c64::new(0.076664, 0.553616),
                c64::new(0.289764, 0.851508),
                c64::new(0.694958, 0.277080),
                c64::new(0.512564, 0.025005),
                c64::new(0.015380, 0.550337),
                c64::new(0.396079, 0.363697),
                c64::new(0.004287, 0.784070),
                c64::new(0.689216, 0.873509),
                c64::new(0.542274, 0.879788),
                c64::new(0.973676, 0.441473),
                c64::new(0.096915, 0.098194),
                c64::new(0.623378, 0.162147),
                c64::new(0.584288, 0.757530),
                c64::new(0.093668, 0.549968),
                c64::new(0.415618, 0.994629),
                c64::new(0.677865, 0.598068),
                c64::new(0.568144, 0.288480),
                c64::new(0.321794, 0.382578),
                c64::new(0.587715, 0.840209),
                c64::new(0.860247, 0.164856),
                c64::new(0.514940, 0.919737),
                c64::new(0.782391, 0.834831),
                c64::new(0.646076, 0.948930),
                c64::new(0.747259, 0.306129),
                c64::new(0.599216, 0.640426),
                c64::new(0.296840, 0.422823),
                c64::new(0.311353, 0.756295),
                c64::new(0.777810, 0.758092),
                c64::new(0.815063, 0.461959),
                c64::new(0.684121, 0.263384),
                c64::new(0.184665, 0.990219),
                c64::new(0.002221, 0.454991),
                c64::new(0.801437, 0.995631),
                c64::new(0.671939, 0.318441),
                c64::new(0.969676, 0.772701),
                c64::new(0.430074, 0.540313),
                c64::new(0.891830, 0.507685),
                c64::new(0.167946, 0.406441),
                c64::new(0.682917, 0.055576),
                c64::new(0.445727, 0.590770),
                c64::new(0.891029, 0.523637),
                c64::new(0.635353, 0.664379),
                c64::new(0.163148, 0.127204),
                c64::new(0.803838, 0.957595),
                c64::new(0.235096, 0.623109),
                c64::new(0.768703, 0.694038),
                c64::new(0.379656, 0.317182),
                c64::new(0.637745, 0.389882),
                c64::new(0.516254, 0.426718),
                c64::new(0.314915, 0.771417),
                c64::new(0.500097, 0.104434),
                c64::new(0.647588, 0.223342),
                c64::new(0.774185, 0.184224),
                c64::new(0.960569, 0.062188),
                c64::new(0.453213, 0.008999),
                c64::new(0.709832, 0.812121),
                c64::new(0.682145, 0.462596),
                c64::new(0.083441, 0.795369),
                c64::new(0.159183, 0.982480),
                c64::new(0.426025, 0.760261),
                c64::new(0.148704, 0.293468),
                c64::new(0.832499, 0.765547),
                c64::new(0.228840, 0.472538),
                c64::new(0.059984, 0.390391),
                c64::new(0.482423, 0.957724),
                c64::new(0.842010, 0.987434),
                c64::new(0.385056, 0.052797),
                c64::new(0.977742, 0.422514),
                c64::new(0.102317, 0.434259),
                c64::new(0.255053, 0.538431),
                c64::new(0.926015, 0.508161),
                c64::new(0.898991, 0.845158),
                c64::new(0.588937, 0.823437),
                c64::new(0.487999, 0.869617),
                c64::new(0.154551, 0.400082),
                c64::new(0.010964, 0.004314),
                c64::new(0.734884, 0.272391),
                c64::new(0.272488, 0.328222),
                c64::new(0.179140, 0.871644),
                c64::new(0.690670, 0.929218),
                c64::new(0.345124, 0.628851),
                c64::new(0.202151, 0.822823),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.735244, 0.648389),
                c64::new(0.404006, 0.742885),
                c64::new(0.841040, 0.372941),
                c64::new(0.376585, 0.171119),
                c64::new(0.379345, 0.290793),
                c64::new(0.974902, 0.284359),
                c64::new(0.339845, 0.372387),
                c64::new(0.510283, 0.097024),
                c64::new(0.313688, 0.202658),
                c64::new(0.901789, 0.659984),
                c64::new(0.161707, 0.397154),
                c64::new(0.366609, 0.727116),
                c64::new(0.969650, 0.997864),
                c64::new(0.562091, 0.464392),
                c64::new(0.235726, 0.003537),
                c64::new(0.342560, 0.113092),
                c64::new(0.866232, 0.116060),
                c64::new(0.050457, 0.039689),
                c64::new(0.059197, 0.395785),
                c64::new(0.729158, 0.362577),
                c64::new(0.219059, 0.033233),
                c64::new(0.037553, 0.799813),
                c64::new(0.703727, 0.548888),
                c64::new(0.538787, 0.150539),
                c64::new(0.773617, 0.052906),
                c64::new(0.689023, 0.666266),
                c64::new(0.488228, 0.036400),
                c64::new(0.699439, 0.589869),
                c64::new(0.994067, 0.012801),
                c64::new(0.230714, 0.986099),
                c64::new(0.558274, 0.294239),
                c64::new(0.797241, 0.917226),
                c64::new(0.721611, 0.000435),
                c64::new(0.792361, 0.968039),
                c64::new(0.440256, 0.682418),
                c64::new(0.525702, 0.179734),
                c64::new(0.864027, 0.799144),
                c64::new(0.828210, 0.069105),
                c64::new(0.448220, 0.890020),
                c64::new(0.253954, 0.189809),
                c64::new(0.274033, 0.096953),
                c64::new(0.929008, 0.295046),
                c64::new(0.301024, 0.798143),
                c64::new(0.308140, 0.832541),
                c64::new(0.773130, 0.536980),
                c64::new(0.022837, 0.687518),
                c64::new(0.849632, 0.856421),
                c64::new(0.125587, 0.859170),
                c64::new(0.679994, 0.715336),
                c64::new(0.262491, 0.105062),
                c64::new(0.078057, 0.659324),
                c64::new(0.041488, 0.683008),
                c64::new(0.547994, 0.040802),
                c64::new(0.034698, 0.190690),
                c64::new(0.272914, 0.140549),
                c64::new(0.316603, 0.867679),
                c64::new(0.320706, 0.533576),
                c64::new(0.238152, 0.751955),
                c64::new(0.335698, 0.117406),
                c64::new(0.953892, 0.288300),
                c64::new(0.796714, 0.844719),
                c64::new(0.108367, 0.975809),
                c64::new(0.155480, 0.781712),
                c64::new(0.402051, 0.903873),
                c64::new(0.457915, 0.555736),
                c64::new(0.663592, 0.583719),
                c64::new(0.950692, 0.983393),
                c64::new(0.093302, 0.447040),
                c64::new(0.722098, 0.611867),
                c64::new(0.063848, 0.063716),
                c64::new(0.897905, 0.015183),
                c64::new(0.545538, 0.806946),
                c64::new(0.411816, 0.995557),
                c64::new(0.439471, 0.609904),
                c64::new(0.721325, 0.224354),
                c64::new(0.543109, 0.625237),
                c64::new(0.491476, 0.517450),
                c64::new(0.048539, 0.347591),
                c64::new(0.361001, 0.007827),
                c64::new(0.976649, 0.193139),
                c64::new(0.161901, 0.907522),
                c64::new(0.652200, 0.511186),
                c64::new(0.316799, 0.007711),
                c64::new(0.530285, 0.254500),
                c64::new(0.465437, 0.978259),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.740842, 0.783866),
                c64::new(0.033385, 0.042925),
                c64::new(0.575596, 0.812848),
                c64::new(0.904745, 0.508162),
                c64::new(0.472407, 0.052847),
                c64::new(0.122701, 0.368541),
                c64::new(0.251666, 0.743719),
                c64::new(0.462145, 0.980271),
                c64::new(0.583311, 0.242405),
                c64::new(0.320579, 0.455996),
                c64::new(0.085088, 0.882172),
                c64::new(0.884200, 0.597737),
                c64::new(0.709772, 0.860837),
                c64::new(0.882575, 0.229343),
                c64::new(0.785658, 0.435296),
                c64::new(0.010676, 0.618796),
                c64::new(0.549745, 0.331950),
                c64::new(0.195679, 0.631380),
                c64::new(0.984996, 0.975261),
                c64::new(0.248163, 0.459661),
                c64::new(0.152879, 0.480971),
                c64::new(0.336315, 0.540395),
                c64::new(0.300499, 0.432347),
                c64::new(0.383623, 0.785044),
                c64::new(0.316381, 0.691250),
                c64::new(0.877756, 0.115295),
                c64::new(0.357677, 0.445853),
                c64::new(0.657477, 0.809514),
                c64::new(0.939017, 0.197572),
                c64::new(0.679222, 0.413273),
                c64::new(0.678824, 0.530290),
                c64::new(0.238824, 0.971717),
                c64::new(0.701926, 0.760123),
                c64::new(0.539227, 0.158329),
                c64::new(0.309239, 0.022726),
                c64::new(0.251704, 0.546854),
                c64::new(0.271569, 0.411367),
                c64::new(0.227215, 0.526785),
                c64::new(0.919411, 0.734374),
                c64::new(0.718807, 0.683201),
                c64::new(0.780541, 0.135578),
                c64::new(0.770898, 0.642002),
                c64::new(0.612331, 0.702958),
                c64::new(0.598458, 0.310000),
                c64::new(0.949549, 0.513887),
                c64::new(0.470771, 0.856472),
                c64::new(0.797922, 0.502809),
                c64::new(0.912868, 0.474236),
                c64::new(0.838611, 0.095393),
                c64::new(0.992794, 0.425593),
                c64::new(0.037090, 0.691406),
                c64::new(0.052643, 0.667864),
                c64::new(0.982393, 0.692716),
                c64::new(0.529457, 0.603488),
                c64::new(0.670193, 0.336718),
                c64::new(0.556342, 0.271673),
                c64::new(0.469207, 0.272442),
                c64::new(0.974720, 0.329922),
                c64::new(0.594939, 0.340691),
                c64::new(0.278813, 0.306808),
                c64::new(0.861661, 0.850804),
                c64::new(0.041141, 0.953087),
                c64::new(0.196310, 0.962632),
                c64::new(0.080603, 0.641448),
                c64::new(0.492321, 0.754173),
                c64::new(0.716660, 0.205709),
                c64::new(0.961207, 0.561262),
                c64::new(0.317460, 0.921434),
                c64::new(0.399585, 0.280991),
                c64::new(0.222316, 0.465773),
                c64::new(0.879807, 0.317272),
                c64::new(0.829285, 0.625259),
                c64::new(0.418587, 0.176443),
                c64::new(0.217018, 0.340260),
                c64::new(0.607509, 0.352298),
                c64::new(0.142951, 0.054512),
                c64::new(0.130994, 0.346119),
                c64::new(0.623081, 0.658988),
                c64::new(0.387066, 0.909583),
                c64::new(0.614997, 0.503082),
                c64::new(0.022170, 0.969789),
                c64::new(0.309229, 0.064041),
                c64::new(0.526816, 0.491999),
                c64::new(0.399190, 0.940576),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.063767, 0.969424),
                c64::new(0.791789, 0.340409),
                c64::new(0.569635, 0.428131),
                c64::new(0.567881, 0.622286),
                c64::new(0.822358, 0.401580),
                c64::new(0.075380, 0.043813),
                c64::new(0.372989, 0.722281),
                c64::new(0.567670, 0.469555),
                c64::new(0.542539, 0.712902),
                c64::new(0.759956, 0.109597),
                c64::new(0.762471, 0.135956),
                c64::new(0.673440, 0.215577),
                c64::new(0.699551, 0.165429),
                c64::new(0.840807, 0.410416),
                c64::new(0.457256, 0.129448),
                c64::new(0.055652, 0.037775),
                c64::new(0.018296, 0.447113),
                c64::new(0.544935, 0.366760),
                c64::new(0.546950, 0.497241),
                c64::new(0.960863, 0.010089),
                c64::new(0.475636, 0.881121),
                c64::new(0.927500, 0.348824),
                c64::new(0.476186, 0.342071),
                c64::new(0.989443, 0.904209),
                c64::new(0.681820, 0.013746),
                c64::new(0.251894, 0.048605),
                c64::new(0.733044, 0.103096),
                c64::new(0.667786, 0.375581),
                c64::new(0.239180, 0.293539),
                c64::new(0.093310, 0.780990),
                c64::new(0.110455, 0.282434),
                c64::new(0.453472, 0.046739),
                c64::new(0.267683, 0.872115),
                c64::new(0.591873, 0.452040),
                c64::new(0.723499, 0.927522),
                c64::new(0.999531, 0.119166),
                c64::new(0.662849, 0.712324),
                c64::new(0.455840, 0.021098),
                c64::new(0.935933, 0.532738),
                c64::new(0.300185, 0.904884),
                c64::new(0.636084, 0.517001),
                c64::new(0.673406, 0.070859),
                c64::new(0.571675, 0.111066),
                c64::new(0.567300, 0.719249),
                c64::new(0.360746, 0.446870),
                c64::new(0.298526, 0.871757),
                c64::new(0.805616, 0.975627),
                c64::new(0.752524, 0.121227),
                c64::new(0.433470, 0.667452),
                c64::new(0.086683, 0.998162),
                c64::new(0.547694, 0.406948),
                c64::new(0.848343, 0.534504),
                c64::new(0.392504, 0.620580),
                c64::new(0.032446, 0.713862),
                c64::new(0.135602, 0.651080),
                c64::new(0.322579, 0.643641),
                c64::new(0.708088, 0.583642),
                c64::new(0.802348, 0.539662),
                c64::new(0.805814, 0.925074),
                c64::new(0.676603, 0.402843),
                c64::new(0.394300, 0.501153),
                c64::new(0.717314, 0.645844),
                c64::new(0.771667, 0.559784),
                c64::new(0.411441, 0.157799),
                c64::new(0.652021, 0.190861),
                c64::new(0.018805, 0.342677),
                c64::new(0.254884, 0.541318),
                c64::new(0.735083, 0.432794),
                c64::new(0.413014, 0.018520),
                c64::new(0.581146, 0.818187),
                c64::new(0.470701, 0.038610),
                c64::new(0.516790, 0.948117),
                c64::new(0.420665, 0.006452),
                c64::new(0.769275, 0.051267),
                c64::new(0.548390, 0.934181),
                c64::new(0.793766, 0.647510),
                c64::new(0.635297, 0.601192),
                c64::new(0.739021, 0.547473),
                c64::new(0.451937, 0.859916),
                c64::new(0.770549, 0.283643),
                c64::new(0.082991, 0.376299),
                c64::new(0.113400, 0.640015),
                c64::new(0.917996, 0.521593),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.163089, 0.273389),
                c64::new(0.811467, 0.633897),
                c64::new(0.237482, 0.157601),
                c64::new(0.876689, 0.419349),
                c64::new(0.377798, 0.350626),
                c64::new(0.398159, 0.260687),
                c64::new(0.168427, 0.267953),
                c64::new(0.617739, 0.621212),
                c64::new(0.474590, 0.457928),
                c64::new(0.589916, 0.095024),
                c64::new(0.560993, 0.042352),
                c64::new(0.243610, 0.399045),
                c64::new(0.104052, 0.822881),
                c64::new(0.600130, 0.714676),
                c64::new(0.228513, 0.791459),
                c64::new(0.112508, 0.591510),
                c64::new(0.530863, 0.351889),
                c64::new(0.281428, 0.139930),
                c64::new(0.533596, 0.437076),
                c64::new(0.479364, 0.294141),
                c64::new(0.689434, 0.192899),
                c64::new(0.957118, 0.600044),
                c64::new(0.888687, 0.262283),
                c64::new(0.751957, 0.362791),
                c64::new(0.465120, 0.534709),
                c64::new(0.245114, 0.408272),
                c64::new(0.110807, 0.079697),
                c64::new(0.902414, 0.237419),
                c64::new(0.137727, 0.858949),
                c64::new(0.648885, 0.874984),
                c64::new(0.776626, 0.657961),
                c64::new(0.003596, 0.388133),
                c64::new(0.414637, 0.883650),
                c64::new(0.864253, 0.306317),
                c64::new(0.974860, 0.163915),
                c64::new(0.133261, 0.134175),
                c64::new(0.106598, 0.639122),
                c64::new(0.983986, 0.236873),
                c64::new(0.988355, 0.949175),
                c64::new(0.473965, 0.287890),
                c64::new(0.014822, 0.112255),
                c64::new(0.319707, 0.598309),
                c64::new(0.695511, 0.720575),
                c64::new(0.892346, 0.174677),
                c64::new(0.333509, 0.767643),
                c64::new(0.877922, 0.811274),
                c64::new(0.177003, 0.725204),
                c64::new(0.562330, 0.927592),
                c64::new(0.677940, 0.473708),
                c64::new(0.931128, 0.769879),
                c64::new(0.705611, 0.743969),
                c64::new(0.940165, 0.029474),
                c64::new(0.624727, 0.146186),
                c64::new(0.884796, 0.728655),
                c64::new(0.270486, 0.631052),
                c64::new(0.238674, 0.530211),
                c64::new(0.380656, 0.118154),
                c64::new(0.320399, 0.615927),
                c64::new(0.775070, 0.067789),
                c64::new(0.617536, 0.188078),
                c64::new(0.423775, 0.974675),
                c64::new(0.995372, 0.984541),
                c64::new(0.195728, 0.500860),
                c64::new(0.488122, 0.174121),
                c64::new(0.995921, 0.874693),
                c64::new(0.822936, 0.576571),
                c64::new(0.221050, 0.729484),
                c64::new(0.621477, 0.962220),
                c64::new(0.435125, 0.614258),
                c64::new(0.197838, 0.553756),
                c64::new(0.806655, 0.700758),
                c64::new(0.271395, 0.119299),
                c64::new(0.116925, 0.143481),
                c64::new(0.706521, 0.121832),
                c64::new(0.402241, 0.227492),
                c64::new(0.629821, 0.534489),
                c64::new(0.107307, 0.514986),
                c64::new(0.741552, 0.516972),
                c64::new(0.320052, 0.096161),
                c64::new(0.600597, 0.605190),
                c64::new(0.045159, 0.975452),
                c64::new(0.727018, 0.293743),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.799229, 0.849440),
                c64::new(0.099408, 0.593789),
                c64::new(0.690383, 0.925689),
                c64::new(0.000777, 0.854768),
                c64::new(0.311094, 0.640667),
                c64::new(0.230836, 0.098775),
                c64::new(0.671578, 0.265600),
                c64::new(0.331314, 0.919857),
                c64::new(0.965351, 0.769148),
                c64::new(0.726478, 0.154061),
                c64::new(0.419581, 0.792037),
                c64::new(0.433292, 0.819654),
                c64::new(0.704997, 0.266928),
                c64::new(0.779640, 0.440752),
                c64::new(0.493419, 0.321085),
                c64::new(0.413336, 0.741590),
                c64::new(0.979612, 0.935775),
                c64::new(0.283523, 0.203253),
                c64::new(0.756698, 0.352677),
                c64::new(0.366930, 0.627752),
                c64::new(0.556794, 0.418028),
                c64::new(0.893619, 0.448382),
                c64::new(0.883760, 0.254408),
                c64::new(0.195357, 0.291014),
                c64::new(0.135402, 0.608488),
                c64::new(0.250140, 0.522668),
                c64::new(0.396578, 0.651867),
                c64::new(0.808536, 0.304212),
                c64::new(0.482107, 0.176170),
                c64::new(0.795186, 0.822652),
                c64::new(0.956763, 0.915771),
                c64::new(0.465354, 0.837144),
                c64::new(0.831110, 0.145040),
                c64::new(0.734009, 0.107015),
                c64::new(0.028545, 0.238481),
                c64::new(0.165469, 0.455555),
                c64::new(0.420377, 0.333274),
                c64::new(0.153989, 0.879627),
                c64::new(0.260940, 0.588463),
                c64::new(0.444215, 0.123470),
                c64::new(0.639518, 0.256302),
                c64::new(0.213616, 0.223893),
                c64::new(0.449585, 0.799953),
                c64::new(0.447798, 0.432008),
                c64::new(0.476905, 0.565120),
                c64::new(0.730520, 0.436029),
                c64::new(0.201089, 0.692167),
                c64::new(0.939982, 0.266081),
                c64::new(0.233604, 0.730289),
                c64::new(0.477286, 0.038042),
                c64::new(0.584854, 0.206653),
                c64::new(0.555491, 0.221860),
                c64::new(0.696481, 0.590005),
                c64::new(0.822097, 0.027754),
                c64::new(0.295132, 0.257796),
                c64::new(0.433091, 0.071554),
                c64::new(0.592868, 0.573857),
                c64::new(0.542998, 0.689213),
                c64::new(0.343058, 0.437967),
                c64::new(0.098596, 0.941911),
                c64::new(0.885031, 0.234322),
                c64::new(0.348074, 0.780896),
                c64::new(0.641003, 0.332914),
                c64::new(0.732398, 0.034702),
                c64::new(0.996535, 0.487708),
                c64::new(0.777322, 0.903445),
                c64::new(0.165493, 0.729339),
                c64::new(0.343228, 0.118311),
                c64::new(0.613768, 0.935661),
                c64::new(0.036355, 0.612588),
                c64::new(0.443394, 0.886640),
                c64::new(0.741300, 0.811158),
                c64::new(0.964496, 0.952623),
                c64::new(0.834799, 0.696315),
                c64::new(0.156811, 0.501020),
                c64::new(0.513243, 0.510997),
                c64::new(0.753263, 0.372360),
                c64::new(0.948367, 0.387329),
                c64::new(0.616189, 0.382091),
                c64::new(0.241661, 0.029919),
                c64::new(0.586426, 0.459524),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.672241, 0.016195),
                c64::new(0.223985, 0.694470),
                c64::new(0.762024, 0.430071),
                c64::new(0.379920, 0.586203),
                c64::new(0.062118, 0.232354),
                c64::new(0.183123, 0.842981),
                c64::new(0.231409, 0.122464),
                c64::new(0.676716, 0.365949),
                c64::new(0.710958, 0.965847),
                c64::new(0.524080, 0.983746),
                c64::new(0.310669, 0.917988),
                c64::new(0.981849, 0.818970),
                c64::new(0.165776, 0.398590),
                c64::new(0.949855, 0.684631),
                c64::new(0.225315, 0.871847),
                c64::new(0.627879, 0.857924),
                c64::new(0.823507, 0.494359),
                c64::new(0.602040, 0.628310),
                c64::new(0.436062, 0.418872),
                c64::new(0.593698, 0.762824),
                c64::new(0.309493, 0.307448),
                c64::new(0.305630, 0.978485),
                c64::new(0.519521, 0.164075),
                c64::new(0.130780, 0.355422),
                c64::new(0.931170, 0.955420),
                c64::new(0.639897, 0.783504),
                c64::new(0.975837, 0.798297),
                c64::new(0.728007, 0.961547),
                c64::new(0.905566, 0.368517),
                c64::new(0.502515, 0.820132),
                c64::new(0.254013, 0.605336),
                c64::new(0.667199, 0.329764),
                c64::new(0.402744, 0.539325),
                c64::new(0.204232, 0.516049),
                c64::new(0.307148, 0.723428),
                c64::new(0.327617, 0.892380),
                c64::new(0.621088, 0.756344),
                c64::new(0.607570, 0.875468),
                c64::new(0.326344, 0.441220),
                c64::new(0.840396, 0.816170),
                c64::new(0.837914, 0.284873),
                c64::new(0.333503, 0.896922),
                c64::new(0.294999, 0.097194),
                c64::new(0.453152, 0.354745),
                c64::new(0.795984, 0.320055),
                c64::new(0.532080, 0.975280),
                c64::new(0.914625, 0.894096),
                c64::new(0.303106, 0.159891),
                c64::new(0.455389, 0.364989),
                c64::new(0.775027, 0.873579),
                c64::new(0.823579, 0.973852),
                c64::new(0.745249, 0.238750),
                c64::new(0.055775, 0.462683),
                c64::new(0.601197, 0.949506),
                c64::new(0.413830, 0.503666),
                c64::new(0.922602, 0.762575),
                c64::new(0.908801, 0.618141),
                c64::new(0.617544, 0.759948),
                c64::new(0.903917, 0.909115),
                c64::new(0.211280, 0.466552),
                c64::new(0.753063, 0.448736),
                c64::new(0.842201, 0.192935),
                c64::new(0.158894, 0.373644),
                c64::new(0.919627, 0.968862),
                c64::new(0.825325, 0.659396),
                c64::new(0.789320, 0.508504),
                c64::new(0.373330, 0.844511),
                c64::new(0.618077, 0.165491),
                c64::new(0.106393, 0.792885),
                c64::new(0.779730, 0.614132),
                c64::new(0.069509, 0.054985),
                c64::new(0.255013, 0.866241),
                c64::new(0.347205, 0.127296),
                c64::new(0.821637, 0.346251),
                c64::new(0.377968, 0.148919),
                c64::new(0.327687, 0.247203),
                c64::new(0.992984, 0.799573),
                c64::new(0.108255, 0.788765),
                c64::new(0.514215, 0.731791),
                c64::new(0.580841, 0.514226),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.885363, 0.687837),
                c64::new(0.551166, 0.497917),
                c64::new(0.825376, 0.185158),
                c64::new(0.045191, 0.719457),
                c64::new(0.452973, 0.349091),
                c64::new(0.661357, 0.599048),
                c64::new(0.991222, 0.543534),
                c64::new(0.975168, 0.325773),
                c64::new(0.462447, 0.624167),
                c64::new(0.483536, 0.218702),
                c64::new(0.201990, 0.668881),
                c64::new(0.077028, 0.989315),
                c64::new(0.855804, 0.885904),
                c64::new(0.089533, 0.460653),
                c64::new(0.578667, 0.405857),
                c64::new(0.601327, 0.146957),
                c64::new(0.661265, 0.659130),
                c64::new(0.490284, 0.578846),
                c64::new(0.986696, 0.309514),
                c64::new(0.477208, 0.265428),
                c64::new(0.979748, 0.260254),
                c64::new(0.996741, 0.981340),
                c64::new(0.660590, 0.753325),
                c64::new(0.009068, 0.196276),
                c64::new(0.452666, 0.974502),
                c64::new(0.707275, 0.469765),
                c64::new(0.977787, 0.306894),
                c64::new(0.183545, 0.394689),
                c64::new(0.564566, 0.242242),
                c64::new(0.008744, 0.926465),
                c64::new(0.728964, 0.260786),
                c64::new(0.008644, 0.199844),
                c64::new(0.869522, 0.570320),
                c64::new(0.642926, 0.152818),
                c64::new(0.560739, 0.735914),
                c64::new(0.553056, 0.796059),
                c64::new(0.296222, 0.502716),
                c64::new(0.971138, 0.725463),
                c64::new(0.717370, 0.437528),
                c64::new(0.938721, 0.781635),
                c64::new(0.646146, 0.450928),
                c64::new(0.816899, 0.183706),
                c64::new(0.668215, 0.594327),
                c64::new(0.800453, 0.757989),
                c64::new(0.874637, 0.532485),
                c64::new(0.353945, 0.197614),
                c64::new(0.418672, 0.150726),
                c64::new(0.797229, 0.437803),
                c64::new(0.535866, 0.718961),
                c64::new(0.651383, 0.553907),
                c64::new(0.686525, 0.063237),
                c64::new(0.220896, 0.586850),
                c64::new(0.759162, 0.440712),
                c64::new(0.459859, 0.363337),
                c64::new(0.724872, 0.732944),
                c64::new(0.916895, 0.809516),
                c64::new(0.881414, 0.287903),
                c64::new(0.029766, 0.786832),
                c64::new(0.322803, 0.370826),
                c64::new(0.970439, 0.027472),
                c64::new(0.067497, 0.125748),
                c64::new(0.175802, 0.880536),
                c64::new(0.600612, 0.027345),
                c64::new(0.257356, 0.999831),
                c64::new(0.726091, 0.442390),
                c64::new(0.336480, 0.731606),
                c64::new(0.833272, 0.576771),
                c64::new(0.727876, 0.447540),
                c64::new(0.334698, 0.543269),
                c64::new(0.756506, 0.721196),
                c64::new(0.063982, 0.026329),
                c64::new(0.340649, 0.276816),
                c64::new(0.829566, 0.630828),
                c64::new(0.184451, 0.657306),
                c64::new(0.813606, 0.393297),
                c64::new(0.113836, 0.230948),
                c64::new(0.303218, 0.011066),
                c64::new(0.429184, 0.856271),
                c64::new(0.029322, 0.582039),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.051513, 0.173577),
                c64::new(0.012726, 0.804310),
                c64::new(0.487114, 0.268684),
                c64::new(0.556880, 0.368487),
                c64::new(0.019847, 0.348510),
                c64::new(0.287455, 0.112647),
                c64::new(0.783608, 0.193450),
                c64::new(0.427013, 0.790532),
                c64::new(0.920399, 0.585603),
                c64::new(0.808125, 0.645392),
                c64::new(0.556897, 0.343202),
                c64::new(0.029028, 0.123232),
                c64::new(0.585008, 0.217228),
                c64::new(0.796710, 0.248294),
                c64::new(0.530208, 0.972322),
                c64::new(0.615333, 0.740084),
                c64::new(0.614021, 0.452392),
                c64::new(0.386514, 0.033972),
                c64::new(0.585173, 0.707130),
                c64::new(0.381525, 0.189543),
                c64::new(0.134625, 0.217450),
                c64::new(0.826885, 0.260257),
                c64::new(0.855527, 0.988953),
                c64::new(0.297988, 0.560438),
                c64::new(0.248692, 0.642258),
                c64::new(0.362976, 0.026407),
                c64::new(0.211771, 0.805659),
                c64::new(0.332951, 0.431480),
                c64::new(0.286264, 0.597393),
                c64::new(0.012763, 0.772031),
                c64::new(0.256025, 0.593176),
                c64::new(0.610110, 0.690080),
                c64::new(0.818794, 0.155757),
                c64::new(0.280947, 0.253321),
                c64::new(0.921852, 0.983459),
                c64::new(0.864363, 0.596182),
                c64::new(0.110890, 0.061237),
                c64::new(0.225254, 0.975588),
                c64::new(0.132167, 0.137406),
                c64::new(0.637429, 0.975110),
                c64::new(0.754447, 0.606436),
                c64::new(0.930467, 0.965452),
                c64::new(0.599985, 0.607495),
                c64::new(0.525392, 0.916377),
                c64::new(0.676208, 0.452939),
                c64::new(0.791249, 0.716835),
                c64::new(0.012510, 0.441904),
                c64::new(0.359214, 0.832472),
                c64::new(0.961906, 0.090752),
                c64::new(0.421605, 0.118564),
                c64::new(0.675362, 0.081498),
                c64::new(0.051258, 0.162947),
                c64::new(0.445172, 0.175669),
                c64::new(0.365096, 0.747562),
                c64::new(0.043119, 0.642289),
                c64::new(0.984830, 0.221750),
                c64::new(0.927521, 0.232740),
                c64::new(0.581800, 0.055723),
                c64::new(0.207550, 0.288908),
                c64::new(0.199472, 0.404946),
                c64::new(0.720699, 0.082526),
                c64::new(0.473826, 0.097967),
                c64::new(0.365177, 0.745804),
                c64::new(0.292566, 0.809429),
                c64::new(0.616526, 0.809386),
                c64::new(0.989486, 0.939155),
                c64::new(0.542755, 0.588260),
                c64::new(0.738519, 0.314831),
                c64::new(0.900225, 0.235369),
                c64::new(0.938270, 0.358360),
                c64::new(0.287220, 0.197597),
                c64::new(0.709490, 0.615926),
                c64::new(0.484982, 0.431445),
                c64::new(0.601948, 0.793442),
                c64::new(0.442526, 0.099519),
                c64::new(0.601774, 0.606681),
                c64::new(0.628607, 0.555036),
                c64::new(0.625620, 0.996953),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.289155, 0.961740),
                c64::new(0.490163, 0.088500),
                c64::new(0.372419, 0.612549),
                c64::new(0.869340, 0.285781),
                c64::new(0.002317, 0.070976),
                c64::new(0.005120, 0.356129),
                c64::new(0.508412, 0.360240),
                c64::new(0.715763, 0.347053),
                c64::new(0.937713, 0.875848),
                c64::new(0.363204, 0.882866),
                c64::new(0.376593, 0.758217),
                c64::new(0.766621, 0.097073),
                c64::new(0.152066, 0.523066),
                c64::new(0.257496, 0.170583),
                c64::new(0.941468, 0.488247),
                c64::new(0.644193, 0.726258),
                c64::new(0.403752, 0.151882),
                c64::new(0.191348, 0.460387),
                c64::new(0.465699, 0.155976),
                c64::new(0.535878, 0.078421),
                c64::new(0.436712, 0.042002),
                c64::new(0.008183, 0.699912),
                c64::new(0.582933, 0.937441),
                c64::new(0.968276, 0.026208),
                c64::new(0.927708, 0.855369),
                c64::new(0.424361, 0.201965),
                c64::new(0.205309, 0.838301),
                c64::new(0.791709, 0.293998),
                c64::new(0.447689, 0.301747),
                c64::new(0.321360, 0.945269),
                c64::new(0.393706, 0.896704),
                c64::new(0.665387, 0.551242),
                c64::new(0.132991, 0.791550),
                c64::new(0.271829, 0.894138),
                c64::new(0.934476, 0.586796),
                c64::new(0.280712, 0.200650),
                c64::new(0.937166, 0.946780),
                c64::new(0.089695, 0.980144),
                c64::new(0.549917, 0.105686),
                c64::new(0.283430, 0.174846),
                c64::new(0.123791, 0.684295),
                c64::new(0.718919, 0.013677),
                c64::new(0.402465, 0.732890),
                c64::new(0.250679, 0.162083),
                c64::new(0.849216, 0.705755),
                c64::new(0.304303, 0.999280),
                c64::new(0.675160, 0.251512),
                c64::new(0.779423, 0.329941),
                c64::new(0.758445, 0.811183),
                c64::new(0.238767, 0.950807),
                c64::new(0.589344, 0.386455),
                c64::new(0.889184, 0.658848),
                c64::new(0.713055, 0.231224),
                c64::new(0.879370, 0.873876),
                c64::new(0.231817, 0.119096),
                c64::new(0.238865, 0.366560),
                c64::new(0.531642, 0.847822),
                c64::new(0.868093, 0.882441),
                c64::new(0.896504, 0.336205),
                c64::new(0.159768, 0.878179),
                c64::new(0.615317, 0.067198),
                c64::new(0.981414, 0.315545),
                c64::new(0.081170, 0.321401),
                c64::new(0.326565, 0.265870),
                c64::new(0.343486, 0.584561),
                c64::new(0.190563, 0.573878),
                c64::new(0.665755, 0.611720),
                c64::new(0.611632, 0.490041),
                c64::new(0.973960, 0.889928),
                c64::new(0.541750, 0.118986),
                c64::new(0.457407, 0.501023),
                c64::new(0.907357, 0.229604),
                c64::new(0.872756, 0.287523),
                c64::new(0.057295, 0.976400),
                c64::new(0.466542, 0.612046),
                c64::new(0.791288, 0.483691),
                c64::new(0.126724, 0.673544),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.847072, 0.082530),
                c64::new(0.183976, 0.432579),
                c64::new(0.037648, 0.047921),
                c64::new(0.510633, 0.086960),
                c64::new(0.245840, 0.448125),
                c64::new(0.934751, 0.032999),
                c64::new(0.469982, 0.188153),
                c64::new(0.040879, 0.717910),
                c64::new(0.402241, 0.370923),
                c64::new(0.529969, 0.942187),
                c64::new(0.450408, 0.533942),
                c64::new(0.657583, 0.093236),
                c64::new(0.154370, 0.981789),
                c64::new(0.636435, 0.253734),
                c64::new(0.940168, 0.372809),
                c64::new(0.908495, 0.928031),
                c64::new(0.460247, 0.293298),
                c64::new(0.980880, 0.388466),
                c64::new(0.527768, 0.141638),
                c64::new(0.866410, 0.752786),
                c64::new(0.597654, 0.265190),
                c64::new(0.626204, 0.650233),
                c64::new(0.062501, 0.080421),
                c64::new(0.838740, 0.788869),
                c64::new(0.745186, 0.801858),
                c64::new(0.239010, 0.532903),
                c64::new(0.645842, 0.829909),
                c64::new(0.395571, 0.600123),
                c64::new(0.409420, 0.502887),
                c64::new(0.083691, 0.975611),
                c64::new(0.922538, 0.904089),
                c64::new(0.569499, 0.793621),
                c64::new(0.379816, 0.098092),
                c64::new(0.958372, 0.986638),
                c64::new(0.749293, 0.545442),
                c64::new(0.478899, 0.059587),
                c64::new(0.379918, 0.404380),
                c64::new(0.606773, 0.643400),
                c64::new(0.194018, 0.385942),
                c64::new(0.813690, 0.041803),
                c64::new(0.349065, 0.282158),
                c64::new(0.942123, 0.460220),
                c64::new(0.996575, 0.819298),
                c64::new(0.158097, 0.152044),
                c64::new(0.550044, 0.914536),
                c64::new(0.441441, 0.587730),
                c64::new(0.541700, 0.265326),
                c64::new(0.846507, 0.885090),
                c64::new(0.014993, 0.503610),
                c64::new(0.310628, 0.935434),
                c64::new(0.125796, 0.258505),
                c64::new(0.298452, 0.274630),
                c64::new(0.719749, 0.816482),
                c64::new(0.513980, 0.792151),
                c64::new(0.491703, 0.166468),
                c64::new(0.042092, 0.379490),
                c64::new(0.583096, 0.090002),
                c64::new(0.201434, 0.389102),
                c64::new(0.964109, 0.840889),
                c64::new(0.901285, 0.438227),
                c64::new(0.613267, 0.987794),
                c64::new(0.003690, 0.311719),
                c64::new(0.680027, 0.823017),
                c64::new(0.809014, 0.344832),
                c64::new(0.482488, 0.192916),
                c64::new(0.600872, 0.221161),
                c64::new(0.954395, 0.816372),
                c64::new(0.231132, 0.813224),
                c64::new(0.887985, 0.686116),
                c64::new(0.367708, 0.928285),
                c64::new(0.029087, 0.585519),
                c64::new(0.050048, 0.752287),
                c64::new(0.246735, 0.830854),
                c64::new(0.439015, 0.088483),
                c64::new(0.571588, 0.233914),
                c64::new(0.761459, 0.042672),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.134721, 0.117495),
                c64::new(0.710757, 0.317919),
                c64::new(0.835706, 0.516099),
                c64::new(0.641673, 0.250428),
                c64::new(0.016463, 0.719739),
                c64::new(0.117612, 0.615379),
                c64::new(0.640435, 0.940191),
                c64::new(0.890575, 0.006322),
                c64::new(0.745815, 0.878253),
                c64::new(0.553585, 0.285384),
                c64::new(0.296112, 0.200575),
                c64::new(0.181999, 0.137015),
                c64::new(0.180876, 0.296113),
                c64::new(0.000414, 0.041770),
                c64::new(0.492262, 0.358157),
                c64::new(0.099339, 0.610093),
                c64::new(0.701999, 0.938898),
                c64::new(0.717859, 0.434411),
                c64::new(0.882205, 0.923658),
                c64::new(0.880753, 0.797947),
                c64::new(0.404337, 0.482164),
                c64::new(0.566132, 0.194229),
                c64::new(0.546535, 0.831949),
                c64::new(0.430047, 0.736391),
                c64::new(0.541614, 0.518207),
                c64::new(0.940717, 0.430477),
                c64::new(0.876921, 0.944464),
                c64::new(0.790170, 0.270346),
                c64::new(0.956553, 0.793296),
                c64::new(0.365244, 0.346208),
                c64::new(0.152071, 0.273308),
                c64::new(0.419713, 0.965446),
                c64::new(0.566255, 0.933235),
                c64::new(0.718807, 0.589764),
                c64::new(0.459619, 0.668015),
                c64::new(0.874463, 0.463188),
                c64::new(0.774406, 0.864261),
                c64::new(0.887313, 0.262216),
                c64::new(0.840591, 0.951061),
                c64::new(0.877501, 0.953548),
                c64::new(0.626295, 0.354906),
                c64::new(0.343095, 0.324552),
                c64::new(0.224416, 0.391491),
                c64::new(0.587904, 0.072234),
                c64::new(0.711380, 0.685617),
                c64::new(0.998138, 0.118515),
                c64::new(0.387209, 0.498534),
                c64::new(0.423211, 0.790793),
                c64::new(0.284595, 0.584091),
                c64::new(0.903383, 0.623149),
                c64::new(0.644151, 0.597755),
                c64::new(0.160947, 0.229379),
                c64::new(0.680522, 0.169160),
                c64::new(0.827088, 0.498378),
                c64::new(0.299603, 0.444109),
                c64::new(0.050926, 0.077077),
                c64::new(0.238776, 0.471601),
                c64::new(0.890046, 0.631329),
                c64::new(0.768302, 0.089965),
                c64::new(0.293780, 0.402001),
                c64::new(0.765867, 0.878908),
                c64::new(0.454987, 0.268004),
                c64::new(0.496033, 0.697280),
                c64::new(0.979176, 0.622676),
                c64::new(0.762197, 0.171706),
                c64::new(0.184477, 0.441072),
                c64::new(0.214003, 0.013070),
                c64::new(0.068075, 0.659970),
                c64::new(0.057238, 0.123197),
                c64::new(0.515342, 0.203352),
                c64::new(0.755313, 0.696643),
                c64::new(0.181033, 0.784380),
                c64::new(0.835228, 0.683653),
                c64::new(0.411146, 0.802499),
                c64::new(0.277201, 0.328529),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.702421, 0.765904),
                c64::new(0.839961, 0.999068),
                c64::new(0.017109, 0.780875),
                c64::new(0.756169, 0.251776),
                c64::new(0.532281, 0.590446),
                c64::new(0.803530, 0.821690),
                c64::new(0.748807, 0.832200),
                c64::new(0.782620, 0.540693),
                c64::new(0.427245, 0.848720),
                c64::new(0.922237, 0.388498),
                c64::new(0.451346, 0.374304),
                c64::new(0.794544, 0.959751),
                c64::new(0.834146, 0.927950),
                c64::new(0.028056, 0.686113),
                c64::new(0.514599, 0.160004),
                c64::new(0.996838, 0.326637),
                c64::new(0.892468, 0.988518),
                c64::new(0.775975, 0.432043),
                c64::new(0.550931, 0.503653),
                c64::new(0.614399, 0.845978),
                c64::new(0.821343, 0.810769),
                c64::new(0.022260, 0.603564),
                c64::new(0.425300, 0.887315),
                c64::new(0.942777, 0.947226),
                c64::new(0.880103, 0.765437),
                c64::new(0.515144, 0.344826),
                c64::new(0.086979, 0.704080),
                c64::new(0.381330, 0.923357),
                c64::new(0.325314, 0.883021),
                c64::new(0.366910, 0.949177),
                c64::new(0.688811, 0.436700),
                c64::new(0.983523, 0.061837),
                c64::new(0.566587, 0.008725),
                c64::new(0.369999, 0.833578),
                c64::new(0.937723, 0.538905),
                c64::new(0.718395, 0.459362),
                c64::new(0.199505, 0.478610),
                c64::new(0.796923, 0.844344),
                c64::new(0.006066, 0.367275),
                c64::new(0.089481, 0.877109),
                c64::new(0.582316, 0.045363),
                c64::new(0.253497, 0.419430),
                c64::new(0.400703, 0.367109),
                c64::new(0.625370, 0.261142),
                c64::new(0.419139, 0.031708),
                c64::new(0.449513, 0.107413),
                c64::new(0.049282, 0.406617),
                c64::new(0.152414, 0.191101),
                c64::new(0.364252, 0.321594),
                c64::new(0.494791, 0.706791),
                c64::new(0.281913, 0.125534),
                c64::new(0.683524, 0.775035),
                c64::new(0.867788, 0.694527),
                c64::new(0.423488, 0.931051),
                c64::new(0.788618, 0.122292),
                c64::new(0.902456, 0.971440),
                c64::new(0.422054, 0.782146),
                c64::new(0.845611, 0.481976),
                c64::new(0.570494, 0.151423),
                c64::new(0.725791, 0.506390),
                c64::new(0.329420, 0.792650),
                c64::new(0.650810, 0.325382),
                c64::new(0.466256, 0.864707),
                c64::new(0.202334, 0.072871),
                c64::new(0.033598, 0.554956),
                c64::new(0.949537, 0.014101),
                c64::new(0.076042, 0.536352),
                c64::new(0.127306, 0.706655),
                c64::new(0.981507, 0.998651),
                c64::new(0.953146, 0.311490),
                c64::new(0.435276, 0.463180),
                c64::new(0.377466, 0.019256),
                c64::new(0.750418, 0.156447),
                c64::new(0.402671, 0.417217),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.288354, 0.069300),
                c64::new(0.637948, 0.869916),
                c64::new(0.192646, 0.602937),
                c64::new(0.523864, 0.227529),
                c64::new(0.751533, 0.841225),
                c64::new(0.232521, 0.433072),
                c64::new(0.511121, 0.002728),
                c64::new(0.748941, 0.771176),
                c64::new(0.323678, 0.031203),
                c64::new(0.741931, 0.831453),
                c64::new(0.925469, 0.970031),
                c64::new(0.905204, 0.590407),
                c64::new(0.599409, 0.863042),
                c64::new(0.507210, 0.588771),
                c64::new(0.878830, 0.890152),
                c64::new(0.332500, 0.324549),
                c64::new(0.238287, 0.462902),
                c64::new(0.719274, 0.917094),
                c64::new(0.035883, 0.591034),
                c64::new(0.437419, 0.505450),
                c64::new(0.191261, 0.675281),
                c64::new(0.006677, 0.900233),
                c64::new(0.320398, 0.174150),
                c64::new(0.039423, 0.512047),
                c64::new(0.945177, 0.277938),
                c64::new(0.426195, 0.939573),
                c64::new(0.333030, 0.921074),
                c64::new(0.367822, 0.257916),
                c64::new(0.668833, 0.636193),
                c64::new(0.813007, 0.930248),
                c64::new(0.855751, 0.573301),
                c64::new(0.597131, 0.445240),
                c64::new(0.208038, 0.008813),
                c64::new(0.626267, 0.802330),
                c64::new(0.397644, 0.780801),
                c64::new(0.026732, 0.524766),
                c64::new(0.993118, 0.864911),
                c64::new(0.275436, 0.659924),
                c64::new(0.207874, 0.935148),
                c64::new(0.580159, 0.458235),
                c64::new(0.923498, 0.233373),
                c64::new(0.590309, 0.007575),
                c64::new(0.573844, 0.650567),
                c64::new(0.413193, 0.085211),
                c64::new(0.053832, 0.158010),
                c64::new(0.483921, 0.800576),
                c64::new(0.567822, 0.116363),
                c64::new(0.941227, 0.715853),
                c64::new(0.790322, 0.404079),
                c64::new(0.296792, 0.646534),
                c64::new(0.947274, 0.268211),
                c64::new(0.737550, 0.933737),
                c64::new(0.759094, 0.961509),
                c64::new(0.475370, 0.616588),
                c64::new(0.774080, 0.619592),
                c64::new(0.793377, 0.310516),
                c64::new(0.561490, 0.743030),
                c64::new(0.661350, 0.913628),
                c64::new(0.400355, 0.312450),
                c64::new(0.009455, 0.504541),
                c64::new(0.678427, 0.031561),
                c64::new(0.338970, 0.809862),
                c64::new(0.727223, 0.558381),
                c64::new(0.693748, 0.997720),
                c64::new(0.593051, 0.966409),
                c64::new(0.674978, 0.561222),
                c64::new(0.230805, 0.928958),
                c64::new(0.319079, 0.186830),
                c64::new(0.951224, 0.201895),
                c64::new(0.502603, 0.098175),
                c64::new(0.363395, 0.219970),
                c64::new(0.915262, 0.306267),
                c64::new(0.751837, 0.067867),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.928632, 0.014294),
                c64::new(0.623636, 0.733350),
                c64::new(0.719040, 0.705779),
                c64::new(0.150044, 0.107562),
                c64::new(0.424208, 0.397618),
                c64::new(0.006779, 0.524027),
                c64::new(0.777143, 0.455639),
                c64::new(0.179864, 0.836513),
                c64::new(0.145424, 0.154232),
                c64::new(0.998484, 0.316628),
                c64::new(0.668597, 0.515389),
                c64::new(0.656614, 0.401649),
                c64::new(0.204017, 0.061244),
                c64::new(0.208303, 0.683078),
                c64::new(0.810729, 0.445351),
                c64::new(0.060864, 0.446500),
                c64::new(0.249369, 0.216105),
                c64::new(0.081067, 0.975701),
                c64::new(0.970867, 0.414631),
                c64::new(0.771116, 0.123571),
                c64::new(0.939932, 0.975482),
                c64::new(0.133711, 0.801477),
                c64::new(0.314497, 0.378443),
                c64::new(0.960412, 0.449325),
                c64::new(0.171597, 0.313348),
                c64::new(0.075398, 0.000693),
                c64::new(0.165520, 0.185899),
                c64::new(0.595150, 0.578187),
                c64::new(0.630520, 0.771998),
                c64::new(0.401871, 0.943809),
                c64::new(0.264347, 0.604535),
                c64::new(0.200030, 0.561144),
                c64::new(0.881772, 0.681708),
                c64::new(0.556242, 0.686678),
                c64::new(0.960705, 0.817919),
                c64::new(0.229932, 0.262362),
                c64::new(0.242204, 0.701172),
                c64::new(0.188162, 0.300419),
                c64::new(0.522348, 0.574836),
                c64::new(0.470990, 0.671590),
                c64::new(0.926416, 0.969330),
                c64::new(0.103682, 0.210421),
                c64::new(0.756438, 0.649655),
                c64::new(0.073072, 0.096394),
                c64::new(0.533524, 0.877982),
                c64::new(0.776069, 0.355412),
                c64::new(0.984760, 0.487311),
                c64::new(0.046849, 0.964595),
                c64::new(0.207738, 0.756174),
                c64::new(0.201928, 0.954603),
                c64::new(0.929979, 0.866708),
                c64::new(0.978434, 0.450073),
                c64::new(0.054542, 0.494125),
                c64::new(0.587824, 0.223926),
                c64::new(0.128772, 0.423246),
                c64::new(0.895176, 0.886990),
                c64::new(0.588247, 0.501663),
                c64::new(0.537568, 0.454227),
                c64::new(0.948885, 0.074807),
                c64::new(0.909492, 0.256017),
                c64::new(0.643756, 0.629815),
                c64::new(0.081138, 0.922447),
                c64::new(0.694561, 0.211555),
                c64::new(0.336716, 0.757590),
                c64::new(0.128734, 0.672381),
                c64::new(0.922387, 0.892989),
                c64::new(0.075463, 0.271327),
                c64::new(0.791137, 0.376125),
                c64::new(0.518887, 0.029713),
                c64::new(0.476135, 0.434405),
                c64::new(0.766806, 0.522100),
                c64::new(0.090336, 0.500756),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.680779, 0.563839),
                c64::new(0.315215, 0.442118),
                c64::new(0.573534, 0.464481),
                c64::new(0.168558, 0.932330),
                c64::new(0.646310, 0.889925),
                c64::new(0.188574, 0.549096),
                c64::new(0.084223, 0.135694),
                c64::new(0.152856, 0.197392),
                c64::new(0.522790, 0.990399),
                c64::new(0.415518, 0.932995),
                c64::new(0.924193, 0.300508),
                c64::new(0.903897, 0.621065),
                c64::new(0.077679, 0.673859),
                c64::new(0.126101, 0.977766),
                c64::new(0.250092, 0.191678),
                c64::new(0.765216, 0.580964),
                c64::new(0.796742, 0.108554),
                c64::new(0.247215, 0.678409),
                c64::new(0.277292, 0.378846),
                c64::new(0.415253, 0.489091),
                c64::new(0.964074, 0.606494),
                c64::new(0.412454, 0.403663),
                c64::new(0.939715, 0.144333),
                c64::new(0.954734, 0.892460),
                c64::new(0.089493, 0.779992),
                c64::new(0.122855, 0.434266),
                c64::new(0.081651, 0.655056),
                c64::new(0.277337, 0.147456),
                c64::new(0.350739, 0.239010),
                c64::new(0.277188, 0.499880),
                c64::new(0.177255, 0.110972),
                c64::new(0.283030, 0.897528),
                c64::new(0.848949, 0.729184),
                c64::new(0.727881, 0.383386),
                c64::new(0.574032, 0.329352),
                c64::new(0.815679, 0.885693),
                c64::new(0.283870, 0.755946),
                c64::new(0.879925, 0.796970),
                c64::new(0.287090, 0.259367),
                c64::new(0.712576, 0.486397),
                c64::new(0.746372, 0.328527),
                c64::new(0.014562, 0.914873),
                c64::new(0.196638, 0.318923),
                c64::new(0.939782, 0.683047),
                c64::new(0.220371, 0.480508),
                c64::new(0.369133, 0.431275),
                c64::new(0.669206, 0.729253),
                c64::new(0.861328, 0.614278),
                c64::new(0.978438, 0.851836),
                c64::new(0.616812, 0.226828),
                c64::new(0.014539, 0.021849),
                c64::new(0.822437, 0.395025),
                c64::new(0.927630, 0.621066),
                c64::new(0.824863, 0.010540),
                c64::new(0.553708, 0.300130),
                c64::new(0.776251, 0.147879),
                c64::new(0.019079, 0.854687),
                c64::new(0.128524, 0.303232),
                c64::new(0.161871, 0.205345),
                c64::new(0.124224, 0.227913),
                c64::new(0.889258, 0.312434),
                c64::new(0.035476, 0.660393),
                c64::new(0.592208, 0.303752),
                c64::new(0.901217, 0.172900),
                c64::new(0.112475, 0.529248),
                c64::new(0.039528, 0.048004),
                c64::new(0.664706, 0.800502),
                c64::new(0.327749, 0.783531),
                c64::new(0.304708, 0.997066),
                c64::new(0.513420, 0.168063),
                c64::new(0.410556, 0.316992),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.787460, 0.030260),
                c64::new(0.801795, 0.627786),
                c64::new(0.285873, 0.686322),
                c64::new(0.775403, 0.687013),
                c64::new(0.094980, 0.264256),
                c64::new(0.079904, 0.162777),
                c64::new(0.072621, 0.754462),
                c64::new(0.990801, 0.572672),
                c64::new(0.831351, 0.710820),
                c64::new(0.858615, 0.377715),
                c64::new(0.027147, 0.665864),
                c64::new(0.217363, 0.866443),
                c64::new(0.707035, 0.821437),
                c64::new(0.345745, 0.969772),
                c64::new(0.235670, 0.079851),
                c64::new(0.568655, 0.400310),
                c64::new(0.649330, 0.916502),
                c64::new(0.042361, 0.773001),
                c64::new(0.313628, 0.904049),
                c64::new(0.615882, 0.800768),
                c64::new(0.875865, 0.374017),
                c64::new(0.431686, 0.335662),
                c64::new(0.948825, 0.759633),
                c64::new(0.881793, 0.023924),
                c64::new(0.274708, 0.589976),
                c64::new(0.482459, 0.362582),
                c64::new(0.966596, 0.006429),
                c64::new(0.999264, 0.059982),
                c64::new(0.752257, 0.934289),
                c64::new(0.591709, 0.597292),
                c64::new(0.231377, 0.146189),
                c64::new(0.241700, 0.960261),
                c64::new(0.448265, 0.522905),
                c64::new(0.581061, 0.329183),
                c64::new(0.460465, 0.933466),
                c64::new(0.229838, 0.866867),
                c64::new(0.689144, 0.692121),
                c64::new(0.486526, 0.578116),
                c64::new(0.891100, 0.969261),
                c64::new(0.702364, 0.198655),
                c64::new(0.327563, 0.945783),
                c64::new(0.411356, 0.033630),
                c64::new(0.489600, 0.976008),
                c64::new(0.116099, 0.740040),
                c64::new(0.488057, 0.313102),
                c64::new(0.863002, 0.190106),
                c64::new(0.625803, 0.133282),
                c64::new(0.686834, 0.937542),
                c64::new(0.257289, 0.801109),
                c64::new(0.378066, 0.190012),
                c64::new(0.558051, 0.715583),
                c64::new(0.000529, 0.004055),
                c64::new(0.424932, 0.457640),
                c64::new(0.017396, 0.744663),
                c64::new(0.800487, 0.419399),
                c64::new(0.087275, 0.801226),
                c64::new(0.237736, 0.527129),
                c64::new(0.822535, 0.777507),
                c64::new(0.820192, 0.525228),
                c64::new(0.209773, 0.617459),
                c64::new(0.053779, 0.949775),
                c64::new(0.115291, 0.855693),
                c64::new(0.567801, 0.888227),
                c64::new(0.271315, 0.092692),
                c64::new(0.493620, 0.911935),
                c64::new(0.037067, 0.477662),
                c64::new(0.159134, 0.931661),
                c64::new(0.997083, 0.439616),
                c64::new(0.117976, 0.646445),
                c64::new(0.154513, 0.504701),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.140138, 0.071674),
                c64::new(0.975502, 0.541554),
                c64::new(0.275948, 0.171194),
                c64::new(0.365138, 0.465518),
                c64::new(0.137805, 0.776210),
                c64::new(0.245550, 0.120454),
                c64::new(0.869530, 0.772682),
                c64::new(0.288553, 0.215842),
                c64::new(0.136829, 0.019483),
                c64::new(0.009384, 0.849946),
                c64::new(0.506560, 0.919152),
                c64::new(0.134280, 0.261640),
                c64::new(0.124997, 0.231915),
                c64::new(0.473885, 0.946449),
                c64::new(0.996759, 0.412659),
                c64::new(0.433863, 0.612710),
                c64::new(0.965073, 0.358656),
                c64::new(0.113152, 0.515233),
                c64::new(0.740752, 0.604666),
                c64::new(0.728119, 0.407021),
                c64::new(0.976962, 0.582090),
                c64::new(0.291462, 0.364608),
                c64::new(0.646786, 0.381119),
                c64::new(0.093932, 0.491290),
                c64::new(0.743225, 0.514136),
                c64::new(0.549217, 0.214169),
                c64::new(0.619430, 0.614920),
                c64::new(0.916178, 0.300604),
                c64::new(0.967381, 0.273828),
                c64::new(0.906354, 0.304740),
                c64::new(0.414396, 0.151451),
                c64::new(0.428243, 0.001305),
                c64::new(0.680172, 0.070404),
                c64::new(0.396098, 0.238335),
                c64::new(0.264861, 0.506710),
                c64::new(0.182229, 0.428885),
                c64::new(0.978060, 0.952945),
                c64::new(0.760246, 0.629492),
                c64::new(0.576727, 0.917568),
                c64::new(0.413628, 0.756818),
                c64::new(0.753249, 0.009970),
                c64::new(0.459267, 0.616996),
                c64::new(0.986139, 0.569455),
                c64::new(0.909302, 0.683984),
                c64::new(0.436890, 0.845689),
                c64::new(0.942748, 0.865839),
                c64::new(0.111274, 0.077348),
                c64::new(0.057448, 0.991013),
                c64::new(0.809427, 0.622816),
                c64::new(0.447975, 0.428509),
                c64::new(0.464079, 0.823557),
                c64::new(0.119724, 0.830360),
                c64::new(0.671749, 0.848045),
                c64::new(0.301023, 0.697826),
                c64::new(0.011135, 0.580421),
                c64::new(0.981546, 0.280422),
                c64::new(0.891786, 0.620942),
                c64::new(0.684455, 0.712140),
                c64::new(0.246158, 0.770796),
                c64::new(0.044686, 0.761402),
                c64::new(0.090128, 0.356289),
                c64::new(0.990248, 0.513372),
                c64::new(0.986424, 0.140793),
                c64::new(0.416767, 0.217947),
                c64::new(0.009809, 0.525178),
                c64::new(0.480546, 0.191721),
                c64::new(0.202437, 0.964897),
                c64::new(0.310777, 0.777483),
                c64::new(0.486027, 0.652922),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.857597, 0.311647),
                c64::new(0.260602, 0.377903),
                c64::new(0.626663, 0.902168),
                c64::new(0.788404, 0.686660),
                c64::new(0.639723, 0.737831),
                c64::new(0.246013, 0.560676),
                c64::new(0.092070, 0.708475),
                c64::new(0.027427, 0.596595),
                c64::new(0.128558, 0.388927),
                c64::new(0.485607, 0.399719),
                c64::new(0.136591, 0.933920),
                c64::new(0.034073, 0.884551),
                c64::new(0.239537, 0.980636),
                c64::new(0.889264, 0.401986),
                c64::new(0.130337, 0.149767),
                c64::new(0.767439, 0.522387),
                c64::new(0.766977, 0.450827),
                c64::new(0.671896, 0.048003),
                c64::new(0.013986, 0.403314),
                c64::new(0.021642, 0.951764),
                c64::new(0.016382, 0.731560),
                c64::new(0.310583, 0.922378),
                c64::new(0.553749, 0.223975),
                c64::new(0.310859, 0.555714),
                c64::new(0.809124, 0.940052),
                c64::new(0.816735, 0.769145),
                c64::new(0.058215, 0.350701),
                c64::new(0.345878, 0.654909),
                c64::new(0.639001, 0.319911),
                c64::new(0.899544, 0.479364),
                c64::new(0.197103, 0.954102),
                c64::new(0.492390, 0.207693),
                c64::new(0.898481, 0.668550),
                c64::new(0.122696, 0.378159),
                c64::new(0.330345, 0.475442),
                c64::new(0.583832, 0.646319),
                c64::new(0.895222, 0.259556),
                c64::new(0.242250, 0.164501),
                c64::new(0.138347, 0.904833),
                c64::new(0.217104, 0.409711),
                c64::new(0.852666, 0.811122),
                c64::new(0.413665, 0.745094),
                c64::new(0.719341, 0.269381),
                c64::new(0.329402, 0.404577),
                c64::new(0.347419, 0.616515),
                c64::new(0.816734, 0.574113),
                c64::new(0.761698, 0.186940),
                c64::new(0.817057, 0.761391),
                c64::new(0.111651, 0.687449),
                c64::new(0.277711, 0.665095),
                c64::new(0.340377, 0.617886),
                c64::new(0.399145, 0.101577),
                c64::new(0.822689, 0.945302),
                c64::new(0.460187, 0.208720),
                c64::new(0.631135, 0.261331),
                c64::new(0.962732, 0.816154),
                c64::new(0.560329, 0.338756),
                c64::new(0.221826, 0.553681),
                c64::new(0.708415, 0.301194),
                c64::new(0.618215, 0.606696),
                c64::new(0.074301, 0.029931),
                c64::new(0.489139, 0.515670),
                c64::new(0.602021, 0.417793),
                c64::new(0.153403, 0.066526),
                c64::new(0.983115, 0.264848),
                c64::new(0.096695, 0.520589),
                c64::new(0.758410, 0.209188),
                c64::new(0.682855, 0.246801),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.232805, 0.576305),
                c64::new(0.207400, 0.598511),
                c64::new(0.854243, 0.115204),
                c64::new(0.393042, 0.396782),
                c64::new(0.017042, 0.667761),
                c64::new(0.801282, 0.340346),
                c64::new(0.003711, 0.259986),
                c64::new(0.694978, 0.303832),
                c64::new(0.001762, 0.935495),
                c64::new(0.353230, 0.511647),
                c64::new(0.414766, 0.486634),
                c64::new(0.958218, 0.023350),
                c64::new(0.123091, 0.942666),
                c64::new(0.341048, 0.734574),
                c64::new(0.073752, 0.141648),
                c64::new(0.964084, 0.961285),
                c64::new(0.255540, 0.534859),
                c64::new(0.321547, 0.411187),
                c64::new(0.436190, 0.702827),
                c64::new(0.497867, 0.625002),
                c64::new(0.458156, 0.612110),
                c64::new(0.239477, 0.226563),
                c64::new(0.993340, 0.693538),
                c64::new(0.640763, 0.933537),
                c64::new(0.231692, 0.273402),
                c64::new(0.399929, 0.814692),
                c64::new(0.459467, 0.270334),
                c64::new(0.907065, 0.488429),
                c64::new(0.598465, 0.477257),
                c64::new(0.854214, 0.197979),
                c64::new(0.616104, 0.445388),
                c64::new(0.257036, 0.470692),
                c64::new(0.592040, 0.680530),
                c64::new(0.853276, 0.804195),
                c64::new(0.077291, 0.991949),
                c64::new(0.870280, 0.695300),
                c64::new(0.595863, 0.598614),
                c64::new(0.342565, 0.275554),
                c64::new(0.940165, 0.526830),
                c64::new(0.648304, 0.588100),
                c64::new(0.556404, 0.114495),
                c64::new(0.003300, 0.474866),
                c64::new(0.957053, 0.341193),
                c64::new(0.714953, 0.461662),
                c64::new(0.544661, 0.666366),
                c64::new(0.293557, 0.771761),
                c64::new(0.809628, 0.824242),
                c64::new(0.119130, 0.223361),
                c64::new(0.125331, 0.630057),
                c64::new(0.556906, 0.025803),
                c64::new(0.921313, 0.125813),
                c64::new(0.330347, 0.458814),
                c64::new(0.829348, 0.818831),
                c64::new(0.481705, 0.466419),
                c64::new(0.076885, 0.750243),
                c64::new(0.226752, 0.695141),
                c64::new(0.043138, 0.049178),
                c64::new(0.423535, 0.206036),
                c64::new(0.759752, 0.178571),
                c64::new(0.410225, 0.347630),
                c64::new(0.263708, 0.842497),
                c64::new(0.893796, 0.130713),
                c64::new(0.754144, 0.144180),
                c64::new(0.658781, 0.523402),
                c64::new(0.530868, 0.404838),
                c64::new(0.833528, 0.820693),
                c64::new(0.935554, 0.303123),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.098898, 0.291565),
                c64::new(0.127684, 0.203323),
                c64::new(0.309607, 0.314736),
                c64::new(0.060626, 0.221379),
                c64::new(0.483895, 0.672487),
                c64::new(0.243821, 0.178579),
                c64::new(0.577209, 0.346306),
                c64::new(0.216545, 0.776561),
                c64::new(0.175957, 0.854670),
                c64::new(0.574775, 0.331719),
                c64::new(0.526802, 0.140727),
                c64::new(0.100710, 0.254408),
                c64::new(0.381700, 0.913495),
                c64::new(0.704171, 0.914087),
                c64::new(0.419427, 0.971748),
                c64::new(0.349818, 0.313148),
                c64::new(0.290198, 0.429943),
                c64::new(0.047751, 0.094872),
                c64::new(0.811178, 0.506003),
                c64::new(0.873735, 0.158410),
                c64::new(0.757864, 0.425735),
                c64::new(0.177057, 0.136891),
                c64::new(0.364484, 0.095229),
                c64::new(0.558330, 0.241812),
                c64::new(0.655079, 0.872026),
                c64::new(0.459313, 0.585551),
                c64::new(0.754606, 0.685536),
                c64::new(0.036234, 0.451956),
                c64::new(0.922808, 0.030559),
                c64::new(0.734693, 0.009678),
                c64::new(0.708131, 0.084668),
                c64::new(0.931658, 0.948045),
                c64::new(0.700392, 0.408606),
                c64::new(0.398315, 0.514785),
                c64::new(0.081578, 0.966499),
                c64::new(0.864563, 0.668021),
                c64::new(0.976224, 0.798854),
                c64::new(0.042229, 0.924166),
                c64::new(0.221770, 0.557293),
                c64::new(0.380843, 0.584176),
                c64::new(0.661663, 0.208928),
                c64::new(0.694916, 0.407374),
                c64::new(0.653842, 0.863777),
                c64::new(0.898236, 0.534947),
                c64::new(0.915596, 0.495553),
                c64::new(0.555592, 0.842158),
                c64::new(0.265006, 0.420740),
                c64::new(0.873024, 0.533431),
                c64::new(0.864627, 0.829643),
                c64::new(0.478702, 0.376117),
                c64::new(0.384400, 0.347533),
                c64::new(0.214649, 0.773535),
                c64::new(0.350049, 0.561640),
                c64::new(0.499972, 0.843294),
                c64::new(0.152344, 0.856472),
                c64::new(0.928886, 0.008301),
                c64::new(0.271550, 0.011560),
                c64::new(0.150235, 0.899952),
                c64::new(0.109420, 0.823146),
                c64::new(0.892450, 0.938720),
                c64::new(0.637133, 0.448833),
                c64::new(0.253142, 0.767440),
                c64::new(0.867487, 0.621166),
                c64::new(0.063119, 0.847866),
                c64::new(0.689416, 0.759780),
                c64::new(0.773086, 0.765058),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.452458, 0.547820),
                c64::new(0.541054, 0.696215),
                c64::new(0.880126, 0.115864),
                c64::new(0.789370, 0.013493),
                c64::new(0.635422, 0.380481),
                c64::new(0.309871, 0.980499),
                c64::new(0.158249, 0.586149),
                c64::new(0.994911, 0.684472),
                c64::new(0.368954, 0.287684),
                c64::new(0.136307, 0.774540),
                c64::new(0.368455, 0.932371),
                c64::new(0.327801, 0.003596),
                c64::new(0.055909, 0.236207),
                c64::new(0.127678, 0.867030),
                c64::new(0.819601, 0.823852),
                c64::new(0.972461, 0.481220),
                c64::new(0.824719, 0.194951),
                c64::new(0.277140, 0.010203),
                c64::new(0.014613, 0.169351),
                c64::new(0.519789, 0.964528),
                c64::new(0.390106, 0.513682),
                c64::new(0.814322, 0.739923),
                c64::new(0.914908, 0.337673),
                c64::new(0.675706, 0.915563),
                c64::new(0.379254, 0.542954),
                c64::new(0.869223, 0.289050),
                c64::new(0.137291, 0.231450),
                c64::new(0.231773, 0.487907),
                c64::new(0.860038, 0.868516),
                c64::new(0.136004, 0.717899),
                c64::new(0.935145, 0.444900),
                c64::new(0.344244, 0.438961),
                c64::new(0.016063, 0.546069),
                c64::new(0.171182, 0.800204),
                c64::new(0.375901, 0.922508),
                c64::new(0.233871, 0.769126),
                c64::new(0.996215, 0.207428),
                c64::new(0.075000, 0.844190),
                c64::new(0.526818, 0.925333),
                c64::new(0.073576, 0.604256),
                c64::new(0.840102, 0.769539),
                c64::new(0.265159, 0.923164),
                c64::new(0.047443, 0.675284),
                c64::new(0.785721, 0.845011),
                c64::new(0.030684, 0.157244),
                c64::new(0.940729, 0.378728),
                c64::new(0.761030, 0.073452),
                c64::new(0.802376, 0.433708),
                c64::new(0.958386, 0.148966),
                c64::new(0.236726, 0.370927),
                c64::new(0.097915, 0.469611),
                c64::new(0.181766, 0.093282),
                c64::new(0.942272, 0.136644),
                c64::new(0.965049, 0.074911),
                c64::new(0.504028, 0.393909),
                c64::new(0.271465, 0.362212),
                c64::new(0.473880, 0.928421),
                c64::new(0.247273, 0.791944),
                c64::new(0.049183, 0.372903),
                c64::new(0.878568, 0.909872),
                c64::new(0.372199, 0.341723),
                c64::new(0.430157, 0.646008),
                c64::new(0.107023, 0.922188),
                c64::new(0.242081, 0.457675),
                c64::new(0.331362, 0.385730),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.386213, 0.380313),
                c64::new(0.383878, 0.674157),
                c64::new(0.550544, 0.781560),
                c64::new(0.463711, 0.366946),
                c64::new(0.208219, 0.928823),
                c64::new(0.841025, 0.849972),
                c64::new(0.461435, 0.813771),
                c64::new(0.794603, 0.830791),
                c64::new(0.705329, 0.868150),
                c64::new(0.563815, 0.172584),
                c64::new(0.747780, 0.904092),
                c64::new(0.617104, 0.393452),
                c64::new(0.288880, 0.952542),
                c64::new(0.848529, 0.620404),
                c64::new(0.117828, 0.478020),
                c64::new(0.758637, 0.081532),
                c64::new(0.474897, 0.117917),
                c64::new(0.663699, 0.987136),
                c64::new(0.235378, 0.047592),
                c64::new(0.120591, 0.675556),
                c64::new(0.170053, 0.636506),
                c64::new(0.891871, 0.908334),
                c64::new(0.885688, 0.775340),
                c64::new(0.750344, 0.982613),
                c64::new(0.048792, 0.306203),
                c64::new(0.903194, 0.990810),
                c64::new(0.518846, 0.276747),
                c64::new(0.812017, 0.107148),
                c64::new(0.219129, 0.880038),
                c64::new(0.514671, 0.191879),
                c64::new(0.771265, 0.000951),
                c64::new(0.971458, 0.472083),
                c64::new(0.017341, 0.537628),
                c64::new(0.011712, 0.845474),
                c64::new(0.971384, 0.904212),
                c64::new(0.587244, 0.493683),
                c64::new(0.873233, 0.799109),
                c64::new(0.184360, 0.942148),
                c64::new(0.008719, 0.190757),
                c64::new(0.680186, 0.095852),
                c64::new(0.252581, 0.428795),
                c64::new(0.273962, 0.458328),
                c64::new(0.127309, 0.207646),
                c64::new(0.559770, 0.031846),
                c64::new(0.734383, 0.367334),
                c64::new(0.055559, 0.288492),
                c64::new(0.254701, 0.074855),
                c64::new(0.678248, 0.257818),
                c64::new(0.014979, 0.218807),
                c64::new(0.456853, 0.485134),
                c64::new(0.801688, 0.040015),
                c64::new(0.475335, 0.016010),
                c64::new(0.811928, 0.989480),
                c64::new(0.410234, 0.138619),
                c64::new(0.618638, 0.136353),
                c64::new(0.705006, 0.847345),
                c64::new(0.399656, 0.560976),
                c64::new(0.353228, 0.538137),
                c64::new(0.732496, 0.418608),
                c64::new(0.413902, 0.495859),
                c64::new(0.875504, 0.480384),
                c64::new(0.076926, 0.415075),
                c64::new(0.438311, 0.213753),
                c64::new(0.817010, 0.407064),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.818807, 0.356581),
                c64::new(0.087022, 0.340993),
                c64::new(0.256544, 0.119007),
                c64::new(0.654615, 0.274354),
                c64::new(0.550146, 0.980704),
                c64::new(0.460586, 0.788776),
                c64::new(0.178380, 0.959158),
                c64::new(0.849689, 0.093853),
                c64::new(0.496622, 0.200960),
                c64::new(0.181062, 0.193027),
                c64::new(0.180593, 0.753521),
                c64::new(0.494335, 0.434076),
                c64::new(0.400492, 0.702027),
                c64::new(0.114057, 0.377841),
                c64::new(0.737800, 0.844015),
                c64::new(0.229027, 0.881637),
                c64::new(0.856988, 0.500834),
                c64::new(0.701550, 0.704984),
                c64::new(0.452745, 0.740888),
                c64::new(0.832512, 0.682428),
                c64::new(0.599581, 0.278271),
                c64::new(0.571865, 0.854290),
                c64::new(0.619819, 0.304887),
                c64::new(0.817864, 0.931829),
                c64::new(0.938161, 0.158445),
                c64::new(0.302613, 0.280865),
                c64::new(0.386840, 0.538870),
                c64::new(0.717666, 0.251018),
                c64::new(0.152778, 0.473429),
                c64::new(0.141575, 0.859729),
                c64::new(0.774090, 0.109539),
                c64::new(0.968625, 0.381673),
                c64::new(0.725729, 0.153549),
                c64::new(0.665831, 0.190365),
                c64::new(0.251885, 0.052860),
                c64::new(0.317803, 0.794153),
                c64::new(0.546734, 0.990296),
                c64::new(0.370426, 0.750447),
                c64::new(0.139265, 0.251563),
                c64::new(0.001849, 0.312077),
                c64::new(0.753600, 0.386599),
                c64::new(0.778721, 0.427982),
                c64::new(0.604517, 0.488052),
                c64::new(0.339569, 0.075269),
                c64::new(0.697206, 0.875211),
                c64::new(0.679898, 0.598174),
                c64::new(0.929893, 0.100704),
                c64::new(0.856897, 0.617276),
                c64::new(0.649747, 0.905103),
                c64::new(0.921305, 0.672133),
                c64::new(0.395744, 0.509464),
                c64::new(0.585934, 0.981896),
                c64::new(0.658867, 0.839548),
                c64::new(0.023311, 0.411135),
                c64::new(0.239438, 0.240733),
                c64::new(0.615399, 0.083178),
                c64::new(0.443822, 0.909183),
                c64::new(0.913556, 0.435211),
                c64::new(0.619827, 0.557358),
                c64::new(0.165276, 0.291985),
                c64::new(0.639731, 0.422752),
                c64::new(0.107090, 0.839078),
                c64::new(0.545747, 0.944014),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.160306, 0.434584),
                c64::new(0.888617, 0.582978),
                c64::new(0.645691, 0.138817),
                c64::new(0.820897, 0.220362),
                c64::new(0.442680, 0.417230),
                c64::new(0.260500, 0.877439),
                c64::new(0.472540, 0.379039),
                c64::new(0.028521, 0.665868),
                c64::new(0.201938, 0.184704),
                c64::new(0.370981, 0.904823),
                c64::new(0.673727, 0.511260),
                c64::new(0.740533, 0.124585),
                c64::new(0.853558, 0.528332),
                c64::new(0.237410, 0.702642),
                c64::new(0.706216, 0.815913),
                c64::new(0.285481, 0.999694),
                c64::new(0.540566, 0.924384),
                c64::new(0.588151, 0.271940),
                c64::new(0.177312, 0.995944),
                c64::new(0.831175, 0.107000),
                c64::new(0.108884, 0.983461),
                c64::new(0.801376, 0.665708),
                c64::new(0.264568, 0.389138),
                c64::new(0.499733, 0.301545),
                c64::new(0.948428, 0.800391),
                c64::new(0.362868, 0.130502),
                c64::new(0.924059, 0.901129),
                c64::new(0.249601, 0.041394),
                c64::new(0.697652, 0.486580),
                c64::new(0.913518, 0.502038),
                c64::new(0.767932, 0.539755),
                c64::new(0.630700, 0.050857),
                c64::new(0.900497, 0.356990),
                c64::new(0.772433, 0.705287),
                c64::new(0.942617, 0.180580),
                c64::new(0.910186, 0.373954),
                c64::new(0.613121, 0.151796),
                c64::new(0.232655, 0.426063),
                c64::new(0.614645, 0.122409),
                c64::new(0.954038, 0.815919),
                c64::new(0.202463, 0.378799),
                c64::new(0.111259, 0.338185),
                c64::new(0.286318, 0.750374),
                c64::new(0.302773, 0.022048),
                c64::new(0.615490, 0.878986),
                c64::new(0.349606, 0.276054),
                c64::new(0.065656, 0.134387),
                c64::new(0.923611, 0.794954),
                c64::new(0.297339, 0.438172),
                c64::new(0.758198, 0.501083),
                c64::new(0.788695, 0.638563),
                c64::new(0.672825, 0.987513),
                c64::new(0.625925, 0.232360),
                c64::new(0.346429, 0.289553),
                c64::new(0.182751, 0.484001),
                c64::new(0.262688, 0.973446),
                c64::new(0.963768, 0.387030),
                c64::new(0.893541, 0.115761),
                c64::new(0.219034, 0.293819),
                c64::new(0.188020, 0.598587),
                c64::new(0.594951, 0.471772),
                c64::new(0.680702, 0.833959),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.036841, 0.554012),
                c64::new(0.217267, 0.677886),
                c64::new(0.962421, 0.382163),
                c64::new(0.099152, 0.565319),
                c64::new(0.993354, 0.368700),
                c64::new(0.298927, 0.075455),
                c64::new(0.088639, 0.421599),
                c64::new(0.038452, 0.196264),
                c64::new(0.741094, 0.005035),
                c64::new(0.687882, 0.321568),
                c64::new(0.706083, 0.827760),
                c64::new(0.105706, 0.481707),
                c64::new(0.082563, 0.850696),
                c64::new(0.361732, 0.954073),
                c64::new(0.929661, 0.105852),
                c64::new(0.476897, 0.750140),
                c64::new(0.856464, 0.199840),
                c64::new(0.454634, 0.720243),
                c64::new(0.318633, 0.999263),
                c64::new(0.226113, 0.941395),
                c64::new(0.623938, 0.004206),
                c64::new(0.939712, 0.693339),
                c64::new(0.375879, 0.218080),
                c64::new(0.659926, 0.663282),
                c64::new(0.317313, 0.908174),
                c64::new(0.699430, 0.828173),
                c64::new(0.038317, 0.667422),
                c64::new(0.398214, 0.992886),
                c64::new(0.160185, 0.934842),
                c64::new(0.242165, 0.740390),
                c64::new(0.302362, 0.598836),
                c64::new(0.792431, 0.842984),
                c64::new(0.962403, 0.229432),
                c64::new(0.461789, 0.369810),
                c64::new(0.798361, 0.452351),
                c64::new(0.223391, 0.573898),
                c64::new(0.381721, 0.072942),
                c64::new(0.334315, 0.735502),
                c64::new(0.869569, 0.561063),
                c64::new(0.863645, 0.806858),
                c64::new(0.007327, 0.815685),
                c64::new(0.521895, 0.578075),
                c64::new(0.187019, 0.037893),
                c64::new(0.068689, 0.255155),
                c64::new(0.560574, 0.444974),
                c64::new(0.753656, 0.494791),
                c64::new(0.518580, 0.518634),
                c64::new(0.968574, 0.345391),
                c64::new(0.288120, 0.444614),
                c64::new(0.329465, 0.155058),
                c64::new(0.344177, 0.544982),
                c64::new(0.602684, 0.819607),
                c64::new(0.988476, 0.560660),
                c64::new(0.864427, 0.488691),
                c64::new(0.960909, 0.584091),
                c64::new(0.384697, 0.999073),
                c64::new(0.317082, 0.223471),
                c64::new(0.933574, 0.440688),
                c64::new(0.782907, 0.615256),
                c64::new(0.319008, 0.612252),
                c64::new(0.466377, 0.530760),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.790118, 0.840122),
                c64::new(0.074273, 0.080222),
                c64::new(0.592436, 0.456623),
                c64::new(0.729329, 0.305745),
                c64::new(0.048285, 0.166177),
                c64::new(0.010648, 0.930173),
                c64::new(0.662583, 0.007011),
                c64::new(0.839485, 0.771252),
                c64::new(0.356505, 0.806122),
                c64::new(0.575927, 0.447686),
                c64::new(0.188236, 0.562915),
                c64::new(0.667218, 0.200158),
                c64::new(0.543735, 0.908211),
                c64::new(0.155615, 0.782023),
                c64::new(0.970960, 0.497195),
                c64::new(0.821891, 0.593604),
                c64::new(0.170449, 0.451393),
                c64::new(0.544218, 0.854445),
                c64::new(0.987164, 0.704795),
                c64::new(0.364546, 0.789337),
                c64::new(0.332949, 0.648587),
                c64::new(0.492926, 0.098968),
                c64::new(0.081727, 0.788144),
                c64::new(0.474131, 0.631215),
                c64::new(0.283096, 0.184928),
                c64::new(0.560974, 0.594577),
                c64::new(0.005540, 0.439198),
                c64::new(0.353721, 0.250289),
                c64::new(0.871863, 0.401736),
                c64::new(0.375179, 0.596502),
                c64::new(0.078085, 0.450148),
                c64::new(0.499148, 0.928855),
                c64::new(0.604782, 0.890541),
                c64::new(0.943865, 0.103422),
                c64::new(0.677014, 0.570201),
                c64::new(0.355293, 0.304826),
                c64::new(0.106290, 0.845411),
                c64::new(0.779928, 0.658228),
                c64::new(0.167167, 0.443837),
                c64::new(0.791891, 0.755804),
                c64::new(0.357731, 0.144494),
                c64::new(0.221511, 0.909803),
                c64::new(0.908758, 0.482111),
                c64::new(0.350651, 0.714419),
                c64::new(0.979006, 0.536138),
                c64::new(0.305743, 0.946639),
                c64::new(0.062863, 0.563200),
                c64::new(0.541004, 0.233887),
                c64::new(0.995982, 0.792898),
                c64::new(0.823303, 0.841811),
                c64::new(0.086561, 0.241551),
                c64::new(0.340164, 0.403718),
                c64::new(0.171850, 0.813410),
                c64::new(0.463218, 0.376034),
                c64::new(0.712241, 0.615310),
                c64::new(0.809444, 0.167278),
                c64::new(0.762444, 0.374177),
                c64::new(0.914282, 0.615331),
                c64::new(0.143935, 0.454042),
                c64::new(0.296887, 0.393304),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.715274, 0.367237),
                c64::new(0.509709, 0.598164),
                c64::new(0.390999, 0.209680),
                c64::new(0.663001, 0.489987),
                c64::new(0.179933, 0.691761),
                c64::new(0.716592, 0.565920),
                c64::new(0.089595, 0.402018),
                c64::new(0.617964, 0.709612),
                c64::new(0.054911, 0.687305),
                c64::new(0.312831, 0.665747),
                c64::new(0.228305, 0.102253),
                c64::new(0.631583, 0.764260),
                c64::new(0.592907, 0.490800),
                c64::new(0.036438, 0.809528),
                c64::new(0.243750, 0.852269),
                c64::new(0.951810, 0.052494),
                c64::new(0.618893, 0.946484),
                c64::new(0.939660, 0.757697),
                c64::new(0.946228, 0.143086),
                c64::new(0.415961, 0.214238),
                c64::new(0.984906, 0.076734),
                c64::new(0.992232, 0.515563),
                c64::new(0.805721, 0.030894),
                c64::new(0.137336, 0.508217),
                c64::new(0.558797, 0.471115),
                c64::new(0.202788, 0.329763),
                c64::new(0.071424, 0.137316),
                c64::new(0.240207, 0.944668),
                c64::new(0.558816, 0.158790),
                c64::new(0.449376, 0.195025),
                c64::new(0.603549, 0.739477),
                c64::new(0.414329, 0.371159),
                c64::new(0.475775, 0.564285),
                c64::new(0.033610, 0.385572),
                c64::new(0.448648, 0.144425),
                c64::new(0.249027, 0.805371),
                c64::new(0.621390, 0.423475),
                c64::new(0.053593, 0.586188),
                c64::new(0.730849, 0.594146),
                c64::new(0.879174, 0.775135),
                c64::new(0.953319, 0.503720),
                c64::new(0.705713, 0.996021),
                c64::new(0.850167, 0.408298),
                c64::new(0.862667, 0.713061),
                c64::new(0.786574, 0.651448),
                c64::new(0.372438, 0.429324),
                c64::new(0.856125, 0.393425),
                c64::new(0.243611, 0.614022),
                c64::new(0.319956, 0.100039),
                c64::new(0.755913, 0.395980),
                c64::new(0.887310, 0.424228),
                c64::new(0.169630, 0.604425),
                c64::new(0.494511, 0.511823),
                c64::new(0.406940, 0.661718),
                c64::new(0.100812, 0.896214),
                c64::new(0.843168, 0.647253),
                c64::new(0.645970, 0.844043),
                c64::new(0.039139, 0.459093),
                c64::new(0.720323, 0.183381),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.425928, 0.107204),
                c64::new(0.800045, 0.099971),
                c64::new(0.070251, 0.151644),
                c64::new(0.371839, 0.662490),
                c64::new(0.738315, 0.351246),
                c64::new(0.432473, 0.729187),
                c64::new(0.419706, 0.995534),
                c64::new(0.362100, 0.552402),
                c64::new(0.460506, 0.763973),
                c64::new(0.820248, 0.185273),
                c64::new(0.543987, 0.960491),
                c64::new(0.146243, 0.996164),
                c64::new(0.319060, 0.749107),
                c64::new(0.765624, 0.873639),
                c64::new(0.733579, 0.813885),
                c64::new(0.810016, 0.844789),
                c64::new(0.960963, 0.062396),
                c64::new(0.985746, 0.784937),
                c64::new(0.920432, 0.570684),
                c64::new(0.821910, 0.376710),
                c64::new(0.163600, 0.281017),
                c64::new(0.305898, 0.418352),
                c64::new(0.066167, 0.793898),
                c64::new(0.060118, 0.498109),
                c64::new(0.200186, 0.687364),
                c64::new(0.025688, 0.953083),
                c64::new(0.321023, 0.687941),
                c64::new(0.339044, 0.964690),
                c64::new(0.119874, 0.658082),
                c64::new(0.122863, 0.713573),
                c64::new(0.363512, 0.545189),
                c64::new(0.118211, 0.086559),
                c64::new(0.463308, 0.968755),
                c64::new(0.941122, 0.359383),
                c64::new(0.665183, 0.556579),
                c64::new(0.680304, 0.405885),
                c64::new(0.831033, 0.924925),
                c64::new(0.966335, 0.707535),
                c64::new(0.169531, 0.414370),
                c64::new(0.944768, 0.391817),
                c64::new(0.109566, 0.792191),
                c64::new(0.900625, 0.485256),
                c64::new(0.684430, 0.350744),
                c64::new(0.823977, 0.056221),
                c64::new(0.280401, 0.008367),
                c64::new(0.501874, 0.720454),
                c64::new(0.217392, 0.517522),
                c64::new(0.840796, 0.398781),
                c64::new(0.986843, 0.905970),
                c64::new(0.014353, 0.173737),
                c64::new(0.490886, 0.185908),
                c64::new(0.628309, 0.192846),
                c64::new(0.811604, 0.933861),
                c64::new(0.535302, 0.027279),
                c64::new(0.931984, 0.482057),
                c64::new(0.038722, 0.441096),
                c64::new(0.137052, 0.307943),
                c64::new(0.612830, 0.617216),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.418849, 0.532531),
                c64::new(0.107083, 0.719746),
                c64::new(0.310694, 0.560926),
                c64::new(0.760125, 0.262740),
                c64::new(0.158550, 0.994097),
                c64::new(0.807434, 0.158468),
                c64::new(0.775618, 0.946512),
                c64::new(0.431907, 0.652202),
                c64::new(0.142663, 0.639367),
                c64::new(0.574674, 0.387925),
                c64::new(0.110711, 0.892480),
                c64::new(0.083943, 0.601967),
                c64::new(0.943322, 0.808827),
                c64::new(0.168371, 0.725480),
                c64::new(0.291024, 0.636374),
                c64::new(0.975400, 0.649293),
                c64::new(0.371338, 0.798654),
                c64::new(0.328305, 0.172540),
                c64::new(0.235682, 0.047897),
                c64::new(0.905067, 0.940245),
                c64::new(0.907505, 0.043832),
                c64::new(0.304953, 0.998240),
                c64::new(0.375923, 0.686134),
                c64::new(0.281803, 0.189080),
                c64::new(0.936808, 0.084883),
                c64::new(0.453281, 0.786526),
                c64::new(0.841850, 0.539031),
                c64::new(0.207174, 0.980019),
                c64::new(0.303841, 0.588644),
                c64::new(0.899236, 0.866050),
                c64::new(0.513926, 0.203281),
                c64::new(0.473419, 0.006041),
                c64::new(0.755681, 0.704911),
                c64::new(0.839672, 0.334013),
                c64::new(0.237145, 0.483412),
                c64::new(0.601244, 0.901770),
                c64::new(0.491642, 0.231243),
                c64::new(0.035101, 0.401534),
                c64::new(0.098882, 0.958966),
                c64::new(0.796004, 0.975521),
                c64::new(0.481170, 0.786754),
                c64::new(0.012739, 0.467880),
                c64::new(0.967237, 0.450599),
                c64::new(0.088282, 0.963175),
                c64::new(0.997335, 0.699526),
                c64::new(0.645252, 0.323436),
                c64::new(0.362289, 0.209948),
                c64::new(0.942874, 0.734774),
                c64::new(0.767611, 0.944060),
                c64::new(0.234817, 0.510999),
                c64::new(0.624267, 0.857866),
                c64::new(0.387731, 0.243747),
                c64::new(0.488888, 0.428010),
                c64::new(0.026640, 0.019524),
                c64::new(0.251537, 0.923842),
                c64::new(0.872278, 0.762427),
                c64::new(0.203000, 0.708612),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.536584, 0.723950),
                c64::new(0.677810, 0.609953),
                c64::new(0.202928, 0.830191),
                c64::new(0.721802, 0.030965),
                c64::new(0.590068, 0.041306),
                c64::new(0.593856, 0.529075),
                c64::new(0.864659, 0.754259),
                c64::new(0.163143, 0.760660),
                c64::new(0.872759, 0.574347),
                c64::new(0.433395, 0.507012),
                c64::new(0.522135, 0.805751),
                c64::new(0.923445, 0.361330),
                c64::new(0.868754, 0.732123),
                c64::new(0.098894, 0.636107),
                c64::new(0.735279, 0.680066),
                c64::new(0.647744, 0.099186),
                c64::new(0.520748, 0.423773),
                c64::new(0.612231, 0.988279),
                c64::new(0.175866, 0.795076),
                c64::new(0.999766, 0.165041),
                c64::new(0.889125, 0.431715),
                c64::new(0.184336, 0.297914),
                c64::new(0.727470, 0.172602),
                c64::new(0.420187, 0.559221),
                c64::new(0.208330, 0.268368),
                c64::new(0.252354, 0.730862),
                c64::new(0.304587, 0.614354),
                c64::new(0.110456, 0.195224),
                c64::new(0.303946, 0.388018),
                c64::new(0.726385, 0.653285),
                c64::new(0.593882, 0.084965),
                c64::new(0.360321, 0.858468),
                c64::new(0.893790, 0.437986),
                c64::new(0.991152, 0.093827),
                c64::new(0.328462, 0.800984),
                c64::new(0.226094, 0.243485),
                c64::new(0.456858, 0.556662),
                c64::new(0.265730, 0.445485),
                c64::new(0.103144, 0.465237),
                c64::new(0.828831, 0.995590),
                c64::new(0.510411, 0.153707),
                c64::new(0.216157, 0.682608),
                c64::new(0.782011, 0.058702),
                c64::new(0.071981, 0.734880),
                c64::new(0.571384, 0.968799),
                c64::new(0.920737, 0.914234),
                c64::new(0.018279, 0.976197),
                c64::new(0.966476, 0.679589),
                c64::new(0.156650, 0.477255),
                c64::new(0.957561, 0.010660),
                c64::new(0.657315, 0.771903),
                c64::new(0.715325, 0.317240),
                c64::new(0.948571, 0.413203),
                c64::new(0.012430, 0.437045),
                c64::new(0.916248, 0.017849),
                c64::new(0.409066, 0.902296),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.128966, 0.393048),
                c64::new(0.991382, 0.363649),
                c64::new(0.689582, 0.438939),
                c64::new(0.981972, 0.534930),
                c64::new(0.260193, 0.915763),
                c64::new(0.825290, 0.384255),
                c64::new(0.885266, 0.929639),
                c64::new(0.507663, 0.820205),
                c64::new(0.535147, 0.621162),
                c64::new(0.123445, 0.270132),
                c64::new(0.108129, 0.207833),
                c64::new(0.628728, 0.039248),
                c64::new(0.579917, 0.971678),
                c64::new(0.527193, 0.917437),
                c64::new(0.132248, 0.774570),
                c64::new(0.802700, 0.289729),
                c64::new(0.863688, 0.281852),
                c64::new(0.768497, 0.046592),
                c64::new(0.560227, 0.073644),
                c64::new(0.587867, 0.253124),
                c64::new(0.406094, 0.731975),
                c64::new(0.867507, 0.015642),
                c64::new(0.686177, 0.383130),
                c64::new(0.723251, 0.745997),
                c64::new(0.027504, 0.055766),
                c64::new(0.167852, 0.894087),
                c64::new(0.145940, 0.576829),
                c64::new(0.015916, 0.885107),
                c64::new(0.779099, 0.877844),
                c64::new(0.458668, 0.519214),
                c64::new(0.916154, 0.111284),
                c64::new(0.836685, 0.631290),
                c64::new(0.728151, 0.354445),
                c64::new(0.058543, 0.521122),
                c64::new(0.657684, 0.582414),
                c64::new(0.342108, 0.324812),
                c64::new(0.672241, 0.744105),
                c64::new(0.446364, 0.686208),
                c64::new(0.965206, 0.475126),
                c64::new(0.344798, 0.578364),
                c64::new(0.008011, 0.357516),
                c64::new(0.034041, 0.499454),
                c64::new(0.339018, 0.350601),
                c64::new(0.115104, 0.869680),
                c64::new(0.270346, 0.103482),
                c64::new(0.827058, 0.369253),
                c64::new(0.703335, 0.220370),
                c64::new(0.161436, 0.829609),
                c64::new(0.316228, 0.518901),
                c64::new(0.313521, 0.431155),
                c64::new(0.359681, 0.306954),
                c64::new(0.311050, 0.915704),
                c64::new(0.383658, 0.937352),
                c64::new(0.889989, 0.170452),
                c64::new(0.559301, 0.473358),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.050173, 0.552781),
                c64::new(0.415845, 0.512301),
                c64::new(0.054231, 0.213533),
                c64::new(0.737206, 0.779989),
                c64::new(0.089992, 0.536487),
                c64::new(0.779028, 0.284559),
                c64::new(0.238610, 0.061747),
                c64::new(0.710270, 0.238035),
                c64::new(0.032906, 0.671454),
                c64::new(0.009978, 0.729838),
                c64::new(0.671447, 0.882419),
                c64::new(0.396256, 0.133264),
                c64::new(0.677539, 0.002765),
                c64::new(0.632125, 0.426313),
                c64::new(0.502694, 0.924364),
                c64::new(0.618373, 0.958592),
                c64::new(0.180755, 0.279852),
                c64::new(0.566814, 0.919856),
                c64::new(0.161129, 0.863001),
                c64::new(0.749486, 0.733981),
                c64::new(0.420560, 0.782312),
                c64::new(0.397212, 0.425525),
                c64::new(0.336070, 0.659079),
                c64::new(0.367184, 0.042728),
                c64::new(0.409056, 0.739271),
                c64::new(0.834689, 0.561427),
                c64::new(0.036659, 0.930406),
                c64::new(0.522245, 0.395179),
                c64::new(0.565026, 0.421176),
                c64::new(0.098957, 0.841092),
                c64::new(0.212424, 0.469148),
                c64::new(0.591057, 0.499176),
                c64::new(0.702326, 0.081997),
                c64::new(0.384239, 0.647011),
                c64::new(0.881257, 0.591259),
                c64::new(0.180229, 0.193953),
                c64::new(0.242542, 0.346969),
                c64::new(0.669821, 0.154080),
                c64::new(0.034888, 0.206132),
                c64::new(0.308462, 0.284540),
                c64::new(0.567083, 0.263070),
                c64::new(0.892258, 0.312336),
                c64::new(0.952831, 0.188356),
                c64::new(0.546890, 0.653276),
                c64::new(0.982915, 0.846217),
                c64::new(0.351776, 0.003294),
                c64::new(0.024888, 0.215093),
                c64::new(0.600126, 0.519162),
                c64::new(0.825102, 0.063609),
                c64::new(0.546078, 0.764150),
                c64::new(0.874095, 0.952595),
                c64::new(0.810068, 0.070169),
                c64::new(0.902064, 0.710634),
                c64::new(0.444590, 0.180842),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.529818, 0.784003),
                c64::new(0.065876, 0.348303),
                c64::new(0.959784, 0.922829),
                c64::new(0.309259, 0.296683),
                c64::new(0.869259, 0.324780),
                c64::new(0.366862, 0.183642),
                c64::new(0.803000, 0.548745),
                c64::new(0.586937, 0.747447),
                c64::new(0.759521, 0.377407),
                c64::new(0.775176, 0.005084),
                c64::new(0.418544, 0.586202),
                c64::new(0.852882, 0.746814),
                c64::new(0.767252, 0.467046),
                c64::new(0.310193, 0.204264),
                c64::new(0.206111, 0.415933),
                c64::new(0.664510, 0.898087),
                c64::new(0.087468, 0.422819),
                c64::new(0.198767, 0.946357),
                c64::new(0.700252, 0.156549),
                c64::new(0.664096, 0.333602),
                c64::new(0.492962, 0.448198),
                c64::new(0.866852, 0.973557),
                c64::new(0.146269, 0.186005),
                c64::new(0.923646, 0.760607),
                c64::new(0.648200, 0.016135),
                c64::new(0.333844, 0.646075),
                c64::new(0.053782, 0.318661),
                c64::new(0.224892, 0.927456),
                c64::new(0.114737, 0.019274),
                c64::new(0.342666, 0.324201),
                c64::new(0.751898, 0.771154),
                c64::new(0.879673, 0.845538),
                c64::new(0.487731, 0.639117),
                c64::new(0.600249, 0.784276),
                c64::new(0.643879, 0.390951),
                c64::new(0.791309, 0.612263),
                c64::new(0.227421, 0.889436),
                c64::new(0.326243, 0.577876),
                c64::new(0.830689, 0.846874),
                c64::new(0.953776, 0.426577),
                c64::new(0.746185, 0.380683),
                c64::new(0.403674, 0.192418),
                c64::new(0.115260, 0.458896),
                c64::new(0.334320, 0.234912),
                c64::new(0.603312, 0.799253),
                c64::new(0.002358, 0.862354),
                c64::new(0.397216, 0.576467),
                c64::new(0.117960, 0.198585),
                c64::new(0.609190, 0.045098),
                c64::new(0.190081, 0.030743),
                c64::new(0.023431, 0.143244),
                c64::new(0.052493, 0.170783),
                c64::new(0.370181, 0.432948),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.692966, 0.106329),
                c64::new(0.107289, 0.819744),
                c64::new(0.269768, 0.861335),
                c64::new(0.008943, 0.833064),
                c64::new(0.883100, 0.120529),
                c64::new(0.505258, 0.452582),
                c64::new(0.499368, 0.426845),
                c64::new(0.857245, 0.065116),
                c64::new(0.036659, 0.496364),
                c64::new(0.843591, 0.155928),
                c64::new(0.000781, 0.990894),
                c64::new(0.853006, 0.816387),
                c64::new(0.071921, 0.902840),
                c64::new(0.854368, 0.134865),
                c64::new(0.209034, 0.605100),
                c64::new(0.445809, 0.439048),
                c64::new(0.923672, 0.697042),
                c64::new(0.488671, 0.859437),
                c64::new(0.110011, 0.989637),
                c64::new(0.489858, 0.856089),
                c64::new(0.603585, 0.952052),
                c64::new(0.256532, 0.621132),
                c64::new(0.324275, 0.178258),
                c64::new(0.204355, 0.972283),
                c64::new(0.118170, 0.201359),
                c64::new(0.167174, 0.705644),
                c64::new(0.850068, 0.243707),
                c64::new(0.948765, 0.928045),
                c64::new(0.887268, 0.380672),
                c64::new(0.599746, 0.801615),
                c64::new(0.757189, 0.659039),
                c64::new(0.483228, 0.252994),
                c64::new(0.114750, 0.712840),
                c64::new(0.184600, 0.021121),
                c64::new(0.800941, 0.484812),
                c64::new(0.494397, 0.435646),
                c64::new(0.896421, 0.310441),
                c64::new(0.308579, 0.421727),
                c64::new(0.523243, 0.851544),
                c64::new(0.223532, 0.770586),
                c64::new(0.331834, 0.492622),
                c64::new(0.383687, 0.727389),
                c64::new(0.478783, 0.741261),
                c64::new(0.422514, 0.470108),
                c64::new(0.345648, 0.733007),
                c64::new(0.621038, 0.540694),
                c64::new(0.060240, 0.192364),
                c64::new(0.108584, 0.429246),
                c64::new(0.272918, 0.501780),
                c64::new(0.856421, 0.933779),
                c64::new(0.383547, 0.565598),
                c64::new(0.205123, 0.471749),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.530644, 0.114391),
                c64::new(0.995212, 0.060741),
                c64::new(0.196191, 0.251464),
                c64::new(0.871896, 0.669238),
                c64::new(0.623634, 0.159381),
                c64::new(0.251083, 0.272553),
                c64::new(0.153685, 0.294151),
                c64::new(0.610836, 0.554587),
                c64::new(0.382380, 0.713336),
                c64::new(0.855581, 0.789283),
                c64::new(0.834056, 0.246915),
                c64::new(0.189225, 0.908385),
                c64::new(0.366736, 0.008198),
                c64::new(0.737318, 0.929174),
                c64::new(0.672374, 0.013510),
                c64::new(0.373511, 0.578757),
                c64::new(0.053234, 0.222877),
                c64::new(0.915233, 0.315481),
                c64::new(0.221711, 0.343100),
                c64::new(0.312995, 0.087510),
                c64::new(0.065323, 0.184284),
                c64::new(0.931357, 0.753462),
                c64::new(0.789698, 0.365413),
                c64::new(0.556487, 0.567628),
                c64::new(0.181109, 0.243681),
                c64::new(0.947106, 0.927725),
                c64::new(0.257872, 0.154825),
                c64::new(0.830974, 0.777344),
                c64::new(0.029995, 0.113224),
                c64::new(0.975448, 0.376700),
                c64::new(0.110346, 0.300270),
                c64::new(0.594238, 0.452685),
                c64::new(0.811570, 0.367539),
                c64::new(0.434646, 0.456926),
                c64::new(0.556643, 0.156577),
                c64::new(0.142664, 0.796419),
                c64::new(0.555489, 0.204816),
                c64::new(0.251996, 0.005737),
                c64::new(0.301605, 0.048812),
                c64::new(0.521452, 0.714892),
                c64::new(0.380390, 0.225937),
                c64::new(0.784832, 0.205877),
                c64::new(0.010905, 0.538770),
                c64::new(0.253868, 0.418586),
                c64::new(0.418340, 0.178341),
                c64::new(0.494828, 0.500012),
                c64::new(0.270986, 0.786761),
                c64::new(0.078481, 0.238570),
                c64::new(0.128007, 0.732474),
                c64::new(0.710010, 0.285319),
                c64::new(0.161119, 0.232155),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.615040, 0.169075),
                c64::new(0.479961, 0.196773),
                c64::new(0.626809, 0.671324),
                c64::new(0.015183, 0.675374),
                c64::new(0.853066, 0.383082),
                c64::new(0.634475, 0.732480),
                c64::new(0.737968, 0.785650),
                c64::new(0.055095, 0.243289),
                c64::new(0.302135, 0.523876),
                c64::new(0.273966, 0.611152),
                c64::new(0.519451, 0.925538),
                c64::new(0.121914, 0.443001),
                c64::new(0.338227, 0.173193),
                c64::new(0.143675, 0.081170),
                c64::new(0.361864, 0.192628),
                c64::new(0.574452, 0.358146),
                c64::new(0.254029, 0.992810),
                c64::new(0.312667, 0.147341),
                c64::new(0.163967, 0.337127),
                c64::new(0.067906, 0.463478),
                c64::new(0.862173, 0.326313),
                c64::new(0.020866, 0.777907),
                c64::new(0.294795, 0.218811),
                c64::new(0.779580, 0.448250),
                c64::new(0.151978, 0.118878),
                c64::new(0.105307, 0.522607),
                c64::new(0.044355, 0.477039),
                c64::new(0.553556, 0.243574),
                c64::new(0.517181, 0.966844),
                c64::new(0.176832, 0.781490),
                c64::new(0.103951, 0.213612),
                c64::new(0.809956, 0.089341),
                c64::new(0.135489, 0.114184),
                c64::new(0.582365, 0.160086),
                c64::new(0.049516, 0.273526),
                c64::new(0.468771, 0.332455),
                c64::new(0.547074, 0.848985),
                c64::new(0.148206, 0.045514),
                c64::new(0.760739, 0.938908),
                c64::new(0.855047, 0.228042),
                c64::new(0.340053, 0.992980),
                c64::new(0.386192, 0.931463),
                c64::new(0.925009, 0.163898),
                c64::new(0.735551, 0.987414),
                c64::new(0.858869, 0.133657),
                c64::new(0.635674, 0.330128),
                c64::new(0.591815, 0.259063),
                c64::new(0.199770, 0.216630),
                c64::new(0.270794, 0.038836),
                c64::new(0.917995, 0.710299),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.013206, 0.769226),
                c64::new(0.982724, 0.204332),
                c64::new(0.832015, 0.230369),
                c64::new(0.722229, 0.943239),
                c64::new(0.164207, 0.795466),
                c64::new(0.870445, 0.218724),
                c64::new(0.905161, 0.733672),
                c64::new(0.793996, 0.840568),
                c64::new(0.568141, 0.209436),
                c64::new(0.939912, 0.015546),
                c64::new(0.708153, 0.119672),
                c64::new(0.112656, 0.091892),
                c64::new(0.789936, 0.998927),
                c64::new(0.600303, 0.584520),
                c64::new(0.032633, 0.801800),
                c64::new(0.640685, 0.208122),
                c64::new(0.914674, 0.772388),
                c64::new(0.461943, 0.399845),
                c64::new(0.289375, 0.498549),
                c64::new(0.865014, 0.199500),
                c64::new(0.913698, 0.244386),
                c64::new(0.674337, 0.877634),
                c64::new(0.292650, 0.317611),
                c64::new(0.141600, 0.161414),
                c64::new(0.276631, 0.996088),
                c64::new(0.657545, 0.044637),
                c64::new(0.401320, 0.015300),
                c64::new(0.482133, 0.906061),
                c64::new(0.528246, 0.093756),
                c64::new(0.463702, 0.584531),
                c64::new(0.549211, 0.621718),
                c64::new(0.486438, 0.499984),
                c64::new(0.954552, 0.056841),
                c64::new(0.786101, 0.515876),
                c64::new(0.536461, 0.113695),
                c64::new(0.942868, 0.413480),
                c64::new(0.819429, 0.432666),
                c64::new(0.256145, 0.574643),
                c64::new(0.104054, 0.868428),
                c64::new(0.755437, 0.700219),
                c64::new(0.381512, 0.537619),
                c64::new(0.088226, 0.441106),
                c64::new(0.050314, 0.735001),
                c64::new(0.251476, 0.475054),
                c64::new(0.678935, 0.061407),
                c64::new(0.497039, 0.530531),
                c64::new(0.085386, 0.399800),
                c64::new(0.891555, 0.439290),
                c64::new(0.214637, 0.314174),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.735857, 0.240597),
                c64::new(0.356875, 0.454911),
                c64::new(0.677605, 0.062164),
                c64::new(0.458087, 0.091385),
                c64::new(0.848670, 0.956103),
                c64::new(0.509684, 0.719534),
                c64::new(0.559033, 0.496636),
                c64::new(0.634391, 0.077391),
                c64::new(0.144552, 0.615835),
                c64::new(0.382440, 0.428077),
                c64::new(0.329797, 0.592698),
                c64::new(0.259466, 0.861564),
                c64::new(0.814550, 0.639481),
                c64::new(0.681772, 0.188467),
                c64::new(0.542203, 0.230573),
                c64::new(0.723169, 0.296597),
                c64::new(0.502171, 0.601625),
                c64::new(0.108210, 0.574614),
                c64::new(0.950486, 0.662628),
                c64::new(0.134496, 0.121186),
                c64::new(0.945668, 0.980793),
                c64::new(0.279623, 0.809355),
                c64::new(0.289832, 0.629227),
                c64::new(0.264471, 0.791683),
                c64::new(0.809414, 0.203814),
                c64::new(0.923640, 0.753235),
                c64::new(0.024518, 0.882491),
                c64::new(0.529799, 0.719094),
                c64::new(0.209641, 0.486209),
                c64::new(0.881656, 0.619114),
                c64::new(0.125449, 0.139146),
                c64::new(0.421330, 0.931407),
                c64::new(0.165221, 0.504492),
                c64::new(0.745031, 0.116208),
                c64::new(0.825442, 0.313819),
                c64::new(0.717770, 0.345277),
                c64::new(0.717937, 0.471781),
                c64::new(0.259691, 0.525759),
                c64::new(0.538932, 0.059867),
                c64::new(0.308357, 0.922209),
                c64::new(0.553270, 0.775689),
                c64::new(0.244659, 0.475483),
                c64::new(0.754569, 0.513202),
                c64::new(0.233177, 0.433069),
                c64::new(0.393786, 0.264758),
                c64::new(0.395247, 0.607339),
                c64::new(0.782158, 0.362668),
                c64::new(0.466013, 0.676191),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.211153, 0.524206),
                c64::new(0.782684, 0.305200),
                c64::new(0.289794, 0.268442),
                c64::new(0.877141, 0.625267),
                c64::new(0.790307, 0.280562),
                c64::new(0.969690, 0.180761),
                c64::new(0.932455, 0.373890),
                c64::new(0.877947, 0.750656),
                c64::new(0.954191, 0.542275),
                c64::new(0.052056, 0.536181),
                c64::new(0.006876, 0.372513),
                c64::new(0.399839, 0.480845),
                c64::new(0.815459, 0.473374),
                c64::new(0.598484, 0.121254),
                c64::new(0.120161, 0.001846),
                c64::new(0.193432, 0.108058),
                c64::new(0.869470, 0.966034),
                c64::new(0.984019, 0.518268),
                c64::new(0.720083, 0.755748),
                c64::new(0.962629, 0.341934),
                c64::new(0.141994, 0.141350),
                c64::new(0.659665, 0.919693),
                c64::new(0.772235, 0.877017),
                c64::new(0.119477, 0.982275),
                c64::new(0.098820, 0.946571),
                c64::new(0.699791, 0.867417),
                c64::new(0.940846, 0.809407),
                c64::new(0.959413, 0.987842),
                c64::new(0.274393, 0.443179),
                c64::new(0.222700, 0.549738),
                c64::new(0.516315, 0.863630),
                c64::new(0.706504, 0.996410),
                c64::new(0.929674, 0.240881),
                c64::new(0.162427, 0.367135),
                c64::new(0.163174, 0.278608),
                c64::new(0.048447, 0.319252),
                c64::new(0.216889, 0.002103),
                c64::new(0.466778, 0.717206),
                c64::new(0.565258, 0.438017),
                c64::new(0.626099, 0.347652),
                c64::new(0.157418, 0.755288),
                c64::new(0.361151, 0.210734),
                c64::new(0.729458, 0.276126),
                c64::new(0.400067, 0.158686),
                c64::new(0.313482, 0.732345),
                c64::new(0.923399, 0.049118),
                c64::new(0.877780, 0.677202),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.489670, 0.014685),
                c64::new(0.954007, 0.386280),
                c64::new(0.508154, 0.083524),
                c64::new(0.102988, 0.822501),
                c64::new(0.914405, 0.376727),
                c64::new(0.629056, 0.889769),
                c64::new(0.755001, 0.312801),
                c64::new(0.842430, 0.072925),
                c64::new(0.178321, 0.332097),
                c64::new(0.597829, 0.044490),
                c64::new(0.330761, 0.450853),
                c64::new(0.622313, 0.877718),
                c64::new(0.800713, 0.276759),
                c64::new(0.241428, 0.832541),
                c64::new(0.697723, 0.989073),
                c64::new(0.730382, 0.635387),
                c64::new(0.553550, 0.538356),
                c64::new(0.474978, 0.556896),
                c64::new(0.821939, 0.844671),
                c64::new(0.528817, 0.795304),
                c64::new(0.573663, 0.943586),
                c64::new(0.128926, 0.236285),
                c64::new(0.606450, 0.158736),
                c64::new(0.154906, 0.231764),
                c64::new(0.029804, 0.707674),
                c64::new(0.946999, 0.522959),
                c64::new(0.103517, 0.674785),
                c64::new(0.142024, 0.702340),
                c64::new(0.900657, 0.824920),
                c64::new(0.675868, 0.941447),
                c64::new(0.838365, 0.916458),
                c64::new(0.004632, 0.576753),
                c64::new(0.547344, 0.124311),
                c64::new(0.454067, 0.579979),
                c64::new(0.702220, 0.513525),
                c64::new(0.412980, 0.564992),
                c64::new(0.126822, 0.088142),
                c64::new(0.186729, 0.766497),
                c64::new(0.070893, 0.077732),
                c64::new(0.668822, 0.563395),
                c64::new(0.599082, 0.662259),
                c64::new(0.016302, 0.611303),
                c64::new(0.819525, 0.216332),
                c64::new(0.879024, 0.973667),
                c64::new(0.953213, 0.414129),
                c64::new(0.199642, 0.400405),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.374217, 0.584874),
                c64::new(0.674097, 0.993344),
                c64::new(0.260273, 0.726468),
                c64::new(0.159321, 0.707261),
                c64::new(0.861934, 0.305156),
                c64::new(0.257695, 0.819030),
                c64::new(0.673227, 0.478773),
                c64::new(0.379518, 0.566754),
                c64::new(0.325058, 0.295923),
                c64::new(0.127626, 0.470031),
                c64::new(0.442566, 0.071991),
                c64::new(0.212180, 0.110231),
                c64::new(0.055312, 0.572069),
                c64::new(0.908728, 0.879889),
                c64::new(0.854632, 0.602182),
                c64::new(0.032382, 0.567270),
                c64::new(0.105613, 0.239383),
                c64::new(0.416218, 0.937289),
                c64::new(0.576225, 0.144356),
                c64::new(0.745276, 0.873887),
                c64::new(0.438998, 0.804666),
                c64::new(0.219396, 0.678971),
                c64::new(0.420545, 0.155666),
                c64::new(0.350061, 0.054623),
                c64::new(0.724278, 0.245731),
                c64::new(0.915217, 0.751912),
                c64::new(0.897378, 0.158476),
                c64::new(0.493065, 0.190242),
                c64::new(0.846690, 0.162171),
                c64::new(0.614193, 0.830638),
                c64::new(0.728447, 0.713174),
                c64::new(0.867831, 0.152079),
                c64::new(0.344324, 0.189384),
                c64::new(0.041797, 0.095245),
                c64::new(0.145474, 0.825597),
                c64::new(0.409952, 0.084094),
                c64::new(0.538063, 0.853134),
                c64::new(0.389526, 0.867265),
                c64::new(0.156201, 0.373831),
                c64::new(0.856444, 0.255195),
                c64::new(0.624446, 0.800430),
                c64::new(0.813866, 0.704184),
                c64::new(0.305135, 0.440342),
                c64::new(0.935756, 0.535515),
                c64::new(0.239918, 0.421123),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.443887, 0.096252),
                c64::new(0.048803, 0.497488),
                c64::new(0.471642, 0.661234),
                c64::new(0.672943, 0.569576),
                c64::new(0.126652, 0.077283),
                c64::new(0.929775, 0.760164),
                c64::new(0.209105, 0.921316),
                c64::new(0.730360, 0.098735),
                c64::new(0.837673, 0.384573),
                c64::new(0.793078, 0.778765),
                c64::new(0.075759, 0.204853),
                c64::new(0.067606, 0.310397),
                c64::new(0.990904, 0.653071),
                c64::new(0.671204, 0.829804),
                c64::new(0.039517, 0.702387),
                c64::new(0.770537, 0.087154),
                c64::new(0.285338, 0.629213),
                c64::new(0.350901, 0.274550),
                c64::new(0.986373, 0.895405),
                c64::new(0.671961, 0.764578),
                c64::new(0.094694, 0.090760),
                c64::new(0.693340, 0.616171),
                c64::new(0.647754, 0.612236),
                c64::new(0.139916, 0.646102),
                c64::new(0.511963, 0.592511),
                c64::new(0.046803, 0.376892),
                c64::new(0.726543, 0.342553),
                c64::new(0.497081, 0.218586),
                c64::new(0.924997, 0.975546),
                c64::new(0.130269, 0.391220),
                c64::new(0.252864, 0.062803),
                c64::new(0.792219, 0.181532),
                c64::new(0.543743, 0.851862),
                c64::new(0.857259, 0.746477),
                c64::new(0.880736, 0.652548),
                c64::new(0.980625, 0.937644),
                c64::new(0.966536, 0.923276),
                c64::new(0.041165, 0.146294),
                c64::new(0.011005, 0.103325),
                c64::new(0.341004, 0.665122),
                c64::new(0.671485, 0.370646),
                c64::new(0.413666, 0.898151),
                c64::new(0.647795, 0.946871),
                c64::new(0.634048, 0.264302),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.399760, 0.545531),
                c64::new(0.302525, 0.349635),
                c64::new(0.689429, 0.842495),
                c64::new(0.533322, 0.010064),
                c64::new(0.819166, 0.243282),
                c64::new(0.583474, 0.841934),
                c64::new(0.176796, 0.576558),
                c64::new(0.975096, 0.985660),
                c64::new(0.838062, 0.680148),
                c64::new(0.874235, 0.481368),
                c64::new(0.963365, 0.562827),
                c64::new(0.709004, 0.504630),
                c64::new(0.723597, 0.477495),
                c64::new(0.900440, 0.436930),
                c64::new(0.038598, 0.623360),
                c64::new(0.148142, 0.588271),
                c64::new(0.021614, 0.156331),
                c64::new(0.246216, 0.335143),
                c64::new(0.335856, 0.499071),
                c64::new(0.491294, 0.703764),
                c64::new(0.803089, 0.309034),
                c64::new(0.608218, 0.082254),
                c64::new(0.201717, 0.668288),
                c64::new(0.249120, 0.151090),
                c64::new(0.906595, 0.457076),
                c64::new(0.373574, 0.028683),
                c64::new(0.420152, 0.922571),
                c64::new(0.204873, 0.461974),
                c64::new(0.030222, 0.637080),
                c64::new(0.180576, 0.714499),
                c64::new(0.519789, 0.687641),
                c64::new(0.971125, 0.477576),
                c64::new(0.063517, 0.906057),
                c64::new(0.023328, 0.537450),
                c64::new(0.242376, 0.024823),
                c64::new(0.241058, 0.739243),
                c64::new(0.074309, 0.920669),
                c64::new(0.092918, 0.988590),
                c64::new(0.656460, 0.399125),
                c64::new(0.445855, 0.015304),
                c64::new(0.980544, 0.780683),
                c64::new(0.709660, 0.150207),
                c64::new(0.929180, 0.512614),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.183885, 0.640048),
                c64::new(0.912031, 0.634759),
                c64::new(0.236413, 0.295296),
                c64::new(0.802898, 0.509686),
                c64::new(0.465896, 0.991385),
                c64::new(0.215434, 0.858177),
                c64::new(0.213919, 0.535239),
                c64::new(0.516533, 0.693449),
                c64::new(0.192257, 0.140350),
                c64::new(0.033248, 0.426737),
                c64::new(0.594861, 0.553121),
                c64::new(0.629219, 0.815631),
                c64::new(0.738466, 0.478181),
                c64::new(0.284469, 0.636524),
                c64::new(0.941980, 0.963783),
                c64::new(0.076833, 0.344528),
                c64::new(0.305275, 0.382909),
                c64::new(0.455290, 0.409875),
                c64::new(0.515029, 0.551295),
                c64::new(0.107499, 0.231463),
                c64::new(0.763160, 0.003572),
                c64::new(0.985063, 0.544932),
                c64::new(0.951567, 0.386958),
                c64::new(0.980338, 0.350313),
                c64::new(0.016406, 0.726972),
                c64::new(0.449129, 0.641696),
                c64::new(0.672999, 0.756411),
                c64::new(0.623663, 0.983450),
                c64::new(0.321635, 0.378232),
                c64::new(0.794536, 0.654493),
                c64::new(0.081418, 0.747104),
                c64::new(0.562292, 0.238510),
                c64::new(0.633112, 0.758092),
                c64::new(0.368399, 0.113202),
                c64::new(0.209711, 0.536470),
                c64::new(0.424816, 0.634438),
                c64::new(0.651365, 0.955733),
                c64::new(0.777340, 0.979498),
                c64::new(0.250119, 0.763171),
                c64::new(0.893011, 0.506565),
                c64::new(0.453844, 0.276880),
                c64::new(0.958790, 0.279738),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.417238, 0.369492),
                c64::new(0.490593, 0.448243),
                c64::new(0.035815, 0.658906),
                c64::new(0.722830, 0.941547),
                c64::new(0.943861, 0.212606),
                c64::new(0.975218, 0.576794),
                c64::new(0.661743, 0.297687),
                c64::new(0.540861, 0.521144),
                c64::new(0.724745, 0.473121),
                c64::new(0.306702, 0.661744),
                c64::new(0.766205, 0.343116),
                c64::new(0.178149, 0.524040),
                c64::new(0.278347, 0.072409),
                c64::new(0.073853, 0.230312),
                c64::new(0.150297, 0.487793),
                c64::new(0.248672, 0.883212),
                c64::new(0.352977, 0.544588),
                c64::new(0.694336, 0.384960),
                c64::new(0.824341, 0.463108),
                c64::new(0.630856, 0.571479),
                c64::new(0.260298, 0.528532),
                c64::new(0.811217, 0.380165),
                c64::new(0.587356, 0.360909),
                c64::new(0.906302, 0.532543),
                c64::new(0.154833, 0.784205),
                c64::new(0.869951, 0.549785),
                c64::new(0.903254, 0.495039),
                c64::new(0.462508, 0.491829),
                c64::new(0.462055, 0.055157),
                c64::new(0.333980, 0.195511),
                c64::new(0.201547, 0.047648),
                c64::new(0.423427, 0.350561),
                c64::new(0.430907, 0.409929),
                c64::new(0.050620, 0.948940),
                c64::new(0.043502, 0.335449),
                c64::new(0.875777, 0.383892),
                c64::new(0.084340, 0.727528),
                c64::new(0.557758, 0.437854),
                c64::new(0.226960, 0.552549),
                c64::new(0.271554, 0.234268),
                c64::new(0.945420, 0.600483),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.505768, 0.657006),
                c64::new(0.317234, 0.834577),
                c64::new(0.256432, 0.284741),
                c64::new(0.546082, 0.830825),
                c64::new(0.563529, 0.064925),
                c64::new(0.449906, 0.731639),
                c64::new(0.215404, 0.615188),
                c64::new(0.262296, 0.981337),
                c64::new(0.603737, 0.561204),
                c64::new(0.537121, 0.982509),
                c64::new(0.210794, 0.680657),
                c64::new(0.204230, 0.075728),
                c64::new(0.256903, 0.771966),
                c64::new(0.785525, 0.818340),
                c64::new(0.195146, 0.225036),
                c64::new(0.098322, 0.031491),
                c64::new(0.468545, 0.601058),
                c64::new(0.175068, 0.323258),
                c64::new(0.744423, 0.888722),
                c64::new(0.355310, 0.442561),
                c64::new(0.825458, 0.854611),
                c64::new(0.265355, 0.372867),
                c64::new(0.574520, 0.213399),
                c64::new(0.358056, 0.333241),
                c64::new(0.181991, 0.071544),
                c64::new(0.479192, 0.159646),
                c64::new(0.314514, 0.137438),
                c64::new(0.434541, 0.705659),
                c64::new(0.177177, 0.880997),
                c64::new(0.532315, 0.194454),
                c64::new(0.663670, 0.527412),
                c64::new(0.807943, 0.019050),
                c64::new(0.193427, 0.231821),
                c64::new(0.634212, 0.356353),
                c64::new(0.069678, 0.726242),
                c64::new(0.176196, 0.066913),
                c64::new(0.499981, 0.104282),
                c64::new(0.451192, 0.411616),
                c64::new(0.624788, 0.879581),
                c64::new(0.338014, 0.138071),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.653213, 0.156558),
                c64::new(0.405071, 0.346611),
                c64::new(0.070587, 0.689220),
                c64::new(0.937365, 0.283641),
                c64::new(0.058099, 0.236126),
                c64::new(0.463782, 0.273806),
                c64::new(0.103982, 0.729582),
                c64::new(0.339173, 0.981693),
                c64::new(0.143861, 0.119670),
                c64::new(0.721600, 0.620586),
                c64::new(0.270474, 0.298087),
                c64::new(0.412060, 0.262356),
                c64::new(0.522878, 0.643121),
                c64::new(0.939208, 0.071407),
                c64::new(0.976928, 0.817970),
                c64::new(0.276530, 0.316723),
                c64::new(0.985798, 0.945611),
                c64::new(0.622870, 0.662908),
                c64::new(0.646896, 0.327989),
                c64::new(0.040252, 0.582125),
                c64::new(0.812273, 0.035676),
                c64::new(0.631451, 0.041672),
                c64::new(0.761114, 0.996644),
                c64::new(0.582146, 0.191825),
                c64::new(0.520360, 0.612590),
                c64::new(0.676146, 0.579723),
                c64::new(0.130856, 0.419919),
                c64::new(0.075372, 0.237200),
                c64::new(0.493157, 0.717249),
                c64::new(0.616828, 0.173934),
                c64::new(0.025733, 0.502773),
                c64::new(0.141155, 0.622883),
                c64::new(0.830478, 0.232179),
                c64::new(0.930286, 0.076636),
                c64::new(0.396307, 0.248675),
                c64::new(0.739178, 0.519687),
                c64::new(0.008844, 0.273440),
                c64::new(0.799089, 0.082913),
                c64::new(0.536671, 0.414616),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.521473, 0.493330),
                c64::new(0.974857, 0.834323),
                c64::new(0.924678, 0.781474),
                c64::new(0.799778, 0.908626),
                c64::new(0.759952, 0.735617),
                c64::new(0.929625, 0.196712),
                c64::new(0.725352, 0.183207),
                c64::new(0.582649, 0.239378),
                c64::new(0.476960, 0.676360),
                c64::new(0.756559, 0.692404),
                c64::new(0.526575, 0.913120),
                c64::new(0.645668, 0.746689),
                c64::new(0.103983, 0.623127),
                c64::new(0.368590, 0.229021),
                c64::new(0.161638, 0.698745),
                c64::new(0.807479, 0.517863),
                c64::new(0.286258, 0.121340),
                c64::new(0.816100, 0.079495),
                c64::new(0.885197, 0.415368),
                c64::new(0.570288, 0.925722),
                c64::new(0.185786, 0.450075),
                c64::new(0.048422, 0.071797),
                c64::new(0.868734, 0.965302),
                c64::new(0.042518, 0.027159),
                c64::new(0.243484, 0.185245),
                c64::new(0.353959, 0.626861),
                c64::new(0.158712, 0.200324),
                c64::new(0.655371, 0.960645),
                c64::new(0.762146, 0.335258),
                c64::new(0.608540, 0.724254),
                c64::new(0.695535, 0.085063),
                c64::new(0.443263, 0.240561),
                c64::new(0.800195, 0.579143),
                c64::new(0.532124, 0.120470),
                c64::new(0.763295, 0.438061),
                c64::new(0.862724, 0.170962),
                c64::new(0.043138, 0.998741),
                c64::new(0.198109, 0.350048),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.372571, 0.448722),
                c64::new(0.123640, 0.080367),
                c64::new(0.897636, 0.279573),
                c64::new(0.757336, 0.043046),
                c64::new(0.263985, 0.848332),
                c64::new(0.809271, 0.363908),
                c64::new(0.177042, 0.695124),
                c64::new(0.975649, 0.907473),
                c64::new(0.388029, 0.293192),
                c64::new(0.591198, 0.887654),
                c64::new(0.831064, 0.205380),
                c64::new(0.459905, 0.467796),
                c64::new(0.262950, 0.409071),
                c64::new(0.910862, 0.806174),
                c64::new(0.465802, 0.969445),
                c64::new(0.076946, 0.555695),
                c64::new(0.467504, 0.279660),
                c64::new(0.346030, 0.808167),
                c64::new(0.030150, 0.819514),
                c64::new(0.849011, 0.274073),
                c64::new(0.810242, 0.361799),
                c64::new(0.543533, 0.552200),
                c64::new(0.836596, 0.916915),
                c64::new(0.774134, 0.080999),
                c64::new(0.653268, 0.024501),
                c64::new(0.471965, 0.849630),
                c64::new(0.654975, 0.859705),
                c64::new(0.844660, 0.439487),
                c64::new(0.659301, 0.686872),
                c64::new(0.086476, 0.925477),
                c64::new(0.780236, 0.225463),
                c64::new(0.064931, 0.924466),
                c64::new(0.501670, 0.743015),
                c64::new(0.496195, 0.534468),
                c64::new(0.167837, 0.509298),
                c64::new(0.265500, 0.710958),
                c64::new(0.904978, 0.269971),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.294044, 0.101433),
                c64::new(0.840744, 0.482138),
                c64::new(0.321340, 0.661387),
                c64::new(0.046925, 0.019995),
                c64::new(0.058222, 0.774151),
                c64::new(0.689841, 0.479795),
                c64::new(0.028144, 0.590773),
                c64::new(0.418230, 0.534886),
                c64::new(0.010664, 0.859802),
                c64::new(0.978053, 0.420512),
                c64::new(0.515001, 0.569142),
                c64::new(0.938735, 0.728374),
                c64::new(0.705333, 0.138013),
                c64::new(0.256271, 0.780427),
                c64::new(0.055680, 0.333958),
                c64::new(0.049076, 0.837780),
                c64::new(0.475005, 0.018740),
                c64::new(0.087773, 0.551546),
                c64::new(0.199519, 0.008091),
                c64::new(0.375711, 0.472807),
                c64::new(0.384777, 0.245652),
                c64::new(0.656408, 0.886297),
                c64::new(0.832098, 0.919534),
                c64::new(0.720171, 0.042488),
                c64::new(0.357659, 0.101234),
                c64::new(0.987358, 0.402306),
                c64::new(0.002892, 0.123673),
                c64::new(0.381156, 0.815712),
                c64::new(0.536368, 0.073339),
                c64::new(0.980000, 0.743241),
                c64::new(0.624982, 0.876012),
                c64::new(0.743195, 0.256734),
                c64::new(0.923790, 0.803760),
                c64::new(0.020420, 0.823294),
                c64::new(0.913883, 0.149200),
                c64::new(0.167479, 0.037469),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.931888, 0.521996),
                c64::new(0.246849, 0.937940),
                c64::new(0.080709, 0.083121),
                c64::new(0.077108, 0.339871),
                c64::new(0.276430, 0.281028),
                c64::new(0.454198, 0.533949),
                c64::new(0.369664, 0.349490),
                c64::new(0.790596, 0.358850),
                c64::new(0.076970, 0.707554),
                c64::new(0.186630, 0.325488),
                c64::new(0.269162, 0.577607),
                c64::new(0.165001, 0.303532),
                c64::new(0.728556, 0.680016),
                c64::new(0.474250, 0.810420),
                c64::new(0.641103, 0.614717),
                c64::new(0.739550, 0.804100),
                c64::new(0.597237, 0.894850),
                c64::new(0.203832, 0.346580),
                c64::new(0.083933, 0.550873),
                c64::new(0.576164, 0.485114),
                c64::new(0.468030, 0.107698),
                c64::new(0.222727, 0.027713),
                c64::new(0.203053, 0.185396),
                c64::new(0.726388, 0.493627),
                c64::new(0.856492, 0.591528),
                c64::new(0.429702, 0.856186),
                c64::new(0.860735, 0.736582),
                c64::new(0.404977, 0.549614),
                c64::new(0.059981, 0.827863),
                c64::new(0.190986, 0.085409),
                c64::new(0.683270, 0.611813),
                c64::new(0.921823, 0.885776),
                c64::new(0.935917, 0.163498),
                c64::new(0.930860, 0.881363),
                c64::new(0.500434, 0.651832),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.477213, 0.816390),
                c64::new(0.696142, 0.274229),
                c64::new(0.473900, 0.191473),
                c64::new(0.756113, 0.893784),
                c64::new(0.228119, 0.215071),
                c64::new(0.178401, 0.847046),
                c64::new(0.368154, 0.431101),
                c64::new(0.931433, 0.444458),
                c64::new(0.392626, 0.443108),
                c64::new(0.802455, 0.827599),
                c64::new(0.504839, 0.435489),
                c64::new(0.911301, 0.019647),
                c64::new(0.528954, 0.184097),
                c64::new(0.374212, 0.419425),
                c64::new(0.318845, 0.853139),
                c64::new(0.761209, 0.699405),
                c64::new(0.822795, 0.816995),
                c64::new(0.906917, 0.330352),
                c64::new(0.397829, 0.673311),
                c64::new(0.325090, 0.795162),
                c64::new(0.523321, 0.803657),
                c64::new(0.063926, 0.901238),
                c64::new(0.702656, 0.551506),
                c64::new(0.775445, 0.396521),
                c64::new(0.345684, 0.683411),
                c64::new(0.832613, 0.422860),
                c64::new(0.949785, 0.716163),
                c64::new(0.122076, 0.036800),
                c64::new(0.376389, 0.026402),
                c64::new(0.784089, 0.449161),
                c64::new(0.546165, 0.199809),
                c64::new(0.201773, 0.853562),
                c64::new(0.817945, 0.246961),
                c64::new(0.766789, 0.364150),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.988552, 0.024220),
                c64::new(0.668978, 0.166909),
                c64::new(0.373566, 0.492455),
                c64::new(0.156334, 0.944764),
                c64::new(0.893443, 0.529912),
                c64::new(0.223432, 0.142919),
                c64::new(0.117640, 0.754037),
                c64::new(0.767211, 0.962383),
                c64::new(0.430687, 0.997779),
                c64::new(0.350008, 0.634207),
                c64::new(0.745317, 0.877145),
                c64::new(0.610312, 0.693158),
                c64::new(0.497559, 0.079633),
                c64::new(0.067073, 0.633330),
                c64::new(0.385002, 0.742704),
                c64::new(0.183908, 0.159583),
                c64::new(0.665184, 0.612644),
                c64::new(0.507063, 0.543285),
                c64::new(0.487266, 0.071591),
                c64::new(0.564632, 0.461084),
                c64::new(0.068025, 0.676927),
                c64::new(0.964044, 0.428076),
                c64::new(0.375566, 0.537645),
                c64::new(0.142064, 0.145362),
                c64::new(0.905332, 0.407875),
                c64::new(0.605614, 0.063398),
                c64::new(0.348924, 0.544658),
                c64::new(0.399783, 0.790985),
                c64::new(0.034866, 0.395968),
                c64::new(0.325829, 0.392267),
                c64::new(0.141683, 0.167293),
                c64::new(0.064855, 0.841444),
                c64::new(0.862007, 0.736700),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.465233, 0.974480),
                c64::new(0.496830, 0.202256),
                c64::new(0.444633, 0.227591),
                c64::new(0.872616, 0.765372),
                c64::new(0.029040, 0.596321),
                c64::new(0.208196, 0.153195),
                c64::new(0.966768, 0.664515),
                c64::new(0.270849, 0.965266),
                c64::new(0.657596, 0.651827),
                c64::new(0.417418, 0.040728),
                c64::new(0.911620, 0.922430),
                c64::new(0.497233, 0.783682),
                c64::new(0.155848, 0.253251),
                c64::new(0.882663, 0.585815),
                c64::new(0.546980, 0.888237),
                c64::new(0.911277, 0.180439),
                c64::new(0.722533, 0.429635),
                c64::new(0.535996, 0.922361),
                c64::new(0.858176, 0.564687),
                c64::new(0.036813, 0.047624),
                c64::new(0.031473, 0.141138),
                c64::new(0.553480, 0.077788),
                c64::new(0.745701, 0.221420),
                c64::new(0.419971, 0.219457),
                c64::new(0.297862, 0.798811),
                c64::new(0.528767, 0.539170),
                c64::new(0.845049, 0.851757),
                c64::new(0.597357, 0.812396),
                c64::new(0.257300, 0.769656),
                c64::new(0.997910, 0.161835),
                c64::new(0.928603, 0.151185),
                c64::new(0.410534, 0.170216),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.263740, 0.070841),
                c64::new(0.085195, 0.953737),
                c64::new(0.935889, 0.492878),
                c64::new(0.325004, 0.323936),
                c64::new(0.673089, 0.950248),
                c64::new(0.241384, 0.187216),
                c64::new(0.380081, 0.918274),
                c64::new(0.113941, 0.375945),
                c64::new(0.127950, 0.031666),
                c64::new(0.236561, 0.511949),
                c64::new(0.858776, 0.686316),
                c64::new(0.407311, 0.411411),
                c64::new(0.862216, 0.678282),
                c64::new(0.071830, 0.117966),
                c64::new(0.908264, 0.931949),
                c64::new(0.451725, 0.418887),
                c64::new(0.474883, 0.286795),
                c64::new(0.419337, 0.317515),
                c64::new(0.833829, 0.188620),
                c64::new(0.290675, 0.763958),
                c64::new(0.916789, 0.929790),
                c64::new(0.918379, 0.966845),
                c64::new(0.604291, 0.183526),
                c64::new(0.660053, 0.360011),
                c64::new(0.467357, 0.581639),
                c64::new(0.822700, 0.719627),
                c64::new(0.402778, 0.989676),
                c64::new(0.831058, 0.695419),
                c64::new(0.412800, 0.110401),
                c64::new(0.711937, 0.124685),
                c64::new(0.731400, 0.651739),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.201815, 0.339267),
                c64::new(0.478435, 0.266306),
                c64::new(0.603137, 0.031932),
                c64::new(0.035291, 0.886259),
                c64::new(0.909158, 0.856333),
                c64::new(0.578671, 0.536461),
                c64::new(0.240898, 0.234194),
                c64::new(0.277868, 0.206047),
                c64::new(0.937686, 0.079509),
                c64::new(0.454081, 0.556229),
                c64::new(0.326920, 0.457140),
                c64::new(0.990671, 0.229664),
                c64::new(0.564838, 0.185324),
                c64::new(0.692371, 0.597092),
                c64::new(0.083861, 0.742127),
                c64::new(0.702010, 0.440175),
                c64::new(0.526973, 0.242738),
                c64::new(0.563775, 0.475515),
                c64::new(0.719758, 0.068025),
                c64::new(0.167863, 0.362157),
                c64::new(0.388790, 0.946578),
                c64::new(0.565807, 0.230416),
                c64::new(0.144110, 0.755131),
                c64::new(0.829153, 0.275558),
                c64::new(0.664987, 0.108553),
                c64::new(0.481917, 0.162064),
                c64::new(0.765184, 0.050974),
                c64::new(0.724045, 0.763459),
                c64::new(0.084706, 0.313820),
                c64::new(0.478119, 0.482271),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.337319, 0.468942),
                c64::new(0.425542, 0.357882),
                c64::new(0.579661, 0.886450),
                c64::new(0.994428, 0.585313),
                c64::new(0.247265, 0.136053),
                c64::new(0.028711, 0.626514),
                c64::new(0.035833, 0.624124),
                c64::new(0.135771, 0.052620),
                c64::new(0.453788, 0.946800),
                c64::new(0.925700, 0.031319),
                c64::new(0.973821, 0.517315),
                c64::new(0.861117, 0.164880),
                c64::new(0.654902, 0.177966),
                c64::new(0.313864, 0.381645),
                c64::new(0.836102, 0.233840),
                c64::new(0.192123, 0.283378),
                c64::new(0.631308, 0.823393),
                c64::new(0.179937, 0.815756),
                c64::new(0.169099, 0.676691),
                c64::new(0.517739, 0.527136),
                c64::new(0.268164, 0.920537),
                c64::new(0.743914, 0.010564),
                c64::new(0.004083, 0.109453),
                c64::new(0.936432, 0.207108),
                c64::new(0.710014, 0.983777),
                c64::new(0.868031, 0.151155),
                c64::new(0.208938, 0.154368),
                c64::new(0.351507, 0.910477),
                c64::new(0.310524, 0.349058),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.089359, 0.592571),
                c64::new(0.025927, 0.993073),
                c64::new(0.266887, 0.011920),
                c64::new(0.311563, 0.746637),
                c64::new(0.285068, 0.913357),
                c64::new(0.312499, 0.722821),
                c64::new(0.717670, 0.149793),
                c64::new(0.465951, 0.844516),
                c64::new(0.939096, 0.976909),
                c64::new(0.239914, 0.766679),
                c64::new(0.664594, 0.398251),
                c64::new(0.083727, 0.254150),
                c64::new(0.143020, 0.115627),
                c64::new(0.696803, 0.086359),
                c64::new(0.947894, 0.981045),
                c64::new(0.879589, 0.719811),
                c64::new(0.038965, 0.666100),
                c64::new(0.153779, 0.348154),
                c64::new(0.358095, 0.464243),
                c64::new(0.403023, 0.060618),
                c64::new(0.312094, 0.472284),
                c64::new(0.685940, 0.427614),
                c64::new(0.859723, 0.232919),
                c64::new(0.731689, 0.421356),
                c64::new(0.164438, 0.873587),
                c64::new(0.536389, 0.688468),
                c64::new(0.626736, 0.791729),
                c64::new(0.435494, 0.213942),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.537652, 0.589619),
                c64::new(0.946855, 0.048099),
                c64::new(0.580112, 0.166462),
                c64::new(0.324205, 0.280354),
                c64::new(0.475494, 0.465311),
                c64::new(0.548022, 0.889453),
                c64::new(0.033595, 0.663259),
                c64::new(0.136410, 0.589449),
                c64::new(0.714292, 0.650982),
                c64::new(0.163666, 0.817209),
                c64::new(0.861672, 0.232646),
                c64::new(0.285274, 0.941984),
                c64::new(0.851216, 0.255581),
                c64::new(0.893365, 0.689433),
                c64::new(0.266747, 0.828572),
                c64::new(0.655859, 0.429806),
                c64::new(0.040382, 0.524036),
                c64::new(0.370495, 0.301692),
                c64::new(0.934222, 0.267697),
                c64::new(0.377701, 0.314634),
                c64::new(0.647210, 0.642849),
                c64::new(0.502407, 0.527397),
                c64::new(0.971237, 0.574455),
                c64::new(0.194412, 0.939835),
                c64::new(0.428054, 0.878591),
                c64::new(0.338179, 0.673912),
                c64::new(0.039063, 0.198542),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.221777, 0.400528),
                c64::new(0.837110, 0.936972),
                c64::new(0.498637, 0.234044),
                c64::new(0.170315, 0.993655),
                c64::new(0.990653, 0.999184),
                c64::new(0.059078, 0.674922),
                c64::new(0.943839, 0.841376),
                c64::new(0.463733, 0.536914),
                c64::new(0.822575, 0.587461),
                c64::new(0.088258, 0.756803),
                c64::new(0.421873, 0.634243),
                c64::new(0.377358, 0.203371),
                c64::new(0.752734, 0.602778),
                c64::new(0.499969, 0.712709),
                c64::new(0.837253, 0.231370),
                c64::new(0.116668, 0.310818),
                c64::new(0.173397, 0.374433),
                c64::new(0.060857, 0.977007),
                c64::new(0.819561, 0.641144),
                c64::new(0.822887, 0.030252),
                c64::new(0.838394, 0.649523),
                c64::new(0.916121, 0.489410),
                c64::new(0.620829, 0.545052),
                c64::new(0.751841, 0.947253),
                c64::new(0.807363, 0.828453),
                c64::new(0.450859, 0.154589),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.699100, 0.951363),
                c64::new(0.563549, 0.047178),
                c64::new(0.759856, 0.504205),
                c64::new(0.950519, 0.603581),
                c64::new(0.720274, 0.838028),
                c64::new(0.039368, 0.103535),
                c64::new(0.954781, 0.972311),
                c64::new(0.385626, 0.187801),
                c64::new(0.206112, 0.539507),
                c64::new(0.115542, 0.211145),
                c64::new(0.386164, 0.517787),
                c64::new(0.580916, 0.060254),
                c64::new(0.184256, 0.719706),
                c64::new(0.540605, 0.878884),
                c64::new(0.405216, 0.081987),
                c64::new(0.602953, 0.948485),
                c64::new(0.114403, 0.903891),
                c64::new(0.018613, 0.258778),
                c64::new(0.124654, 0.055391),
                c64::new(0.510379, 0.779819),
                c64::new(0.535666, 0.465172),
                c64::new(0.657444, 0.325994),
                c64::new(0.629904, 0.934739),
                c64::new(0.985249, 0.928895),
                c64::new(0.190160, 0.401857),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.604496, 0.439327),
                c64::new(0.914930, 0.024786),
                c64::new(0.276434, 0.677078),
                c64::new(0.986048, 0.643022),
                c64::new(0.571138, 0.208059),
                c64::new(0.295947, 0.313493),
                c64::new(0.441540, 0.176906),
                c64::new(0.707673, 0.677433),
                c64::new(0.375721, 0.861414),
                c64::new(0.844488, 0.139881),
                c64::new(0.756368, 0.717143),
                c64::new(0.948742, 0.985242),
                c64::new(0.502844, 0.515428),
                c64::new(0.718708, 0.372507),
                c64::new(0.010345, 0.423695),
                c64::new(0.356272, 0.181515),
                c64::new(0.494079, 0.209055),
                c64::new(0.658812, 0.253919),
                c64::new(0.687930, 0.599062),
                c64::new(0.638538, 0.420199),
                c64::new(0.746242, 0.007273),
                c64::new(0.148347, 0.856613),
                c64::new(0.257613, 0.961503),
                c64::new(0.528096, 0.539232),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.070610, 0.603847),
                c64::new(0.002458, 0.280009),
                c64::new(0.119855, 0.567809),
                c64::new(0.785196, 0.108852),
                c64::new(0.164837, 0.592943),
                c64::new(0.326320, 0.748832),
                c64::new(0.099473, 0.163483),
                c64::new(0.855275, 0.184303),
                c64::new(0.463606, 0.582601),
                c64::new(0.814641, 0.027090),
                c64::new(0.007846, 0.811801),
                c64::new(0.479140, 0.018916),
                c64::new(0.461830, 0.068934),
                c64::new(0.848401, 0.995212),
                c64::new(0.129596, 0.807371),
                c64::new(0.690149, 0.497024),
                c64::new(0.412093, 0.620214),
                c64::new(0.780180, 0.749085),
                c64::new(0.821760, 0.123452),
                c64::new(0.766459, 0.213236),
                c64::new(0.588871, 0.233157),
                c64::new(0.366032, 0.772210),
                c64::new(0.528094, 0.468377),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.998811, 0.151823),
                c64::new(0.153496, 0.173089),
                c64::new(0.777784, 0.561901),
                c64::new(0.196152, 0.291610),
                c64::new(0.815561, 0.186349),
                c64::new(0.362361, 0.617128),
                c64::new(0.611270, 0.096765),
                c64::new(0.106938, 0.385818),
                c64::new(0.199039, 0.383701),
                c64::new(0.148565, 0.310793),
                c64::new(0.951014, 0.944885),
                c64::new(0.897417, 0.373009),
                c64::new(0.198774, 0.744056),
                c64::new(0.544202, 0.295716),
                c64::new(0.743012, 0.458318),
                c64::new(0.544675, 0.605523),
                c64::new(0.907680, 0.298343),
                c64::new(0.322987, 0.639377),
                c64::new(0.910781, 0.274136),
                c64::new(0.185915, 0.744621),
                c64::new(0.838579, 0.264727),
                c64::new(0.731056, 0.417156),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.530267, 0.197380),
                c64::new(0.226052, 0.296029),
                c64::new(0.284460, 0.375842),
                c64::new(0.367103, 0.924426),
                c64::new(0.822326, 0.700994),
                c64::new(0.108366, 0.549252),
                c64::new(0.447332, 0.034698),
                c64::new(0.700999, 0.579940),
                c64::new(0.891061, 0.931660),
                c64::new(0.290266, 0.383912),
                c64::new(0.577130, 0.740764),
                c64::new(0.499824, 0.060088),
                c64::new(0.495415, 0.891506),
                c64::new(0.451586, 0.209770),
                c64::new(0.023542, 0.315256),
                c64::new(0.833144, 0.394419),
                c64::new(0.764398, 0.649331),
                c64::new(0.224859, 0.445504),
                c64::new(0.571073, 0.019700),
                c64::new(0.310973, 0.096345),
                c64::new(0.268852, 0.996233),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.644001, 0.182244),
                c64::new(0.647874, 0.832067),
                c64::new(0.569676, 0.516088),
                c64::new(0.389917, 0.694640),
                c64::new(0.973880, 0.100981),
                c64::new(0.121832, 0.899526),
                c64::new(0.981247, 0.591717),
                c64::new(0.881897, 0.798091),
                c64::new(0.639378, 0.327739),
                c64::new(0.600453, 0.097110),
                c64::new(0.646777, 0.252624),
                c64::new(0.123707, 0.116175),
                c64::new(0.146758, 0.177720),
                c64::new(0.857192, 0.146874),
                c64::new(0.097650, 0.161668),
                c64::new(0.448814, 0.135467),
                c64::new(0.393067, 0.884283),
                c64::new(0.914110, 0.846350),
                c64::new(0.759353, 0.071437),
                c64::new(0.996620, 0.055950),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.728995, 0.845020),
                c64::new(0.990701, 0.252026),
                c64::new(0.824465, 0.361300),
                c64::new(0.763808, 0.469939),
                c64::new(0.894858, 0.255854),
                c64::new(0.653015, 0.661021),
                c64::new(0.056561, 0.598045),
                c64::new(0.175273, 0.986933),
                c64::new(0.192212, 0.578786),
                c64::new(0.591390, 0.136326),
                c64::new(0.162523, 0.633553),
                c64::new(0.657352, 0.979075),
                c64::new(0.701467, 0.306737),
                c64::new(0.372409, 0.698891),
                c64::new(0.937645, 0.634370),
                c64::new(0.919380, 0.416858),
                c64::new(0.372741, 0.475970),
                c64::new(0.260335, 0.404998),
                c64::new(0.180611, 0.112005),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.814781, 0.486892),
                c64::new(0.070927, 0.526587),
                c64::new(0.520279, 0.069979),
                c64::new(0.712456, 0.411117),
                c64::new(0.796706, 0.210675),
                c64::new(0.137444, 0.320210),
                c64::new(0.610915, 0.173164),
                c64::new(0.939402, 0.725923),
                c64::new(0.253125, 0.909806),
                c64::new(0.652449, 0.191755),
                c64::new(0.131386, 0.765295),
                c64::new(0.707202, 0.876745),
                c64::new(0.707284, 0.233998),
                c64::new(0.639369, 0.749559),
                c64::new(0.840714, 0.309042),
                c64::new(0.604209, 0.017491),
                c64::new(0.325224, 0.878176),
                c64::new(0.261566, 0.500890),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.919724, 0.062919),
                c64::new(0.176956, 0.340665),
                c64::new(0.923453, 0.173053),
                c64::new(0.212610, 0.959185),
                c64::new(0.995610, 0.609521),
                c64::new(0.713215, 0.873191),
                c64::new(0.566007, 0.690059),
                c64::new(0.010516, 0.637544),
                c64::new(0.659578, 0.089076),
                c64::new(0.715069, 0.630759),
                c64::new(0.543477, 0.085693),
                c64::new(0.673411, 0.034171),
                c64::new(0.604034, 0.193489),
                c64::new(0.283909, 0.240958),
                c64::new(0.684924, 0.883630),
                c64::new(0.030740, 0.530821),
                c64::new(0.864281, 0.164948),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.057777, 0.371621),
                c64::new(0.276834, 0.604932),
                c64::new(0.321289, 0.795083),
                c64::new(0.863373, 0.691706),
                c64::new(0.819117, 0.439222),
                c64::new(0.607592, 0.118105),
                c64::new(0.019902, 0.698416),
                c64::new(0.946729, 0.362846),
                c64::new(0.856545, 0.546844),
                c64::new(0.369417, 0.790704),
                c64::new(0.443440, 0.589923),
                c64::new(0.662327, 0.754029),
                c64::new(0.619686, 0.457036),
                c64::new(0.288752, 0.565755),
                c64::new(0.827299, 0.904174),
                c64::new(0.869594, 0.402969),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.139517, 0.676169),
                c64::new(0.504913, 0.135384),
                c64::new(0.298331, 0.180278),
                c64::new(0.483749, 0.285515),
                c64::new(0.438249, 0.339124),
                c64::new(0.342086, 0.835658),
                c64::new(0.975726, 0.808771),
                c64::new(0.910044, 0.239789),
                c64::new(0.242144, 0.853117),
                c64::new(0.891421, 0.339500),
                c64::new(0.616380, 0.450252),
                c64::new(0.210675, 0.446397),
                c64::new(0.074969, 0.837579),
                c64::new(0.217164, 0.400573),
                c64::new(0.777591, 0.348604),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.495632, 0.077062),
                c64::new(0.974407, 0.459050),
                c64::new(0.517030, 0.835661),
                c64::new(0.596032, 0.961335),
                c64::new(0.675459, 0.847674),
                c64::new(0.749591, 0.027275),
                c64::new(0.129578, 0.819247),
                c64::new(0.794775, 0.701672),
                c64::new(0.021770, 0.332271),
                c64::new(0.100434, 0.809707),
                c64::new(0.953545, 0.514408),
                c64::new(0.862212, 0.847182),
                c64::new(0.216898, 0.855612),
                c64::new(0.891567, 0.243043),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.441988, 0.999107),
                c64::new(0.832639, 0.884452),
                c64::new(0.320947, 0.780186),
                c64::new(0.810076, 0.236526),
                c64::new(0.537672, 0.740795),
                c64::new(0.680193, 0.066289),
                c64::new(0.874927, 0.227058),
                c64::new(0.694998, 0.999270),
                c64::new(0.884695, 0.089978),
                c64::new(0.459663, 0.511654),
                c64::new(0.613541, 0.367113),
                c64::new(0.729744, 0.552765),
                c64::new(0.566712, 0.090803),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.202611, 0.195663),
                c64::new(0.711010, 0.367417),
                c64::new(0.922125, 0.424018),
                c64::new(0.525708, 0.552767),
                c64::new(0.401893, 0.721927),
                c64::new(0.809928, 0.540102),
                c64::new(0.112140, 0.010517),
                c64::new(0.623596, 0.889269),
                c64::new(0.137125, 0.282061),
                c64::new(0.268159, 0.548416),
                c64::new(0.829648, 0.013244),
                c64::new(0.277403, 0.997901),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.059393, 0.220650),
                c64::new(0.267126, 0.022048),
                c64::new(0.627902, 0.176709),
                c64::new(0.074405, 0.525112),
                c64::new(0.420115, 0.146938),
                c64::new(0.393349, 0.066754),
                c64::new(0.463582, 0.148698),
                c64::new(0.283088, 0.992110),
                c64::new(0.238601, 0.012526),
                c64::new(0.065825, 0.877910),
                c64::new(0.473864, 0.323552),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.553993, 0.740662),
                c64::new(0.403079, 0.881710),
                c64::new(0.004518, 0.735658),
                c64::new(0.788280, 0.594436),
                c64::new(0.315087, 0.244893),
                c64::new(0.928472, 0.704937),
                c64::new(0.661671, 0.093359),
                c64::new(0.990800, 0.340999),
                c64::new(0.379550, 0.092898),
                c64::new(0.200277, 0.267814),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.070966, 0.101361),
                c64::new(0.025529, 0.604415),
                c64::new(0.419968, 0.636520),
                c64::new(0.186913, 0.557707),
                c64::new(0.825992, 0.520800),
                c64::new(0.583048, 0.054066),
                c64::new(0.496266, 0.328172),
                c64::new(0.173648, 0.966839),
                c64::new(0.558996, 0.879698),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.972306, 0.356104),
                c64::new(0.036298, 0.759947),
                c64::new(0.922970, 0.971384),
                c64::new(0.731822, 0.457290),
                c64::new(0.880327, 0.811134),
                c64::new(0.451752, 0.046667),
                c64::new(0.561549, 0.995209),
                c64::new(0.069626, 0.012071),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.402153, 0.778096),
                c64::new(0.702851, 0.824055),
                c64::new(0.446759, 0.551222),
                c64::new(0.001990, 0.298375),
                c64::new(0.688905, 0.630581),
                c64::new(0.255168, 0.560828),
                c64::new(0.939632, 0.289745),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.731109, 0.623374),
                c64::new(0.508514, 0.160079),
                c64::new(0.583604, 0.710679),
                c64::new(0.201480, 0.233400),
                c64::new(0.868764, 0.920140),
                c64::new(0.295113, 0.366947),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.606475, 0.104967),
                c64::new(0.264909, 0.750597),
                c64::new(0.591990, 0.000629),
                c64::new(0.004423, 0.995822),
                c64::new(0.890303, 0.247400),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.554399, 0.010273),
                c64::new(0.543164, 0.208611),
                c64::new(0.977420, 0.224906),
                c64::new(0.467072, 0.164433),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.282165, 0.831008),
                c64::new(0.050223, 0.674817),
                c64::new(0.944237, 0.737266),
            ],
            [
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.000000, 0.000000),
                c64::new(0.329299, 0.333973),
                c64::new(0.626537, 0.651087),
            ],
        ];
        let h = Mat::from_fn(n, n, |i, j| h[i][j]);

        let mut q = Mat::from_fn(n, n, |i, j| {
            if i == j {
                c64::faer_one()
            } else {
                c64::faer_zero()
            }
        });
        let mut w = Col::zeros(n);
        let mut t = h.clone();
        let params = EvdParams {
            recommended_shift_count: None,
            recommended_deflation_window: None,
            blocking_threshold: Some(15),
            nibble_threshold: Some(14),
        };
        let (_, n_aed, n_sweep) = multishift_qr(
            true,
            t.as_mut(),
            Some(q.as_mut()),
            w.as_mut(),
            0,
            n,
            f64::EPSILON,
            f64::MIN_POSITIVE,
            Parallelism::None,
            make_stack!(multishift_qr_req::<c64>(
                n,
                n,
                true,
                true,
                Parallelism::None,
                params,
            )),
            params,
        );
        // asserts to ensure that we don't mess up the shift computation and slow down convergence
        assert!(n_aed <= 50);
        assert!(n_sweep <= 25);

        for j in 0..n {
            for i in j + 1..n {
                t.write(i, j, c64::faer_zero());
            }
        }

        let h_reconstructed = &q * &t * q.adjoint();

        for i in 0..n {
            for j in 0..n {
                assert_approx_eq!(h_reconstructed.read(i, j), h.read(i, j));
            }
        }
    }

    #[test]
    fn test_gh_84_cplx_20x20() {
        let a = mat![
            [
                0.0, -1.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
                0.0, 0.0, 0.0, 0.0,
            ],
            [
                -1.0,
                0.0,
                1.414213562373095,
                0.0,
                4.2913523532056946e-17,
                0.0,
                -1.044653549357173e-17,
                0.0,
                -3.1114138757927777e-18,
                0.0,
                -1.7041915655062705e-17,
                0.0,
                -1.5407439555097887e-33,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
            ],
            [
                0.0,
                1.4142135623730951,
                0.0,
                1.8708286933869704,
                0.0,
                5.584290580070408e-17,
                0.0,
                4.247123568135012e-17,
                0.0,
                3.383773620463798e-17,
                0.0,
                -7.996931848726948e-17,
                0.0,
                2.137243672031505e-16,
                -2.1697413347872967e-16,
                2.1669070851885488e-16,
                8.793731080691344e-17,
                -6.366867306877532e-17,
                -2.166907085188549e-16,
                -1.5209416587779695e-16,
            ],
            [
                0.0,
                0.0,
                1.8708286933869707,
                0.0,
                0.8864052604279185,
                0.0,
                -5.4482028175548746e-18,
                0.0,
                -1.6227019814468362e-18,
                0.0,
                -8.887904793467559e-18,
                0.0,
                -1.5407439555097887e-33,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
            ],
            [
                0.0,
                0.0,
                0.0,
                0.8864052604279183,
                0.0,
                -1.780121110720045,
                0.0,
                2.4905953568839346e-17,
                0.0,
                1.9539297406873977e-17,
                0.0,
                -4.520412950660578e-17,
                0.0,
                7.136040597386633e-17,
                -3.3287408030765184e-17,
                3.9021417760079386e-17,
                2.050611271670174e-18,
                8.594289573868776e-17,
                -3.9021417760079386e-17,
                -2.225425376396652e-17,
            ],
            [
                0.0,
                0.0,
                0.0,
                0.0,
                -1.7801211107200454,
                0.0,
                -0.27971669721424774,
                0.0,
                1.3907420351599062e-17,
                0.0,
                7.617407843277181e-17,
                0.0,
                6.162975822039155e-33,
                2.413689799097554e-35,
                0.0,
                7.173100169147405e-36,
                2.1640125834190973e-35,
                1.746648713648135e-36,
                -7.173100169147405e-36,
                -2.3028287065724625e-36,
            ],
            [
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                -0.27971669721424774,
                0.0,
                2.08179724791363,
                0.0,
                3.128025644367276e-17,
                0.0,
                1.4857837468143015e-17,
                3.0814879110195774e-33,
                -4.353679358901167e-17,
                1.575970043415495e-17,
                3.371331087010223e-17,
                -3.158189568582724e-17,
                3.363028106986343e-17,
                -3.371331087010223e-17,
                1.0436797917561959e-16,
            ],
            [
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                2.08179724791363,
                0.0,
                -0.591353947688746,
                0.0,
                -1.6942472264780397e-16,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
            ],
            [
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                -0.5913539476887459,
                0.0,
                1.382991720854028,
                0.0,
                2.9339388813743063e-16,
                0.0,
                -2.5386000785908e-17,
                4.58003519909708e-17,
                -3.248053804275205e-17,
                -1.7259032483343233e-17,
                -6.634605605063898e-17,
                3.248053804275205e-17,
                6.644449342304512e-17,
            ],
            [
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                1.3829917208540286,
                0.0,
                0.5110217026036908,
                0.0,
                4.2381199337456166e-17,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
            ],
            [
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.5110217026036906,
                0.0,
                -1.2687886196697642,
                0.0,
                -1.9866208356455852e-17,
                6.074934010535109e-17,
                -7.39350135280486e-17,
                1.5423322066319723e-17,
                -9.775009908321562e-17,
                7.39350135280486e-17,
                2.485877967834109e-16,
            ],
            [
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                -1.2687886196697646,
                0.0,
                -1.1161328626787337e-16,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
            ],
            [
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                -4.319252649148509e-17,
                0.0,
                2.095595710655279e-17,
                1.2259068349524595e-18,
                1.2354010708239072e-17,
                -6.000254847480346e-18,
                -7.043660603033051e-18,
                -1.2354010708239072e-17,
                2.8117846829535463e-17,
            ],
            [
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                -3.3963911681598194e-17,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
            ],
            [
                0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
                0.0, 0.0, 0.0, 0.0,
            ],
            [
                0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
                0.0, 0.0, 0.0, 0.0,
            ],
            [
                0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
                0.0, 0.0, 0.0, 0.0,
            ],
            [
                0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
                0.0, 0.0, 0.0, 0.0,
            ],
            [
                0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
                0.0, 0.0, 0.0, 0.0,
            ],
            [
                0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
                0.0, 0.0, 0.0, 0.0,
            ],
        ];
        let mut a = zipped!(a.as_ref()).map(|unzipped!(x)| c64::new(x.read(), 0.0));

        let mut w = Col::zeros(20);
        lahqr(
            false,
            a.as_mut(),
            None,
            w.as_mut(),
            0,
            20,
            f64::EPSILON,
            f64::MIN_POSITIVE,
        );
    }
}
