% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/idf.R
\docType{class}
\name{Idf}
\alias{Idf}
\title{Read, Modify, and Run an EnergyPlus Model}
\description{
eplusr provides parsing EnergyPlus Input Data File (IDF) files and strings
in a hierarchical structure, which was extremely inspired by
\href{https://openstudio-sdk-documentation.s3.amazonaws.com/cpp/OpenStudio-2.5.1-doc/utilities/html/idf_page.html}{OpenStudio utilities library},
but with total different data structure under the hook.
}
\section{Overview}{


eplusr uses \code{Idf} class to present the whole IDF file and use \link{IdfObject}
to present a single object in IDF. Both \code{Idf} and \link{IdfObject} contain member
functions for helping modify the data in IDF so it complies with the
underlying IDD (EnergyPlus Input Data Dictionary).

Under the hook, eplusr uses a SQL-like structure to store both IDF and IDD
data in different \link[data.table:data.table]{data.table::data.table}s. So to modify an EnergyPlus model
in eplusr is equal to change the data in those IDF tables accordingly, in the
context of specific IDD data. This means that a corresponding \link{Idd} object is
needed whenever creating an \code{Idf} object. eplusr provides several
\link[=use_idd]{helpers} to easily download IDD files and create \link{Idd} objects.

All IDF reading process starts with function \code{\link[=read_idf]{read_idf()}} which returns an
\code{Idf} object. \code{Idf} class provides lots of methods to programmatically query
and modify EnergyPlus models. Below is the detailed documentation on each
method.
}

\section{Usage}{


\preformatted{
model <- read_idf(path)
model$version()
model$path()
model$group_name(all = FALSE, sorted = TRUE)
model$class_name(all = FALSE, sorted = TRUE, by_group = FALSE)
model$is_valid_group(group, all = FALSE)
model$is_valid_class(class, all = FALSE)
model$definition(class)
model$object_id(class = NULL, simplify = FALSE)
model$object_name(class = NULL, simplify = FALSE)
model$is_valid_id(id)
model$is_valid_name(name)
model$object_num(class = NULL)
model$object(which)
model$object_unique(class)
model$objects(which)
model$objects_in_class(class)
model$objects_in_group(group)
model$object_relation(which, direction = c("all", "ref_to", "ref_by", "node"), recursive = FALSE, depth = 1L)
model$objects_in_relation(which, direction = c("ref_to", "ref_by", "node"), class = NULL, recursive = FALSE, depth = 1L)
model$search_object(pattern, class = NULL, ignore.case = FALSE, perl = FALSE, fixed = FALSE, useBytes = FALSE)
model$ClassName
model[[ClassName]]
model$dup(...)
model$add(..., .default = TRUE, .all = FALSE)
model$set(..., .default = TRUE)
model$del(..., .ref_by = FALSE, .ref_to = FALSE, .recursive = FALSE, .force = FALSE)
model$insert(..., .unique = TRUE)
model$load(..., .unique = TRUE, .default = TRUE)
model$update(..., .default = TRUE)
model$rename(...)
model$paste(in_ip = FALSE, ver = NULL, unique = TRUE)
model$search_value(pattern, class = NULL, ignore.case = FALSE, perl = FALSE, fixed = FALSE, useBytes = FALSE)
model$replace_value(pattern, class = NULL, replacement, ignore.case = FALSE, perl = FALSE, fixed = FALSE, useBytes = FALSE)
model$is_unsaved()
model$validate(level = eplusr_option("validate_level"))
model$is_valid(level = eplusr_option("validate_level"))
model$to_table(which = NULL, class = NULL, string_value = TRUE, unit = FALSE, wide = FALSE)
model$to_string(which = NULL, class = NULL, comment = TRUE, header = TRUE, format = eplusr_option("save_format"), leading = 4L, sep_at = 29L)
model$save(path = NULL, format = eplusr_option("save_format"), overwrite = FALSE, copy_external = TRUE)
model$run(weather = NULL, dir = NULL, wait = TRUE, force = FALSE, copy_external = FALSE)
model$clone(deep = TRUE)
model$print(zoom = c("object", "class", "group", "field"), order = TRUE)
print(model)
}
}

\section{Basic Info}{
\preformatted{model <- read_idf(path)
model$version()
model$path()
model$group_name(all = FALSE, sorted = TRUE)
model$class_name(all = FALSE, sorted = TRUE, by_group = FALSE)
model$is_valid_group(group, all = FALSE)
model$is_valid_class(class, all = FALSE)
}

\code{$version()} returns the version of current model in a
\link[base:numeric_version]{numeric_version} format. This makes it easy to
direction compare versions of different model, e.g. \code{model1$version() > 8.6}
or \code{model1$version() > model2$version()}.

\code{$path()} returns the full path of current model or \code{NULL} if the \code{Idf}
object is created using a character vector and not saved locally.

\code{$group_name()} returns all groups the model contains when \code{all} is \code{FALSE}
or all groups the underlying \link{Idd} object contains when \code{all} is \code{TRUE}.

\code{$class_name()} returns all classes the model contains when \code{all} is \code{FALSE}
or all classes the underlying \link{Idd} object contains when \code{all} is \code{TRUE}.

\code{$is_valid_group()} returns \code{TRUE}s if given group names are valid for
current model (when \code{all} is \code{FALSE}) or current underlying \link{Idd} object
(when \code{all} is \code{TRUE}).

\code{$is_valid_class()} returns \code{TRUE}s if given class names are valid for
current model (when \code{all} is \code{FALSE}) or underlying \link{Idd} object (when \code{all}
is \code{TRUE}).

\strong{Arguments}:
\itemize{
\item \code{path}: Either a path, a connection, or literal data (either a single
string or a raw vector) to an EnergyPlus Input Data File (IDF).
\item \code{all}: If \code{FALSE}, only values in current \code{Idf} object will be returned. If
\code{TRUE}, all values in the underlying \link{Idd} will be returned. For
\code{$is_valid_group()} and \code{$is_valid_class()}, \code{all} equals to \code{TRUE} means
that input group or class names are checked in all existing ones in the
underlying \link{Idd} object. Default: \code{FALSE}.
\item \code{sorted}: Only applicable when \code{all} is \code{FALSE}. If \code{TRUE}, duplications in
returned group or class names are removed, and unique names are further
sorted according to their occurrences in the underlying \link{Idd} object.
Default: \code{TRUE}.
\item \code{by_group}: Only applicable when \code{all} or \code{sorted} is \code{TRUE}. If \code{TRUE}, a
list is returned which separate class names by the group they belong to.
\item \code{group}: A character vector of valid group names.
\item \code{class}: A character vector of valid class names.
}
}

\section{Definition}{
\preformatted{model$definition(class)
}

\code{$definition()} returns an \link{IddObject} of given class. \link{IddObject} contains
all data used for parsing and creating an \link{IdfObject}. For details, please
see \link{IddObject} class.

\strong{Arguments}:
\itemize{
\item \code{class}: A \strong{single} string of valid class name in current IDD.
}
}

\section{Object Info}{
\preformatted{model$object_id(class = NULL, simplify = FALSE)
model$is_valid_id(id)
model$object_name(class = NULL, simplify = FALSE)
model$is_valid_name(name)
model$object_num(class = NULL)
}

\code{$object_id()} returns an integer vector (when \code{simplify} is \code{TRUE}) or a
named list (when \code{simplify} is \code{FALSE}) of all object IDs in specified
classes. The returned list is named using specified class names.

\code{$object_name()} returns a character vector (when \code{simplify} is \code{TRUE}) or a
named list (when \code{simplify} is \code{FALSE}) of all object names in specified
classes. The returned list is named using specified class names.

\code{$is_valid_id()} and \code{$is_valid_name()} returns a logical vector whether the
given integer vector or character vector contains valid object IDs or names
respectively. Note that for \code{$is_valid_name()}, object name matching is
\strong{case-insensitive}.

\code{$object_num()} returns an integer vector of object numbers in specified
classes.

\strong{Arguments}:
\itemize{
\item \code{id}: An integer vector to check.
\item \code{name}: A character vector to check.
\item \code{class}: A character vector that contains valid class names. If \code{NULL}, all
classes in current \code{Idf} object are used. Default: \code{NULL}.
\item \code{simplify}: If \code{TRUE}, an integer vector (for \code{$object_id()}) or a
character vector (for \code{$object_name}()) is returned. If \code{FALSE}, a list
with each element being the data per class is returned. If \code{class} is
\code{NULL}, the order of classes returned is the same as that in the underlying
\link{Idd} object. Default: \code{FALSE}.
}
}

\section{Object Relation}{
\preformatted{model$object_relation(which, direction = c("all", "ref_to", "ref_by", "node"), recursive = TRUE, depth = 1L)
}

Many fields in \link{Idd} can be referred by others. For example, the \code{Outside Layer} and other fields in \code{Construction} class refer to the \code{Name} field
in \code{Material} class and other material related classes. Here it means that
the \code{Outside Layer} field \strong{refers to} the \code{Name} field and the \code{Name} field
is \strong{referred by} the \code{Outside Layer}. In EnergyPlus, there is also a
special type of field called \code{Node}, which together with \code{Branch} and
\code{BranchList} define the topography of the HVAC connections. A outlet node of
a component can be referred by another component as its inlet node, but can
also exists independently, such as zone air node.

\code{$object_relation()} provides a simple interface to get this kind of
relation. It takes a single object ID or name and also a relation direction,
and returns an \code{IdfRelation} object which contains data presenting such
relation above. For instance, if \code{model$object_relation("WALL-1", "ref_to")}
gives results below:\preformatted{-- Refer to Others ------------------------
  Class: <Construction>
  \- Object [ID:2] <WALL-1>
     \- 2: "WD01";        !- Outside Layer
        v~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        \- Class: <Material>
           \- Object [ID:1] <WD01>
              \- 1: "WD01";        !- Name
}

This means that the value \code{"WD01"} of \code{Outside Layer} in a construction named
\code{WALL-1} refers to a material named \code{WD01}. All those objects can be further
easily extracted using \code{$objects_in_relation()} method described below.

\strong{Arguments}:
\itemize{
\item \code{which}: Either a single integer of object ID or a string of object name.
\item \code{direciton}: The relation direction to extract. Should be either \code{"all"},
\code{"ref_to"} or "ref_by".
\item \code{recursive}: If \code{TRUE}, the relation is searched recursively. A simple
example of recursive reference: one material named \code{mat} is referred by a
construction named \code{const}, and \code{const} is also referred by a surface named
\code{surf}.
\item \code{depth}: Only applicable when \code{recursive} is \code{TRUE}. This is a depth to
when searching value relations recursively. If \code{NULL}, all recursive
relations are returned. Default: \code{1}.
}
}

\section{Object Query}{


\preformatted{
model$object(which)
model$objects(which)
model$object_unique(class)
model$objects_in_class(class)
model$objects_in_group(group)
model$objects_in_relation(which, direction = c("ref_to", "ref_by", "node"), class = NULL, recursive = FALSE, depth = 1L)
model$search_object(pattern, class = NULL, ignore.case = FALSE, perl = FALSE, fixed = FALSE, useBytes = FALSE)
model$ClassName
model[[ClassName]]
}

\code{$object()} returns an \link{IdfObject} specified by an object ID or name. Note
that unlike object ID, which is always unique across the whole \code{Idf} object,
sometimes different objects can have the same name. If the name given matches
multiple objects, an error is issued showing what objects are matched by the
same name. This behavior is consistent in all methods that take an object
name(s) as input.

\code{$object_unique()} returns the \link{IdfObject} in unique-object class, e.g.
\code{SimulationControl} class. This makes it easy to directly extract and modify
those unique objects, e.g.
\code{model$object_unique("SimulationContrl")$set(...)}. Note that if there are
multiple objects in that unique-object class, an error is issued. This makes
sure that \code{$object_unique()} always returns a single \link{IdfObject}.

\code{$objects()} returns a named \strong{list} of \link{IdfObject}s specified by object IDs
or names.

\code{$objects_in_class()} returns a named \strong{list} of all \link{IdfObject}s in
specified class.

\code{$objects_in_group()} returns a named \strong{list} of all \link{IdfObject}s in
specified group.

\code{$objects_in_relation()} returns a named \strong{list} of \link{IdfObject}s that have
specified relations with given object. The first element of returned list is
always the \link{IdfObject} of specified object. If that object does not have
specified relation with other objects in specified \code{class}, a list that only
contains that \link{IdfObject} is returned. For instance, assume that \code{const} is a
valid object name in \code{Construction} class,
\code{model$objects_in_relation("const", "ref_by", "BuildingSurface:Detailed")}
will return a named list of an \link{IdfObject} named \code{const} and also all other
\link{IdfObject}s in \code{BuildingSurface:Detailed} that refer to field values in
\code{const}. Similarly, \code{model$objects_in_relation("const", "ref_to", "Material")}
will return a named list of an \link{IdfObject} named \code{const} and also all other
\link{IdfObject}s in \code{Material} class that \code{const} refers to. This makes it easy
to directly extract groups of related objects and then use \code{$insert()} method
described below to insert them. For example, copy a construction named
\code{const} from an \code{Idf} object \code{model1} to another \code{Idf} object \code{model2} is
simply to do \code{model2$insert(model1$objects_in_relation("const", "ref_to"))}.

There are lots of recursive references in a model. For instance, a material
can be referred by a construction, that construction can be referred by a
building surface, and that building surface can be referred by a window on
that surface. These objects related recursively can be extracted by setting
\code{recursive} to \code{TRUE}.

\code{$search_object()} returns a named \strong{list} of \link{IdfObject}s whose names meet
the given pattern in specified classes.

eplusr also provides custom S3 method of \code{$} and \code{[[} to make it more
convenient to get \link{IdfObject}s in specified class. Basically, \code{model$ClassName} and
\code{model[[ClassName]]}, where \code{ClassName} is a single valid class name, is
equivalent to \code{model$objects_in_class(ClassName)} if \code{ClassName} is not an
unique-object class and \code{model$object_unique(ClassName)} if \code{ClassName} is an
unique-object class. For convenience, underscore-style names are allowed, e.g.
\code{BuildingSurface_Detailed} is equivalent to \code{BuildingSurface:Detailed}. For
instance, \code{model$BuildingSurface_Detailed} and also
\code{model[["BuildingSurface:Detailed"]]} will return all \link{IdfObject}s in
\code{BuildingSurface:Detailed} class; \code{model$Building} and also
\code{model[["Building"]]} will return the \link{IdfObject} in \code{Building} class which
is an unique-object class.

\strong{Note}: The returned list from \code{$objects()}, \code{$objects_in_class()} and
other methods is named using the names of returned \link{IdfObject}s in that list.
This will makes it easy to using object name to do further subsetting, e.g.
\code{model$objects_in_class("Material")$mat} will return an \link{IdfObject} named
\code{mat} in \code{Material} class, and \code{model$objects_in_class("Material")[[1]]} will
return the first material in \code{model}. If returned \link{IdfObject}s belongs to a
class that does not have name attribute, such like \code{Version},
\code{SimulationControl} and etc., \code{NA} is assigned as the name.

\link{IdfObject} class provides more detailed methods to modify a single object in
an \code{Idf}. For detailed explanations, please see \link{IdfObject} class.

\strong{Arguments}:
\itemize{
\item \code{which}: A single object ID or name for \code{$object()} and
\code{$objects_in_relation()}; an integer vector of object IDs or a character
vector of object names for \code{$objects()}.
\item \code{class}: A single string of class name for \code{$object_unique()} and
\code{$objects_in_class()}; a character vector of class names for
\code{$objects_in_relation()} and \code{$search_object()}.
\item \code{group}: A single string of group name.
\item \code{pattern}, \code{ignore.case}, \code{perl}, \code{fixed} and \code{useBytes}: All of them are
directly passed to \link[base:grep]{base::grepl}.
\item \code{ClassName}: A single string of class name. For \code{[[}, \code{ClassName} can
be an underscore-style class name, where all characters other than letters
and numbers are replaced by underscores \code{_}.
\item \code{direciton}: The relation direction to extract. Should be either \code{"ref_to"}
or "ref_by".
\item \code{recursive}: If \code{TRUE}, the relation is searched recursively, e.g. one
material named \code{mat} is referred by a construction named \code{const}, and
\code{const} is also referred by a surface named \code{surf}, all \code{mat}, \code{const} and
\code{surf} are returned.
\item \code{depth}: Only applicable when \code{recursive} is \code{TRUE}. This is a depth to
when searching value relations recursively. If \code{NULL}, all recursive
relations are returned. Default: \code{1}.
\item \code{ignore.case}, \code{perl}, \code{fixed} and \code{useBytes}: All are directly passed to
\link[base:grep]{base::grepl}.
}
}

\section{Object Modification}{

\subsection{Duplicate Objects}{\preformatted{model$dup(...)
}

\code{$dup()} takes integer vectors of object IDs and character vectors of object
names, duplicates objects specified, and returns a list of newly created
\link{IdfObject}s. The names of input are used as new names for created
\link{IdfObject}s. If input is not named, new names are the names of duplicated
objects with a suffix \code{"_1"}, \code{"_2"} and etc, depending on how many times
that object has been duplicated. Note an error will be issued if trying to
assign a new name to an object which does not have name attribute.

\strong{Note}:
\itemize{
\item Assign newly added objects with an existing name in current \code{Idf} object is
prohibited if current validation level includes object name conflicting
checking. For details, please see \code{level_checks()}.
}

\strong{Argument}:
\itemize{
\item \code{...}: Integer vectors of object IDs and character vectors of valid
object names. If input has names, they will be used as the names of newly
created objects.
}

\strong{Usage}:
\itemize{
\item Without new names: \code{model$dup(c("name1", "name2"), 6:10)}.
\item With new names: \code{model$dup(c(new_name1 = "name1", new_name2 = "name2"), new_name3 = 6)}.
\item Variable inputs: \code{a <- c("name1", new_name2 = "name2"); b <- 10:20; c <- c(new_name3 = 10); model$dup(a, b, c)}.
}
}

\subsection{Add Objects}{\preformatted{model$add(..., .default = TRUE, .all = FALSE)
}

\code{$add()} takes object definitions in list format, adds corresponding
objects in specified classes, returns a list of newly added \link{IdfObject}s.
Every list should be named with a valid class name. Underscore-style class
name is allowed. Names in list element are treated as field names. Values
without names will be inserted according to their position. There is a
special element named \code{.comment} in each list, which will be used as the
comments of newly added object.

\strong{Note}:
\itemize{
\item Empty objects can be added using an empty list, e.g. \code{model$add(building = list())}. All empty fields will be filled with corresponding default value
if \code{.default} is \code{TRUE}, leaving other fields as blank. However, adding
blank objects may not be successful if required fields are not valued and
current validate level includes missing-required-field checking. For what
kind of validation components to be performed during modifications, please
see \code{\link[=level_checks]{level_checks()}}.
\item Field name matching is \strong{case-insensitive}. For convenience,
underscore-style field names are also allowed, e.g. \code{eNd_MoNtH} is
equivalent to \code{End Month}.
\item There is no need to give all field values if only specific fields are
interested, unless other fields are not required. For example, to define a
new object in \code{RunPeriod} class, the following is enough:
\code{model$add(RunPeriod = list(begin_month = 1, begin_day_of_month = 1, end_month = 1, end_day_of_month = 31), .default = TRUE)}.
\item If not all field names are given, positions of those values without field
names are determined after those values with names. E.g. in
\code{model$add(Construction = list("out_layer", name = "name"))}, \code{"out_layer"}
will be treated as the value of field \code{Outside Layer} in \code{Construction}
class, as value of field \code{Name} has been given as \code{"name"}.
}

\strong{Arguments}:
\itemize{
\item \code{...}: Lists of object definitions. Each list should be named with a valid
class name. There is a special element \code{.comment} in each list, which will
be used as the comments of newly added object.
\item \code{.default}: If \code{TRUE}, default values are used for those blank fields if
possible. If \code{FALSE}, each required field in input object must have one
value. Otherwise, an error will be issued during validation. Default:
\code{TRUE}.
\item \code{.all}: If \code{TRUE}, all fields are added, otherwise only minimum fields are
added. Default: \code{FALSE}.
}

\strong{Usage}:
\itemize{
\item Empty object with default values: \code{model$add(Building = list(), .default = TRUE)}.
\item Empty object with comments: \code{model$add(Building = list(.comment = c("This", "is", "a", "comment")))}.
\item Empty object with all fields: \code{model$add(Building = list(), .all = TRUE)}.
\item New objects: \code{model$add(RunPeriod = list("rp", 1, 1, end_month = 2, 1, "Monday"), list(Construction = list("const", "mat"), Material = list("mat")))}.
\item New objects with comments: \code{model$add(RunPeriod = list("rp", 1, 1, 2, 1, .comment = "comment1"))}.
\item Variable inputs: \code{x <- list(Construction = list("const"), Building = list()); model$add(x)}.
}
}

\subsection{Set Values of Existing Objects}{\preformatted{model$set(..., .default = TRUE)
}

\code{$set()} takes new field value definitions in list format, sets new
values for fields in objects specified, and returns a list of modified
\link{IdfObject}s. Every list in \code{$set()} should be named with a
valid object name. Object ID can also be used but have to be combined with
prevailing two periods \code{..}, e.g. \code{..10} indicates the object with ID \code{10}.
Similar to \code{$add()}, a special element \code{.comment} in each list will be used
as the \strong{new} comments for modified object, overwriting the old ones. Names
in list element are treated as field names.

\strong{Note}:
\itemize{
\item You can delete a field by assigning \code{NULL} to it, e.g. \code{list(fld = NULL)} means to delete the value of field \code{fld}. If \code{.default} is FALSE,
also \code{fld} is not a required field and the index of \code{fld} is larger than
the number minimum fields required for that class, it will be deleted.
Otherwise it will be left as blank. If \code{.default} is \code{TRUE}, that field
will be filled with default value if applicable and left as blank if not.
\item New fields that currently do not exist in that object can also be set. They
will be automatically added on the fly.
\item Field name matching is \strong{case-insensitive}. For convenience,
underscore-style field names are also allowed, e.g. \code{eNd_MoNtH} is
equivalent to \code{End Month}.
\item If not all field names are given, positions of those values without field
names are determined after those values with names. E.g. in
\code{model$set(Construction = list("out_layer", name = "name"))}, \code{"out_layer"}
will be treated as the value of field \code{Outside Layer} in \code{Construction}, as
value of field \code{Name} has been given as \code{"name"}.
}

\strong{Arguments}:
\itemize{
\item \code{...}: Lists of object definitions. Each list should be named with a valid
object name or a valid object ID denoted in style \code{..1}, \code{..2} and etc.
There is a special element \code{.comment} in each list, which will be used as
new comments of modified object, overwriting existing ones.
\item \code{.default}: If \code{TRUE}, default values are used for those blank fields if
possible. Default: \code{TRUE}.
}

\strong{Usage}:
\itemize{
\item Specify object with name: \code{model$set(Object_Name = list(val1, val2, val3))}.
\item Specify object with ID: \code{model$set(..8 = list(val1))}.
\item Overwrite existing object comments: \code{model$set(..8 = list(.comment = c("new", "comment")))}.
\item Delete field value: \code{model$set(Object_Name = list(Field_1 = NULL), .default = FALSE)}.
\item Assign default field value: \code{model$set(Object_Name = list(Field_1 = NULL), .default = TRUE)}.
\item Variable input: \code{a <- list(Object_Name = list(Field_1 = val1)); model$set(a, .default = TRUE)}.
\item Set all values of field \code{fld} in a class \code{cls}:
}\preformatted{ids <- model$object_id("cls", simplify = TRUE)
val <- rep(list(list(fld = val)), times = length(ids))
names(val) <- paste0("..", ids)
model$set(val)
}

}

\subsection{Deleting Existing Objects}{\preformatted{model$del(..., .ref_by = FALSE, .ref_to = FALSE, .recursive = FALSE, .force = FALSE)
}

\code{$del()} takes integer vectors of object IDs and character vectors of object
names, and deletes objects specified. If \code{.ref_by} is \code{TRUE}, objects
whose fields refer to input objects will also be deleted. IF \code{.ref_to} is
\code{TRUE}, objects whose fields are referred by input objects will also be
deleted.

\strong{Note}:
\itemize{
\item If current \link[=level_checks]{validate level} includes reference checking,
objects will not be allowed to be deleted if they are referred by other
objects. For example, an error will be issued if you want to delete one
material that is referred by other constructions, because doing so will
result in invalid field value references. You may bypass this if you really
want to by setting \code{.force} to \code{TRUE}.
\item When \code{.ref_by} or \code{.ref_to} is \code{TRUE}, objects are only deleted when they
only have relation with input objects. For example, a construction \code{const}
consist of 4 different materials. If \code{.ref_to} is \code{TRUE}, that 4 materials
will only be deleted when they are only used in \code{const}, but not used in
any other objects.
\item There are recursively reference relations in \code{Idf} object. For example, one
material's name is referenced by one construction, and that construction's
name can be referred by another surface. You can delete all of them by
setting \code{.recursive} to \code{TRUE}.
}

\strong{Arguments}:
\itemize{
\item \code{.ref_by}: If \code{TRUE}, objects whose fields refer to input objects will
also be deleted. Default: \code{FALSE}.
\item \code{.ref_to}: If \code{TRUE}, objects whose fields are referred by input objects
will also be deleted. Default: \code{FALSE}.
\item \code{.recursive}: If \code{TRUE}, relation searching is performed recursively, in
case that objects whose fields refer to target object are also referred by
another object, and also objects whose fields are referred by target object
are also referred by another object. Default: \code{FALSE}.
\item \code{.force}: If \code{TRUE}, objects are deleted even if they are referred by other
objects.
}

\strong{Usage}:
\itemize{
\item Specify object with name: \code{model$del("Object_Name1", "Object_Name2")}.
\item Specify object with ID: \code{model$del(1, 2, 10)}.
\item Delete objects even they are referred by other objects: \code{model$del(1:5, .force = TRUE)}
\item Delete objects and also other objects that refer to them: \code{model$del(2, "Object_Name1", .ref_by = TRUE)}
\item Delete objects and also other objects that refer to them recursively:
\code{model$del(1:5, .ref_by = TRUE, .recursive = TRUE)}
\item Delete objects and also other objects that input objects refer to: \code{model$del(1:5, .ref_to = TRUE)}
\item Variable input: \code{x <- c("Object_Name1", "Object_Name2"); y <- c(1:5); model$del(x, y)}.
}
}

\subsection{Rename Objects}{\preformatted{model$rename(...)
}

\code{$rename()} takes named character vectors of object names and named integer
vectors of object IDs, renames specified object to names of input vectors and
returns a list of renamed \link{IdfObject}s. An error will be issued if trying to
"rename" an object which does not have name attribute.

\strong{Argument}:
\itemize{
\item \code{...}: Integer vectors of valid object IDs and character vectors of valid
object names. Each element should be named. That name is used at the new
object name.
}

\strong{Usage}:
\itemize{
\item Rename objects: \code{model$rename(c(new_name1 = "name1", new_name2 = "name2"), new_name3 = 6)}.
\item Variable inputs: \code{a <- c(new_name1 = "name1", new_name2 = "name2"); b <- c(new_name3 = 10); model$rename(a, b)}.
}
}

\subsection{Insert Objects}{\preformatted{model$insert(..., .unique = TRUE)
}

\code{$insert()} takes \link{IdfObject}s or lists of \link{IdfObject}s as input, inserts
them into current Idf, and returns a list of inserted \link{IdfObject}s.

\strong{Note}:
\itemize{
\item You cannot insert an \link{IdfObject} which comes from a different version than
current \code{Idf} object.
\item If input \link{IdfObject} has the same name as one \link{IdfObject} in current \code{Idf}
object but field values are not equal, an error may be issued if current
validation level includes conflicted-name checking. For what kind of
validation components to be performed during modifications, please see
\code{\link[=level_checks]{level_checks()}}.
}

\strong{Argument}:
\itemize{
\item \code{...}: \link{IdfObject}s or lists of \link{IdfObject}s.
\item \code{.unique}: If there are duplications in input \link{IdfObject}s or there is same
object in current \code{Idf} object, duplications in input are removed. Default:
\code{TRUE}.
}

\strong{Usage}:
\itemize{
\item Insert objects without new names: \code{model1$insert(model2$Material)}.
\item Insert an object without new name: \code{model1$insert(my_material = model2$Material[[1]])}.
\item Insert objects but keep duplications: \code{model1$insert(model1$Output_Variable)}.
\item Variable input: \code{mat <- model2$Material; names(mat) <- c("mat1", "mat2"); model1$insert(mat)}.
}
}

\subsection{Load Objects from characters or data.frames}{\preformatted{model$load(..., .unique = TRUE, .default = TRUE)
}

\code{$load()} is similar to \code{$insert()} except it takes directly character
vectors or data.frames of \link{IdfObject} definitions, insert corresponding
objects into current \code{Idf} object and returns a list of newly added
\link{IdfObject}s. This makes it easy to create objects using the output from
\code{$to_string()} and \code{$to_table} method from \link{Idd}, \link{IddObject}, also
Idf and \link{IdfObject} class.

For object definitions in character vector format, they follow the same rules
as normal IDF file. Each object starts with a class name and a comma (\code{,}),
separates each values with a comma (\code{,}) and ends with a semicolon (\code{;}).
Noted that you can also provide headers to indicate if input objects are
presented in IP units, using \code{!-Option ViewInIPunits}. If this header does
not exist, then all values are treated as in SI units.

For object definitions in data.frame format, it is highly recommended to use
\code{$to_table()} method in \link{Idd}, \link{IddObject}, \code{Idf} and \link{IdfObject} class. A
valid definition requires at least three columns described below. Note that
column order does not matter.
\itemize{
\item \code{class}: \strong{Mandatory}. Character type. Valid class names in the underlying
\link{Idd} object. You can get all valid class names using
\code{use_idd(model$version())$class_name()}
\item \code{index}: \strong{Mandatory}. Integer type. Valid field indexes for each class.
\item \code{value}: \strong{Mandatory}. Character type or list type. The value of each field
to be added.
\itemize{
\item If \code{value} is a character column, usually when \code{string_value} is \code{TRUE}
in method \code{$to_table()} in \code{Idf} and \link{IdfObject} class. Each value should
be given as a string even if the corresponding field is a numeric type.
\item If \code{value} is a list column, usually when \code{string_value} is set to
\code{FALSE} in method \code{$to_table()} in \code{Idf} and \link{IdfObject} class. Each value
should have the right type  as the corresponding field definition.
Otherwise, errors will be issued during if current validation level
includes invalid-type checking. For what kind of validation components to
be performed during modifications, please see \code{\link[=level_checks]{level_checks()}}.
}
\item \code{id}: \strong{Optional}. Integer type. If input data.frame includes multiple
object definitions in a same class, the value in \code{id} will be used to
distinguish each definition. If \code{id} column does not exists, it assumes
that each definition is separated by \code{class} column and will issue an error
if there is any duplication in the \code{index} column.
}

\strong{Note}:
\itemize{
\item \code{$load()} assume all definitions are from the same version as current \code{Idf}
object. If input definition is from different version, parsing error may
occur.
}

\strong{Argument}:
\itemize{
\item \code{...}: Character vectors or data.frames of object definitions For details,
see above.
\item \code{.unique}: If there are duplications in input \link{IdfObject}s or there is same
object in current Idf, duplications in input are removed. Default: \code{TRUE}.
\item \code{.default}: If \code{TRUE}, default values are used for those blank fields if
possible. Default: \code{TRUE}.
}

\strong{Usage}:
\itemize{
\item Load objects from string definitions:\preformatted{model$load(c(
    "Material,",
    "    mat,                     !- Name",
    "    MediumSmooth,            !- Roughness",
    "    0.667,                   !- Thickness {m}",
    "    0.115,                   !- Conductivity {W/m-K}",
    "    513,                     !- Density {kg/m3}",
    "    1381;                    !- Specific Heat {J/kg-K}",

    "Construction,
     const,
     mat;
    "
))
}
\item Load objects from data.frame definitions:\preformatted{dt <- model1$to_table(class = "Material")
dt[field == "thickness", value := "0.5"]
model$load(dt)
}
}

}

\subsection{Update Objects from characters or data.frames}{\preformatted{model$update(..., .unique = TRUE, .default = TRUE)
}

\code{$update()} is similar to \code{$load()} except it update field values.  This
makes it easy to update object values using the output from \code{$to_string()}
and \code{$to_table} method from \link{Idd}, \link{IddObject}, also Idf and \link{IdfObject}
class.

For object definitions in character vector format, object names are used to
locate which objects to update. Objects that have name attribute should have
valid names. This means that there is no way to update object names using
character vector format, but it can be achieved using data.frame format as it
uses object IDs instead of object names to locate objects.

For object definitions in data.frame format, it is highly recommended to use
\code{$to_table()} method in \link{Idd}, \link{IddObject}, \code{Idf} and \link{IdfObject} class. A
valid definition requires four columns described below. Note that
column order does not matter.
\itemize{
\item \code{id}: Integer type. The value in \code{id} should be valid object IDs for current
\code{Idf}.
\item \code{class}: Character type. Valid class names for current \code{Idf}.
\item \code{index}: Integer type. Valid field indexes for each class. If input field
does not exist in specified object, it will be added on the fly.
\item \code{value}: Character type or list type. The value of each field
to be added.
\itemize{
\item If \code{value} is a character column, usually when \code{string_value} is \code{TRUE}
in method \code{$to_table()} in \code{Idf} and \link{IdfObject} class. Each value should
be given as a string even if the corresponding field is a numeric type.
\item If \code{value} is a list column, usually when \code{string_value} is set to
\code{FALSE} in method \code{$to_table()} in \code{Idf} and \link{IdfObject} class. Each value
should have the right type  as the corresponding field definition.
Otherwise, errors will be issued during if current validation level
includes invalid-type checking. For what kind of validation components to
be performed during modifications, please see \code{\link[=level_checks]{level_checks()}}.
}
}

\strong{Note}:
\itemize{
\item \code{$update()} assume all definitions are from the same version as current
\code{Idf} object. If input definition is from different version, parsing error
may occur.
}

\strong{Argument}:
\itemize{
\item \code{...}: Character vectors or data.frames of object definitions For details,
see above.
\item \code{.default}: If \code{TRUE}, default values are used for those blank fields if
possible. Default: \code{TRUE}.
}

\strong{Usage}:
\itemize{
\item Update objects from string definitions:\preformatted{str <- model$Material[[1]]$to_string()
str[4] <- "0.8"
model$update(str)
}
\item Update objects from data.frame definitions:\preformatted{dt <- model1$to_table(class = "Material")
dt[field == "thickness", value := "0.5"]
model$update(dt)
}
}

}

\subsection{Paste Objects from IDF Editor}{\preformatted{model$paste(in_ip = FALSE, ver = NULL, unique = TRUE)
}

\code{$paste()} reads the contents (from clipboard) of copied objects from IDF
Editor (after hitting \code{Copy Obj} button), parses it and inserts corresponding
objects into current Idf. As IDF Editor only available on Windows platform,
\code{$paste()} only works on Windows too.

\strong{Note}:
\itemize{
\item There is no version data copied to the clipboard when copying objects in
IDF Editor. It is possible that IDF Editor opens an IDF with different
version than current IDF. Please check the version before running
\code{$paste()}, or explicitly specify the version of file opened by IDF Editor
using \code{ver} parameter. Parsing error may occur if there is a version
mismatch.
}

\strong{Arguments}:
\itemize{
\item \code{in_ip}: Set to \code{TRUE} if the IDF file is open with \code{Inch-Pound}
view option toggled. Numeric values will automatically converted to SI
units if necessary. Default: \code{FALSE}.
\item \code{ver}: The version of IDF file opened by IDF Editor, e.g. 8.6, "8.8.0". If
\code{NULL}, assume that the file has the same version as current Idf object.
Default: \code{NULL}.
\item \code{unique}: If there are duplications in copied objects from IDF Editor or
there is same object in current Idf, duplications in input are removed.
Default: \code{TRUE}.
}

\strong{Usage}:
\itemize{
\item Paste objects from same version: \code{model$paste()}.
\item Paste objects from different version: \code{model$paste(ver = "version")}.
\item Paste objects that are viewed in IP units in IDF Editor: \code{model$paste(in_ip = TRUE)}.
\item Paste objects but also keep duplications: \code{model$paste(unique = FALSE)}.
}
}

\subsection{Search and Replace Values}{\preformatted{model$search_value(pattern, class = NULL, ignore.case = FALSE, perl = FALSE, fixed = FALSE, useBytes = FALSE)
model$replace_value(pattern, class = NULL, replacement, ignore.case = FALSE, perl = FALSE, fixed = FALSE, useBytes = FALSE)
}

\code{$search_value()} returns a list of \link{IdfObject}s that contain values which
match the given pattern. If no matched found, \code{NULL} is returned invisibly.

\code{$replace_value()} returns a list of \link{IdfObject}s whose values have been
replace with given pattern. If no matched found, \code{NULL} is returned
invisibly.

\strong{Note}:
\itemize{
\item During matching, all values are treated as characters, including numeric
values.
\item Replacing values using regular expression is not recommended, because it is
error prone. Validation rules also apply during replacing.
}

\strong{Arguments}:
\itemize{
\item \code{class}: A character vector of invalid class names in current \code{Idf} object
to search for values. If \code{NULL}, all classes are used.
\item \code{pattern}, \code{replacement}, \code{ignore.case}, \code{perl}, \code{fixed} and \code{useBytes}:
All of them are directly passed to \link[base:grep]{base::grepl} and
\link[base:grep]{base::gsub}.
}

\strong{Usage}:
\itemize{
\item Search values that contains \code{supply}: \code{model$search_value("supply")}
\item Search values that contains \code{supply} or \code{demand} in class \code{Branch}: \code{model$search_value("supply|demand", "Branch")}
\item Search values that contains \code{win} and replace them with \code{windows}: \code{model$replace_value("win", "windows")}
}
}
}

\section{Validation}{
\preformatted{model$validate(level = eplusr_option("validate_level"))
model$is_valid(level = eplusr_option("validate_level"))
}

\code{$validate()} checks if there are errors in current \code{Idf} under specified
validation level and returns an \code{IdfValidity} object which contains data of
invalid field values. Different validation result examples are shown below:
\itemize{
\item No error is found:\preformatted{v No error found.
}

Above result shows that there is no error found after conducting all
validation checks in specified validation level.
\item Errors are found:\preformatted{ x [2] Errors found during validation.
=========================================================================

-- [2] Invalid Autocalculate Field --------------------------------------
   Fields below cannot be `autocalculate`:

    Class: <AirTerminal:SingleDuct:VAV:Reheat>
    \- Object [ID:176] <SPACE5-1 VAV Reheat>
       +- 17: AUTOCALCULATE, !- Maximum Flow per Zone Floor Area During Reheat {m3/s-m2}
       \- 18: AUTOCALCULATE; !- Maximum Flow Fraction During Reheat
}
}

Above validation results show that after all validation components performed
under current validation level, 2 invalid field values are found. All of them
are in object named \code{SPACE5-1 VAV Reheat} with ID \code{176}. They are invalid
because those two fields do not have an autocalculatable attribute but are
given \code{AUTOCALCULATE} value. Knowing this info, one simple way to fix the
error is to set those two fields to correct value by doing \code{idf$set(..176 = list(}Maximum Flow per Zone Floor Area During Reheat\code{= "autosize",}Maximum Flow Fraction During Reheat\code{ = "autosize" ))}

\code{$is_valid()} returns \code{TRUE} if there is no error in current \code{Idf} object
under specified validation level and \code{FALSE} otherwise.

Underneath, an \code{IdfValidity} object which \code{$validate()} returns is a list of
13 element as shown below. Each element or several elements represents the
results from a single validation checking component. In total, There are 10
different validation check components. To get the meaning of each component,
please see \code{\link[=level_checks]{level_checks()}} and \code{\link[=custom_validate]{custom_validate()}}.
\itemize{
\item \code{missing_object}
\item \code{duplicate_object}
\item \code{conflict_name}
\item \code{incomplete_extensible}
\item \code{missing_value}
\item \code{invalid_autosize}
\item \code{invalid_autocalculate}
\item \code{invalid_character}
\item \code{invalid_numeric}
\item \code{invalid_integer}
\item \code{invalid_choice}
\item \code{invalid_range}
\item \code{invalid_reference}
}

Except \code{missing_object}, which is a character vector, all other elements
are \link[data.table:data.table]{data.table} with 9 columns containing data of
invalid field values:
\itemize{
\item \code{object_id}: IDs of objects that contain invalid values
\item \code{object_name}: names of objects that contain invalid values
\item \code{class_id}: indexes of classes that invalid objects belong to
\item \code{class_name}: names of classes that invalid objects belong to
\item \code{field_id}: indexes (at Idd level) of object fields that are invalid
\item \code{field_index}: indexes of object fields in corresponding that are invalid
\item \code{field_name}: names (without units) of object fields that are invalid
\item \code{units}: SI units of object fields that are invalid
\item \code{ip_units}: IP units of object fields that are invalid
\item \code{type_enum}: An integer vector indicates types of invalid fields
\item \code{value_id}: indexes (at Idf level) of object field values that are invalid
\item \code{value_chr}: values (converted to characters) of object fields that are
invalid
\item \code{value_num}: values (converted to numbers in SI units) of object fields
that are invalid
}

Knowing the internal structure of \code{IdfValidity}, it is easy to extract
invalid \link{IdfObject}s you interested in. For example, you can get all IDs of
objects that contain invalid value references using
\code{model$validate()$invalid_reference$object_id}. Then using \code{$set()} method
to correct them.
}

\section{Data Extraction}{
\preformatted{model$to_table(which = NULL, class = NULL, string_value = TRUE, unit = FALSE, wide = FALSE)
model$to_string(which = NULL, class = NULL, comment = TRUE, header = TRUE, format = eplusr_option("save_format"), leading = 4L, sep_at = 29L)
}

\code{$to_table()} returns a \link[data.table:data.table]{data.table} that contains
core data of specified objects. It has 6 columns:
\itemize{
\item \code{id}: Integer type. Object IDs.
\item \code{name}: Character type. Object names.
\item \code{class}: Character type. Current class name.
\item \code{index}: Integer type. Field indexes.
\item \code{field}: Character type. Field names.
\item \code{value}: Character type if \code{string_value} is \code{TRUE} or list type if
\code{string_value} is \code{FALSE}. Field values.
}

\code{$to_string()} returns the text format of an IDF file.

\strong{Arguments}:
\itemize{
\item \code{which}: Either an integer vector of valid object IDs or a character vector
of valid object names. If \code{NULL}, the whole \code{Idf} object is converted.
Default: \code{NULL}.
\item \code{class}: A character vector of class names. If \code{NULL}, all classed in
current \code{Idf} object is converted. Default: \code{NULL}.
\item \code{string_value}: If \code{TRUE}, all field values are returned as character. If
\code{FALSE}, \code{value} column in returned \link[data.table:data.table]{data.table}
is a list column with each value stored as corresponding type. Note that if
the value of numeric field is set to \code{"Autosize"} or \code{"Autocalculate"}, it
is left as it is, leaving the returned type being a string instead of a
number.  Default: \code{TRUE}.
\item \code{unit}: Only applicable when \code{string_value} is \code{FALSE}. If \code{TRUE}, values
of numeric fields are assigned with units using \code{\link[units:set_units]{units::set_units()}} if
applicable. Default: \code{FALSE}.
\item \code{wide}: Only applicable if target objects belong to a same class. If
\code{TRUE}, a wide table will be returned, i.e. first three columns are always
\code{id}, \code{name} and \code{class}, and then every field in a separate column.
Default: \code{FALSE}.
\item \code{comment}: If \code{FALSE}, all comments will not be included. Default: \code{TRUE}.
\item \code{header}: If \code{FALSE}, the header will not be included. Default: \code{TRUE}.
\item \code{format}: Specific format used when formatting. For details, please see
\code{$save()}. Default: \code{eplusr_option("save_format")}
\item \code{leading}: Leading spaces added to each field. Default: \code{4L}.
\item \code{sep_at}: The character width to separate value string and field string.
Default: \code{29L} which is the same as IDF Editor.
}
}

\section{Save}{
\preformatted{model$is_unsaved()
model$save(path = NULL, format = eplusr_option("save_format"), overwrite = FALSE, copy_external = TRUE)
}

\code{$is_unsaved()} returns \code{TRUE} if there are modifications on the model since
it was read or since last time it was saved and \code{FALSE} otherwise.

\code{$save()} saves the \code{Idf} object as an IDF file.

\strong{Arguments}:
\itemize{
\item \code{path}: A path where to save the model. If \code{NULL}, the path of the \code{Idf}
itself, i.e. \code{model$path()}, will be used.
\item \code{format}: A string to specify the saving format. Should be one of \code{"asis"},
\code{"sorted"}, \code{"new_top"}, and \code{"new_bot"}.
\itemize{
\item If \code{"asis"}, the model will be saved in the same format as it was when
first read. If the model does not contain any format saving option, which
is typically the case when the model was not saved using eplusr or
IDFEditor, \code{"sorted"} will be used.
\item \code{"sorted"}, \code{"new_top"} and \code{"new_bot"} are the same as the save options
\code{"Sorted"}, \code{"Original with New at Top"}, and \code{"Original with New at Bottom"} in IDFEditor. Default: \code{eplusr_option("save_format")}.
}
\item \code{overwrite}: Whether to overwrite the file if it already exists. Default:
\code{FALSE}.
\item \code{copy_external}: If \code{TRUE}, the external files that current Idf depends on
will also be copied into the same directory. The values of file paths in
the Idf will be changed into relative path automatically. This makes it
possible to create fully reproducible simulation conditions. Currently,
only \code{Schedule:File} class is supported. Default: \code{FALSE}.
}
}

\section{Clone}{
\preformatted{model$clone(deep = TRUE)
}

\code{$clone()} returns the exactly the same cloned model. Because \code{Idf} uses
\code{\link[R6:R6Class]{R6::R6Class()}} under the hook which has "modify-in-place" semantics, \code{idf_2 <- idf_1} does not copy \code{idf_1} at all but only create a new binding to
\code{idf_1}.  Modify \code{idf_1} will also affect \code{idf_2} as well, as these two are
exactly the same thing underneath. In order to create a complete cloned copy,
use \code{$clone(deep = TRUE)}.

\strong{Arguments}:
\itemize{
\item \code{deep}: Has to be \code{TRUE} if a complete cloned copy is desired. Default:
\code{TRUE}.
}
}

\section{Run Model}{
\preformatted{model$run(weather, dir = NULL, wait = TRUE, force = FALSE, copy_external = FALSE)
}

\code{$run()} calls corresponding version of EnergyPlus to run the current model
together with specified weather. The model and the weather used will be
copied into the output directory. An \link{EplusJob} object is returned which
provides detailed info of the simulation and methods to collect simulation
results. Please see \link{EplusJob} for details.

\strong{Note}:
\itemize{
\item eplusr uses the EnergyPlus command line interface which was introduced
since EnergyPlus 8.3.0. So \code{$run()} only supports models with version no
lower than 8.3.0.
\item eplusr uses the EnergyPlus SQL output for extracting simulation results. In
order to do so, an object in \code{Output:SQLite} class with \code{Option Type} value
being \code{SimpleAndTabular} will be automatically created if it does not
exists.
}

\strong{Arguments}:
\itemize{
\item \code{weather}: A path to an \code{.epw} file or an \link{Epw} object.
\item \code{dir}: The directory to save the simulation results. If \code{NULL}, the model
folder will be used. Default: NULL
\item \code{wait}: Whether to wait until the simulation completes and print the
standard output and error of EnergyPlus to the screen. If \code{FALSE}, the
simulation will run in the background. Default is \code{TRUE}.
\item \code{force}: Only applicable when the last simulation runs with \code{wait} equals
to \code{FALSE} and is still running. If \code{TRUE}, current running job is
forced to stop and a new one will start. Default: \code{FALSE}.
\item \code{copy_external}: If \code{TRUE}, the external files that current \code{Idf} object
depends on will also be copied into the simulation output directory. The
values of file paths in the Idf will be changed automatically. Currently,
only \code{Schedule:File} class is supported.  This ensures that the output
directory will have all files needed for the model to run. Default is
\code{FALSE}.
}
}

\section{Print}{
\preformatted{model$print(zoom = c("object", "class", "group", "field"), order = TRUE)
print(model)
}

\code{$print()} prints the \code{Idf} object according to different detail level
specified using the \code{zoom} argument.

With the default \code{zoom} level \code{object}, contents of the \code{Idf} object is
printed in a similar style as you see in IDF Editor, with additional heading
lines showing \code{Path}, \code{Version} of the \code{Idf} object. Class names of objects
are ordered by group and the number of objects in classes are shown in square
bracket.

\strong{Arguments}:
\itemize{
\item \code{zoom}: Control how detailed of the Idf object should be printed. Should be
one of \code{"group"}, \code{"class"}, \code{"object"} and \code{"field"}. Default: \code{"group"}.
\itemize{
\item \code{"group"}: all group names current existing are shown with prevailing
square bracket showing how many **<C>**lasses existing in that group.
\item \code{"class"}: all class names are shown with prevailing square bracket
showing how many **<O>**bjects existing in that class, together with
parent group name of each class.
\item \code{"object"}: all object IDs and names are shown, together with parent
class name of each object.
\item \code{"field"}: all object IDs and names, field names and values are shown,
together with parent class name of each object.
}
\item \code{order}: Only applicable when \code{zoom} is \code{"object"} or \code{"field"}. If \code{TRUE},
objects are shown as the same order in the IDF. If \code{FALSE}, objects are
grouped and ordered by classes. Default: \code{TRUE}.
}
}

\examples{
\dontrun{
# ===== CREATE =====
# read an IDF file
idf <- read_idf(system.file("extdata/1ZoneUncontrolled.idf", package = "eplusr"),
    idd = use_idd(8.8, download = "auto"))

# ===== MODEL BASIC INFO =====
# get version
idf$version()

# get path
idf$path()

# get names of all groups in current model
str(idf$group_name())

# get names of all defined groups in the IDD
str(idf$group_name(all = TRUE))

# get names of all classes in current model
str(idf$class_name())

# get names of all defined classes in the IDD
str(idf$class_name(all = TRUE))

# check if input is a valid group name in current model
idf$is_valid_group("Schedules")
idf$is_valid_group("Compliance Objects")

# check if input is a valid group name in IDD
idf$is_valid_group("Compliance Objects", all = TRUE)

# check if input is a valid class name in current model
idf$is_valid_class("Building")
idf$is_valid_class("ShadowCalculation")

# check if input is a valid class name in IDD
idf$is_valid_class("ShadowCalculation", all = TRUE)

# ===== OBJECT DEFINITION (IDDOBJECT) =====
# get the a list of underlying IddObjects
idf$definition("Version")

# ===== OBJECT INFO =====
# get IDs of objects in classes
idf$object_id(c("Version", "Zone"))

# when `simplify` is TRUE, an integer vector will be returned instead of a
# named list
idf$object_id(c("Version", "Zone"), simplify = TRUE)

# get names of objects in classes
# NA will be returned if targeted class does not have a name attribute
idf$object_name(c("Building", "Zone", "Version"))

# if `simplify` is TRUE, a character vector will be returned instead of a
# named list
idf$object_name(c("Building", "Zone", "Version"), simplify = TRUE)

# get number of objects in classes
idf$object_num(c("Zone", "Schedule:Compact"))

# check if input is a valid object ID, i.e. there is an object whose ID is
# the same with input integer
idf$is_valid_id(c(51, 1000))

# check if input is a valid object name, i.e., there is an object whose name is
# the same with input string
idf$is_valid_name(c("Simple One Zone (Wireframe DXF)", "ZONE ONE"))

# ===== OBJECT QUERY =====
# get single object using object ID or name
# NOTE: object name matching is case-insensitive
idf$object(3)
idf$object("simple one zone (wireframe dxf)")

# get objects using object IDs or names
# NOTE: object name matching is case-insensitive
idf$objects(c(3,10))
idf$objects(c("Simple One Zone (Wireframe DXF)", "zone one"))

# the names of returned list are object names
names(idf$objects(c("Simple One Zone (Wireframe DXF)", "zone one")))

# get all objects in classes in a named list
idf$objects_in_class("Zone")
names(idf$objects_in_class("Zone"))

# OR using shortcuts
idf$Zone
idf[["Zone"]]

# get a single object in unique-object class
idf$object_unique("SimulationControl")
idf$SimulationControl
idf[["SimulationControl"]]

# search objects using regular expression
length(idf$search_object("R13"))
names(idf$search_object("R13"))

# search objects using regular expression in specifc class
length(idf$search_object("R13", class = "Construction"))
names(idf$search_object("R13", class = "Construction"))

# get more controls on matching
length(idf$search_object("r\\\\d", ignore.case = TRUE, class = "Construction"))
names(idf$search_object("r\\\\d", ignore.case = TRUE, class = "Construction"))

# ===== DUPLICATE OBJECTS =====
# duplicate objects in "Construction" class
names(idf$Construction)
idf$dup("R13WALL")

# new objects will have the same names as the duplicated objects but with a
# suffix "_1", "_2" and etc.
names(idf$Construction)

# new names can also be explicitly specified
idf$dup(My_R13Wall = "R13WALL")

# duplicate an object multiple times
idf$dup(rep("R13WALL", times = 10))

# ===== ADD OBJECTS =====
# add two new objects in "RunPeriod" class
idf$add(
    RunPeriod = list("rp_test_1", 1, 1, 2, 1,
        .comment = c("Comment for new object 1", "Another comment")
    ),

    RunPeriod = list(name = "rp_test_2",
        begin_month = 3,
        begin_day_of_month = 1,
        end_month = 4,
        end_day_of_month = 1,
        .comment = "Comment for new object 2"
     )
)

# ===== LOAD OBJECTS =====
# load objects from character vector
idf$load("RunPeriod, rp_test_3, 1, 1, 2, 1;")

# load objects from data.frames
dt <- idf$definition("RunPeriod")$to_table()

## (a) values can be supplied as characters
dt[1:5, value := c("rp_test_4", "2", "1", "3", "1")]
idf$load(dt)

## (b) values can be supplied as list
dt[1:5, value := list("rp_test_5", 3, 1, 4, 1)]
idf$load(dt)

# ===== INSERT OBJECTS =====
# insert objects from other Idf object
idf_1 <- read_idf(system.file("extdata/1ZoneUncontrolled.idf", package = "eplusr"),
    idd = use_idd(8.8, download = "auto"))

idf_1$object_name("Material")

# rename material name from "C5 - 4 IN HW CONCRETE" to "test", otherwise
# insertion will be aborted as there will be two materials with the same name
# in the idf
idf_1$Material$`C5 - 4 IN HW CONCRETE`$set(name = "test")

# insert the object
idf$insert(idf_1$Material$test)

# check if material named "test" is there
idf$object_name("Material")

# $insert() is useful when importing design days from a ".ddy" file
idf$insert(read_idf("foo.ddy"))

# ===== SET OBJECTS =====
# set the thickness of newly inserted material "test" to 0.2 m
idf$set(test = list(thickness = 0.2))
idf$Material$test$Thickness

# set thermal absorptance of all material to 0.85
val <- rep(list(list(thermal_absorptance = 0.85)), idf$object_num("Material"))
names(val) <- idf$object_name("Material", simplify = TRUE)
idf$set(val)

# check results
lapply(idf$Material, function (mat) mat$Thermal_Absorptance)

# reset thermal absorptance of all material to the default
val <- rep(list(list(thermal_absorptance = NULL)), idf$object_num("Material"))
names(val) <- idf$object_name("Material", simplify = TRUE)
idf$set(val)

# check results
lapply(idf$Material, function (mat) mat$Thermal_Absorptance)

# ===== UPDATE OBJECTS =====
# update roughness of new added material "test" to "Smooth"
idf$update("Material, test, smooth;")

# update solar absorptance of all materials to 0.8
dt <- idf$to_table(class = "Material", string_value = TRUE)
idf$update(dt[field == "Solar Absorptance", value := "0.8"])

# ===== RENAME OBJECTS =====
idf$rename(new_test = "test")
idf$object_name("Material")

# ===== DELELTE OBJECTS =====
# delete the added run period "rp_test_1", "rp_test_2" and "new_test" from above
idf$del(c("new_test", "rp_test_1", "rp_test_2"))
names(idf$Material)
names(idf$RunPeriod)

# In "final" validate level, delete will be aborted if the target objects are
# referenced by other objects.
# get objects that referenced material "R13LAYER"
eplusr_option("validate_level")

idf$Material_NoMass$R13LAYER$ref_by_object()
length(idf$Material_NoMass$R13LAYER$ref_by_object())

idf$del("R13LAYER") # will give an error in "final" validate level

# objects referencing target objects can also be deleted by setting
# `referenced` to TRUE
idf$del("R13LAYER", .ref_by = TRUE) # will give an error in "final" validate level

# it is possible to force delete objects
idf$del("R13LAYER", .ref_by = TRUE, .force = TRUE)

# ===== SEARCH ADN REPLACE OBJECT VALUES =====
# get objects whose field values contains both "VAV" and "Node"
idf$search_value("WALL")
length(idf$search_value("WALL"))
names(idf$search_value("WALL"))

# replace values using regular expression
idf$replace_value("WALL", "A_WALL")

# ===== VALIDATE MODEL =====
# check if there are errors in current model
idf$validate()
idf$is_valid()

# change validate level to "none", which will enable invalid modifications
eplusr_option(validate_level = "none")

# change the outside layer of floor to an invalid material
idf$set(FLOOR = list(outside_layer = "wrong_layer"))

# change validate level back to "final" and validate the model again
eplusr_option(validate_level = "final")

idf$validate()
idf$is_valid()

# get IDs of all objects that contains invalid reference fields
idf$validate()$invalid_reference$object_id

# fix the error
idf$set(..16 = list(outside_layer = idf$Material[[1]]$name()))
idf$validate()
idf$is_valid()

# ===== FORMAT MODEL =====
# get text format of the model
head(idf$to_string())

# get text format of the model, excluding the header and all comments
head(idf$to_string(comment = FALSE, header = FALSE))

# ===== SAVE MODEL =====
# check if the model has been modified since read or last saved
idf$is_unsaved()

# save and overwrite current model
idf$save(overwrite = TRUE)

# save the model with newly created and modified objects at the top
idf$save(overwrite = TRUE, format = "new_top")

# save the model to a new file
idf$save(path = file.path(tempdir(), "test.idf"))

# save the model to a new file and copy all external csv files used in
# "Schedule:File" class into the same folder
idf$save(path = file.path(tempdir(), "test1.idf"), copy_external = TRUE)

# the path of this model will be changed to the saved path
idf$path()

# ===== CLONE MODEL =====
# Idf object are modified in place and has reference semantic.
idf_2 <- idf
idf_2$object_name("Building")
idf$object_name("Building")

# modify idf_2 will also affect idf as well
idf_2$Building$set(name = "Building_Name_Changed")
idf_2$object_name("Building")
idf$object_name("Building")

# in order to make a copy of an Idf object, use $clone() method
idf_3 <- idf$clone(deep = TRUE)
idf_3$Building$set(name = "Building_Name_Changed_Again")
idf_3$object_name("Building")

idf$object_name("Building")

# ===== RUN MODEL =====
if (is_avail_eplus(8.8)) {

    # save the model to tempdir()
    idf$save(file.path(tempdir(), "test_run.idf"))

    # use the first epw file in "WeatherData" folder in EnergyPlus v8.8
    # installation path
    epw <- list.files(file.path(eplus_config(8.8)$dir, "WeatherData"),
        pattern = "\\\\.epw$", full.names = TRUE)[1]
    basename(epw)
    # [1] "USA_CA_San.Francisco.Intl.AP.724940_TMY3.epw"

    # if `dir` is NULL, the directory of IDF file will be used as simulation
    # output directory
    job <- idf$run(epw, dir = NULL)

    # run simulation in the background
    idf$run(epw, dir = tempdir(), wait = FALSE)

    # copy all external files into the directory run simulation
    idf$run(epw, dir = tempdir(), copy_external = TRUE)

    # check for simulation errors
    job$errors()

    # get simulation status
    job$status()

    # get output directory
    job$output_dir()

    # re-run the simulation
    job$run()

    # get simulation results
    job$report_data()
}

# ===== PRINT MODEL =====
idf$print("group")
idf$print("class")
idf$print("object")
idf$print("field")
}
}
\seealso{
\link{IdfObject} class
}
\author{
Hongyuan Jia
}
