% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/LCx.R
\name{LC_probit}
\alias{LC_probit}
\title{Lethal Concentration Probit}
\usage{
LC_probit(formula, data, p = seq(1, 99, 1),
 weights = NULL, het_sig = NULL, conf_level = NULL)
}
\arguments{
\item{formula}{an object of class 'formula' or one that can be coerced to that class): a symbolic description of the model to be fitted. The details of model specification are given under Details.}

\item{data}{an optional data frame, list or environment (or object coercible by as.data.frame to a data frame) containing the variables in the model. If not found in data, the variables are taken from environment(formula), typically the environment from which LC is called.}

\item{p}{Lethal Concentration (LC) values for given p, example will return a LC50 value if p equals 50. If more than one LC value wanted specify by creating a vector.}

\item{weights}{vector of 'prior weights' to be used in the fitting process. Should be a numeric vector, if set to NULL weights will not be used.}

\item{het_sig}{signficance level from person's goodness-of-fit test that is used to decide if a hetrogentiy factor is used. NULL is set to 0.15.}

\item{conf_level}{Adjust confidence level as necessary or NULL set at 0.95.}
}
\value{
Returns a data frame with predicted LC for given p level, lower CL (LCL), upper CL (UCL), LCL and UCL distance away from LC (LCL_dis & UCL_dis; important for creating a plot), Pearson's goodness-of-fit test, slope, intercept, slope and intercept p values and standard error, and LC variance.
}
\description{
Calculates lethal concentration (LC) and
its fiducial confidence limits (CL) using a probit analysis
according to Finney 1971, Wheeler et al. 2006, and Robertson et al. 2007.
}
\examples{
head(lampreytox)

# within the dataframe used, control dose, unless produced a value
# during experimentation, are removed from the dataframe,
# as glm cannot handle values of infinite. Other statistical programs
# make note of the control dose but do not include within analysis

#calculate LC50 and LC99

m <- LC_probit((dead / total) ~ log10(dose), p = c(50, 99),
         weights = lampreytox[c(1:19), ]$total,
         data = lampreytox[c(1:19), ])

#view calculated LC50 and LC99 for seasonal toxicity of a pisicide,
#to lamprey in 2011

m

#dose-response curve can be plotted using 'ggplot2'

library(ggplot2)

p1 <- ggplot(data = lampreytox[c(1:19), ],
             aes(x = log10(dose), y = (dead / total))) +
  geom_point() +
  geom_smooth(method = "glm",
            method.args = list(family = binomial(link = "probit")),
            aes(weight = total), colour = "#FF0000", se = TRUE)

p1

#calculate LC50s and LC99s for multiple toxicity tests, June, August, and September

j <- LC_probit((dead / total) ~ log10(dose), p = c(50, 99),
        weights = lampreytox[c(20:38), ]$total,
        data = lampreytox[c(20:38), ])

a <- LC_probit((dead / total) ~ log10(dose), p = c(50, 99),
        weights = lampreytox[c(39:51), ]$total,
        data = lampreytox[c(39:51), ])

s <- LC_probit((dead / total) ~ log10(dose), p = c(50, 99),
        weights = lampreytox[c(52:64), ]$total,
        data = lampreytox[c(52:64), ])

#group results together in a dataframe to plot with 'ggplot2'

results <- rbind(m[, c(1, 3:8, 11)], j[,c(1, 3:8, 11)],
                 a[, c(1, 3:8, 11)], s[, c(1, 3:8, 11)])

results$month <- with(results, factor(c("May", "May", "June", "June",
                                        "August", "August", "September",
                                        "September"),
                                        levels = c("May", "June",
                                        "August", "September")))

p2 <- ggplot(data = results, aes(x = month, y = dose,
                             group = factor(p), fill = factor(p))) +
  geom_col(position = position_dodge(width = 0.9), colour = "#000000") +
  geom_errorbar(aes(ymin = (dose - LCL_dis), ymax = (dose + UCL_dis)),
                size = 0.4, width = 0.06,
                position = position_dodge(width = 0.9))

p2
}
\references{
Finney, D.J., 1971. Probit Analysis, Cambridge University Press, Cambridge, England, ISBN: 052108041X

Wheeler, M.W., Park, R.M., and Bailey, A.J., 2006. Comparing median lethal concentration values using confidence interval overlap or ratio tests, Environ. Toxic. Chem. 25(5), 1441-1444.10.1897/05-320R.1

Robertson, J.L., Savin, N.E., Russell, R.M. and Preisler, H.K., 2007. Bioassays with arthropods. CRC press. ISBN: 9780849323317
}
