% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bayesCox.R
\name{bayesCox}
\alias{bayesCox}
\title{Fit Bayesian Cox Model for Interval Censored Survival Data}
\usage{
bayesCox(formula, data, grid = NULL, out = "mcmc.txt",
         model = c("TimeIndep", "TimeVarying", "Dynamic"),
         base.prior = list(), coef.prior = list(),
         gibbs = list(), control = list())
}
\arguments{
\item{formula}{A formula object, with the response on the left of a '~'
operator, and the terms on the right. The response must be a survival
object as returned by the function \code{Surv} with \code{type =
"interval2"}. \code{help(Surv)} for details.}

\item{data}{A data.frame in which to interpret the variables named in the
\code{formula}.}

\item{grid}{Vector of pre-specified time grid points for model fitting.
It will be automatically set up from data if it is left unspecified
in the function call. By default, it consists of all the unique
finite endpoints (rounded to two significant numbers) of the
censoring intervals after time zero.  The \code{grid} specified in
the function call determines the location of possible jumps. It should
be sorted increasingly and cover all the finite non-zero endpoints of
the censoring intervals. Inappropriate \code{grid} specified will be
taken care by the function internally.}

\item{out}{Name of Markov chain Monte Carlo (MCMC) samples output file.
Each row contains one MCMC sample information. The file is needed for
those functions further summarizing estimation results in this
package. See Section Details for details.}

\item{model}{Model type to fit. Available options are \code{"TimeIndep"},
\code{"TimeVarying"}, and \code{"Dynamic"}. Partial matching on the
name is allowed.}

\item{base.prior}{List of options for prior of baseline lambda. Use
\code{list(type = "Gamma", shape = 0.1, rate = 0.1)} for all models;
\code{list(type = "Const", value = 1)} for \code{Dynamic} model when
\code{intercept = TRUE}.}

\item{coef.prior}{List of options for prior of coefficient beta. Use
\code{list(type = "Normal", mean = 0, sd = 1)} for \code{TimeIndep}
model; \code{list(type = "AR1", sd = 1)} for \code{TimeVarying} and
\code{Dynamic} models; \code{list(type = "HAR1", shape = 2, scale =
1)} for \code{TimeVarying} and \code{Dynamic} models.}

\item{gibbs}{List of options for Gibbs sampler.}

\item{control}{List of general control options.}
}
\value{
An object of S3 class \code{bayesCox} representing the fit.
}
\description{
Fit Bayesian Cox model with time-independent, time-varying or dynamic
covariate coefficient. The fit is done within a Gibbs sampling framework.
The reversible jump algorithm is employed for the dynamic coefficient
model. The baseline hazards are allowed to be either time-varying or
dynamic.
}
\details{
To use default hyper parameters in the specification of either
\code{base.prior} or \code{coef.prior}, one only has to supply the name of
the prior, e.g., \code{list(type = "Gamma")}, \code{list(type = "HAR1")}.

The \code{gibbs} argument is a list of components:
\describe{
    \item{iter:}{Number of iterations, default 3000;}
    \item{burn:}{Number of burning, default 500;}
    \item{thin:}{Number of thinning, default 1;}
    \item{verbose:}{A logical value, default \code{TRUE}. If
        \code{TRUE}, print the iteration;}
    \item{nReport:}{Print frequency, default 100.}
}

The \code{control} argument is a list of components:
\describe{
    \item{intercept:}{A logical value, default \code{FALSE}. If
        \code{TRUE}, the model will estimate the intercept, which is the
        log of baseline hazards. If \code{TRUE}, please remember to turn
        off the direct estimation of baseline hazards, i.e.,
        \code{base.prior = list(type = "Const")}}
    \item{a0:}{Multiplier for initial variance in time-varying or dynamic
        models, default 100;}
    \item{eps0:}{Size of auxiliary uniform latent variable in dynamic model,
        default 1.}
}

For users interested in extracting MCMC sampling information from the
output files, the detail of the output files is presented as follows: Let
\eqn{k} denote the number of time points (excluding time zero) specified
in grid, \eqn{ck} equal \eqn{1} for model with time-invariant coefficients;
\eqn{ck} equal \eqn{k} otherwise, and \eqn{p} denote the number of
covariates.  Then the each sample saved in each row consists of the
following possible parts.
\describe{
    \item{Part 1:}{The first \eqn{k} numbers represent the jump size of
baseline hazard function at each time grid.  If we take the column mean
of the first \eqn{k} columns of the output file, we will get the same
numbers with \code{obj$est$lambda}, where \code{obj} is the \code{bayesCox}
object returned by the function.}
    \item{Part 2:}{The sequence from \eqn{(k + 1) to (k + ck * p)}
represent the coefficients of covariates at the time grid.  The first
\eqn{k} numbers in the sequence are the coefficients for the first covariate
at the time grid; The second \eqn{k} numbers' sub-sequence are the
coefficients for the second covariate and so on.}
    \item{Part 3:}{The sequence from \eqn{(k + ck * p + 1)} to
\eqn{(k + ck * p + p)} represents the sampled latent variance of
coefficients.}
    \item{Part 4:}{The sequence from \eqn{(k + ck * p + p + 1)} to
\eqn{(k + 2 * ck * p + p)} represents the indicator of whether there is
a jump of the covariate coefficients at the time grid.  Similar with Part 2,
the first k numbers' sub-sequence is for the first covariate, the second
\eqn{k} numbers' sub-sequence is for the second covariate, and so on.}
}
For the model with time-independent coefficients, the output file only
has Part 1 and Part 2 in each row; For time-varying coefficient model,
the output file has Part 1, 2, and 3; The output file for the dynamic
model has all the four parts.  Note that the dynamic baseline hazard will
be taken as one covariate.  So \eqn{p} needs being replaced with
\eqn{(p + 1)} for model with dynamic baseline hazard rate.
No function in the package actually needs the Part 1 from the output file
now; The Part 2 is used by function \code{coef} and \code{survCurve};
The Part 3 is needed by function \code{nu}; Function \code{jump} extracts
the Part 4.
}
\examples{
\dontrun{
library(dynsurv)
set.seed(1216)

############################################################################
### Attach one of the following two data sets
############################################################################

### breast cancer data
data(bcos)       # attach bcos and bcos.grid
mydata <- bcos
mygrid <- bcos.grid
myformula <- Surv(left, right, type = "interval2") ~ trt

### tooth data
## data(tooth)   # attach tooth and tooth.grid
## mydata <- tooth
## mygrid <- tooth.grid
## myformula <- Surv(left, rightInf, type = "interval2") ~ dmf + sex

############################################################################
### Fit Bayesian Cox models
############################################################################

## Fit time-independent coefficient model
fit0 <- bayesCox(myformula, mydata, out = "tiCox.txt", model = "TimeIndep",
                 base.prior = list(type = "Gamma", shape = 0.1, rate = 0.1),
                 coef.prior = list(type = "Normal", mean = 0, sd = 1),
                 gibbs = list(iter = 100, burn = 20, thin = 1,
                              verbose = TRUE, nReport = 20))
plotCoef(coef(fit0, level = 0.9))

## Plot the estimated survival function for given new data
newDat <- data.frame(trt = c("Rad", "RadChem"))
row.names(newDat) <- unique(newDat$trt)
plotSurv(survCurve(fit0, newDat), conf.int = TRUE)

## Fit time-varying coefficient model
fit1 <- bayesCox(myformula, mydata, out = "tvCox.txt", model = "TimeVary",
                 base.prior = list(type = "Gamma", shape = 0.1, rate = 0.1),
                 coef.prior = list(type = "AR1", sd = 1),
                 gibbs = list(iter = 100, burn = 20, thin = 1,
                              verbose = TRUE, nReport = 20))
plotCoef(coef(fit1))
plotSurv(survCurve(fit1), conf.int = TRUE)

## Fit dynamic coefficient model with time-varying baseline hazards
fit2 <- bayesCox(myformula, mydata, out = "dynCox1.txt", model = "Dynamic",
                 base.prior = list(type = "Gamma", shape = 0.1, rate = 0.1),
                 coef.prior = list(type = "HAR1", shape = 2, scale = 1),
                 gibbs = list(iter = 100, burn = 20, thin = 1,
                              verbose = TRUE, nReport = 20))
plotCoef(coef(fit2))
plotJumpTrace(jump(fit2))
plotJumpHist(jump(fit2))
plotNu(nu(fit2))
plotSurv(survCurve(fit2), conf.int = TRUE)

## Plot the coefficient estimates from three models together
plotCoef(rbind(coef(fit0), coef(fit1), coef(fit2)))

## Fit dynamic coefficient model with dynamic hazards (in log scales)
fit3 <- bayesCox(myformula, mydata, out = "dynCox2.txt", model = "Dynamic",
                 base.prior = list(type = "Const"),
                 coef.prior = list(type = "HAR1", shape = 2, scale = 1),
                 gibbs = list(iter = 100, burn = 20, thin = 1,
                              verbose = TRUE, nReport = 20),
                 control = list(intercept = TRUE))
plotCoef(coef(fit3))
plotJumpTrace(jump(fit3))
plotJumpHist(jump(fit3))
plotNu(nu(fit3))
plotSurv(survCurve(fit3), conf.int = TRUE)

## Plot the estimated survival function and the difference
plotSurv(survCurve(fit3, newdata = newDat, type = "survival"),
         legendName = "Treatment", conf.int = TRUE)
plotSurv(survDiff(fit3, newdata = newDat, type = "survival"),
         legendName = "Treatment", conf.int = TRUE, smooth = TRUE)
}
}
\references{
X. Wang, M.-H. Chen, and J. Yan (2013). Bayesian dynamic regression
models for interval censored survival data with application to children
dental health. Lifetime data analysis, 19(3), 297--316.

X. Wang, X. Sinha, J. Yan, and M.-H. Chen (2014). Bayesian inference of
interval-censored survival data. In: D. Chen, J. Sun, and K. Peace,
Interval-censored time-to-event data: Methods and applications, 167--195.

X. Wang, M.-H. Chen, and J. Yan (2011). Bayesian dynamic
regression models for interval censored survival data. Technical Report 13,
Department of Statistics, University of Connecticut.

D. Sinha, M.-H. Chen, and S.K. Ghosh (1999). Bayesian analysis and model
selection for interval-censored survival data. \emph{Biometrics} 55(2),
585--590.
}
\seealso{
\code{\link{coef.bayesCox}}, \code{\link{jump.bayesCox}},
    \code{\link{nu.bayesCox}}, \code{\link{plotCoef}},
    \code{\link{plotJumpTrace}}, \code{\link{plotNu}},
    \code{\link{survCurve}}, \code{\link{survDiff}}, and
    \code{\link{plotSurv}}.
}
\keyword{Bayesian}
\keyword{Cox}
\keyword{censor}
\keyword{dynamic}
\keyword{interval}
