% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bayesCox.R
\name{bayesCox}
\alias{bayesCox}
\title{Fit Bayesian Cox Model for Interval Censored Survival Data}
\usage{
bayesCox(formula, data, grid, out,
         model = c("TimeIndep", "TimeVarying", "Dynamic"),
         base.prior = list(), coef.prior = list(),
         gibbs = list(), control = list())
}
\arguments{
\item{formula}{A formula object, with the response on the left of a '~'
operator, and the terms on the right. The response must be a survival
object as returned by the \code{Surv} function.}

\item{data}{A data.frame in which to interpret the variables named in the
\code{formula}.}

\item{grid}{Vector of pre-specified time grid points.}

\item{out}{Name of Markov chain Monte Carlo (MCMC) samples output file.}

\item{model}{Model type to fit.}

\item{base.prior}{List of options for prior of baseline lambda. Use
\code{list(type = "Gamma", shape = 0.1, rate = 0.1)} for all models;
\code{list(type = "Const", value = 1)} for \code{Dynamic} model when
\code{intercept = TRUE}.}

\item{coef.prior}{List of options for prior of coefficient beta. Use
\code{list(type = "Normal", mean = 0, sd = 1)} for \code{TimeIndep} model;
\code{list(type = "AR1", sd = 1)} for \code{TimeVarying} and \code{Dynamic}
models; \code{list(type = "HAR1", shape = 2, scale = 1)} for
\code{TimeVarying} and \code{Dynamic} models.}

\item{gibbs}{List of options for Gibbs sampler.}

\item{control}{List of general control options.}
}
\value{
An object of S3 class \code{bayesCox} representing the fit.
}
\description{
Fit Bayesian Cox model with time-independent, time-varying or dynamic
covariate coefficient. The fit is done within a Gibbs sampling framework.
The reversible jump algorithm is employed for the dynamic coefficient
model. The baseline hazards are allowed to be either time-varying or
dynamic.
}
\details{
To use default hyper parameters in the specification of either
\code{base.prior} or \code{coef.prior}, one only has to supply the name of
the prior, e.g., \code{list(type = "Gamma")}, \code{list(type = "HAR1")}.

The \code{gibbs} argument is a list of components:
\describe{
    \item{iter:}{number of iterations, default 3000;}
    \item{burn:}{number of burning, default 500;}
    \item{thin:}{number of thinning, default 1;}
    \item{verbose:}{a logical value, default \code{TRUE}. If
        \code{TRUE}, print the iteration;}
    \item{nReport:}{print frequency, default 100.}
}

The \code{control} argument is a list of components:
\describe{
    \item{intercept:}{a logical value, default \code{FALSE}. If
        \code{TRUE}, the model will estimate the intercept, which is the
        log of baseline hazards. If \code{TRUE}, please remember to turn
        off the direct estimation of baseline hazards, i.e.,
        \code{base.prior = list(type = "Const")}}
    \item{a0:}{multiplier for initial variance in time-varying or dynamic
        models, default 100;}
    \item{eps0:}{size of auxiliary uniform latent variable in dynamic model,
        default 1.}
}
}
\examples{

\dontrun{
############################################################################
### Attach one of the following two data sets
############################################################################

## breast cancer data
data(bcos) ## attach bcos and bcos.grid
mydata <- bcos
mygrid <- bcos.grid
myformula <- Surv(left, right, type = "interval2") ~ trt

## tooth data
## data(tooth) ## load tooth and tooth.grid
## mydata <- tooth
## mygrid <- tooth.grid
## myformula <- Surv(left, right, type = "interval2") ~ dmf + sex

############################################################################
### Fit Bayesian Cox models
############################################################################

## Fit time-independent coefficient model
fit0 <- bayesCox(myformula, mydata, mygrid, out = "tiCox.txt",
                 model = "TimeIndep",
                 base.prior = list(type = "Gamma", shape = 0.1, rate = 0.1),
                 coef.prior = list(type = "Normal", mean = 0, sd = 1),
                 gibbs = list(iter = 100, burn = 20, thin = 1,
                              verbose = TRUE, nReport = 5))
plotCoef(coef(fit0, level = 0.9))

## Fit time-varying coefficient model
fit1 <- bayesCox(myformula, mydata, mygrid, out = "tvCox.txt",
                 model = "TimeVarying",
                 base.prior = list(type = "Gamma", shape = 0.1, rate = 0.1),
                 coef.prior = list(type = "AR1", sd = 1),
                 gibbs = list(iter = 100, burn = 20, thin = 1,
                              verbose = TRUE, nReport = 5))
plotCoef(coef(fit1))

## Fit dynamic coefficient model with time-varying baseline hazards
fit2 <- bayesCox(myformula, mydata, mygrid, out = "dynCox1.txt",
                 model = "Dynamic",
                 base.prior = list(type = "Gamma", shape = 0.1, rate = 0.1),
                 coef.prior = list(type = "HAR1", shape = 2, scale = 1),
                 gibbs = list(iter = 100, burn = 20, thin = 1,
                              verbose = TRUE, nReport = 5))
plotCoef(coef(fit2))
plotJumpTrace(jump(fit2))
plotJumpHist(jump(fit2))
plotNu(nu(fit2))

## Plot the coefficient estimates from three models together
plotCoef(rbind(coef(fit0), coef(fit1), coef(fit2)))

## Fit dynamic coefficient model with dynamic hazards (in log scales)
fit3 <- bayesCox(myformula, mydata, mygrid, out = "dynCox2.txt",
                 model = "Dynamic",
                 base.prior = list(type = "Const"),
                 coef.prior = list(type = "HAR1", shape = 2, scale = 1),
                 gibbs = list(iter = 100, burn = 20, thin = 1,
                              verbose = TRUE, nReport=5),
                 control = list(intercept = TRUE))
plotCoef(coef(fit3))
plotJumpTrace(jump(fit3))
plotJumpHist(jump(fit3))
plotNu(nu(fit3))

## Plot the estimated survival function and hazard function
newDat <- bcos[c(1, 47), ]
row.names(newDat) <- c("Rad", "RadChem")
plotSurv(survCurve(fit3, newdata = newDat, type = "survival"),
         legendName = "Treatment", conf.int = TRUE)
plotSurv(survDiff(fit3, newdata = newDat, type = "cumhaz"),
         legendName = "Treatment", conf.int = TRUE, smooth = TRUE)
}

}
\references{
X. Wang, M.-H. Chen, and J. Yan (2011). Bayesian dynamic
regression models for interval censored survival data. Under review.

D. Sinha, M.-H. Chen, and S.K. Ghosh (1999). Bayesian analysis and model
selection for interval-censored survival data. \emph{Biometrics} 55(2),
585--590.
}
\seealso{
\code{\link{coef.bayesCox}}, \code{\link{jump.bayesCox}},
\code{\link{nu.bayesCox}}, \code{\link{plotCoef}},
\code{\link{plotJumpTrace}}, \code{\link{plotNu}},
\code{\link{survCurve}}, \code{\link{survDiff}}, and
\code{\link{plotSurv}}.
}
\keyword{Bayesian}
\keyword{Cox}
\keyword{censor}
\keyword{dynamic}
\keyword{interval}

