% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/likelihoods_WEV.R
\name{LogLikWEV}
\alias{LogLikWEV}
\alias{LLWEV}
\alias{LLdynWEV}
\alias{LL2DSD}
\alias{LogLikdynWEV}
\alias{LogLikWEV2DSD}
\title{Log-Likelihood functions for the dynWEV and 2DSD models of confidence}
\usage{
LogLikWEV(data, paramDf, model = "dynaViTE", simult_conf = FALSE,
  precision = 1e-05, stop_on_error = TRUE, data_names = list(), ...)
}
\arguments{
\item{data}{a dataframe where each row is one trial. Containing following variables:
\itemize{
\item condition    (not necessary; convertible to integer (e.g. factor); for different levels of stimulus quality),
\item rating       (convertible to integer (e.g. factor); discrete confidence judgments),
\item rt           (numeric; giving reaction times for decision task),
\item stimulus     (values at least convertible to c(-1,1); stimulus category (direction of evidence accumulation))
\item response     (characters in \code{"upper"}, \code{"lower"} (or convertible to); direction of decision; correct answers are "lower" for stimulus=-1; and "upper" for stimulus=1),

}}

\item{paramDf}{list or data.frame with one row. Names should match the names of
\link{dynaViTE} and \link{2DSD} model specific parameter names. For different
stimulus quality/mean drift rates, names should be \code{v1}, \code{v2}, \code{v3},....
Different \code{sv} and/or \code{s} parameters are possible with \code{sv1}, \code{sv2}, \code{sv3}... (\code{s1}, \code{s2}, \code{s3},...
respectively) with equally many steps as for drift rates. Additionally, the confidence
thresholds should be given by names with \code{thetaUpper1}, \code{thetaUpper2},..., \code{thetaLower1},... or,
for symmetric thresholds only by \code{theta1}, \code{theta2},...
(see Details for the correspondence to the data)}

\item{model}{character scalar. One of "dynWEV" or "2DSD" for the model to fit.}

\item{simult_conf}{logical. Whether in the experiment confidence was reported simultaneously
with the decision, as then decision and confidence judgment are assumed to have happened
subsequent before response and computations are different, when there is an observable
interjudgment time (then \code{simult_conf} should be \code{FALSE}).}

\item{precision}{numerical scalar. Precision of calculation for integration over z and t0.}

\item{stop_on_error}{logical. If TRUE an error in the function will be returned in case of
invalid parameters. Otherwise, the output will be 0 without error.}

\item{data_names}{list. Possibility of giving alternative column names for the variables
in the data. By default column names are identical to the
ones given in the data argument description.}

\item{...}{Possibility of giving alternative variable names in data frame
(in the form \code{condition = "SOA"}).}
}
\value{
Numeric scalar. The summed Log-likelihood of the data given the parameters in the respective model. If one or more row-wise probabilities is <=0,
the function returns -1e+12.
}
\description{
Computes the Log-likelihood for given data and parameters in the
dynWEV model (Hellmann et al., 2023) and the 2DSD model
(Pleskac & Busemeyer, 2010). It is a wrapped version of the
respective densities \code{\link{dWEV}} and \code{\link{d2DSD}},
where one can find more information about the parameters
(\code{z} is always given relatively, in the likelihood).
The function is mainly used in \code{\link{fitRTConf}} but
exported for individual usage in other contexts.
}
\details{
Note, that the requirements on the format of the columns for the likelihood functions
are much stricter, than in \code{\link{fitRTConf}}.
This is because the function is very frequently calls in the optimization routines of the
fitting process and the preprocessing steps are
therefore included in that function.

\strong{rating, condition}. If integer, values should range from 1 to number of possible
ratings/conditions. If a factor, the number of levels should be
equal to number of possible ratings/conditions. This should be consistent with the
parameter vector. The confidence thresholds should be named
as \code{thetaUpper1}, \code{thetaLower1},.... (or \code{theta1},... for symmetric thresholds), with the
number of ratings -1 and the mean drift rates (and possibly the
standard deviation in drift rates) should be denoted as \code{v1}, \code{v2},...
(and \code{sv1}, \code{sv2},.../\code{s1}, \code{s2}, ...) with the number equal to the number of conditions.
If only one condition is used \code{v} will be accepted as well as \code{v1}.

\strong{stimulus, response}. stimulus should always be given in numerical format with values -1 and 1.
response should always be given as a character vector with \code{"lower"} and \code{"upper"} as values.
This corresponds to the situation of Ratcliff's diffusion model (Ratcliff, 1978), where stimulus is the sign of the mean drift direction and
the response is the \code{"upper"} or \code{"lower"} boundary that is first hit by the evidence accumulation. A correct decision is therefore \code{"lower"},
if stimulus is -1, and \code{"upper"}, if stimulus is 1.
}
\examples{
# 1. Generate data from an artificial participants
# Get random drift direction (i.e. stimulus category) and
# stimulus discriminability (two steps: hard, easy)
stimulus <- sample(c(-1, 1), 200, replace=TRUE)
discriminability <- sample(c(1, 2), 200, replace=TRUE)
# generate data for participant 1
data <- rWEV(200, a=2,v=stimulus*discriminability*0.5,
             t0=0.2,z=0.5, sz=0.1,sv=0.1, st0=0,  tau=4, s=1, w=0.3)
# discretize confidence ratings (only 2 steps: unsure vs. sure)
data$rating <- as.numeric(cut(data$conf, breaks = c(-Inf, 1, Inf), include.lowest = TRUE))
data$participant = 1
data$stimulus <- stimulus
data$discriminability <- discriminability
data <- data[data$response!=0, ] # drop not finished decision processes
data <- data[,-3] # drop conf measure (unobservable variable)
head(data)

# 2. Define some parameter set in a data.frame
paramDf <- data.frame(a=2.5,v1=0.5, v2=1, t0=0.1,z=0.7,
                      sz=0,sv=0.2, st0=0,  tau=3, w=0.3,
                      theta1=0.8, svis=0.5, sigvis=0.8)

# 3. Compute log likelihood for parameter and data
LogLikWEV(data, paramDf, model="dynWEV", condition="discriminability")
# adding the hypothetical interjudgment time to response times
# results in the same log likelihood as before when simult_conf=TRUE
data$rt <- data$rt + paramDf$tau
LogLikWEV(data, paramDf, model="dynWEV", condition="discriminability", simult_conf=TRUE)

# the same function for "2DSD" model
paramDf <- data.frame(a=2.5,v1=0.5, v2=1, t0=0.1,z=0.7,
                      sz=0,sv=0.2, st0=0,  tau=3, theta1=0.8)
LogLikWEV(data, paramDf, model="2DSD", condition="discriminability", simult_conf=TRUE)
# this results in the same log likelihood as before

}
\references{
Hellmann, S., Zehetleitner, M., & Rausch, M. (2023). Simultaneous modeling of choice, confidence and response time in visual perception. \emph{Psychological Review} 2023 Mar 13. doi: 10.1037/rev0000411. Epub ahead of print. PMID: 36913292.
}
\author{
Sebastian Hellmann.
}
