#' @title Function config
#' @description Compute the internal runtime parameter list of
#' \code{\link{make}()}. This could save time if you are planning
#' multiple function calls of functions like \code{\link{outdated}()}
#' or \code{\link{plot_graph}()}. Drake needs to import and cache files
#' and objects to compute the configuration list, which in turn
#' supports user-side functions to help with visualization and parallelism.
#' The result differs from
#' \code{\link{make}(..., imports_only = TRUE)}
#' in that the graph includes both the targets and the imports,
#' not just the imports.
#' @export
#' @seealso \code{\link{workplan}}, \code{\link{make}}, \code{\link{plot_graph}}
#' @examples
#' \dontrun{
#' load_basic_example()
#' con <- config(my_plan)
#' outdated(my_plan, config = con)
#' missed(my_plan, config = con)
#' max_useful_jobs(my_plan, config = con)
#' plot_graph(my_plan, config = con)
#' dataframes_graph(my_plan, config = con)
#' }
#' @param plan same as for \code{\link{make}}
#' @param targets same as for \code{\link{make}}
#' @param envir same as for \code{\link{make}}
#' @param verbose same as for \code{\link{make}}
#' @param hook same as for \code{\link{make}}
#' @param parallelism same as for \code{\link{make}}
#' @param jobs same as for \code{\link{make}}
#' @param packages same as for \code{\link{make}}
#' @param prework same as for \code{\link{make}}
#' @param prepend same as for \code{\link{make}}
#' @param command same as for \code{\link{make}}
#' @param args same as for \code{\link{make}}
#' @param recipe_command same as for \code{\link{make}}
#' @param cache same as for \code{\link{make}}
#' @param timeout same as for \code{\link{make}}
#' @param cpu same as for \code{\link{make}}
#' @param elapsed same as for \code{\link{make}}
#' @param retries same as for \code{\link{make}}
#' @param clear_progress logical, whether to clear
#' the cached progress of the targets readable by
#' @param graph igraph object representing the workflow plan network
#' \code{\link{progress}()}
config <- function(
  plan = workplan(),
  targets = drake::possible_targets(plan),
  envir = parent.frame(),
  verbose = TRUE,
  hook = function(code){
    force(code)
  },
  cache = drake::get_cache(verbose = verbose),
  parallelism = drake::default_parallelism(),
  jobs = 1,
  packages = rev(.packages()),
  prework = character(0),
  prepend = character(0),
  command = drake::default_Makefile_command(),
  args = drake::default_Makefile_args(
    jobs = jobs,
    verbose = verbose
  ),
  recipe_command = drake::default_recipe_command(),
  timeout = Inf,
  cpu = timeout,
  elapsed = timeout,
  retries = 0,
  clear_progress = FALSE,
  graph = NULL
){
  force(envir)
  seed <- get_valid_seed()
  plan <- sanitize_plan(plan)
  targets <- sanitize_targets(plan, targets)
  parallelism <- match.arg(
    parallelism,
    choices = parallelism_choices(distributed_only = FALSE)
  )
  prework <- add_packages_to_prework(
    packages = packages,
    prework = prework
  )
  if (is.null(cache)) {
    cache <- recover_cache()
  }
  # A storr_rds() cache should already have the right hash algorithms.
  cache <- configure_cache(
    cache = cache,
    clear_progress = clear_progress,
    overwrite_hash_algos = FALSE
  )
  if (is.null(graph)){
    graph <- build_graph(plan = plan, targets = targets,
      envir = envir, verbose = verbose, jobs = jobs)
  }
  list(
    plan = plan, targets = targets, envir = envir, cache = cache,
    parallelism = parallelism, jobs = jobs, verbose = verbose, hook = hook,
    prepend = prepend, prework = prework, command = command,
    args = args, recipe_command = recipe_command, graph = graph,
    short_hash_algo = cache$get("short_hash_algo", namespace = "config"),
    long_hash_algo = cache$get("long_hash_algo", namespace = "config"),
    inventory = cache$list(), seed = seed,
    inventory_filemtime = cache$list(namespace = "filemtime"),
    timeout = timeout, cpu = cpu, elapsed = elapsed, retries = retries
  )
}

add_packages_to_prework <- function(packages, prework) {
  packages <- c("methods", packages) %>% unique
  if (!length(packages))
    return(prework)
  paste0("if(!R.utils::isPackageLoaded(\"", packages, "\")) require(",
    packages, ")", sep = "") %>% c(prework)
}

#' @title Internal function do_prework
#' @export
#' @description Run the \code{prework} of a \code{\link{make}()}.
#' For internal use only.
#' the only reason this function is exported
#' is to set up PSOCK clusters efficiently.
#' @param config internal configuration list
#' @param verbose_packages logical, whether to print
#' package startup messages
do_prework <- function(config, verbose_packages) {
  wrapper <- ifelse(verbose_packages, invisible,
    base::suppressPackageStartupMessages)
  for (code in config$prework) wrapper(eval(parse(text = code),
    envir = config$envir))
}

inventory <- function(config) {
  config$inventory <- config$cache$list()
  config$inventory_filemtime <- config$cache$list(namespace = "filemtime")
  config
}

#' @title Function \code{possible_targets}
#' @description internal function, returns the list of
#' possible targets that you can select with the \code{targets}
#' argument to \code{\link{make}()}.
#' @seealso \code{\link{make}}
#' @export
#' @return character vector of possible targets
#' @param plan workflow plan data frame
#' @examples
#' \dontrun{
#' load_basic_example()
#' possible_targets(my_plan)
#' }
possible_targets <- function(plan = workplan()) {
  plan <- sanitize_plan(plan)
  c(as.character(plan$output), as.character(plan$target))
}

store_config <- function(config) {
  save_these <- setdiff(names(config), "envir")  # envir could get massive.
  lightly_parallelize(
    save_these,
    function(item){
      config$cache$set(
        key = item,
        value = config[[item]],
        namespace = "config"
      )
    },
    jobs = config$jobs
  )
}
