# NOTE: if some of these tests fail, remove the 
# '.drake/' caches from all parent and ancestor
# directories. I need them to be clear to fully
# test the behavior of the cache functions.

context("cache")

test_that("uncache_imported works", {
  debug_cleanup()
  debug_setup()
  p = example_plan("debug")
  make(p, verbose = F)
  x = get_cache()
  expect_true("f" %in% cached())
  expect_true("f" %in% x$list("depends"))
  uncache_imported(x)
  expect_false("f" %in% cached())
  expect_false("f" %in% x$list("depends"))
  debug_cleanup()
})

test_that("can find cache while in project root", {
  clean(destroy=TRUE)
  expect_error(loadd())
  p = example_plan()
  make(p, verbose = F)
  expect_equal(cached(), letters[1:6])
  expect_equal(readd(a), 14)
  y = "a"
  expect_equal(readd(y, character_only = T), 14)
  expect_false(any(letters[1:6] %in% ls()))
  loadd(c, d)
  expect_true(all(c("c", "d") %in% ls()))
  expect_equal(c(c, d), c(8, 2))
  rm(c)
  rm(d)
  expect_false(any(letters[1:6] %in% ls()))
  loadd()
  expect_true(all(letters[1:6] %in% ls()))
  expect_equal(a, readd(a))
  expect_equal(e, readd(e))
  rm(list = letters[1:6])
  expect_false(any(letters[1:6] %in% ls()))
  wd = getwd()
  expect_equal(wd, find_project())
  expect_equal(file.path(wd, cache_path), find_cache())
  expect_equal(built(), letters[1:6])
  expect_equal(imported(), character(0))
  expect_true(is.list(session()))
  expect_true(is.data.frame(status()))
  expect_error(a)
  expect_error(b)
  loadd("a")
  expect_true(is.numeric(a))
  loadd(list = c("a", "b"))
  expect_true(is.numeric(b))
  rm(list = intersect(ls(), letters[1:6]))
  expect_false(any(letters[1:6] %in% ls()))
  loadd("a", "b")
  expect_true(all(letters[1:2] %in% ls()))
  clean(destroy=TRUE)
})
 
test_that("can find cache in subdir", {
  debug_cleanup()
  debug_setup()
  p = example_plan("debug")
  make(p, verbose = F)
  bt = c("'d'", "'e'", letters[1:5], "final")
  imp = c("'input'", "f", "g", "global", "h", "i", "j", "y")
  ch = sort(c(bt, imp))
  expect_equal(imported(), imp)
  wd = getwd()
  for(mydir in c("d1", "d2")){
    dir.create(mydir)
    setwd(mydir)
  }
  expect_equal(cached(), character(0))
  expect_equal(built(), character(0))
  expect_equal(imported(), character(0))
  expect_equal(cached(search = T), ch)
  expect_equal(built(search = T), bt)
  expect_equal(imported(search = T), imp)
  expect_error(readd("a"))
  expect_true(is.numeric(readd("a", search = T)))
  expect_error(a)
  expect_error(loadd("a"))
  expect_silent(loadd("a", search = T))
  expect_true(is.numeric(a))
  expect_equal(find_cache(), file.path(wd, cache_path))
  expect_equal(find_project(), wd)
  expect_error(o <- session())
  expect_error(o <- status())
  expect_true(is.list(session(search = T)))
  expect_true(is.data.frame(status(search = T)))
  expect_error(loadd())
  expect_silent(loadd(search = T))
  setwd(wd)
  unlink("d1", recursive = T)
  debug_cleanup()
})

test_that("can find cache in other dir", {
  debug_cleanup()
  p = example_plan("debug")
  wd = getwd()
  dir.create("d1")
  setwd("d1")
  debug_setup()
  make(p, verbose = F)
  setwd(wd)
  deep = file.path("d1", "d2")
  dir.create(deep)
  bt = c("'d'", "'e'", letters[1:5], "final")
  imp = c("'input'", "f", "g", "global", "h", "i", "j", "y")
  ch = sort(c(bt, imp))
  expect_equal(cached(search = T), character(0)) # see comment at top of file
  expect_equal(built(search = T), character(0)) # see comment at top of file
  expect_equal(imported(search = T), character(0))
  expect_equal(cached(path = "d1", search = T), ch)
  expect_equal(built(path = "d1", search = T), bt)
  expect_equal(imported(path = "d1", search = T), imp)
  expect_equal(cached(path = deep, search = T), ch)
  expect_equal(built(path = deep, search = T), bt)
  expect_equal(imported(path = deep, search = T), imp)
  expect_equal(cached(path = deep, search = F), character(0))
  expect_equal(built(path = deep, search = F), character(0))
  expect_equal(imported(path = deep, search = F), character(0))
  expect_error(readd("a", search = T)) # see comment at top of file
  expect_true(is.numeric(readd("a", path = "d1", search = T)))
  expect_true(is.numeric(readd("a", path = deep, search = T)))
  expect_error(readd("a", path = deep, search = F))
  expect_error(a)
  expect_error(loadd("a", search = T)) # see comment at top of file
  expect_silent(loadd("a", path = "d1", search = T))
  expect_true(is.numeric(a))
  rm(a)
  expect_error(a)
  expect_error(loadd("a", path = deep, search = F))
  expect_silent(loadd("a", path = deep, search = T))
  expect_true(is.numeric(a))
  rm(a)
  expect_error(a)
  expect_silent(loadd("a", path = "d1", search = F))
  expect_true(is.numeric(a))
  rm(a)
  rm(f)
  rm(global)
  expect_error(loadd(imported_only = T, path = deep, search = F))
  expect_error(f(1))
  loadd(imported_only = T, path = deep, search = T)
  expect_true(is.numeric(f(1)))
  expect_error(a)
  expect_error(loadd("a", imported_only = T, path = deep, search = T))
  expect_error(a)
  loadd("a", path = deep, search = T)
  expect_true(is.numeric(a))
  rm(a)
  expect_equal(find_cache(path = "d1"), file.path("d1", cache_path))
  expect_equal(find_project(path = "d1"), "d1")
  expect_equal(find_cache(path = deep), file.path("d1", cache_path))
  expect_equal(find_project(path = deep), "d1")
  expect_false(identical(find_cache(), file.path("d1", cache_path)))
  expect_false(identical(find_project(), "d1"))
  expect_true(is.list(session(path = "d1", search = T)))
  expect_true(is.data.frame(status(path = "d1", search = T)))
  setwd(wd)
  expect_error(loadd(search = T)) # see comment at top of file
  expect_error(loadd(search = F))
  expect_silent(loadd(path = "d1", search = T))
  expect_silent(loadd(path = "d1", search = F))
  expect_silent(loadd(path = deep, search = T))
  expect_error(loadd(path = deep, search = F))
  unlink("d1", recursive = T)
  debug_cleanup()
})

test_that("readd & loadd load cached functions/objects correctly", {
  debug_cleanup()
  debug_setup()
  p = example_plan("debug")
  h = function(x){2*x + 5 + global}
  global = 1
  make(p, verbose = F)
  rm(list = ls())
  expect_equal(readd(global), 1)
  expect_true(is.function(readd(f)))
  expect_error(readd(f)(1))
  expect_equal(readd(a), c(1:5, 1:25))
  expect_false(any(c("global", "a", "f") %in% ls()))
  loadd()
  expect_equal(global, 1)
  expect_true(is.function(f))
  expect_true(is.numeric(older <- f(1)))
  global = 2
  expect_true(is.numeric(newer <- f(1)))
  expect_true(older != newer)
  expect_equal(a, c(1:5, 1:25))
  debug_cleanup()
})
